/*////////////////////////////////////////////////////////////////////////
Copyright (c) 1998-2000 Electrotechnical Laboratry (ETL), AIST, MITI
Copyright (c) 1998-2000 Yutaka Sato
Copyright (c) 2001-2002 National Institute of Advanced Industrial Science and Technology (AIST)

Permission to use, copy, modify, and distribute this material for any
purpose and without fee is hereby granted, provided that the above
copyright notice and this permission notice appear in all copies.
AIST MAKES NO REPRESENTATIONS ABOUT THE ACCURACY OR SUITABILITY OF THIS
MATERIAL FOR ANY PURPOSE.  IT IS PROVIDED "AS IS", WITHOUT ANY EXPRESS
OR IMPLIED WARRANTIES.
/////////////////////////////////////////////////////////////////////////
Content-Type:	program/C; charset=US-ASCII
Program:	sslway.c (SSL encoder/decoder with SSLeay/openSSL)
Author:		Yutaka Sato <ysato@etl.go.jp>
Description:

  Given environment:
    file descriptor 0 is a socket connected to a client
    file descriptor 1 is a socket connected to a server

  Commandline argument:
    -cert file -- certificate (possibly with private key) of this SSLway
                  to be shown to a peer
    -key file -- private key file (if not included in the -cert file)
    -pass arg -- the source of passphrase, pass:string or file:path

    -CAfile file -- the name of file contains a CA's certificate
    -CApath dir -- directory contains CA's certificate files each named with
                   `X509 -hash -noout < certificate.pem`

    -Vrfy -- peer's certificate must be shown and must be authorized
    -vrfy -- peer's certificate, if shown, must be authorized
    -Auth -- peer must show its certificate, but it can be unauthorized
    -auth -- just record the peer's certificate, if exists, into log
             -- equals to obsoleted "-client_auth".

    -vd  -- detailed logging
    -vu  -- logging with trace (former default)
    -vt  -- terse logging (current default)
    -vs  -- disalbe any logging

    -ht  through pass if the request is in bare HTTP protocol (GET,HEAD,POST)

    Following options can be omitted when the sslway is called from DeleGate
    with FCL, FSV or FMD since it will be detected automatically.

    -co  apply SSL for the connection to the server [default for FSV]
    -ac  aplly SSL for the accepted connection from the client [default for FCL]
    -ad  accept either SSL or through by auto-detection of SSL-ClientHello
    -ss  negotiate by AUTH SSL for FTP (implicit SSL for data-connection)
    -st  accept STARTTLS (protocol is auto-detect) and SSL tunneling
    -St  require STARTTLS first (protocol is auto-detect) and PBSZ+PROT for FTP
    -{ss|st|St}/protocol enable STARTTLS for the protocol {SMTP,POP,IMAP,FTP}
    -bugs

  Usage:
    delegated FSV=sslway
    delegated FCL=sslway ...

  How to make:
    - do make at .. or ../src directory
    - edit Makefile.go to let SSLEAY points to the directory of libssl.a
    - make -f Makefile.go sslway

History:
	980412	created
	980428	renamed from "sslrelay" to "sslway"
//////////////////////////////////////////////////////////////////////#*/
#include <stdio.h>
#include <string.h>
char **move_envarg();
extern char *getenv();
extern char *wordscanX();

#define LSILENT	-1
#define LERROR	0
#define LTRACE	1
#define LDEBUG	2
static int loglevel = LERROR;
#define ERROR	(loglevel < LERROR)?0:DOLOG
#define TRACE	(loglevel < LTRACE)?0:DOLOG
#define DEBUG	(loglevel < LDEBUG)?0:DOLOG

static FILE *stdctl;
static char *client_host;
static int PID;
static DOLOG(fmt,a,b,c,d,e,f,g,h,i)
	char *fmt,*a,*b,*c,*d,*e,*f,*g,*h,*i;
{	char xfmt[256],head[256];

	sprintf(head,"## SSLway[%d](%s)",PID,client_host);
	sprintf(xfmt,"%%s %s\n",fmt);
	syslog_ERROR(xfmt,head,a,b,c,d,e,f,g,h,i);
}

/*
#include "ssl.h"
*/
#define SSL_FILETYPE_PEM 1
#define SSL_VERIFY_NONE			0x00
#define SSL_VERIFY_PEER			0x01
#define SSL_VERIFY_FAIL_IF_NO_PEER_CERT	0x02
#define SSL_VERIFY_CLIENT_ONCE		0x04

typedef void SSL_CTX;
typedef void SSL_METHOD;
typedef void SSL;
typedef void X509;
#define BIO_NOCLOSE 0
X509 *PEM_read_bio_X509();
typedef void BIO;
BIO *BIO_new_fp();
X509 *SSL_get_peer_certificate();
SSL_METHOD *SSLv2_method();
SSL_METHOD *SSLv3_method();
SSL_METHOD *SSLv3_server_method();
SSL_METHOD *SSLv3_client_method();
SSL_METHOD *SSLv23_server_method();
SSL_METHOD *SSLv23_client_method();
SSL_CTX *SSL_CTX_new();
SSL *SSL_new();
char *X509_NAME_oneline();
void *X509_verify_cert_error_string();
void *X509_STORE_CTX_get_current_cert();
void *X509_get_subject_name();

static SSL_CTX *ssl_new(serv)
{	SSL_CTX *ctx;
	SSL_METHOD *meth;

	SSL_library_init();
	SSL_load_error_strings();
	if( serv )
		meth = SSLv23_server_method();
	else	meth = SSLv23_client_method();
	ctx = SSL_CTX_new(meth);
	return ctx;
}
static ssl_printf(ssl,fd,fmt,a,b,c,d,e,f,g)
	SSL *ssl;
	char *fmt,*a,*b,*c,*d,*e,*f,*g;
{	char buf[0x4000];

	sprintf(buf,fmt,a,b,c,d,e,f,g);
	if( ssl )
		SSL_write(ssl,buf,strlen(buf));
	else	write(fd,buf,strlen(buf));
}
static ssl_prcert(ssl,show,outssl,outfd,what)
	SSL *ssl;
	char *what;
{	X509 *peer;
	char subjb[256],*sb,issrb[256],*is;
	char *dp,ident[256];

	ident[0] = 0;
	if( peer = SSL_get_peer_certificate(ssl) ){
		sb = X509_NAME_oneline(X509_get_subject_name(peer),subjb,1024);
		is = X509_NAME_oneline(X509_get_issuer_name(peer),issrb,1024);
		if( show ){
			ssl_printf(outssl,outfd,
				"##[SSLway: %s's certificate]\r\n",what);
			ssl_printf(outssl,outfd,"## Subject: %s\r\n",sb);
			ssl_printf(outssl,outfd,"## Issuer: %s\r\n",is);
		}
		ERROR("%s's cert. = **subject<<%s>> **issuer<<%s>>",what,sb,is);
		if( dp = (char*)strcasestr(sb,"/email=") )
			wordscanY(dp+7,ident,sizeof(ident),"^/");
		else	strcpy(ident,"someone");
		X509_free(peer);
	}else{
		TRACE("%s's cert. = NONE",what);
		strcpy(ident,"anonymous");
		sb = "";
		is = "";
	}
	if( stdctl ){
		fprintf(stdctl,"CFI/1.0 200- Ident:%s\r\n",ident);
		fprintf(stdctl,"CFI/1.0 200 Certificate:%s//%s\r\n",sb,is);
		fflush(stdctl);
	}
}
static SSL *ssl_conn(ctx,confd)
	SSL_CTX *ctx;
{	SSL *conSSL;

	conSSL = SSL_new(ctx);
	SSL_set_connect_state(conSSL);
	SSL_set_fd(conSSL,SocketOf(confd));
	if( SSL_connect(conSSL) < 0 ){
		ERROR("connect failed");
		ERR_print_errors_fp(stderr);
		return NULL;
	}else{
		TRACE("connected");
		return conSSL;
	}
}
static SSL *ssl_acc(ctx,accfd)
	SSL_CTX *ctx;
{	SSL *accSSL;

	accSSL = SSL_new(ctx);
	SSL_set_accept_state(accSSL);
	SSL_set_fd(accSSL,SocketOf(accfd));
	if( SSL_accept(accSSL) < 0 ){
		ERROR("accept failed");
		ERR_print_errors_fp(stderr);
		ssl_printf(accSSL,0,"SSLway: accept failed\n");
		return NULL;
	}else{
		TRACE("accepted");
		return accSSL;
	}
}
static ssl_setCAs(ctx,file,dir)
	SSL_CTX *ctx;
	char *file,*dir;
{	char xfile[1024],xdir[1024];

	if( LIBFILE_IS(file,xfile) ) file = xfile;
	if( LIBFILE_IS(dir, xdir)  ) dir = xdir;

	if( !SSL_CTX_load_verify_locations(ctx,file,dir)
	 || !SSL_CTX_set_default_verify_paths(ctx) ){
		ERROR("CAs not found or wrong: [%s][%s]",file,dir);
	}else{
		TRACE("CAs = [%s][%s]",file,dir);
	}
}

typedef struct {
	char	*c_cert;	/* cetificate file */
	char	*c_key;		/* private key */
} CertKey1;
typedef struct {
	CertKey1 v_ck[8];
	int	 v_Ncert;
	int	 v_Nkey;
} CertKeyV;
typedef struct {
	CertKeyV x_certkey;
	char	*x_pass;	/* to decrypt the cert */
	char	*x_CApath;	/* CA's certificates */
	char	*x_CAfile;	/* A CA's certificate */
	int	 x_do_SSL;	/* use SSL */
	int	 x_do_STLS;	/* enable STARTTLS */
	int	 x_nego_FTPDATA;
	int	 x_verify;
} SSLContext;

static char sv_cert_default[] = "server-cert.pem";
static char sv_key_default[] = "server-key.pem";
static char sv_certkey_default[] = "server.pem";
static char cl_cert_default[] = "client-cert.pem";
static char cl_key_default[] = "client-key.pem";
static char cl_certkey_default[] = "client.pem";
static char *stls_proto;

static SSLContext sslctx[2] = {
	{ { {sv_cert_default, sv_key_default} } },
	{ { {cl_cert_default, cl_key_default} } },
};
static int   acc_bareHTTP = 0;
static int   verify_depth = -1;
static int   do_showCERT = 0;
static char *cipher_list = NULL;

#define XACC	0
#define XCON	1

#define sv_Cert		sslctx[XACC].x_certkey
#define sv_Ncert	sslctx[XACC].x_certkey.v_Ncert
#define sv_Nkey		sslctx[XACC].x_certkey.v_Nkey
#define sv_cert		sslctx[XACC].x_certkey.v_ck[sv_Ncert].c_cert
#define sv_key		sslctx[XACC].x_certkey.v_ck[sv_Nkey].c_key
#define sv_pass		sslctx[XACC].x_pass
#define cl_CApath	sslctx[XACC].x_CApath
#define cl_CAfile	sslctx[XACC].x_CAfile
#define do_accSSL	sslctx[XACC].x_do_SSL
#define do_accSTLS	sslctx[XACC].x_do_STLS
#define cl_vrfy		sslctx[XACC].x_verify
#define cl_nego_FTPDATA	sslctx[XACC].x_nego_FTPDATA

#define cl_Cert		sslctx[XCON].x_certkey
#define cl_Ncert	sslctx[XCON].x_certkey.v_Ncert
#define cl_Nkey		sslctx[XCON].x_certkey.v_Nkey
#define cl_cert		sslctx[XCON].x_certkey.v_ck[cl_Ncert].c_cert
#define cl_key		sslctx[XCON].x_certkey.v_ck[cl_Nkey].c_key
#define cl_pass		sslctx[XCON].x_pass
#define sv_CApath	sslctx[XCON].x_CApath
#define sv_CAfile	sslctx[XCON].x_CAfile
#define do_conSSL	sslctx[XCON].x_do_SSL
#define do_conSTLS	sslctx[XCON].x_do_STLS
#define sv_vrfy		sslctx[XCON].x_verify
#define sv_nego_FTPDATA	sslctx[XCON].x_nego_FTPDATA

#define ST_OPT		1
#define ST_FORCE	2
#define ST_AUTO		4 /* auto-detection of SSL by Client_Hello */
#define ST_SSL		8 /* AUTH SSL for FTP */

static passfilename(keyfile,passfile)
	char *keyfile,*passfile;
{	char *dp;

	strcpy(passfile,keyfile);
	dp = strrchr(passfile,'.');
	strcpy(dp,".pas");
}
static Freadline(path,line,size)
	char *path,*line;
{	FILE *fp;
	int rcc;
	char *dp;

	fp = fopen(path,"r");
	if( fp == NULL )
		return -1;

	if( 0 < (rcc = fread(line,1,size,fp)) )
		line[rcc] = 0;
	else	line[0] = 0;
	fclose(fp);
	if( dp = strpbrk(line,"\r\n") )
		*dp = 0;
	return strlen(line);
}
static scanpass(arg)
	char *arg;
{	char *file,path[1024],*pass,passb[128];

	if( strncmp(arg,"file:",5) == 0 ){
		file = arg+5;
		if( LIBFILE_IS(file,path) )
			file = path;
		passb[0] = 0;
		Freadline(file,passb,sizeof(passb));
		pass = passb;
	}else
	if( strncmp(arg,"pass:",5) == 0 ){
		pass = arg + 5;
	}else{
		ERROR("Usage: -pass { file:path | pass:string }");
		return;
	}
	if( pass[0] ){
		sv_pass = cl_pass = strdup(pass);
	}
}

static setcert1(ctx,certfile,keyfile,clnt)
	SSL_CTX *ctx;
	char *certfile;
	char *keyfile;
{	int code = 0;
	char cwd[1024];
	char xkeyfile[1024],xcertfile[1024];
	char *dp,passfile[1024],pass[128];

	if( LIBFILE_IS(keyfile,xkeyfile) )
		keyfile = xkeyfile;
	if( LIBFILE_IS(certfile,xcertfile) )
		certfile = xcertfile;

	if( dp = strrchr(keyfile,'.') ){
		passfilename(keyfile,passfile);
		if( 0 <= Freadline(passfile,pass,sizeof(pass)) ){
			if( clnt )
				cl_pass = strdup(pass);
			else	sv_pass = strdup(pass);
		}
	}

	getcwd(cwd,sizeof(cwd));
	if( SSL_CTX_use_certificate_file(ctx,certfile,SSL_FILETYPE_PEM) ){
		DEBUG("certfile loaded: %s",certfile);
	}else{
		ERROR("certfile not found or wrong: %s [at %s]",certfile,cwd);
		code = -1;
	}
	if( SSL_CTX_use_RSAPrivateKey_file(ctx,keyfile,SSL_FILETYPE_PEM) ){
		DEBUG("keyfile loaded: %s",keyfile);
	}else{
		ERROR("keyfile not found or wrong: %s [at %s]",keyfile,cwd);
		code = -1;
	}
	if( !SSL_CTX_check_private_key(ctx) ){
		ERROR("key does not match cert: %s %s",keyfile,certfile);
		code = -1;
	}
	return code;
}
static setcerts(ctx,certv,clnt)
	SSL_CTX *ctx;
	CertKeyV *certv;
{	int certx;
	int code;
	CertKey1 *cert1;

	for( certx = 0; certx <= certv->v_Ncert; certx++ ){
		cert1 = &certv->v_ck[certx];
		code = setcert1(ctx,cert1->c_cert,cert1->c_key,clnt);
		if( code != 0 )
			return code;
	}
	return 0;
}
extern void *RSA_generate_key();
static void *tmprsa_key;
static void *tmprsa_callback(ctx,exp,bits)
	void *ctx;
{
	if( bits != 512 && bits != 1024 ){
		bits = 512;
	}
	if( tmprsa_key == NULL ){
		tmprsa_key = RSA_generate_key(bits,0x10001,NULL,NULL);
	}
	return tmprsa_key;
}
static verify_callback(ok,ctx)
	void *ctx;
{	int err,depth;
	char *errsym;
	void *cert;
	char subjb[256];

	cert =   X509_STORE_CTX_get_current_cert(ctx);
	err =    X509_STORE_CTX_get_error(ctx);
	depth =  X509_STORE_CTX_get_error_depth(ctx);
	X509_NAME_oneline(X509_get_subject_name(cert),subjb,sizeof(subjb));
	errsym = X509_verify_cert_error_string(err);
	ERROR("depth=%d/%d ok=%d %d:\"%s\" %s",
		depth,verify_depth,ok,err,errsym,subjb);

	if( !ok ){
		if( depth <= verify_depth )
			ok = 1;
	}
	return ok;
}
static writes(what,ssl,confd,buf,rcc)
	char *what;
	SSL *ssl;
	char *buf;
{	int wcc,rem;

	rem = rcc;
	while( 0 < rem ){
		if( ssl )
			wcc = SSL_write(ssl,buf,rem);
		else	wcc = write(confd,buf,rem);
		if( wcc == rem )
			DEBUG("%s: %d/%d -> %d%s",what,rem,rcc,wcc,ssl?"/SSL":"");
		else	ERROR("%s? %d/%d -> %d%s",what,rem,rcc,wcc,ssl?"/SSL":"");
		if( wcc <= 0 )
			break;
		rem -= wcc;
	}
}
static ssl_relay(accSSL,accfd,conSSL,confd)
	SSL *accSSL,*conSSL;
{	int fdv[2],rfdv[2],nready,rcc,wcc;
	char buf[8*1024];

	fdv[0] = accfd;
	fdv[1] = confd;

	if( cl_nego_FTPDATA )
		nego_FTPDATAcl(conSSL,"",0);

	for(;;){
		nready = 0;
		rfdv[0] = rfdv[1] = 0;
		if( accSSL && SSL_pending(accSSL) ){
			rfdv[0] = 1;
			nready++;
		}
		if( conSSL && SSL_pending(conSSL) ){
			rfdv[1] = 1;
			nready++;
		}
		if( nready == 0 ){
			nready = PollIns(0,2,fdv,rfdv);
			if( nready <= 0 )
				break;
		}

		if( rfdv[0] ){
			if( accSSL )
				rcc = SSL_read(accSSL,buf,sizeof(buf));
			else	rcc = read(accfd,buf,sizeof(buf));
			if( rcc <= 0 )
				break;
			if( sv_nego_FTPDATA )
				rcc = nego_FTPDATAsv(accSSL,buf,rcc);
			writes("C-S",conSSL,confd,buf,rcc);
		}
		if( rfdv[1] ){
			if( conSSL )
				rcc = SSL_read(conSSL,buf,sizeof(buf));
			else	rcc = read(confd,buf,sizeof(buf));
			if( rcc <= 0 )
				break;
			if( cl_nego_FTPDATA )
				nego_FTPDATAcl(conSSL,buf,rcc);
			writes("S-C",accSSL,accfd,buf,rcc);
		}
	}
}

/*
 * STARTTLS:
 *   RFC2487 SMTP
 *   RFC2595 IMAP and POP3
 *   RFC2228,draft-murray-auth-ftp-ssl-07 FTP
 */
static char *relay_opening(proto,fs,tc,buf,bsize)
	char *proto;
	FILE *fs,*tc;
	char *buf;
{	char msgb[1024];

	for(;;){
		if( fgets(buf,bsize,fs) == NULL )
			return NULL;
		fputs(buf,tc);
		fflush(tc);

		if( proto != NULL )
			break;
		if( strncmp(buf,"220",3) == 0 ){
			if( buf[3] == '-' ){
				do {
					fgets(msgb,sizeof(msgb),fs);
					fputs(msgb,tc);
				} while( msgb[3] == '-' );
				fflush(tc);
			}
			if( strstr(buf,"FTP") )
				proto = "FTP";
			else
			proto = "SMTP";
			break;
		}else
		if( strncasecmp(buf,"+OK",3) == 0 ){
			proto = "POP3";
			break;
		}else
		if( strncasecmp(buf,"* OK",4) == 0 ){
			proto = "IMAP";
			break;
		}else{
			return NULL;
		}
	}
	return proto;
}
static isinSSL(fd)
{	unsigned char buf[6];
	int rcc,leng,type,vmaj,vmin;

	buf[0] = 0x7F;
	RecvPeek(fd,buf,1);
	if( (buf[0] & 0x80) || buf[0] < 0x20 ){
		ERROR("STARTTLS got binary [%X] from client",0xFF&buf[0]);
		if( buf[0] == 0x80 ){
			rcc = RecvPeek(fd,buf,5);
			ERROR("SSL Hello?%d [%X %d %d %d %d]",rcc,buf[0],
				buf[1],buf[2],buf[3],buf[4]);
			leng = (0x7F&buf[0]) << 8 | buf[1];
			type = buf[2];
			if( type == 1 ){ /* SSLv3 ClientHello */
				vmaj = buf[3];
				vmin = buf[4];
				return 1;
			}
		}
	}
	return 0;
}
static starttls(accfd,confd)
{	FILE *fc,*tc,*fs,*ts;
	int fdv[2],rfdv[2];
	char buf[1024],com[32],arg[32];
	char *msg,msgb[1024],*dp;
	char *proto;
	int xi;

	fdv[0] = accfd;
	fdv[1] = confd;
	fc = fdopen(fdv[0],"r"); setbuf(fc,NULL);
	tc = fdopen(fdv[0],"w");
	fs = fdopen(fdv[1],"r"); setbuf(fs,NULL);
	ts = fdopen(fdv[1],"w");

	proto = stls_proto;
	if( do_conSSL && do_conSTLS ){
		proto = relay_opening(proto,fs,tc,buf,sizeof(buf));
		if( proto == NULL )
			return -1;

		ERROR("STARTTLS to server -- %s",proto);
		if( strcasecmp(proto,"FTP") == 0 ){
			if( do_conSTLS & ST_SSL )
				fputs("AUTH SSL\r\n",ts);
			else{
				fputs("AUTH TLS\r\n",ts);
				if( do_conSTLS & ST_FORCE )
					cl_nego_FTPDATA = 1;
			}
		}else
		if( strcasecmp(proto,"SMTP") == 0 ){
			fputs("STARTTLS\r\n",ts);
		}else
		if( strncasecmp(proto,"POP",3) == 0 ){
			fputs("STLS\r\n",ts);
		}else
		if( strcasecmp(proto,"IMAP") == 0 ){
			fputs("stls0 STARTTLS\r\n",ts);
		}
		fflush(ts);
		if( fgets(buf,sizeof(buf),fs) == NULL )
			return -1;
		if( dp = strpbrk(buf,"\r\n") )
			*dp = 0;
		ERROR("STARTTLS to server -- %s",buf);
	}
	if( do_accSSL && do_accSTLS ){
	  for( xi = 0; ; xi++ ){
	    PollIns(0,2,fdv,rfdv);
	    if( rfdv[0] ){
		if( xi == 0 /* && accept implicit SSL too */ ){
			if( isinSSL(fdv[0]) )
				return 0;
			if( do_accSTLS == ST_AUTO ){
				ERROR("SSL-autodetect C-S: not in SSL");
				do_accSSL = 0;
				return 0;
			}
		}
		if( fgets(buf,sizeof(buf),fc) == NULL )
			return -1;
		dp = wordscanX(buf,com,sizeof(com));
		wordscanX(dp,arg,sizeof(arg));
		ERROR("STARTTLS prologue: C-S: [%s][%s]",com,arg);

		/* SMTP */
		if( strcasecmp(com,"EHLO") == 0 ){
			write(accfd,"250 STARTTLS\r\n",14);
			continue;
		}
		if( strcasecmp(com,"STARTTLS") == 0 ){
			msg = "220 Ready to start TLS\r\n";
			write(accfd,msg,strlen(msg));
			ERROR("STARTTLS from SMTP client -- OK");
			break;
		}

		/* POP3 */
		if( strcasecmp(com,"STLS") == 0 ){
			msg = "+OK Begin TLS negotiation\r\n";
			write(accfd,msg,strlen(msg));
			ERROR("STARTTLS from POP client -- OK");
			break;
		}

		/* IMAP */
		if( strcasecmp(arg,"CAPABILITY") == 0 ){
			msg = "* CAPABILITY STARTTLS\r\n";
			write(accfd,msg,strlen(msg));
			sprintf(msgb,"%s OK CAPABILITY\r\n",com);
			write(accfd,msgb,strlen(msgb));
			continue;
		}
		if( strcasecmp(arg,"STARTTLS") == 0 ){
			sprintf(msgb,"%s OK Begin TLS negotiation\r\n",com);
			write(accfd,msgb,strlen(msgb));
			ERROR("STARTTLS from IMAP client -- OK");
			break;
		}

		/* FTP */
		if( strcasecmp(com,"AUTH") == 0 )
		if( strcasecmp(arg,"TLS") == 0 || strcasecmp(arg,"SSL") == 0 ){
			msg = "234 OK\r\n";
			write(accfd,msg,strlen(msg));
			ERROR("AUTH %s from FTP client -- 234 OK",arg);
			if( strcasecmp(arg,"TLS") == 0 && do_accSTLS == ST_FORCE )
				sv_nego_FTPDATA = 1;
			break;
		}

		/* HTTP */
		if( strcasecmp(com,"CONNECT") == 0 ){
			if( proto == 0 ){
				proto = "http";
			}
		}

		if( do_accSTLS == 2 ){
			ERROR("STARTTLS required");
			if( proto != 0 && strcasecmp(proto,"IMAP") == 0 )
				fprintf(tc,"%s BAD do STARTTLS first.\r\n",com);
			else
			if( proto != 0 && strcasecmp(proto,"POP") == 0 )
				fprintf(tc,"+ERR do STLS first.\r\n");
			else
			fprintf(tc,"530 do STARTTLS first.\r\n");
			fflush(tc);
			return -1;
		}
		fputs(buf,ts);
		fflush(ts);
	    }
	    if( rfdv[1] ){
		if( xi == 0 ){
			if( isinSSL(fdv[1]) ) /* will not match */
				return 0;
			if( do_accSTLS == ST_AUTO ){
				ERROR("SSL-autodetect S-C: not in SSL");
				do_accSSL = 0;
				return 0;
			}
		}
		if( proto == NULL ){
			proto = relay_opening(proto,fs,tc,buf,sizeof(buf));
			if( proto == NULL )
				return -1;
			ERROR("STARTTLS to client -- %s",proto);
		}else{
		if( fgets(buf,sizeof(buf),fs) == NULL )
			return -1;
		fputs(buf,tc);
		}
		/* HTTP */
		if( proto != NULL && streq(proto,"http") ){
			if( buf[0] == '\r' || buf[1] == '\n' ){
				ERROR("STARTTLS prologue: S-C HTTP-CONNECT DONE");
				fflush(tc);
				break;
			}
		}
		if( dp = strpbrk(buf,"\r\n") )
			*dp = 0;
		ERROR("STARTTLS prologue: S-C: %s",buf);
		fflush(tc);
	    }
	  }
	}
	return 0;
}
static nego_FTPDATAsv(accSSL,buf,len)
	SSL *accSSL;
	char *buf;
{	char com[32],arg[32],*dp,*msg;

	buf[len] = 0;
	dp = wordscanX(buf,com,sizeof(com));
	wordscanX(dp,arg,sizeof(arg));
	if( strcasecmp(com,"PBSZ") == 0 ){
		msg = "200 OK\r\n";
		SSL_write(accSSL,msg,strlen(msg));
		ERROR("PBSZ %s from FTP client -- 200 OK",arg);
		len = 0;
	}
	else
	if( strcasecmp(com,"PROT") == 0 ){
		msg = "200 OK\r\n";
		SSL_write(accSSL,msg,strlen(msg));
		ERROR("PROT %s from FTP client -- 200 OK",arg);
		len = 0;
		sv_nego_FTPDATA = 0;
	}
	return len;
}
#define FTP_LOGIN_OK	"230"
static nego_FTPDATAcl(conSSL,sbuf,len)
	SSL *conSSL;
	char *sbuf;
{	char *msg,buf[64],resp[64];
	int rcc;

	if( len != 0 )
	if( strncmp(sbuf,FTP_LOGIN_OK,strlen(FTP_LOGIN_OK)) != 0 )
		return;

	msg = "PBSZ 0\r\n";
	SSL_write(conSSL,msg,strlen(msg));
	rcc = SSL_read(conSSL,buf,sizeof(buf));
	buf[rcc] = 0;
	linescanX(buf,resp,sizeof(resp));
	ERROR("STARTTLS/FTP PBSZ 0 -> %s",resp);
	if( atoi(resp) != 200 )
		return;

	msg = "PROT P\r\n";
	SSL_write(conSSL,msg,strlen(msg));
	rcc = SSL_read(conSSL,buf,sizeof(buf));
	buf[rcc] = 0;
	linescanX(buf,resp,sizeof(resp));
	ERROR("STARTTLS/FTP PROT P -> %s",resp);
	if( atoi(resp) == 200 )
		cl_nego_FTPDATA = 0;
}

static HTTP_CAresp(fd,certfile)
	char *certfile;
{	FILE *tc,*cfp;
	X509 *cert;
	BIO *in,*out;

	tc = fdopen(fd,"w");
	cfp = fopen(certfile,"r");
	if( cfp == NULL )
		return -1;

	fprintf(tc,"HTTP/1.0 200 ok\r\n");
	fprintf(tc,"MIME-Version: 1.0\r\n");
	fprintf(tc,"Content-Type: application/x-x509-ca-cert\r\n");
	fprintf(tc,"\r\n");

	in = BIO_new_fp(cfp,BIO_NOCLOSE);
	cert = PEM_read_bio_X509(in,NULL,NULL);
	out = BIO_new_fp(tc,BIO_NOCLOSE);
	i2d_X509_bio(out,cert);

	BIO_free(in);
	BIO_free(out);
	fclose(tc);
	return 0;
}
static CArequest(accfd,isHTTP,certfile)
	int *isHTTP;
	char *certfile;
{	char method[8],line[1024],url[1024];
	int rcc;

	setNonblockingIO(accfd,1);
	rcc = RecvPeek(accfd,method,6);
	setNonblockingIO(accfd,0);

	if( rcc <= 0 )
		return 0;

	method[rcc] = 0;

	if( strncmp(method,"GET ",4) == 0 ){
		setNonblockingIO(accfd,1);
		rcc = RecvPeek(accfd,line,16);
		setNonblockingIO(accfd,0);
		line[rcc] = 0;

		wordscanX(line+4,url,sizeof(url));
		if( strcmp(url,"/-/ca.der") == 0 ){
			HTTP_CAresp(0,certfile);
			TRACE("sent cert");
			return 1;
		}
		*isHTTP = 1;
	}else
	if( strncmp(method,"HEAD ",5) == 0 || strncmp(method,"POST ",5) == 0 )
		*isHTTP = 1;
	return 0;
}
static rand_seed()
{	int seed[8],si;

	seed[0] = Gettimeofday(&seed[1]);
	seed[2] = getpid();
	seed[3] = getuid();
	seed[4] = (int)seed;
	seed[5] = (int)rand_seed;
	RAND_seed(seed,sizeof(seed));
	for( si = 0; si < 8; si++ )
		seed[si] = 0;
}
static _passwd(what,pass,keyfile,buf,siz,vrfy)
	char *what,*pass,*keyfile;
	char *buf;
{	char passfile[1024];

	if( pass ){
		TRACE("passphrase for %s -- OK",keyfile);
		strcpy(buf,pass);
		return strlen(pass);
	}else{
		passfilename(keyfile,passfile);
		ERROR("passphrase for %s -- ERROR: '%s' file not found and SSL_%s_KEY_PASSWD undefined",
			keyfile,passfile,what);
		return -1;
	}
}
static sv_passwd(buf,siz,vrfy)
	char *buf;
{
	return _passwd("SERVER",sv_pass,sv_key,buf,siz,vrfy);
}
static cl_passwd(buf,siz,vrfy)
	char *buf;
{
	return _passwd("CLIENT",cl_pass,cl_key,buf,siz,vrfy);
}

static put_help()
{
	syslog_ERROR("SSLway 2002-12-23 <ysato@delegate.org>\r\n");
}
main(ac,av)
	char *av[];
{	int ai;
	char *arg;
	int accfd,confd;
	SSL_CTX *ctx;
	SSL *accSSL,*conSSL;
	char *env;
	int fdv[2],rfdv[2];
	int vflag;
	int ctrlopt = 0;
	int vflags = 0;
	int nodelay = 1;

	CFI_init(ac,av);
	av = move_envarg(ac,av,NULL,NULL,NULL);

	if( env = getenv("CFI_TYPE") ){
		if( strcmp(env,"FCL") == 0 ){
			DEBUG("CFI_TYPE=%s: -ac is assumed",env);
			do_accSSL = 1;
		}else
		if( strcmp(env,"FSV") == 0 || strcmp(env,"FMD") == 0 ){
			DEBUG("CFI_TYPE=%s: -co is assumed",env);
			do_conSSL = 1;
		}
	}
	if( env = getenv("CFI_STAT") ){
		int fd;
		fd = atoi(env);
		stdctl = fdopen(fd,"w");
		fprintf(stdctl,"CFI/1.0 100 start\r\n");
		fflush(stdctl);
	}
	if( env = getenv("SSL_KEY_PASSWD") )
		sv_pass = cl_pass = env;
	if( env = getenv("SSL_CLIENT_KEY_PASSWD") )
		cl_pass = env;
	if( env = getenv("SSL_SERVER_KEY_PASSWD") )
		sv_pass = env;

	PID = getpid();
	if( (client_host = getenv("REMOTE_HOST")) == 0 )
		client_host = "?";

	accfd = dup(0);
	confd = dup(1);

	if( env = getenv("SSL_CIPHER") ) cipher_list = env;

	if( env = getenv("SSL_CERT_FILE") )
		sv_cert = sv_key = cl_cert = cl_key = env;

	if( env = getenv("SSL_SERVER_KEY_FILE" ) ) sv_key  = env;
	if( env = getenv("SSL_SERVER_CERT_FILE") ) sv_cert = env;

	if( env = getenv("SSL_CLIENT_KEY_FILE" ) ) cl_key  = env;
	if( env = getenv("SSL_CLIENT_CERT_FILE") ) cl_cert = env;

	for( ai = 1; ai < ac; ai++ ){
		arg = av[ai];
		if( strcmp(arg,"-help") == 0 || strcmp(arg,"-v") == 0 ){
			put_help();
			exit(0);
		}else 
		if( strncmp(arg,"-vv",3) == 0 || strncmp(arg,"-vd",3) == 0 ){
			loglevel = LDEBUG;
		}else
		if( strncmp(arg,"-vu",3) == 0 ){
			loglevel = LTRACE;
		}else
		if( strncmp(arg,"-vt",3) == 0 ){
			loglevel = LERROR;
		}else
		if( strncmp(arg,"-vs",3) == 0 ){
			loglevel = LSILENT;
		}else
		if( strncasecmp(arg,"-ss",3) == 0 ){
			do_accSTLS = do_conSTLS = ST_SSL;
			if( arg[3] == '/' && arg[4] != 0 )
				stls_proto = strdup(arg+4);
		}else
		if( strncasecmp(arg,"-st",3) == 0 ){
			do_accSTLS = do_conSTLS = arg[1]=='s'?1:2;
			if( arg[3] == '/' && arg[4] != 0 )
				stls_proto = strdup(arg+4);
		}else
		if( strncasecmp(arg,"-ad",3) == 0 ){
			do_accSTLS = do_conSTLS = ST_AUTO;
		}else
		if( strncmp(arg,"-ac",3) == 0 ){
			do_accSSL = 1;
			if( strncmp(arg+3,"/st",3) == 0 ) do_accSTLS = 1;
		}else
		if( strncmp(arg,"-co",3) == 0 ){
			do_conSSL = 1;
			if( strncmp(arg+3,"/st",3) == 0 ) do_conSTLS = 1;
		}else
		if( strncmp(arg,"-ht",3) == 0 ){
			acc_bareHTTP = 1;
		}else
		if( strncmp(arg,"-show",3) == 0 ){
			do_showCERT = 1;
		}else
		if( strcmp(arg,"-CApath") == 0 ){
			if( ac <= ai + 1 ){
				ERROR("Usage: %s directory-name",arg);
				return -1;
			}
			cl_CApath = sv_CApath = av[++ai];
		}else
		if( strcmp(arg,"-CAfile") == 0 ){
			if( ac <= ai + 1 ){
				ERROR("Usage: %s file-name",arg);
				return -1;
			}
			cl_CAfile = sv_CAfile = av[++ai];
		}else
		if( strcasecmp(arg,"-vrfy")==0 || strcasecmp(arg,"-auth")==0 ){
			vflag = SSL_VERIFY_PEER
				| SSL_VERIFY_CLIENT_ONCE;
			if( arg[1] == 'V' || arg[1] == 'A' )
				vflag |= SSL_VERIFY_FAIL_IF_NO_PEER_CERT;
			if( arg[1] == 'V' || arg[1] == 'v' )
				verify_depth = -1;
			else	verify_depth = 10;
			cl_vrfy = vflag;
			sv_vrfy = vflag;
		}else
		if( strcasecmp(arg,"-verify") == 0 ){
			if( ac <= ai + 1 ){
				ERROR("Usage: %s max-depth",arg);
				return -1;
			}
			verify_depth = atoi(av[++ai]);
			vflag = SSL_VERIFY_PEER | SSL_VERIFY_CLIENT_ONCE;
			if( arg[1] == 'V' )
				vflag |= SSL_VERIFY_FAIL_IF_NO_PEER_CERT;
			cl_vrfy = vflag;
			sv_vrfy = vflag;
		}else
		if( strcmp(arg,"-client_auth") == 0 ){
			verify_depth = 10;
			cl_vrfy = SSL_VERIFY_PEER
				| SSL_VERIFY_CLIENT_ONCE;
		}else
		if( strcmp(arg,"-cipher") == 0 ){
			if( ac <= ai + 1 ){
				ERROR("Usage: %s cipher-list",arg);
				return -1;
			}
			cipher_list = av[++ai];
		}else
		if( strcmp(arg,"-certkey") == 0
		 || strcmp(arg,"-cert") == 0 ){
			if( ac <= ai + 1 ){
				ERROR("Usage: %s cert-key-file-name",arg);
				return -1;
			}
			if( sv_cert != sv_cert_default ){
				sv_Ncert++;
				sv_Nkey++;
			}
			if( cl_cert != cl_cert_default ){
				cl_Ncert++;
				cl_Nkey++;
			}
			sv_cert = sv_key = cl_cert = cl_key = av[++ai];
		}
		else
		if( strcmp(arg,"-key") == 0 ){
			if( ac <= ai + 1 ){
				ERROR("Usage: %s key-file-name",arg);
				return -1;
			}
			sv_key = cl_key = av[++ai];
		}
		else
		if( strcmp(arg,"-pass") == 0 ){
			if( ac <= ai + 1 ){
				ERROR("Usage: %s {pass:str|file:path}");
				return -1;
			}
			scanpass(av[++ai]);
		}
		else
		if( strcmp(arg,"-bugs") == 0 ){
			ctrlopt = 0x000FFFFFL; /* SSL_OP_ALL */
		}
		else
		if( strcmp(arg,"-delay") == 0 ){
			nodelay = 0;
		}
	}

	accSSL = NULL;
	conSSL = NULL;

	if( do_conSSL || do_accSSL )
	{
		rand_seed();
		TRACE("start");
	}

	if( nodelay ){
		set_nodelay(accfd,1);
		set_nodelay(confd,1);
	}

	fdv[0] = accfd;
	fdv[1] = confd;

	if( acc_bareHTTP ){
		int isHTTP;
		if( 0 < PollIns(100,2,fdv,rfdv) && 0<rfdv[0] && rfdv[1] <= 0 ){
			isHTTP = 0;
			if( CArequest(accfd,&isHTTP,sv_cert) )
				return 0;
			if( isHTTP ){
				 /* ... through raw HTTP request ...
				do_accSSL = 0;
				 */
			}
		}
	}

	if( do_conSSL && do_conSTLS || do_accSSL && do_accSTLS ){
		if( starttls(accfd,confd) < 0 )
			return -1;
	}

	if( do_conSSL ){
		ctx = ssl_new(0);
		SSL_CTX_set_default_passwd_cb(ctx,cl_passwd);
		if( cipher_list )
			SSL_CTX_set_cipher_list(ctx,cipher_list);
		if( cl_cert != cl_cert_default || LIBFILE_IS(cl_cert,NULL) )
			setcerts(ctx,&cl_Cert,1);

		if( sv_CAfile || sv_CApath )
			ssl_setCAs(ctx,sv_CAfile,sv_CApath);
		if( sv_vrfy )
			SSL_CTX_set_verify(ctx,sv_vrfy,verify_callback);

		conSSL = ssl_conn(ctx,confd);
		if( conSSL == NULL )
			return -1;
	}

	if( do_accSSL ){
		ctx = ssl_new(1);
		if( ctrlopt )
			SSL_CTX_ctrl(ctx,32/*SSL_CTRL_OPTIONS*/,ctrlopt,NULL);
		SSL_CTX_set_default_passwd_cb(ctx,sv_passwd);
		if( cipher_list )
			SSL_CTX_set_cipher_list(ctx,cipher_list);
		if( setcerts(ctx,&sv_Cert,0) < 0 )
			return -1;
		SSL_CTX_set_tmp_rsa_callback(ctx,tmprsa_callback);

		if( cl_CAfile || cl_CApath )
			ssl_setCAs(ctx,cl_CAfile,cl_CApath);
		if( cl_vrfy )
			SSL_CTX_set_verify(ctx,cl_vrfy,verify_callback);

/*
{
int store;
store = SSL_CTX_get_cert_store(ctx);
X509_STORE_set_flags(store, vflags);
}
*/

		accSSL = ssl_acc(ctx,accfd);
		if( accSSL == NULL )
			return -1;
	}

	if( conSSL )
		ssl_prcert(conSSL,do_showCERT,NULL,  accfd,"server");
	if( accSSL )
		ssl_prcert(accSSL,do_showCERT,accSSL,accfd,"client");

	ssl_relay(accSSL,accfd,conSSL,confd);

	if( do_conSSL || do_accSSL )
		TRACE("done");
	return 0;
}
