# -*- coding: EUC-JP -*-
#
#  hanayu.py - a HANAYU compatible Saori module for ninix
#  Copyright (C) 2002, 2003 by Shyouzou Sugitani <shy@debian.or.jp>
#  Copyright (C) 2002, 2003 by MATSUMURA Namihiko <nie@counterghost.net>
#
#  This program is free software; you can redistribute it and/or modify it
#  under the terms of the GNU General Public License (version 2) as
#  published by the Free Software Foundation.  It is distributed in the
#  hope that it will be useful, but WITHOUT ANY WARRANTY; without even the
#  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
#  PURPOSE.  See the GNU General Public License for more details.
#
#  $Id: hanayu.py,v 1.16 2003/12/22 02:39:29 shy Exp $
#

# TODO:
# - usetime, line ʳηΥդؤб.
# - ȥνĽ.(GTK+бԤ.)

import sys
import os
import string
import StringIO
import math
import time
import codecs

if os.environ.has_key('DISPLAY'):
    if not sys.modules.has_key('gtk'):
        try:
            import pygtk
            pygtk.require("2.0")
        except ImportError:
            pass
    import gtk
    import pango
else:
    gtk = None

import ninix.pix

import locale

class Saori:
    __DBNAME = 'HANAYU.db'
    def __init__(self):
        self.loaded = 0
        self.graphs = {}
        self.data = {}
    def load(self, dir=os.curdir):
        self.dir = dir
        self.dbpath = os.path.join(self.dir, self.__DBNAME)
        result = 0
        if not gtk:
            pass
        elif self.loaded:
            result = 2
        else:
            self.graphs = {}
            self.data = self.read_hanayu_txt(self.dir)
            if self.data:
                self.loaded = 1
                self.read_db()
                result = 1
        return result
    def read_hanayu_txt(self, dir):
        graphs = {}
        try:
            file = open(os.path.join(dir, 'hanayu.txt'), 'r')
            data = {}
            name = ''
            while 1:
                line = file.readline()
                if not line:
                    if data:
                        graphs[name] = data
                    break # EOF
                line = string.strip(line)
                if not line:
                    continue
                if line[:2] == '//':
                    continue
                comma = string.find(line, ',')
                if line[0] == '[': # bln.txt like format
                    graphs[name] = data
                    data = {}
                    end = string.find(line, ']')
                    if end < 0:
                        end = len(line)
                    name = line[1:end]
                elif line == '{': # surfaces.txt like format
                    graphs[name] = data
                    data = {}
                    name = tmp_name
                elif line == '}': # surfaces.txt like format
                    graphs[name] = data
                    data = {}
                elif comma >= 0:
                    key = string.strip(line[:comma])
                    value = string.strip(line[comma+1:])
                    data[key] = value
                elif not name:
                    tmp_name = line
            return graphs
        except:
            return None
    def unload(self):
        if self.loaded == 0:
            return 0
        for name in self.graphs.keys():
            self.graphs[name].destroy()
        self.graphs = {}
        self.data = {}
        self.loaded = 0
        self.write_db()
        return 1
    def time_to_key(self, secs, offset):
        year = int(time.strftime("%Y", time.localtime(secs)))
        month = int(time.strftime("%m", time.localtime(secs)))
        day = int(time.strftime("%d", time.localtime(secs)))
        target_time = time.mktime((year, month, day+offset, 0, 0, 0, -1, -1, -1))
        year = int(time.strftime("%Y", time.localtime(target_time)))
        month = int(time.strftime("%m", time.localtime(target_time)))
        day = int(time.strftime("%d", time.localtime(target_time)))
        key = str(year*10000 + month*100 + day)
        return key, year, month, day
    def read_db(self):
        self.seven_days = []
        current_time = self.last_update = time.time()
        for index in range(-6, 1):
            key, year, month, day = self.time_to_key(current_time, index)
            self.seven_days.append([key, year, month, day, 0.0])
        try:
            file = open(self.dbpath)
        except:
            return
        else:
            line = file.readline()
            line = string.strip(line)
            if not line or line != '# Format: v1.0':
                return
            while 1:
                line = file.readline()
                if not line: # EOF
                    break
                line = string.strip(line)
                if not line:
                    continue
                comma = string.find(line, ',')
                if comma < 0:
                    continue
                key = line[:comma]
                for index in range(7):
                    if self.seven_days[index][0] == key:
                        self.seven_days[index][4] = float(line[comma+1:])
            file.close()
    def update_db(self):
        current_time = time.time()
        old_seven_days = []
        old_seven_days.extend(self.seven_days)
        self.seven_days = []
        for index in range(-6, 1):
            key, year, month, day = self.time_to_key(current_time, index)
            self.seven_days.append([key, year, month, day, 0.0])
        for i in range(7):
            key = old_seven_days[i][0]
            for j in range(7):
                if self.seven_days[j][0] == key:
                    self.seven_days[j][4] = old_seven_days[i][4]
                    if j == 6:
                        self.seven_days[j][4] = self.seven_days[j][4] + \
                                                (current_time - self.last_update) / (60.0 * 60.0)
        self.last_update = current_time
    def write_db(self):
        self.update_db()
        try:
            file = open(self.dbpath, 'w')
        except IOError:
            print 'HANAYU: cannot write database (ignored)'
        else:
            file.write('# Format: v1.0\n')
            for index in range(7):
                file.write('%s, %s\n' % (self.seven_days[index][0], self.seven_days[index][4]))
            file.close()
    def request(self, req):
        type, argument = self.evaluate_request(req)
        header = StringIO.StringIO(req)
        line = header.readline()
        if not type:
            return 'SAORI/1.0 400 Bad Request\r\n\r\n'
        elif type == 'GET Version':
            return 'SAORI/1.0 204 No Content\r\n\r\n'
        elif type == 'EXECUTE':
            if len(argument) == 0:
                return 'SAORI/1.0 400 Bad Request\r\n\r\n'
            command = argument[0]
            if command == 'show':
                if len(argument) >= 2:
                    name = argument[1]
                    if self.graphs.has_key(name):
                        self.graphs[name].destroy()
                else:
                    name = ''
                if not self.data.has_key(name):
                    return 'SAORI/1.0 400 Bad Request\r\n\r\n'
                if self.data[name].has_key('graph') and \
                       self.data[name]['graph'] in ['line', 'bar', 'radar', 'radar2']:
                    type = self.data[name]['graph']
                else:
                    type = 'usetime'
                if type == 'usetime':
                    self.update_db()
                    new_args = []
                    for index in range(7):
                        date = str(self.seven_days[index][2]) + \
                               '/' + \
                               str(self.seven_days[index][3])
                        new_args.append(date)
                        hours = self.seven_days[index][4]
                        new_args.append(hours)
                    self.graphs[name] = Line(self.dir, self.data[name], new_args, 0, 24)
                elif type == 'line':
                    self.graphs[name] = Line(self.dir, self.data[name], argument[2:])
                elif type == 'bar':
                    self.graphs[name] = Bar(self.dir, self.data[name], argument[2:])
                elif type == 'radar':
                    self.graphs[name] = Radar(self.dir, self.data[name], argument[2:])
                elif type == 'radar2':
                    self.graphs[name] = Radar2(self.dir, self.data[name], argument[2:])
            elif command == 'hide':
                if len(argument) >= 2:
                    name = argument[1]
                else:
                    name = ''
                if self.graphs.has_key(name):
                    self.graphs[name].destroy()
                else:
                    return 'SAORI/1.0 400 Bad Request\r\n\r\n'
            else:
                return 'SAORI/1.0 400 Bad Request\r\n\r\n'
            return 'SAORI/1.0 204 No Content\r\n\r\n'
    def evaluate_request(self, req):
        type = None
        argument = []
        charset = 'Shift_JIS' # default
        header = StringIO.StringIO(req)
        line = header.readline()
        if not line:
            return type, argument
        if line[-1] == '\n':
            line = line[:-1]
        line = string.strip(line)
        if not line:
            return type, argument
        for request in ['EXECUTE', 'GET Version']:
            if line[:len(request)] == request:
                type = request
                break
        while 1:
            line = header.readline()
            if not line:
                break # EOF
            if line[-1] == '\n':
                line = line[:-1]
            line = string.strip(line)
            if not line:
                continue
            colon = string.find(line, ':')
            if colon >= 0:
                key = string.strip(line[:colon])
                value = string.strip(line[colon+1:])
                if key == 'Charset':
                    charset = value
                    try:
                        codecs.lookup(charset)
                    except:
                        sys.stderr.write("Unsupported charset %s" % repr(charset))
                if key[:8] == 'Argument': ## FIXME
                    argument.append(value)
                else:
                    continue
        for i in range(len(argument)):
            argument[i] = unicode(argument[i], charset, 'ignore')
        return type, argument

class Graph:
    width = 450
    height = 340
    def __init__(self, dir, data, args=[], min=None, max=None):
        self.dir = dir
        self.data = data
        self.args = args
        self.min = min
        self.max = max
        self.create_window()
        self.draw_title()
        self.draw_frame()
        self.draw_graph()
        self.window.show()
    def create_window(self):
        self.window = gtk.Window()
        self.window.set_title(unicode('ͮ', 'EUC-JP').encode(locale.nl_langinfo(locale.CODESET)))
        self.window.set_decorated(gtk.FALSE)
        self.window.set_resizable(gtk.FALSE)
        self.window.connect("delete_event", self.delete)
        self.window.connect("button_press_event", self.button_press)
        self.window.set_events(gtk.gdk.BUTTON_PRESS_MASK)
        self.x = gtk.gdk.screen_width()/2
        self.y = gtk.gdk.screen_height()/4
        self.window.move(self.x, self.y)
        self.darea = gtk.DrawingArea()
        self.darea.set_events(gtk.gdk.EXPOSURE_MASK)
        self.darea.connect("expose_event", self.redraw)
        self.darea.set_size_request(self.width, self.height)
        self.darea.show()
        self.window.add(self.darea)
        self.darea.realize()
        self.layout = pango.Layout(self.darea.get_pango_context())
        pixbuf = None
        if self.data.has_key('background.filename'):
            path = os.path.join(self.dir,
                                string.replace(self.data['background.filename'],
                                               '\\', '/'))
            
            pixbuf = ninix.pix.create_pixbuf_from_file(path, 'png')
        self.surface_pixmap = gtk.gdk.Pixmap(None, self.width, self.height, gtk.gdk.visual_get_best_depth())
        cmap = self.darea.get_colormap()
        self.surface_gc = self.surface_pixmap.new_gc()
        self.surface_gc.foreground = cmap.alloc_color("#FFFFFF")
        self.surface_pixmap.draw_rectangle(self.surface_gc, gtk.TRUE, 0, 0,
                                           self.width, self.height)
        if pixbuf:
            width = pixbuf.get_width()
            height = pixbuf.get_height()
            xoffset = (self.width - width)/2
            yoffset = (self.height - height)/2
            pixbuf.render_to_drawable(self.surface_pixmap, self.surface_gc,
                                      0, 0, xoffset, yoffset, width, height,
                                      gtk.gdk.RGB_DITHER_NONE, 0, 0)
        self.darea.window.draw_drawable(self.surface_gc, self.surface_pixmap,
                                        0, 0, 0, 0, self.width, self.height)
    def draw_title(self):
        fontcolor_r = fontcolor_g = fontcolor_b = 0
        if self.data.has_key('font.color'):
            fontcolor_r = int(self.data['font.color'][:2], 16)
            fontcolor_g = int(self.data['font.color'][2:4], 16)
            fontcolor_b = int(self.data['font.color'][4:6], 16)
        else:
            if self.data.has_key('font.color.r'):
                fontcolor_r = int(self.data['font.color.r'])
            if self.data.has_key('font.color.g'):
                fontcolor_g = int(self.data['font.color.g'])
            if self.data.has_key('font.color.b'):
                fontcolor_b = int(self.data['font.color.b'])
        cmap = self.darea.get_colormap()
        self.surface_gc.foreground = cmap.alloc_color(
            "#%02x%02x%02x" % (fontcolor_r, fontcolor_g, fontcolor_b))
        if self.data.has_key('title'):
            self.title = unicode(self.data['title'], 'Shift_JIS', 'ignore')
        self.font_size = 12
        self.font_desc = pango.FontDescription()
        self.font_desc.set_family('Sans') # FIXME
        self.font_desc.set_size(self.font_size * 1024) # size * PANGO_SCALE
        text = ""
        i = 0
        while i < len(self.title):
            if text:
                text = text + '\n'
            text = text + self.title[i]
            i = i + 1
        self.layout.set_font_description(self.font_desc)
        self.layout.set_wrap(pango.WRAP_CHAR)
        self.layout.set_alignment(pango.ALIGN_CENTER)
        try:
            self.layout.set_text(text)
        except:
            self.layout.set_text(text, -1) # pygtk <= 1.99.14
        self.surface_pixmap.draw_layout(self.surface_gc, 20, 58, self.layout)
    def draw_frame(self):
        pass
    def draw_graph(self):
        pass
    def redraw(self, widget, event):
        x, y, w, h = event.area
        if self.surface_pixmap:
            self.darea.window.draw_drawable(self.surface_gc, self.surface_pixmap,
                                            x, y, x, y, w, h)
    def button_press(self, window, event):
        if event.type == gtk.gdk.BUTTON_PRESS:
            self.window.begin_move_drag(event.button, int(event.x_root), int(event.y_root), gtk.get_current_event_time())
        elif event.type == gtk.gdk._2BUTTON_PRESS: # double click
            self.destroy()
        return gtk.TRUE
    def delete(self, window, event):
        return gtk.TRUE
    def destroy(self):
        if self.window:
            self.window.destroy()
            self.window = None
            self.timeout_id = None

class Line(Graph):
    def draw_frame(self):
        framecolor_r = framecolor_g = framecolor = 0
        if self.data.has_key('frame.color'):
            framecolor_r = int(self.data['frame.color'][:2], 16)
            framecolor_g = int(self.data['frame.color'][2:4], 16)
            framecolor_b = int(self.data['frame.color'][4:6], 16)
        else:
            if self.data.has_key('frame.color.r'):
                framecolor_r = int(self.data['frame.color.r'])
            if self.data.has_key('frame.color.g'):
                framecolor_g = int(self.data['frame.color.g'])
            if self.data.has_key('frame.color.b'):
                framecolor_b = int(self.data['frame.color.b'])
        cmap = self.darea.get_colormap()
        self.surface_gc.foreground = cmap.alloc_color(
            "#%02x%02x%02x" % (framecolor_r, framecolor_g, framecolor_b))
        frame_width = 2
        if self.data.has_key('frame.width'):
            frame_width = int(self.data['frame.width'])
        self.surface_gc.line_width = frame_width
        self.surface_pixmap.draw_line(self.surface_gc,
                                      60, 48, 60, 260)
        self.surface_pixmap.draw_line(self.surface_gc,
                                      60, 260, 420, 260)
    def draw_graph(self):
        num = len(self.args)/2
        step = 368 / num 
        for index in range(num):
            try:
                self.layout.set_text(self.args[index*2])
            except:
                self.layout.set_text(self.args[index*2], -1) # pygtk <= 1.99.14
            w, h = self.layout.get_pixel_size()
            pos_x = 60 + index*step + step/2 - w/2
            pos_y = 268
            self.surface_pixmap.draw_layout(self.surface_gc, pos_x, pos_y, self.layout)
        if self.min != None:
            min = self.min
        else:
            min = self.args[1]
            for index in range(2, num):
                if self.args[index*2] < min:
                    min = self.args[index*2]
        if self.max != None:
            max = self.max
        else:
            max = self.args[1]
            for index in range(2, num):
                if self.args[index*2] > max:
                    max = self.args[index*2]
        linecolor_r = linecolor_g = linecolor = 0
        if self.data.has_key('line.color'):
            linecolor_r = int(self.data['line.color'][:2], 16)
            linecolor_g = int(self.data['line.color'][2:4], 16)
            linecolor_b = int(self.data['line.color'][4:6], 16)
        else:
            if self.data.has_key('line.color.r'):
                linecolor_r = int(self.data['line.color.r'])
            if self.data.has_key('line.color.g'):
                linecolor_g = int(self.data['line.color.g'])
            if self.data.has_key('line.color.b'):
                linecolor_b = int(self.data['line.color.b'])
        cmap = self.darea.get_colormap()
        self.surface_gc.foreground = cmap.alloc_color(
            "#%02x%02x%02x" % (linecolor_r, linecolor_g, linecolor_b))
        line_width = 2
        if self.data.has_key('line.width'):
            line_width = int(self.data['line.width'])
        self.surface_gc.line_width = line_width
        for index in range(1, num):
            src_x = 60 + (index-1)*step + step/2
            src_y = 220 - int(168*self.args[(index-1)*2+1]/(max-min))
            dst_x = 60 + index*step + step/2
            dst_y = 220 - int(168*self.args[index*2+1]/(max-min))
            self.surface_pixmap.draw_line(self.surface_gc,
                                          src_x, src_y, dst_x, dst_y)
        for index in range(num):
            image, mask = None, None
            if self.args[index*2+1] == min and self.data.has_key('mark0.filename'):
                path = os.path.join(self.dir,
                                    string.replace(self.data['mark0.filename'],
                                                   '\\', '/'))
                if os.path.exists(path):
                    image, mask = ninix.pix.create_pixmap_from_file(path)
            elif self.args[index*2+1] == max and self.data.has_key('mark2.filename'):
                path = os.path.join(self.dir,
                                    string.replace(self.data['mark2.filename'],
                                                   '\\', '/'))
                if os.path.exists(path):
                    image, mask = ninix.pix.create_pixmap_from_file(path)
            elif self.data.has_key('mark1.filename'):
                path = os.path.join(self.dir,
                                    string.replace(self.data['mark1.filename'],
                                                   '\\', '/'))
                if os.path.exists(path):
                    image, mask = ninix.pix.create_pixmap_from_file(path)
            if image:
                w, h = image.get_size()
                x = 60 + index*step + step/2 - w/2
                y = 220 - int(168*self.args[index*2+1]/(max-min)) - h/2
                gc = self.surface_pixmap.new_gc()
                gc.set_clip_mask(mask)
                gc.set_clip_origin(x, y)
                self.surface_pixmap.draw_drawable(gc, image, 0, 0, x, y, w, h)

class Bar(Graph):
    def draw_graph(self):
        barcolor_r = barcolor_g = barcolor_b = 0
        if self.data.has_key('bar.color'):
            barcolor_r = int(self.data['bar.color'][:2], 16)
            barcolor_g = int(self.data['bar.color'][2:4], 16)
            barcolor_b = int(self.data['bar.color'][4:6], 16)
        ##else:
        ##    if self.data.has_key('bar.color.r'):
        ##        barcolor_r = int(self.data['bar.color.r'])
        ##    if self.data.has_key('bar.color.g'):
        ##        barcolor_g = int(self.data['bar.color.g'])
        ##    if self.data.has_key('bar.color.b'):
        ##        barcolor_b = int(self.data['bar.color.b'])
        cmap = self.darea.get_colormap()
        self.surface_gc.foreground = cmap.alloc_color(
            "#%02x%02x%02x" % (barcolor_r, barcolor_g, barcolor_b))
        if self.data.has_key('bar.width'):
            pass # FIXME
        ### NOT YET ###

class Radar(Graph):
    width = 288
    height = 288
    def __init__(self, dir, data, args=[]):
        Graph.__init__(self, dir, data, args)
    def draw_graph(self):
        ##print 'RADAR:', self.args
        pass

class Radar2(Graph):
    width = 288
    height = 288
    def __init__(self, dir, data, args=[]):
        Graph.__init__(self, dir, data, args)

if __name__ == "__main__":
    USAGE= "Usage: hanayu.py <dir>"
    if len(sys.argv) == 2:
        saori = Saori()
        data = saori.read_hanayu_txt(sys.argv[1])
        print data
    else:
        print USAGE
