/*////////////////////////////////////////////////////////////////////////
Copyright (c) 1994-1999 Yutaka Sato
Copyright (c) 1994-1999 Electrotechnical Laboratry (ETL), AIST, MITI

Permission to use, copy, and distribute this material for any purpose
and without fee is hereby granted, provided that the above copyright
notice and this permission notice appear in all copies.
ETL MAKES NO REPRESENTATIONS ABOUT THE ACCURACY OR SUITABILITY OF THIS
MATERIAL FOR ANY PURPOSE.  IT IS PROVIDED "AS IS", WITHOUT ANY EXPRESS
OR IMPLIED WARRANTIES.
/////////////////////////////////////////////////////////////////////////
Content-Type:	program/C; charset=US-ASCII
Program:	hostlist.c
Author:		Yutaka Sato <ysato@etl.go.jp>
Description:

    Each element in HostList is one of follows:

	IP-address matching with mask:
		host/255.255.255.0
		host/@C

	IP-address range:
		192.31.200.[65-94]
		192.31.[197-206]
		150.29.[0-255]

	Domain name pattern macthing by name:
		*.ac.jp
		*.jp

    A special host-name "?" matches with hosts which name cannot be
    retrieved with DNS, thus unknown.

    Each element can be prefixed with UserList.
    A special user-name "?" matches when the user cannot be
    identified with IDENTd.

    Special elements which add a flavor of composit operation :-)

	AND: A,&,B -- if A is false, false as a whole (can be "&{A,B}" ???)
	OR : A,|,B -- if A is true, ture as a whole (short cut evaluation)
	NEG: A,!,B -- negate the result of A, then evaluate B (can be "!{A,B}")

History:
	940623	added simple masking (suggested by <yoko@huie.hokudai.ac.jp>)
	940807	separated from access.c
//////////////////////////////////////////////////////////////////////#*/
#include "ystring.h"
#include <ctype.h>
#include "hostlist.h"
extern char *frex_create(),*frex_match();
extern char *VA_inAddr();
#define inAddr(addr)	VA_inAddr(addr)

#define HL_INCSIZE	8

static char opsyms[] = "+!-?";
#define ON_EQ	0	/* ON  if matched */
#define ON_NEQ	1	/* ON  if not matched */
#define OFF_EQ	2	/* OFF if matched */
#define OFF_NEQ	3	/* OFF if not matched */

#define OP_AND	4
#define OP_OR	5
#define OP_NEG	6
#define OP_HOUR	7

#define	DO_NOP	0
#define	DO_ON	1
#define DO_OFF	2

struct {
	double	currentTime;
	VAddr	clientsideHost;
	VAddr	clientSockHost;
} clientInfo = { 0 };

#define CurrentTime	clientInfo.currentTime
#define ClientsideHost	clientInfo.clientsideHost
#define ClientSockHost	clientInfo.clientSockHost
VA_HL_pushClientInfo(Now,peerhost,sockhost)
	double Now;
	VAddr *peerhost,*sockhost;
{
	CurrentTime = Now;
	ClientsideHost = *peerhost;
	ClientSockHost = *sockhost;
}
HL_popClientInfo()
{
	ClientsideHost = AddrNull;
	ClientSockHost = AddrNull;
}

static matchProto(protolist,proto)
	char *protolist[],*proto;
{	int isin,pi;
	char *proto1;
	int op,match;

	if( *protolist[0] == '!' )
		isin = 1;
	else	isin = 0;

	for( pi = 0; proto1 = protolist[pi]; pi++ ){
		if( *proto1 == '!' ){
			proto1++;
			op = OFF_EQ;
		}else	op = ON_EQ;

		if( streq(proto1,"*") )
			match = 1;
		else	match = streq(proto1,proto);

		if( match && op == ON_EQ )	isin = 1; else
		if( match && op == OFF_EQ )	isin = 0;
		Verbose("--> %d : %d : %s : %s : %s\n",
			isin, match, op==OFF_EQ?"NEQ":"EQ",
			proto, proto1);
	}
	return isin;
}
static char **makeUserRexp(uc,userlist)
	char *userlist[];
{	char **rexpv;
	char *user;
	int ui;

	rexpv = (char**)malloc(sizeof(char*)*uc);
	for( ui = 0; ui < uc; ui++ ){
		user = userlist[ui];
		if( *user == '!' )
			user++;
		if( !streq(user,"*") && strpbrk(user,"*[") ){
			sv1log("##REGEXP COMPILED: %s\n",user);
			rexpv[ui] = frex_create(user);
		}else	rexpv[ui] = 0;
	}
	return rexpv;
}
static matchUser(userlist,rexplist,username)
	char *userlist[];
	char *rexplist[];
	char *username;
{	int isin,ui;
	char *user;
	char *rexp,*tail;
	int op,match;

	if( *userlist[0] == '!' )
		isin = 1;
	else	isin = 0;

	for( ui = 0; user = userlist[ui]; ui++ ){
		if( *user == '!' ){
			user++;
			op = OFF_EQ;
		}else	op = ON_EQ;

		if( streq(user,"*") )
			match = 1;
		else
		if( rexp = rexplist[ui] ){
			tail = frex_match(rexp,username);
			match = tail && *tail == 0;
			sv1log("##REGEXP EXECUTED ==> %d : %s : %s\n",
				match,username,user);
		}
		else	match = strcmp(user,username) == 0;

		if( match && op == ON_EQ )	isin = 1; else
		if( match && op == OFF_EQ )	isin = 0;
		Verbose("--> %d : %d : %s : %s : %s\n",
			isin, match, op==OFF_EQ?"NEQ":"EQ",
			username,user);
	}
	return isin;
}
static matchPort(portlist,portnum)
	char *portlist[];
{	int isin,pi;
	char *ports;
	int op,match;
	int port0,port1;

	if( *portlist[0] == '!' )
		isin = 1;
	else	isin = 0;

	for( pi = 0; ports = portlist[pi]; pi++ ){
		if( *ports == '!' ){
			ports++;
			op = OFF_EQ;
		}else	op = ON_EQ;

		if( streq(ports,"*") )
			match = 1;
		else{
			switch( sscanf(ports,"%d-%d",&port0,&port1) ){
			case 0: match = 0; break;
			case 1: match = port0 == portnum; break;
			case 2: match = port0 <= portnum && portnum <= port1;
				break;
			}
		}

		if( match && op == ON_EQ )	isin = 1; else
		if( match && op == OFF_EQ )	isin = 0;
		Verbose("--> %d : %d : %s : %d : %s\n",
			isin, match, op==OFF_EQ?"NEQ":"EQ",
			portnum,ports);
	}
	return isin;
}

static match_addrrange(Hp,Vaddr)
	Host *Hp;
	VAddr *Vaddr;
{	int ai;
	int addr = Vaddr->I3;
	Uchar a1;

	for(ai = 0; ai < 4; ai++){
		a1 = (addr >> ((3-ai)*8)) & 0xFF;
		if( a1 < Hp->ranges[ai].h_low || Hp->ranges[ai].h_high < a1 )
			return 0;
	}
	return 1;
}
static default_netmask(Mask,Vaddr)
	VAddr *Mask,*Vaddr;
{	unsigned int ip4c,mask1;

	if( Mask->I3 != CLASS_MASK )
		return 0;

	ip4c = (Vaddr->I3 >> 24) & 0xFF;
	if( ip4c < 128 ) mask1 = 0xFF000000; else
	if( ip4c < 192 ) mask1 = 0xFFFF0000; else
	       mask1 = 0xFFFFFF00;
	*Mask = MaskZero;
	Mask->I3 = mask1;
	return 1;
}
static matchTimePeriod(period,mask)
	char *period;
{	char hour[8];
	int hc,h1,h2;
	int mc,m1,m2;

	/*
	 * -T.000-499/1000
	 * -T.0-4/10
	 */
	if( 0 < mask ){
		sscanf(period,"%d-%d",&m1,&m2);
		mc = (int)((CurrentTime - (int)CurrentTime) * mask) % mask;
		if( m1 <= mc && mc <= m2 )
			return 1;
		else	return 0;
	}

	StrftimeLocal(hour,sizeof(hour),"%H",(int)CurrentTime);
	hc = atoi(hour);

	h1 = 0;
	h2 = 23;
	sscanf(period,"%d-%d",&h1,&h2);

	if( h1 <= h2 ){
		if( h1 <= hc && hc <= h2 )
			return 1; 
	}else{
		if( h1 <= hc && hc <= 23 || 0 <= hc && hc <= h2 )
			return 1;
	}
	return 0;
}

static notResolved(Vaddr,name)
	VAddr *Vaddr;
	char *name;
{
	if( AddrInvalid(*Vaddr) )
		return 1;

	if( isinetAddr(name) != 0 ) /* name is in "dd.dd.dd.dd" IP-address */
		return 1;
	return 0;
}
static char *scanVdomain(hostname,hostnameb,vdomain)
	char *hostname,*hostnameb,*vdomain;
{	char *vdom,*vp,*sp,*dp,ch;

	*vdomain = 0;
	vdom = strstr(hostname,".-");
	if( vdom == 0 )
		return 0;

	strcpy(hostnameb,hostname);
	hostname = hostnameb;
	vp = sp = strstr(hostname,".-");
	dp = vdomain;
	for( vp += 2; ch = *vp; vp++ ){
		if( isalnum(ch) )
			*dp++ = ch;
	}
	if( dp == vdomain )
		return 0;

	*dp = 0;
	while( *sp++ = *vp++ );
	return vdom;
}

#define HostOutgoingIF(hp)	(hp[0]=='.' && hp[1]=='o' && hp[2]==0)
#define HostIncomingIF(hp)	(hp[0]=='.' && hp[1]=='i' && hp[2]==0\
			      || hp[0]=='-' && hp[1]==0)
#define HostOnClientside(hp)	(hp[0]=='-' && hp[1]=='C' && hp[2]==0)
#define TimePeriod(hp)		(hp[0]=='-' && hp[1]=='T' && hp[2]=='.')
#define HostUnknown(hp)		(hp[0]=='?' && hp[1]==0)
#define HostMatchExactly(hp,hn)	(strcasecmp(hp,hn) == 0)

static VA_hostIsinList(hostlist,proto,cname,alias,Vaddr,username)
	HostList *hostlist;
	char *proto,*cname,*alias;
	VAddr *Vaddr;
	char *username;
{	int portnum = Vaddr->a_port;
	int hi,nhosts;
	VAddr hosti,mask,ahost,chost;
	Host *Hp;
	char *hostpat;
	int op;
	int isin,match,do_onoff;
	char hostnameb[256],*vdom,vdomb[128],*vdom1;
	char *hostname;
	char *domhost;

	hosti = *Vaddr;

	if( scanVdomain(cname,hostnameb,vdomb) ){
		cname = hostnameb;
		vdom = vdomb;
	}else	vdom = 0;

	strtolower(cname,hostnameb);
	cname = hostnameb;

	op = 0;
	isin = 0;
	nhosts = hostlist->hl_cnt;

	for( hi = 1; hi <= nhosts; hi++ ){
		Hp = hostlist->hl_List[hi];
		op = Hp->h_op;
		if( op == OP_AND ){
			if( isin ){
				Verbose("[%d/%d] &=> 1 AND succeed, cont.\n",
					hi,nhosts);
				isin = 0;
				continue;
			}else{
				Verbose("[%d/%d] &=> 0 AND failed, break\n",
					hi,nhosts);
				for( hi++; hi <= nhosts; hi++ ){
					Hp = hostlist->hl_List[hi];
					op = Hp->h_op;
					if( Hp->h_op == OP_OR )
						break;
				}
				if( op != OP_OR )
					break;
				Verbose("[%d/%d] |=> found OR.\n",hi,nhosts);
				continue;
			}
		}
		if( op == OP_OR ){
			if( isin ){
				Verbose("[%d] |=> OR 1 succeed, break\n",hi);
				break;
			}else{
				Verbose("[%d] |=> OR 0 ineffective, cont.\n",hi);
				continue;
			}
		}
		if( op == OP_NEG ){
			isin = !isin;
			Verbose("[%d/%d] !=> %d NEGATE\n",hi,nhosts,isin);
			continue;
		}

		vdom1 = Hp->h_vdomain;
		if( vdom == 0 && vdom1 != 0
		 || vdom != 0 && vdom1 == 0
		 || vdom != 0 && vdom1 != 0 && !streq(vdom,vdom1) )
			continue;

		if( isin && (op == ON_EQ || op == ON_NEQ) ||
		   !isin && (op ==OFF_EQ || op==OFF_NEQ) ){
			/* nothing will happen */
			continue;
		}
		match = 0;

		mask = Hp->h_mask;
		hostpat = Hp->h_name;

		domhost = 0;
		if( hostpat[0] == '*' && hostpat[1] == '.' )
			domhost = hostpat + 2;

		if( Hp->h_asis && alias )
			hostname = alias;
		else	hostname = cname;

		if( TimePeriod(hostpat) ){
			match = matchTimePeriod(hostpat+3,mask.I3);
			Verbose("[%d/%d] TimePeriod = %s / %d = %d\n",
				hi,nhosts,hostpat,mask.I3,match);
			goto ACCUM;
		}

		if( HostOnClientside(hostpat) ){
			Verbose("[%d/%d] -C (ClientHost) = %s\n",hi,nhosts,
				inAddr(&ClientsideHost));
			default_netmask(&mask,&ClientsideHost);
			AddrAND(chost,mask,ClientsideHost);
		}else
		if( HostOutgoingIF(hostpat) ){
			VAddr ohost;
			if( AddrInvalid(hosti) ){
				chost = hosti; /* any different from hosti */
				chost.I3 = ~chost.I3;
				ohost = chost;
				mask = AddrNull;
			}else{
				default_netmask(&mask,&hosti);
				VA_hostIFto(&hosti,&mask,&ohost);
				AddrAND(chost,ohost,mask);
			}
			Verbose("[%d/%d] .o (OutgoingIF) = %s -> %s\n",hi,nhosts,
				inAddr(&ohost),inAddr(&hosti));
		}else
		if( HostIncomingIF(hostpat) ){
			Verbose("[%d/%d] .i (IncomingIF) = %s\n",hi,nhosts,
				inAddr(&ClientSockHost));
			default_netmask(&mask,&ClientSockHost);
			AddrAND(chost,mask,ClientSockHost);
		}else	AddrAND(chost,mask,Hp->h_addr);
		AddrAND(ahost,hosti,mask);

		if( HostUnknown(hostpat) && notResolved(&hosti,hostname) ){
			Verbose("UNKNOWN HOST\n");
			match = 1;
		}else
		if( domhost && HostMatchExactly(domhost,hostname) ){
			Verbose("[%d/%d] HOST==DOMAIN MATCH: %s %c= %s ?\n",
				hi,nhosts,hostname,opsyms[op],hostpat);
			match = 1;
		}else
		if( HostMatchExactly(hostpat,hostname) ){
			Verbose("[%d/%d] EXACT NAME MATCH: %s %c= %s ?\n",
				hi,nhosts,hostname,opsyms[op],hostpat);
			match = 1;
		}else
		if( !AddrEQ(ahost,AddrZero) && !AddrEQ(chost,AddrZero) ){
			Verbose("[%d/%d] ADDR MATCH: %s %c= %s ?\n",hi,nhosts,
				inAddr(&ahost),opsyms[op],inAddr(&chost));
			match = AddrEQ(ahost,chost);
		}else
		if( !AddrEQ(ahost,AddrZero) && Hp->ranges[0].h_high ){
			Verbose("[%d/%d] ADDR RANGE: %s[%s] %c= %s ?\n",hi,nhosts,
				hostname,inAddr(&ahost),opsyms[op],hostpat);
			match = match_addrrange(Hp,&ahost);
		}else
		if( !match ){
			Verbose("[%d/%d] REGEXP NAME MATCHING: %s %c= %s ?\n",
				hi,nhosts,hostname,opsyms[op],hostpat);
			match = rexpmatch(hostpat,hostname)
		     		|| streq(hostpat,hostname);

			if( !match )
			if( hostname != alias )
			if( alias != NULL )
			if( !strcaseeq(alias,cname) ){
				Verbose("[%d/%d] ALIAS MATCHING: %s %c= %s ?\n",
					hi,nhosts,alias,opsyms[op],hostpat);
				match = rexpmatch(hostpat,alias)
					|| strcaseeq(hostpat,alias);

				if( !match )
				if( domhost )
					match = strcaseeq(domhost,alias);
			}
		}
		if( match && Hp->h_proto ){
			if( proto != ANYP && !matchProto(Hp->h_proto,proto) ){
				Verbose("--> proto name unmatch: %s\n",proto);
				match = 0;
			}
		}
		if( match && Hp->h_port ){
			if( portnum && !matchPort(Hp->h_port,portnum) ){
				Verbose("--> port num. unmatch: %d\n",portnum);
				match = 0;
			}
		}
		if( match && Hp->h_user ){
			if( username == 0 )
				username = "?";
			if( !matchUser(Hp->h_user,Hp->h_userRexp,username) ){
				Verbose("--> user name unmatch: %s\n",username);
				match = 0;
			}
		}

ACCUM:
		if( match && op == ON_EQ  || !match && op == ON_NEQ )
			do_onoff = DO_ON;
		else
		if( match && op == OFF_EQ || !match && op == OFF_NEQ )
			do_onoff = DO_OFF;
		else	do_onoff = DO_NOP;

		switch( do_onoff ){
			case DO_ON:	isin = 1; break;
			case DO_OFF:	isin = 0; break;
		}
		Verbose("[%d/%d] ==> %d (%s %s)\n",
			hi,nhosts,isin,hostlist->hl_what,hostname);
	}
	return isin ? hi : 0;
}
hostIsinList(hostlist,proto,hostname,portnum,username)
	HostList *hostlist;
	char *proto,*hostname;
	char *username;
{	VAddr Vaddr;
	char *vdom,vdomb[256],hostnameb[512],*rhostname;
	char primaryname[512];
	char *vhost;

	if( vdom = scanVdomain(hostname,hostnameb,vdomb) )
		rhostname = hostnameb;
	else	rhostname = hostname;

	if( hostname[0] == '-' && !isMYSELF(hostname) ){
		vhost = hostname + 1;
		hostname += 1;
		Vaddr = AddrNull;
	}else{
		vhost = rhostname;
		primaryname[0] = 0;
		VA_gethostVAddr(0,rhostname,primaryname,&Vaddr);
		if( primaryname[0] ){
			hostname = primaryname;
			if( vdom ) strcat(hostname,vdom);
		}
	}
	Vaddr.a_port = portnum;
	return VA_hostIsinList(hostlist,proto,hostname,vhost,&Vaddr,username);
}
pathIsinList(hostlist,path)
	HostList *hostlist;
	char *path[];
{
}

static addUSER1(user,uv,uc)
	char *user,*uv[];
	int *uc;
{
	uv[*uc] = user;
	*uc += 1;
	return 0;
}
static addStrVec1(str,sv,sc)
	char *str,*sv[];
	int *sc;
{
	sv[*sc] = str;
	*sc += 1;
	return 0;
}

static scan_addrrange(Hp,addr)
	Host *Hp;
	char *addr;
{	char abuf[128],*ap,*dp;
	int ai,iaddr,lo,hi;

	strcpy(abuf,addr);
	ap = abuf;
	for( ai = 0; ai < 4;  ){
		if( dp = strchr(ap,'.') )
			*dp++ = 0;
		if( *ap == '[' ){
			if( sscanf(ap+1,"%d-%d",&lo,&hi) != 2 )
				goto error;
		}else	lo = hi = atoi(ap);

		Hp->ranges[ai].h_low = lo;
		Hp->ranges[ai].h_high = hi;
		ai++;
		if( dp == 0 )
			break;
		else	ap = dp;
	}
	for(; ai < 4; ai++ ){
		Hp->ranges[ai].h_low = 0;
		Hp->ranges[ai].h_high  = 255;
	}
	return 0;
error:
	sv1log("ERROR address range syntax: %s\n",addr);
	return -1;
}

static addHL1(hostmask,hostlist,op,users)
	char *hostmask;
	HostList *hostlist;
	char *users;
{	char op_hostmask[2048];

	if( users )
		sprintf(op_hostmask,"%c%s@%s",op,users,hostmask);
	else	sprintf(op_hostmask,"%c%s",op,hostmask);
	addHostList1(op_hostmask,hostlist);
	return 0;
}
static char *hostpart(hostmask,split)
	char *hostmask;
{	char *dp;

	if( dp = strchr(hostmask,'@') )
	if( dp == hostmask || dp[-1] != '/' ){ /* NOT host/@Cn */
		if( split )
			*dp = 0;
		return dp + 1;
	}
	return 0;
}

extern char *DELEGATE_LOCALNET;
static addHL(hostmask,hostlist)
	char *hostmask;
	HostList *hostlist;
{	char list[2048],*dp;
	int op;
	char *hosts,*users;

	if( strchr(opsyms,*hostmask) )
		op = *hostmask++;
	else	op = '+';
	strcpy(list,hostmask);

	if( dp = hostpart(list,1) ){
		hosts = dp;
		users = list;
	}else{
		hosts = list;
		users = 0;
	}

	if( streq(hosts,".localnet") ){
		strcpy(list,DELEGATE_LOCALNET);
		hosts = list;
	}else
	if( *hosts != '{' )
		return 0;

	Verbose("addHL: %c [ %s ] @ [ %s ]\n",op,users?users:"*",hosts);
	scan_commaListL(hosts,0,addHL1,hostlist,op,users);
	return 1;
}

extern char **dupv();

addHostList1(hostmask,hostlist)
	char *hostmask;
	HostList *hostlist;
{	int idx,siz,nsiz;
	Host *Hp,**oHlist,**nHlist;

	if( hostmask[0] == '!' && hostmask[1] == '!' )
		hostmask += 2;
	if( hostmask[0] == '+' && hostmask[1] == '!' )
		hostmask += 1;
	if( addHL(hostmask,hostlist) )
		return 0;
	if( hostlist->hl_cnt == 0 && *hostmask == '!' )
		addHostList1("*",hostlist);

	Hp = (Host*)calloc(sizeof(Host),1);
	idx = ++hostlist->hl_cnt;
	siz = hostlist->hl_size;
	if( siz <= idx ){
		oHlist = hostlist->hl_List;
		if( hostlist->hl_inc )
			nsiz = siz + hostlist->hl_inc;
		else	nsiz = siz + HL_INCSIZE;
		while( nsiz <= idx )
			nsiz++;
		hostlist->hl_size = nsiz;
		if( oHlist == (Host**)0 )
			nHlist = (Host**)calloc(nsiz,sizeof(Host*));
		else	nHlist = (Host**)realloc(oHlist,nsiz*sizeof(Host*));
		hostlist->hl_List = nHlist;
	}
	hostlist->hl_List[idx] = Hp;
	makeHost(hostlist,Hp,0,hostmask);
	return 0;
}

canonWildcardAddr(hostname)
	char *hostname;
{	char *hp,ch,*bp,buff[128],*ap,addr[128];
	int noct,octv,olen,isdg;

	if( strchr(hostname,'*') && strchr(hostname,'.') ){
		hp = hostname;
		ap = addr;
		noct = 0;

		while( *hp ){
			bp = buff;
			isdg = 1;
			while( ch = *hp ){
				hp++;
				*bp++ = ch;
				if( ch == '.' ) 
					break;
				if( !isdigit(ch) )
					isdg = 0;
			}
			*bp = 0;

			if( buff[0]=='*' && (buff[1]==0||buff[1]=='.') ){
				sprintf(ap,"[0-255]%c",buff[1]);
			}else
			if( isdigit(buff[0]) && isdg ){
				octv = atoi(buff);
				if( octv < 0 || 255 < octv )
					goto NOTADDR;
				strcpy(ap,buff);
			}else	goto NOTADDR;
			ap += strlen(ap);

			noct++;
			if( 4 <= noct && *hp != 0 )
				goto NOTADDR;
		}
		if( 2 <= noct && noct <= 4 ){
			*ap = 0;
			strcpy(hostname,addr);
		}
	} NOTADDR:;
}

static isnetmasklen(mask)
	char *mask;
{	int masklen;

	if( isdigit(mask[0]) && mask[1] == 0
	 || isdigit(mask[0]) && isdigit(mask[1]) && mask[2] == 0 ){
		masklen = atoi(mask);
		if( 1 <= masklen && masklen <= 31 )
			return masklen;
	}
	return 0;
}

makeHost(hostlist,Hp,lev,hostmask)
	HostList *hostlist;
	Host *Hp;
	char *hostmask;
{	char *mp,hostnameb[1024],primaryname[512];
	VAddr addr,mask;
	int submask,mask1;
	char proto[256],hostmaskb[1024];
	char hostmaskb2[1024],vdomain[256];
	char *route;

	proto[0] = 0;
	if( strstr(hostmask,"://") ){
		sscanf(hostmask,"%[^:]://%s",proto,hostmaskb);
		hostmask = hostmaskb;
	}
	if( route = strstr(hostmask,"//") ){
		*route = 0;
		Hp->h_route = (Host*)calloc(sizeof(Host),1);
		makeHost(hostlist,Hp->h_route,lev+1,route+2);
	}

	if( hostmask[0] == '&' && hostmask[1] == 0 ){
		Hp->h_op = OP_AND;
		return;
	}
	if( hostmask[0] == '|' && hostmask[1] == 0 ){
		Hp->h_op = OP_OR;
		return;
	}
	if( hostmask[0] == '!' && hostmask[1] == 0 ){
		Hp->h_op = OP_NEG;
		return;
	}

	if( hostmask[0] == '!' && hostmask[1] == '!' )
		hostmask += 2;

	if( *hostmask == '!' ){ Hp->h_op = OFF_EQ;  hostmask++; }else
/*
	if( *hostmask == '!' ){ Hp->h_op = ON_NEQ;  hostmask++; }else
	if( *hostmask == '?' ){ Hp->h_op = OFF_NEQ; hostmask++; }else
	if( *hostmask == '-' ){ Hp->h_op = OFF_EQ;  hostmask++; }else
*/
			      { Hp->h_op = ON_EQ;
				if(*hostmask=='+')  hostmask++; }

	vdomain[0] = 0;
	if( scanVdomain(hostmask,hostmaskb2,vdomain) )
		hostmask = hostmaskb2;

	mask = MaskZero;
	if( mp = strchr(hostmask,'/') ){
		*mp++ = 0;
		if( TimePeriod(hostmask) ){
			mask.I3 = atoi(mp);
		}else
		if( submask = isnetmasklen(mp) ){
			mask.I3 = 0xFFFFFFFF << (32-submask);
		}else
		if( strchr(mp,'.') ) /* address in dot-notaion */
			VA_strtoVAddr(mp,&mask);
		else
		if( mp[0] == '@' ){
			switch( mp[1] ){
			    case 'A': mask1 = 0xFF000000; break;
			    case 'B': mask1 = 0xFFFF0000; break;
			    case 'C': mask1 = 0xFFFFFF00; break;
			    default:  mask1 = CLASS_MASK; break;
			}
			mask.I3 = mask1;
			if( submask = atoi(&mp[2]) )
				mask.I3 = mask.I3 >> submask;
		}else{
			sscanf(mp,"%x",&mask.I3);
		}
	}else{
		mask.I3 = 0xFFFFFFFF;
	}
	Hp->h_mask = mask;

	strcpy(hostnameb,hostmask);

	if( streq(hostmask,".") ){
		char me[128];
		GetHostname(me,sizeof(me));
		strcpy(hostnameb,me);
	}

	{	char *dp,*uv[128],users[2048];
		int ui;

		if( dp = strchr(hostnameb,'@') ){
			if( hostnameb[0] == '{' ){
				char *pp;
				strcpy(users,hostnameb+1);
				if( pp = strchr(users,'}') )
					*pp = 0;
				else	*strchr(users,'@') = 0;
			}else{
				strcpy(users,hostnameb);
				*strchr(users,'@') = 0;
			}
			ui = 0;
			scan_commaList(users,2,addUSER1,uv,&ui);
			uv[ui] = 0;
			Hp->h_user = dupv(uv,0);
			Hp->h_userRexp = makeUserRexp(ui,uv);

			strcpy(hostnameb,dp+1);

			if( !hostlist->hl_noIdent )
			if( hostnameb[0] != '!' )
				enableClientIdent(hostnameb);

			if( hostnameb[0] == '!' ){
				strcpy(hostnameb,hostnameb+1);
				if( Hp->h_op == ON_EQ )
					Hp->h_op = OFF_EQ;
				else	Hp->h_op = ON_EQ;
			}
			/* USER part of USER@!host should be ignored ?? */
		}
	}
	{	char *dp,*pv[128],ports[2048];
		int pi;

		if( dp = strchr(hostnameb,':') ){
			*dp++ = 0;
			strcpy(ports,dp);
			pi = 0;
			scan_commaList(ports,2,addStrVec1,pv,&pi);
			pv[pi] = 0;
			Hp->h_port = dupv(pv,0);
		}
	}
	if( proto[0] != 0 ){
		char *pv[128],protos[2048];
		int pi;

		pi = 0;
		strcpy(protos,proto);
		scan_commaList(protos,2,addStrVec1,pv,&pi);
		pv[pi] = 0;
		Hp->h_proto = dupv(pv,0);
	}

	primaryname[0] = 0;
	canonWildcardAddr(hostnameb);

	if( HostOnClientside(hostnameb) ){
		Verbose("%s[%d]: %c (ClientHost) = -C / %x\n",
			hostlist->hl_what,hostlist->hl_cnt,
			opsyms[Hp->h_op],Hp->h_mask.I3);
	}else
	if( HostIncomingIF(hostnameb) ){
		Verbose("%s[%d]: %c (IncomingIF) = %s / %x\n",
			hostlist->hl_what,hostlist->hl_cnt,
			opsyms[Hp->h_op],hostnameb,Hp->h_mask.I3);
	}else
	if( HostOutgoingIF(hostnameb) ){
		Verbose("%s[%d]: %c (OutgoingIF) = %s / %x\n",
			hostlist->hl_what,hostlist->hl_cnt,
			opsyms[Hp->h_op],hostnameb,Hp->h_mask.I3);
	}else
	if( TimePeriod(hostnameb) ){
		Verbose("%s[%d]: %c (TimePeriod) = %s / %d\n",
			hostlist->hl_what,hostlist->hl_cnt,
			opsyms[Hp->h_op],hostnameb,Hp->h_mask.I3);
	}else
	if( hostnameb[0] == '-' && !isMYSELF(hostnameb) ){
		Verbose("%s[%d]: %c %s\n",
			hostlist->hl_what,hostlist->hl_cnt,
			opsyms[Hp->h_op], hostnameb);
		if( VA_strtoVAddr(hostnameb+1,&addr) )
			Hp->h_addr = addr;
		else{
			Hp->h_asis = 1;
			strcpy(hostnameb,hostnameb+1);
			/* to suppress resolution on reload which will
			 * always result in 'unknown' */
			if( !VA_gethostVAddr(0,hostnameb,primaryname,NULL) ){
				sv1log("Virtual Host Name: '%s'\n",hostnameb);
				sethostcache_predef(hostnameb);
			}
		}
	}else
	if( strchr(hostnameb,'[') ){
		if( scan_addrrange(Hp,hostnameb) == 0 )
		Verbose("%s[%d]: %c %s\n",
			hostlist->hl_what,hostlist->hl_cnt,
			opsyms[Hp->h_op],hostnameb);
	}else
	if( strchr(hostnameb,'*') || streq(hostnameb,"?") ){
		Verbose("%s[%d]: %c %s\n",
			hostlist->hl_what,hostlist->hl_cnt,
			opsyms[Hp->h_op], hostnameb);
	}else
	if( !VA_gethostVAddr(0,hostnameb,primaryname,&addr) ){
		sv1log("ERROR %s[%d] %s ? unknown\n",
			hostlist->hl_what,hostlist->hl_cnt, hostnameb);
	}else{
		if( primaryname[0] )
			strcpy(hostnameb,primaryname);
		Hp->h_addr = addr;
		default_netmask(&Hp->h_mask,&addr);

		Verbose("%s[%d]: %c %s = %x / %x\n",
			hostlist->hl_what,hostlist->hl_cnt,
			opsyms[Hp->h_op], hostnameb,
			addr.I3,Hp->h_mask.I3);
	}

	Hp->h_name = strdup(hostnameb);
	Hp->h_vdomain = vdomain[0] ? strdup(vdomain) : 0;
}


typedef struct {
	char	 *name;
	char	 *path;
	HostList *hlist;
} PathList; 
static PathList *pathlists;
static int pathlists_siz;
static int pathlists_cnt;

int makePathList(what,path)
	char *what,*path;
{	HostList *HL;
	PathList *PL;
	int nsize,hi;

	if( pathlists_siz <= pathlists_cnt ){
		if( pathlists_siz == 0 ){
			pathlists_siz = 32;
			nsize = sizeof(PathList) * pathlists_siz;
			pathlists = (PathList*)malloc(nsize);
		}else{
			pathlists_siz += 32;
			nsize = sizeof(PathList) * pathlists_siz;
			pathlists = (PathList*)realloc(pathlists,nsize);
		}
	}
	for( hi = 0; hi < pathlists_cnt; hi++ )
		if( strcmp(path,pathlists[hi].path) == 0 )
			return hi+1;

	HL = (HostList*)calloc(sizeof(HostList),1);
	HL->hl_what = strdup(what);

	scan_commaListL(path,0,addHostList1,HL);

	PL = &pathlists[pathlists_cnt++];
	PL->hlist = HL;
	PL->path = strdup(path);

	return pathlists_cnt;
}
matchPathList(hlid,path)
	char *path;
{
}
matchPath1(hlid,user,host,port)
	char *user,*host;
{	HostList *HL;
	int match;

	HL = pathlists[hlid-1].hlist;
	match = hostIsinList(HL,"*",host,port,user);
Verbose("###### matchPath1(%d,%s,%s:%d) = %d\n",hlid,user,host,port,match);
	return match;
}
