/*///////////////////////////////////////////////////////////////////////
Copyright (c) 1993-1999 Electrotechnical Laboratry (ETL), AIST, MITI
Copyright (c) 1993-1999 Yutaka Sato

Permission to use, copy, and distribute this material for any purpose
and without fee is hereby granted, provided that the above copyright
notice and this permission notice appear in all copies.
ETL MAKES NO REPRESENTATIONS ABOUT THE ACCURACY OR SUITABILITY OF THIS
MATERIAL FOR ANY PURPOSE.  IT IS PROVIDED "AS IS", WITHOUT ANY EXPRESS
OR IMPLIED WARRANTIES.
/////////////////////////////////////////////////////////////////////////
Content-Type:	program/C; charset=US-ASCII
Program:	JIS.c
Author:		Yutaka Sato <ysato@etl.go.jp>
Description:
History:
	930923	extracted from codeconv.c of cosmos
//////////////////////////////////////////////////////////////////////#*/
#include <string.h>
#include <ctype.h>
char *malloc();
#include <stdio.h>
extern FILE *fopen_LIBPATH();
extern char *getTMPDIR();
int CCX_debug = 0;

/*
 *	STATUS CHANGE CODES
 */
#define ESC		'\033'
#define TO_2BCODE	'$'
#define TO_1BCODE	'('

#define TO_KANA		'\016'
#define TO_KANAOUT	'\017'

#define TO_KANJI	"\033$B"
#define TO_ASCII	"\033(B"

#define IS_SJIS_LO(lo)	((0x40<=lo)&&(lo!=0x7F)&&(lo<=0xFC))
#define IS_SJIS_HI1(hi) ((0x81<=hi)&&(hi<=0x9F))	/* 1st lev. */
#define IS_SJIS_HI2(hi) ((0xE0<=hi)&&(hi<=0xEF))	/* 2nd lev. */
#define IS_SJIS_HI(hi)	(IS_SJIS_HI1(hi)||IS_SJIS_HI2(hi))
#define IS_SJIS_2B(hi,lo,in_sjis)\
	(!IS_SJIS_LO(lo) ? 0:\
	IS_SJIS_HI1(hi) ? (in_sjis = 1):\
	in_sjis && IS_SJIS_HI2(hi))

#define IS_SJIS_1B(ch)	(0xA1 <= (ch) && (ch) <= 0xDF)

#define SJIS_DAKUTEN	0xDE
#define SJIS_HANDAKU	0xDF
static char *SJIS_1B_TO_JIS[] = {
/* 0xA1-A7 */       "!#", "!V", "!W", "!\"","!%", "%r", "%!",
/* 0xA8-AF */ "%#", "%%", "%'", "%)", "%c", "%e", "%g", "%C",
/* 0xB0-B7 */ "!<", "%\"","%$", "%&", "%(", "%*", "%+", "%-",
/* 0xB8-BF */ "%/", "%1", "%3", "%5", "%7", "%9", "%;", "%=",
/* 0xC0-C7 */ "%?", "%A", "%D", "%F", "%H", "%J", "%K", "%L",
/* 0xC8-CF */ "%M", "%N", "%O", "%R", "%U", "%X", "%[", "%^",
/* 0xD0-D7 */ "%_", "%`", "%a", "%b", "%d", "%f", "%h", "%i",
/* 0xD8-DF */ "%j", "%k", "%l", "%m", "%o", "%s", "!+", "!,"
};
static char *SJIS_1B_TO_DAKUTEN[] = {
/* 0xA1-A7 */       0,    0,    0,    0,    0,    0,    0,
/* 0xA8-AF */ 0,    0,    0,    0,    0,    0,    0,    0,
/* 0xB0-B7 */ 0,    0,    0,    "%t", 0,    0,    "%,", "%.",
/* 0xB8-BF */ "%0", "%2", "%4", "%6", "%8", "%:", "%<", "%>",
/* 0xC0-C7 */ "%@", "%B", "%E", "%G", "%I", 0,    0,    0,
/* 0xC8-CF */ 0,    0,    "%P", "%S", "%V", "%Y", "%\\",0,
/* 0xD0-D7 */ 0,    0,    0,    0,    0,    0,    0,    0,
/* 0xD8-DF */ 0,    0,    0,    0,    0,    0,    0,    0
};
static char *SJIS_1B_TO_HANDAKU[] = {
/* 0xA1-A7 */       0,    0,    0,    0,    0,    0,    0,
/* 0xA8-AF */ 0,    0,    0,    0,    0,    0,    0,    0,
/* 0xB0-B7 */ 0,    0,    0,    0,    0,    0,    0,    0,
/* 0xB8-BF */ 0,    0,    0,    0,    0,    0,    0,    0,
/* 0xC0-C7 */ 0,    0,    0,    0,    0,    0,    0,    0,
/* 0xC8-CF */ 0,    0,    "%Q", "%T", "%W", "%Z", "%]", 0,
/* 0xD0-D7 */ 0,    0,    0,    0,    0,    0,    0,    0,
/* 0xD8-DF */ 0,    0,    0,    0,    0,    0,    0,    0
};

#define EUC_HANKAKU_HI	0x8E
#define IS_EUC_HANKAKU(hi,lo)	(hi==EUC_HANKAKU_HI && IS_SJIS_1B(lo))

#define IS_EUC_LOS(lo)	((0x21<=lo)&&(lo<=0x7E))	/* standard */
#define IS_EUC_LOX(lo)	((0xA1<=lo)&&(lo<=0xFE))	/* extended */
#define IS_EUC_HI(hi)	((0xA1<=hi)&&(hi<=0xFE))
#define IS_EUC(hi,lo)\
	(IS_EUC_HANKAKU(hi,lo) || \
	IS_EUC_HI(hi) && (IS_EUC_LOS(lo) || IS_EUC_LOX(lo)))


IS_SJIS_CHAR(ch1,ch2,in_sjis)
{
	if( ch1 & 0x80 ){
		if( IS_SJIS_LO(ch2) ){
			if( IS_SJIS_HI1(ch1) || in_sjis && IS_SJIS_HI2(ch1) )
				return 2;
		}
		if( in_sjis && IS_SJIS_1B(ch1) )
			return 1;
	}
	return 0;
}
IS_SJIS_STR(str)
	unsigned char *str;
{	unsigned char *s,ch;
	int is_sjis = 0;

	s = str;
	while( ch = *s++ ){
		if( ch & 0x80 )
			if( !IS_EUC_HANKAKU(ch,*s) )
			if( IS_SJIS_2B(ch,*s,is_sjis) )
				return 1;
	}
	return 0;
}

unsigned char *
SJIS_TO_JIS1(HI,LO,JCODE)
	register unsigned char HI,LO,*JCODE;
{
	HI -= (HI <= 0x9F) ? 0x71 : 0xB1;
	HI = (HI << 1) + 1;
	if( 0x7F < LO )
		LO--;
	if( 0x9E <= LO ){
		LO -= 0x7D;
		HI++;
	}else	LO -= 0x1F;
	JCODE[0] = HI;
	JCODE[1] = LO;
	return JCODE;
}
unsigned char *
JIS_TO_SJIS1(HI,LO,SJCODE)
	register unsigned char HI,LO,*SJCODE;
{
	if( HI & 1 )
		LO += 0x1F;
	else	LO += 0x7D;
	if( 0x7F <= LO )
		LO++;

	HI = (((int)HI - 0x21) >> 1) + 0x81;
	if( 0x9F < HI )
		HI += 0x40;
	SJCODE[0] = HI;
	SJCODE[1] = LO;
	return SJCODE;
}

static char *sjis_1b_to_jis(ch1,ch2,cat)
	int *cat;
{	unsigned char c1,c2;
	char *js;

	c1 = (0x80 | ch1) - 0xA1;
	c2 =  0x80 | ch2;
	if( c2 == SJIS_DAKUTEN && (js = SJIS_1B_TO_DAKUTEN[c1])
	 || c2 == SJIS_HANDAKU && (js = SJIS_1B_TO_HANDAKU[c1]) ){
		*cat = 1;
		return js;
	}else{
		*cat = 0;
		return SJIS_1B_TO_JIS[c1];
	}
}
static EUC_hankaku_TO_JIS(spp,dpp)
	unsigned char **spp,**dpp;
{	unsigned char *sp,*dp;
	unsigned char ch1,ch2;
	char *js;
	int cat;

	sp = *spp;
	dp = *dpp;
	if( !IS_EUC_HANKAKU(sp[0],sp[1]) )
		return 0;

	ch1 = sp[1];
	if( sp[2] && sp[3] )
		ch2 = sp[3];
	else	ch2 = 0;
	js = sjis_1b_to_jis(ch1,ch2,&cat);
	if( cat )
		sp += 2;

	strcpy(dp,js);
	dp += 2; *dpp = dp;
	sp += 2; *spp = sp;
	return 1;
}

#define IS_JIS_HI(c1)	(0x20 < (c1) && (c1) < 0x7F)
#define IS_JIS_LO(c1)	(0x20 < (c1) && (c1) < 0x7F)
#define	IS_JIS7(c1,c2)	(IS_JIS_HI(c1) && IS_JIS_LO(c2))
#define SO		('N'-0x40)
#define SI		('O'-0x40)
#define NBSP		0xA0	/* non-breaking space */


#define sputc(sp,ch)	(sp?(*sp++ = ch):ch)

static istag(str)
	char *str;
{	char ch,*s;

	for( s = str; ch = *s; s++ ){
		if( ch == '>' || isspace(ch) )
			return str < s;
		if( !isalpha(ch) )
			return 0;
	}
	return 0;
}

FIX_2022(src,dst,ctype)
	char *src,*dst,*ctype;
{	int in2B;
	char ch1,ch2,*sp,*dp;
	int bad;
	int isHTML,len,ech;

	in2B = 0;
	sp = src;
	dp = dst;
	bad = 0;

	isHTML = strcasecmp(ctype,"text/html") == 0;

	while( ch1 = *sp++ ){
		if( ch1 == ESC ){
			if( *sp == TO_2BCODE ){
				if( sp[1] == 'B' || sp[1] == '@' ){
					in2B = 1;
					sputc(dp, ch1);
					sputc(dp, *sp++);
					sputc(dp, *sp++);
					continue;
				}
			}else
			if( *sp == TO_1BCODE ){
				if( sp[1] == 'B' || sp[1] == 'J' ){
					in2B = 0;
					sputc(dp, ch1);
					sputc(dp, *sp++);
					sputc(dp, *sp++);
					continue;
				}
			}
		}

		if( in2B ){
			ch2 = sp[0];
			if( ch1 <= 0x20
			||  ch2 <= 0x20
			||  isHTML && ch1=='<' && sp[0]=='/' && istag(&sp[1])
			||  isHTML && ch2=='<' && sp[1]=='/' && istag(&sp[2]) ){
				in2B = 0;
				sputc(dp, ESC);
				sputc(dp, TO_1BCODE);
				sputc(dp, 'B');
				sputc(dp, ch1);
				bad = 1;
				continue;
			}

			if( isHTML && ch1 == '&' )
			if( len = isHTMLentity(sp,&ech) )
			if( sp[len] != 0 ){
				ch1 = ech;
				sp += len;
				bad = 1;
			}

			ch2 = *sp++;

			if( isHTML && ch2 == '&' )
			if( len = isHTMLentity(sp,&ech) )
			if( sp[len] != 0 ){
				ch2 = ech;
				sp += len;
				bad = 1;
			}

			sputc(dp, ch1);
			sputc(dp, ch2);
		}else{
			sputc(dp, ch1);
		}
	}
	sputc(dp, 0);
	return bad;
}

static is_EUC_JP(euc)
	char *euc;
{	char *cp;
	int ch1,ch2;

	for( cp = euc; ch1 = *cp & 0xFF; cp++ ){
		if( ch1 & 0x80 ){
			ch2 = cp[1] & 0xFF;
			if( !IS_EUC(ch1,ch2) )
				return 0;
			cp++;
		}
	}
	return 1;
}


#define CC_THRU		0
#define CC_ASCII	1
#define CC_JIS2B7	2
#define CC_JIS7K	3	/* with 7bit/1byte kana by ESC(I */
#define CC_SJIS		4
#define CC_EUCJP	5
#define CC_JIS7KANA	6
#define CC_JIS7ROMA	7
#define CC_UTF8		8
#define JIS2B8(code)	(code == CC_SJIS || code == CC_EUCJP)
#define JIS2B(code)	(code == CC_JIS2B7 || JIS2B8(code))

struct {
	char   *cs_name;
	char   *cs_formname;
	short	cs_charcode;
} charsets[] = {
	{0},
	{"jis",		"ISO-2022-JP",	CC_JIS2B7	},
	{"euc",		"x-euc-jp",	CC_EUCJP	},
	{"euc-jp",	"EUC-JP",	CC_EUCJP	},
	{"sjis",	"x-sjis",	CC_SJIS		},
	{"Shift_JIS",	"Shift_JIS",	CC_SJIS		},
	{"utf8",	"UTF-8",	CC_UTF8		},
	{0},
}; 

typedef struct {
	short	cc_size;
	short	cc_OUT; /* target charcode of conversion */
	short	cc_out;	/* current output charcode */
	short	cc_outx; /* chaset-name index in chasets[] */
	short	cc_indef; /* default input charcode */
	short	cc_in;	/* current input charcode */
	short	cc_previn; /* previous non-ASCII input charcode */
	char	cc_inswitch[16]; /* detected input character type */
unsigned char	cc_pinb[8];
	int	cc_pinx;
	int	cc_nonASCII;
} CCX;
#define cc_UTF8		cc_inswitch[CC_UTF8]
#define cc_SJIS		cc_inswitch[CC_SJIS]
#define cc_JIS7		cc_inswitch[CC_JIS2B7]
#define cc_EUCJP	cc_inswitch[CC_EUCJP]

#define setccin(ccx,cc)	{ \
	if( ccx->cc_in != ccx->cc_previn && JIS2B(ccx->cc_in) )\
		ccx->cc_previn = ccx->cc_in; \
	ccx->cc_in = cc; \
	ccx->cc_nonASCII += (cc != CC_ASCII); \
	ccx->cc_inswitch[cc] = 1; \
}

#define pushpending(ccx,ch) (ccx->cc_pinb[ccx->cc_pinx++] = ch)
#define EOB	-1
#define CHn(n) \
	((pilen <= pix+n) ? EOB : \
	(pix+n < pin) ? ccx->cc_pinb[pix+n] : cinb[pix+n-pin])
#define CH1 CHn(0)
#define CH2 CHn(0)
#define CH3 CHn(1)
#define CH4 CHn(2)

#define inNonASCII(ccx,code) \
	(  ccx->cc_in == code \
	|| ccx->cc_in == CC_ASCII && ccx->cc_previn == code )

#define CC_CHECKLEN	8
#define CC_BEEUC	4

/* distinguish SJIS from EUC */
static is_SJIS(ccx,cinb,pix,pin,pilen)
	CCX *ccx;
	unsigned char *cinb;
{	int cn,ci,ch1,ch2;
	int cs = -1;
	int n2K = 0,n1K = 0,n2B = 0;
	int nJ8 = 0,nSJ = 0,nEJ = 0;
	int as_sjis = 1;
	int is2B;

	if( ccx->cc_nonASCII == 0 )
		cn = 1024;
	else	cn = 128;

	for( ci = 0; ci < cn; ){
		ch1 = CHn(ci);
		ch2 = 0;
		ci++;
		if( ch1 == EOB )
			break;
		if( ch1 == 033 ){
			if( nJ8 == 0 ){
				cs = CC_JIS2B7;
				break;
			}
		}
		if( ch1 == '\r' || ch1 == '\n' ){
			if( 0 < nJ8 && ccx->cc_nonASCII != 0 )
				break;
		}

		if( (ch1 & 0x80) == 0 )
			continue;

		ch2 = CHn(ci);
		ci++;
		if( ch2 == EOB )
			break;

		is2B = 0;
		if( nJ8 == nEJ ){
			if( IS_EUC(ch1,ch2) ){
				is2B = 1;
				nEJ++;
				if( IS_EUC_HANKAKU(ch1,ch2) )
					n2K++;
			}
		}
		if( nJ8 == nSJ ){
			if( IS_SJIS_2B(ch1,ch2,as_sjis) ){
				is2B = 1;
				nSJ++;
				n2B++;
			}
			if( IS_SJIS_1B(ch1) ){
				nSJ++;
				n1K++;
				if( !is2B )
					ci--;
			}
		}
		nJ8++;
		if( nSJ != nEJ || nSJ < nJ8 || CC_CHECKLEN <= nJ8 )
			break;
	}
	if( 0 < nJ8 ){
		if( nSJ == nEJ ){
			if( inNonASCII(ccx,CC_SJIS) ) cs = CC_SJIS; else
			if( inNonASCII(ccx,CC_EUCJP) ) cs = CC_EUCJP; else
			if( nJ8 == nSJ ){
				if( ccx->cc_indef == CC_EUCJP
				 || ccx->cc_indef == CC_SJIS )
					cs = ccx->cc_indef;
				else
				/* long indistinguish SJIS is not likely to be */
				if( CC_BEEUC < nJ8 )
					cs = CC_EUCJP;
				else	cs = CC_SJIS;
			}
		}
		if( cs == -1 ){
			if( nJ8 == nSJ ){ cs = CC_SJIS; }else
			if( nJ8 == nEJ ){ cs = CC_EUCJP; }
		}
	}

	if( CCX_debug )
	fprintf(stderr,"#### %3d %3d <%02X %02X> J%d S%d(%d) E%d(%d) [%s]\n",
		ccx->cc_nonASCII,ci,ch1,ch2,
		nJ8, nSJ,n1K, nEJ,n2K,
		cs==CC_SJIS?"SJIS":(cs==CC_EUCJP?"EUC":"?"));

	return cs == CC_SJIS;
}

CCXwithSJIS(ccx)
	CCX *ccx;
{
	return ccx->cc_SJIS;
}
CCXclear(ccx)
	CCX *ccx;
{
	bzero(ccx,ccx->cc_size);
}
CCXactive(ccx)
	CCX *ccx;
{
	return ccx->cc_OUT != CC_THRU;
}
char *CCXcharset(ccx)
	CCX *ccx;
{
	if( ccx->cc_outx )
		return charsets[ccx->cc_outx].cs_formname;
	else	return 0;
}
char *CCXident(ccx)
	CCX *ccx;
{
	if( ccx->cc_UTF8 ) return "UTF-8";
	if( ccx->cc_SJIS ) return "Shift_JIS";
	if( ccx->cc_EUCJP) return "EUC-JP";
	if( ccx->cc_JIS7 ) return "ISO-2022-JP";
	return "US-ASCII";
}
static ccx_codex(ccn,cxp)
	char *ccn;
	int *cxp;
{	int cx,charcode;
	char *name;

	charcode = -1;
	for( cx = 1; name = charsets[cx].cs_name; cx++ ){
		if( strcaseeq(ccn,name)
		 || strcaseeq(ccn,charsets[cx].cs_formname) ){
			charcode = charsets[cx].cs_charcode;
			break;
		}
	}
	if( charcode == -1 ){
		cx = 0;
		switch( *ccn ){
			case 'u':           charcode = CC_UTF8;   break;
			case 'j': case 'J': charcode = CC_JIS2B7; break;
			case 'k': case 'K': charcode = CC_JIS7K;  break;
			case 's': case 'S': charcode = CC_SJIS;   break;
				  case 'U':
			case 'e': case 'E': charcode = CC_EUCJP;  break;
			case 't': case 'T': charcode = CC_THRU;   break;
			default: return -1;
		}
	}
	*cxp = cx;
	return charcode;
}
CCX_setincode(ccx,ccn)
	CCX *ccx;
	char *ccn;
{	int charcode,cx;

	charcode = ccx_codex(ccn,&cx);
	if( 0 <= charcode ){
		ccx->cc_in = charcode;
		ccx->cc_previn = charcode;
	}
}
int CCXcreate(from,to,ccx)
	char *from,*to;
	CCX *ccx;
{	int charcode,icc;
	int cx,icx;

	charcode = ccx_codex(to,&cx);
	if( charcode < 0 )
		return 0;

	bzero(ccx,sizeof(CCX));
	ccx->cc_size = sizeof(CCX);
	ccx->cc_OUT = charcode;
	ccx->cc_out = CC_ASCII;
	ccx->cc_outx = cx;

	icc = ccx_codex(from,&icx);
	if( icc < 0 )
		icc = CC_ASCII;
	ccx->cc_indef = icc;
	ccx->cc_in = icc;
	ccx->cc_previn = icc;

	ccx->cc_nonASCII = 0;
	return sizeof(CCX);
}
CCX *CCXnew(from,to)
	char *from,*to;
{	CCX ccxbuf,*ccx;
	int size;

	if( size = CCXcreate(from,to,&ccxbuf) ){
		ccx = (CCX*)malloc(size);
		bcopy(&ccxbuf,ccx,size);
		return ccx;
	}else	return NULL;
}

#define toJIS7(ccx,op) {\
	if( ccx->cc_out != CC_JIS2B7 ){ \
		*op++ = 033; \
		*op++ = '$'; \
		*op++ = 'B'; \
		ccx->cc_out = CC_JIS2B7; \
	}}

#define toJIS7K(ccx,op) {\
	if( ccx->cc_out != CC_JIS7K ){ \
		*op++ = 033; \
		*op++ = '('; \
		*op++ = 'I'; \
		ccx->cc_out = CC_JIS7K; \
	}}

#define toASCII(ccx,op) { \
	if( ccx->cc_out == CC_JIS2B7 || ccx->cc_out == CC_JIS7K ) \
	if( !JIS2B8(ccx->cc_OUT) ){ \
		*op++ = 033; \
		*op++ = '('; \
		*op++ = 'B'; \
	} \
	ccx->cc_out = CC_ASCII; }

#define IS_UTF8_CONT(ch)	((ch & 0xC0) == 0x80)

CCXexec(ccx,cinb,ilen,out,osiz)
	CCX *ccx;
	unsigned char *cinb,*out;
{	int ch1,ch2,ch3,ch4;
	unsigned char *pinb,*op;
	int pilen,pix,pin;
	int insjis;
	char *js;
	int cat;
	int codesw;
	int outlen;
	unsigned char *outtop,outbuf[2048];

	if( out == cinb ){
		if( osiz <= sizeof(outbuf) )
			outtop = outbuf;
		else	outtop = (unsigned char*)malloc(osiz); 
	}else	outtop = out;
	op = outtop;

	pin = ccx->cc_pinx; ccx->cc_pinx = 0;
	pinb = ccx->cc_pinb;
	pilen = pin + ilen;
	pix = 0;

	while( pix < pilen && (op-outtop) < osiz ){
		ch1 = CH1;
		pix++;
		ch2 = CH2;

		if( ch1 == 033 ){
			/* single ESC from keyboard input must be passed
			 * thru ...  */
			if( ch2 == EOB /* && not buffer full ... */ ){
				pushpending(ccx,ch1);
				break;
			}
			if( ch2 == '$' || ch2 == '(' ){
				ch3 = CH3;
				if( ch3 == EOB ){
					pushpending(ccx,ch1);
					pushpending(ccx,ch2);
					break;
				}

				codesw = 0;
				if( ch2 == '$' ){
					if( ch3 == 'B' || ch3 == '@' )
						codesw = CC_JIS2B7;
				}else
				if( ch2 == '(' ){
					if( ch3 == 'B' ){
						codesw = CC_ASCII;
					}else
					if( ch3 == 'J' ){
						codesw = CC_ASCII;
						/* CC_JIS7ROMA */
					}else
					if( ch3 == 'I' ){
						codesw = CC_JIS7KANA;
					}
				}
				if( codesw ){
					setccin(ccx,codesw);
					if( JIS2B8(ccx->cc_OUT)
					 || ccx->cc_OUT == CC_UTF8
					 || ccx->cc_OUT == CC_JIS2B7
					 && codesw == CC_JIS7KANA
					){
						pix += 2;
						continue;
					}
				}
			}
		}

		if( ccx->cc_in == CC_JIS7KANA )
		if( (ch1 & 0x80) == 0 && IS_SJIS_1B(ch1 | 0x80) ){
			switch( ccx->cc_OUT ){
			    case CC_JIS7K:
				*op++ = ch1;
				break;

			    case CC_JIS2B7:
				toJIS7(ccx,op);
				js = sjis_1b_to_jis(ch1,ch2,&cat);
				*op++ = js[0];
				*op++ = js[1];
				if( cat ) pix++;
				break;

			    case CC_EUCJP:
				*op++ = EUC_HANKAKU_HI;
				*op++ = 0x80 | ch1;
				break;

			    default:
				*op++ = 0x80 | ch1;
				break;
			}
			continue;
		}

		if( ccx->cc_in == CC_JIS2B7 && IS_JIS_HI(ch1) ){
			if( ch2 == EOB ){
				pushpending(ccx,ch1);
				break;
			}
			if( IS_JIS7(ch1,ch2) ){
				switch( ccx->cc_OUT ){
					case CC_SJIS:
						JIS_TO_SJIS1(ch1,ch2,op);
						op += 2;
						break;
					case CC_EUCJP:
						*op++ = 0x80 | ch1;
						*op++ = 0x80 | ch2;
						break;
					case CC_UTF8:
						op += EUCtoUTF8((ch1<<8)|ch2,op);
						break;
					default:
						*op++ = ch1;
						*op++ = ch2;
						break;
				}
				pix++;
				continue;
			}
		}

		if( !ccx->cc_SJIS && !ccx->cc_EUCJP )
		if( (ch1&0xF0) == 0xE0 ){
			int isUTF8 = 0,insch = 0;

			ch3 = CH3;
			ch4 = CH4;
			if( ccx->cc_UTF8 )
			if( ch2 == EOB || ch3 == EOB
			 || ch4 == EOB && (ch2 == '\n' || ch3 == '\n') ){
				pushpending(ccx,ch1);
				if( ch2 != EOB ) pushpending(ccx,ch2);
				if( ch3 != EOB ) pushpending(ccx,ch3);
				break;
			}

			if( ch2 == '\n'
			 && ch3 != EOB && IS_UTF8_CONT(ch3)
			 && ch4 != EOB && IS_UTF8_CONT(ch4)
			){
				syslog_DEBUG("##CCX UTF8-RECOVER[%x %x %x %x]\n",
					ch1,ch2,ch3,ch4);
				ch2 = ch3;
				ch3 = ch4;
				insch = CH2;
				isUTF8 = 1;
			}else
			if( ch3 == '\n'
			 && ch2 != EOB && IS_UTF8_CONT(ch2)
			 && ch4 != EOB && IS_UTF8_CONT(ch4)
			){
				syslog_DEBUG("##CCX UTF8-RECOVER[%x %x %x %x]\n",
					ch1,ch2,ch3,ch4);
				ch3 = ch4;
				insch = CH3;
				isUTF8 = 1;
			}else
			if( ch2 != EOB && IS_UTF8_CONT(ch2) )
			if( ch3 != EOB && IS_UTF8_CONT(ch3) )
			if( ch4 == EOB || (ch4&0x80)==0 || (ch4&0xF0)==0xE0 ){
				isUTF8 = 1;
			}
			if( isUTF8 ){
			    setccin(ccx,CC_UTF8);
			    switch( ccx->cc_OUT ){
				case CC_JIS2B7:
				    toJIS7(ccx,op);
				    op += UTF8toLocal(ch1,ch2,ch3,CC_JIS2B7,op);
				    break;
				case CC_EUCJP:
				    op += UTF8toLocal(ch1,ch2,ch3,CC_EUCJP,op);
				    break;
				case CC_SJIS:
				    op += UTF8toLocal(ch1,ch2,ch3,CC_SJIS,op);
				    break;
				default:
				    *op++ = ch1;
				    *op++ = ch2;
				    *op++ = ch3;
				    break;
			    }
			    pix += 2;
			    if( insch ){
				*op++ = insch;
				pix += 1;
			    }
			    continue;
			}
		}

		if( IS_SJIS_HI(ch1) || IS_EUC_HI(ch1) ){
			if( ch2 == EOB ){
				pushpending(ccx,ch1);
				break;
			}

			if( IS_SJIS_1B(ch1) )
			if( ccx->cc_in != CC_EUCJP )
			if( inNonASCII(ccx,CC_SJIS)
			 || is_SJIS(ccx,cinb,pix-1,pin,pilen) ){
				setccin(ccx,CC_SJIS);

				switch( ccx->cc_OUT ){
				    case CC_JIS7K:
					toJIS7K(ccx,op);
					*op++ = 0x7F & ch1;
					break;
				    case CC_JIS2B7:
					toJIS7(ccx,op);
					js = sjis_1b_to_jis(ch1,ch2,&cat);
					*op++ = js[0];
					*op++ = js[1];
					if( cat ) pix++;
					break;

				    case CC_EUCJP:
					*op++ = EUC_HANKAKU_HI;
					*op++ = ch1;
					break;

				    default:
					*op++ = ch1;
					break;
				}
				continue;
			}

			if( IS_EUC_HANKAKU(ch1,ch2) )
			if( inNonASCII(ccx,CC_EUCJP)
			 || !is_SJIS(ccx,cinb,pix-1,pin,pilen) ){
				setccin(ccx,CC_EUCJP);
				ch3 = CH3;
				ch4 = CH4;
				pix++;

				switch( ccx->cc_OUT ){
				    case CC_SJIS:
					*op++ = ch2;
					break;

				    case CC_JIS7K:
					toJIS7K(ccx,op);
					*op++ = 0x7F & ch2;
					break;

				    case CC_JIS2B7:
					ch1 = ch2;
					if( IS_EUC_HANKAKU(ch3,ch4) )
						ch2 = ch4;
					else	ch2 = -1;

					toJIS7(ccx,op);
					js = sjis_1b_to_jis(ch1,ch2,&cat);
					*op++ = js[0];
					*op++ = js[1];
					if( cat ) pix += 2;
					break;

				    default:
					*op++ = ch1;
					*op++ = ch2;
					break;
				}
				continue;
			}

			if( IS_EUC(ch1,ch2) )
			if( inNonASCII(ccx,CC_EUCJP)
			 || !is_SJIS(ccx,cinb,pix-1,pin,pilen) ){
				setccin(ccx,CC_EUCJP);
				pix++;

				switch( ccx->cc_OUT ){
				    case CC_JIS2B7:
				    case CC_JIS7K:
					toJIS7(ccx,op);
					*op++ = 0x7F & ch1;
					*op++ = 0x7F & ch2;
					break;

				    case CC_SJIS:
					JIS_TO_SJIS1(0x7F&ch1,0x7F&ch2,op);
					op += 2;
					break;

				    case CC_UTF8:
					op += EUCtoUTF8((ch1<<8)|ch2,op);
					break;

				    default:
					*op++ = ch1;
					*op++ = ch2;
					break;
				}
				continue;
			}

			insjis = inNonASCII(ccx,CC_SJIS);
			if( !insjis && ccx->cc_nonASCII == 0 )
				insjis = is_SJIS(ccx,cinb,pix-1,pin,pilen);

			if( IS_SJIS_2B(ch1,ch2,insjis) )
			if( inNonASCII(ccx,CC_SJIS)
			 || is_SJIS(ccx,cinb,pix-1,pin,pilen) ){
				setccin(ccx,CC_SJIS);
				pix++;

				switch( ccx->cc_OUT ){
				    case CC_JIS2B7:
				    case CC_JIS7K:
					toJIS7(ccx,op);
					SJIS_TO_JIS1(ch1,ch2,op);
					op += 2;
					break;

				    case CC_EUCJP:
					SJIS_TO_JIS1(ch1,ch2,op);
					*op++ |= 0x80;
					*op++ |= 0x80;
					break;

				    case CC_UTF8:
					SJIS_TO_JIS1(ch1,ch2,op);
					op += EUCtoUTF8((op[0]<<8)|op[1],op);
					break;

				    default:
					*op++ = ch1;
					*op++ = ch2;
					break;
				}
				continue;
			}
		}
		setccin(ccx,CC_ASCII);
		toASCII(ccx,op);
		*op++ = ch1;
	}
	if( ilen == 0 )
		toASCII(ccx,op);
	*op = 0;

	outlen = op - outtop;
	if( out == cinb ){
		bcopy(outtop,out,outlen+1);
		if( outtop != outbuf )
			free(outtop);
	}else	outlen = op - out;
	return outlen;
}

strip_ISO2022JP(str)
	char *str;
{	char ch,*sp,*dp;

	dp = str;
	for( sp = str; ch = *sp++; ){
		if( ch == 033 )
		if( *sp == TO_2BCODE && (sp[1]=='@' || sp[1]=='B')
		 || *sp == TO_1BCODE && (sp[1]=='J' || sp[1]=='B')
		){
			sp += 2;
			continue;
		}
		*dp++ = ch;
	}
	*dp = 0;
}


int utf8n;
int utf8err;

fixUTF8(us,leng)
	unsigned char *us;
{	int cx,uc4,ucs;
	int error;

	error = 0;
	for( cx = 1; cx < leng; cx++ ){
		uc4 = us[cx];
		if( (uc4&0xC0) != 0x80 ){
			if( uc4 == '\n' && (us[cx+1]&0xC0) == 0x80 ){
syslog_DEBUG("{#CCX:UTF-8:FIXED:%d/%d:%x,%x#}\n",cx+1,leng,uc4,us[cx+1]);
				us[cx] = us[cx+1];
				us[cx+1] = uc4;
			}else{
syslog_DEBUG("{#CCX:UTF-8:ERROR:%d/%d:%x,%x#}\n",cx+1,leng,uc4,us[cx+1]);
				error++;
				break;
			}
		}
	}
	return error;
}
fromUTF8(us,lengp,spp)
	unsigned char *us;
	int *lengp;
	char **spp;
{	unsigned char uc0,ucx;
	int leng,mask,lx;
	unsigned int uc4;

	uc4 = 0;
	uc0 = *us;
	if( (uc0 & 0x80) == 0 ){ leng = 1; }else
	if( (uc0 & 0x40) == 0 ){ leng = 1;
			if( spp ){
sprintf(*spp,"{#CCX:UTF-8:ERROR:%d/%d:%x,%x#}",1,1,uc0,us[1]);
			utf8err++;
			*spp += strlen(*spp);
			uc0 = '?';
			}
	}else
	if( (uc0 & 0x20) == 0 ){ leng = 2; }else
	if( (uc0 & 0x10) == 0 ){ leng = 3; }else
	if( (uc0 & 0x08) == 0 ){ leng = 4; }else
	if( (uc0 & 0x04) == 0 ){ leng = 5; }else
	if( (uc0 & 0x02) == 0 ){ leng = 6; }else{
		leng = 1;
	}

	fixUTF8(us,leng);
	if( leng == 1 ){
		uc4 = uc0;
	}else{
		mask = 0xFF >> (leng+1);
		uc4 = uc0 & mask;
		for( lx = 1; lx < leng; lx++ ){
			ucx = us[lx];
			if( (ucx & 0xC0) != 0x80 ){
				return uc0;
			}
			uc4 = (uc4 << 6) | (0x3F & ucx);
		}
	}
	*lengp = leng;
	return uc4;
}
toUTF8(uc,us)
	unsigned int uc;
	unsigned char *us;
{	int tag,leng,lx;

	if( uc < 0x0000080 ){ leng = 1; }else
	if( uc < 0x0000800 ){ leng = 2; }else
	if( uc < 0x0010000 ){ leng = 3; }else
	if( uc < 0x0200000 ){ leng = 4; }else
	if( uc < 0x4000000 ){ leng = 5; }else
			      leng = 6;
	if( leng == 1 ){
		*us++ = uc;
	}else{
		for( lx = leng-1; 0 < lx; lx-- ){
			us[lx] = 0x80 | (uc & 0x3F);
			uc = uc >> 6;
		}
		tag = 0x3F << (8-leng);
		us[0] = tag | uc;
	}
	return leng;
}

#define JUMAP	"sjis-jis-ucs.ccx"
typedef struct {
unsigned short	u_l2umap[0x10000];
unsigned short	u_u2lmap[0x10000];
	int	u_init;
} UcsX;
#define L2Umap	ucx->u_l2umap
#define U2Lmap	ucx->u_u2lmap
static UcsX *UcsXtab[8];
UcsX *UCSx(){
	if( UcsXtab[0] == 0 )
		UcsXtab[0] = (UcsX*)calloc(sizeof(UcsX),1);
	return UcsXtab[0];
}
static UcxInit(ucx)
	UcsX *ucx;
{	unsigned short l2ucnt[0x10000];
	unsigned short u2lcnt[0x10000];

	if( ucx->u_init == 0 ){
		ucx->u_init = 1;
		bzero(l2ucnt,sizeof(l2ucnt));
		bzero(u2lcnt,sizeof(u2lcnt));
		loadUnicodeMappings(JUMAP,ucx,l2ucnt,u2lcnt);
	}
}
UCSinit(){
	UcsX *ucx = UCSx();
	UcxInit(ucx);
}

static UTF8toLocal(ch1,ch2,ch3,charset,op)
	unsigned char *op;
{	UcsX *ucx = UCSx();
	unsigned char buf[8];
	int len;
	unsigned int ucs,euc;

	UcxInit(ucx);
	buf[0] = ch1;
	buf[1] = ch2;
	buf[2] = ch3;
	buf[3] = 0;
	ucs = fromUTF8(buf,&len,NULL);
	euc = U2Lmap[ucs];
	if( euc  ){
		switch( charset ){
			case CC_JIS2B7:
				*op++ = 0x7F & (euc >> 8);
				*op++ = 0x7F & euc;
				len = 2;
				break;
			case CC_EUCJP:
				*op++ = 0x80 | (euc >> 8);
				*op++ = 0x80 | euc;
				len = 2;
				break;
			case CC_SJIS:
				*op++ = euc >> 8;
				*op++ = euc;
				len = 2;
				break;
			default:
				len = 0;
				break;
		}
		*op = 0;
		return len;
	}else{
		switch( charset ){
			case CC_JIS2B7:
				sprintf(op,"\".");
				break;
			default:
sprintf(op,  "{#CCX:UTF-8:NOMAP:UTF8toLocal:0x%04x:%x,%x,%x#}",ucs,ch1,ch2,ch3);
				break;
		}
		return strlen(op);
	}
}
EUCtoUTF8(euc,us)
	char *us;
{	UcsX *ucx = UCSx();
	unsigned int ucs;
	int len;

	UcxInit(ucx);
	ucs = L2Umap[euc];
	if( ucs == 0 ){
		if( (euc & 0x8080) == 0 )
			ucs = L2Umap[ 0x8080 | euc];
		else	ucs = L2Umap[~0x8080 & euc];
	}
	if( ucs ){
		len = toUTF8(ucs,us);
		return len;
	}else{
sprintf(us,  "{#CCX:UTF-8:NOMAP:LocalToUTF8:%x#}",euc);
		return strlen(us);
	}
	return 0;
}

static dumpCharMapCompiled(file,ucx)
	char *file;
	UcsX *ucx;
{	FILE *mfp;
	int lch,uch,rcc;
	unsigned char buff[0x40000],*bp;

	if( mfp = fopen(file,"w") ){
		bp = buff;
		for( lch = 0; lch < 0x10000; lch++ )
		if( uch = L2Umap[lch] ){
			*bp++ = lch >> 8; *bp++ = lch;
			*bp++ = uch >> 8; *bp++ = uch;
		}
		fwrite(buff,1,bp-buff,mfp);
		syslog_DEBUG("##CCX DUMPED %s %d\n",file,bp-buff);
		fclose(mfp);
		chmodIfShared(file,0644);
	}
}
static loadCharMap(file,ucx)
	char *file;
	UcsX *ucx;
{	FILE *mfp;
	char xpath[1024];
	int rcc;
	unsigned int lch,uch;
	unsigned char buff[0x40000],*bp;

	if( mfp = fopen_LIBPATH(file,"r",xpath) ){
		rcc = fread(buff,1,sizeof(buff),mfp);
		syslog_DEBUG("##CCX LOADED %s %d\n",xpath,rcc);
		for( bp = buff; bp < &buff[rcc]; bp += 4 ){
			lch = (bp[0] << 8) | bp[1];
			uch = (bp[2] << 8) | bp[3];
			L2Umap[lch] = uch;
			U2Lmap[uch] = lch;
		}
		fclose(mfp);
		return 1;
	}
	return 0;
}
static statDump(l2umap,l2ucnt)
	unsigned short l2umap[],l2ucnt[];
{	int filled,unified;
	unsigned int uc4,lc4;

	unified = 0;
	for( lc4 = 0; lc4 < 0x10000; lc4++ ){
		if( 0 < l2umap[lc4] && l2umap[lc4] < 0x100 ){
/*
syslog_ERROR("##CCX UNIFIED: %x -> %x\n",lc4,l2umap[lc4]);
*/
			unified++;
		}
	}
	filled = 0;
	for( uc4 = 0; uc4 < 0x10000; uc4++ ){
		if( 1 < l2ucnt[uc4] ){
/*
syslog_ERROR("##CCX DUPLICATE: %x (%d)\n",uc4,l2ucnt[uc4]);
*/
		}
		if( l2ucnt[uc4] == 1 )
			filled++;
	}
syslog_ERROR("##CCX UNIFIED=%d FILLED=%d\n",unified,filled);
}
static loadMapping1(fp,file,ucx,l2ucnt,u2lcnt)
	FILE *fp;
	char *file;
	UcsX *ucx;
	unsigned short l2ucnt[],u2lcnt[];
{	char line[1024];
	unsigned int lc1,lc2,uc1;

	while( fgets(line,sizeof(line),fp) ){
		if( sscanf(line,"0x%x 0x%x 0x%x",&lc1,&lc2,&uc1) == 3 ){
			lc1 = lc2 | 0x8080;
		}else
		if( sscanf(line,"0x%x 0x%x",&lc1,&uc1) != 2 )
			continue;
/*
		if( 0x80 <= uc1 )
*/
		{
			if( L2Umap[lc1] && L2Umap[lc1] != uc1 ){
				syslog_ERROR("##CCX L2U OVR: %x -> %x %x\n",
					lc1,L2Umap[lc1],uc1);
			}
			l2ucnt[lc1] += 1;
			L2Umap[lc1] = uc1;

			if( U2Lmap[uc1] && U2Lmap[uc1] != lc1 ){
				syslog_ERROR("##CCX U2L OVR: %x -> %x %x\n",
					uc1,U2Lmap[uc1],lc1);
			}
			u2lcnt[uc1] += 1;
			U2Lmap[uc1] = lc1;
		}
	}
	syslog_ERROR("##CCX Local to Unicode:\n"); statDump(L2Umap,l2ucnt);
	syslog_ERROR("##CCX Unicode to Local:\n"); statDump(U2Lmap,u2lcnt);
	syslog_ERROR("##CCX loaded: %s\n",file);
	return 1;
}
static loadMap1(file,ucx,l2ucnt,u2lcnt)
	char *file;
	UcsX *ucx;
	unsigned short l2ucnt[],u2lcnt[];
{	FILE *fp;
	char xpath[1024];

	if( fp = fopen_LIBPATH(file,"r",xpath) ){
		loadMapping1(fp,xpath,ucx,l2ucnt,u2lcnt);
		fclose(fp);
		return 1;
	}
	return 0;
}
extern char *strrpbrk();
static loadUnicodeMappings(jumap,ucx,l2ucnt,u2lcnt)
	UcsX *ucx;
	char *jumap;
	unsigned short l2ucnt[],u2lcnt[];
{	FILE *fp,*mfp;
	char *map,*getenv();
	int loaded = 0;
	char bpath[1024],*tmpdir,*xp;

	if( isFullpath(jumap) )
		strcpy(bpath,jumap);
	else{
		tmpdir = getTMPDIR();
		if( tmpdir == 0 )
			tmpdir = "/tmp";
		sprintf(bpath,"%s/%s",tmpdir,jumap);
	}
	if( (xp = strrpbrk(bpath,"./\\")) && *xp == '.' )
		strcpy(xp,".ccb");
	else	strcat(bpath,".ccb");
	syslog_ERROR("##CCX %s %s\n",jumap,bpath);

	if( loadCharMap(bpath,ucx) ){
		loaded = 1;
		goto EXIT;
	}

	if( map = getenv("MAPUNICODE") )
		loaded += loadMap1(map,ucx,l2ucnt,u2lcnt);

	loaded += loadMap1(jumap,ucx,l2ucnt,u2lcnt);
/*	loaded += loadMap1("JIS0208.TXT",ucx,l2ucnt,u2lcnt); */
/*	loaded += loadMap1("JIS0201.TXT",ucx,l2ucnt,u2lcnt); */

	if( loaded )
		dumpCharMapCompiled(bpath,ucx);

EXIT:
	return loaded;
}

static scanCharMapping(in,ucx,l2ucnt,u2lcnt)
	FILE *in;
	UcsX *ucx;
	unsigned short l2ucnt[],u2lcnt[];
{	char line[1024];
	char map1[1024];
	char *sp;
	unsigned int uc4,lc4;
	int len;

	while( fgets(line,sizeof(line),in) ){
	    for( sp = line; sp && *sp; sp = strpbrk(sp," \t\r\n") ){
		if( isspace(*sp) )
			sp++;
		*map1 = 0;
		sscanf(sp,"%s",map1);
		if( *map1 == 0 )
			break;
		if( isxdigit(map1[0]) && map1[4] == ':' ){
			map1[4] = 0;
			sscanf(map1,"%x",&lc4);
			uc4 = fromUTF8(&map1[5],&len,NULL);

if( (lc4 & 0x8080) == 0 && (lc4 & 0xFF00) != 0 ) lc4 |= 0x8080;

			l2ucnt[uc4] += 1;
			if( l2ucnt[uc4] == 1 )
				L2Umap[lc4] = uc4;
			u2lcnt[lc4] += 1;
			if( u2lcnt[lc4] == 1 )
				U2Lmap[uc4] = lc4;
		}
	    }
	}
	statDump(L2Umap,l2ucnt);
}
dumpUTF8mapping(){
	unsigned int uc4;
	for( uc4 = 0x001; uc4 < 0x080; uc4++ ){
		printf("%04x:%c\r\n",uc4,uc4);
	}
	for( uc4 = 0x080; uc4 < 0x800; uc4++ ){
		printf("%04x:%c%c\r\n",
			uc4,
			0xC0|0x1F&(uc4>>6),
			0x80|0x3F&(uc4)
		);
	}
	for( uc4 = 0x800; uc4 < 0x10000; uc4++ ){
		printf("%04x:%c%c%c\r\n",
			uc4,
			0xE0|0x0F&(uc4>>12),
			0x80|0x3F&(uc4>>6),
			0x80|0x3F&(uc4)
		);
	}
}
dumpJIS7mapping()
{	int ch,ch1,ch2,nc;

	nc = 0;
	for( ch = 0; ch < 0x8000; ch++ ){
		ch1 = 0xFF & (ch >> 8);
		ch2 = 0xFF & ch;
		if( !IS_JIS7(ch1,ch2) )
			continue;
		if( nc != 0 && nc % 8 == 0 )
			printf("\r\n");
		nc++;
		printf("%02x%02x:\033$B%c%c\033(B ",ch1,ch2,ch1,ch2);
	}
	printf("\r\n");
}
dumpEUCJPmapping()
{	int ch,ch1,ch2,nc;

	for( ch = 0x8080; ch < 0x10000; ch++ ){
		ch1 = 0xFF & (ch >> 8);
		ch2 = 0xFF & ch;
		if( !IS_EUC(ch1,ch2) )
			continue;
		if( nc != 0 && nc % 8 == 0 )
			printf("\r\n");
		nc++;
		printf("%02x%02x:%c%c ",ch1,ch2,ch1,ch2);
	}
	printf("\r\n");
}
dumpCharMapping(code)
	char *code;
{
	switch( *code ){
		case 'u': dumpUTF8mapping(); break;
		case 'j': dumpJIS7mapping(); break;
		case 'e': dumpEUCJPmapping(); break;
	}
}
loadCharMapping(code,ifp)
	char *code;
	FILE *ifp;
{	UcsX *ucx = UCSx();
	unsigned short l2ucnt[0x10000];
	unsigned short u2lcnt[0x10000];

	bzero(l2ucnt,sizeof(l2ucnt));
	bzero(u2lcnt,sizeof(u2lcnt));
	scanCharMapping(ifp,ucx,L2Umap,l2ucnt,U2Lmap,u2lcnt);
	dumpCharMapCompiled(JUMAP,ucx);
}


int repair_JIS = 0;
#ifdef XXXXXXXXXXXXXXXXXX /* OBSOLETE */
static unsigned char *repairJIStoEUC(src,dstp)
	unsigned char *src,**dstp;
{	unsigned char *s,*d,ch1,ch2;

	d = *dstp;
	s = src;
	while( (ch1 = s[0]) && (ch2 = s[1]) ){
		s += 2;
		if( ch1 == '(' )
		if( ch2 == 'B' || ch2 == 'J' ){
			*dstp = d;
			return s;
		}
		if( !IS_JIS7(ch1,ch2) )
			return 0;

		*d++ = 0x80 | ch1;
		*d++ = 0x80 | ch2;
	}
	return 0;
}
int TREAT_SJIS = 1;
unsigned char *
SJIS_TO_EUC(src,dst)
	unsigned char *src,*dst;
{	register unsigned char hi,lo,*sp,*dp;
	register int in_sjis = 0;

	in_sjis = IS_SJIS_STR(src);
	for(sp=src, dp=dst; hi = sp[0]; ){
		lo = sp[1];
		if( TREAT_SJIS && IS_SJIS_2B(hi,lo,in_sjis) ){
			SJIS_TO_JIS1(hi,lo,dp);
			dp[0] |= 0x80;
			dp[1] |= 0x80;
			dp += 2;
			sp += 2;
		}else	*dp++ = *sp++;
	}
	*dp = 0;
	return dst;
}
unsigned char *
EUC_TO_SJIS(src,dst)
	unsigned char *src,*dst;
{	register unsigned char *sp,*dp;

	for(sp=src, dp=dst; *sp; ){
		if( *sp & 0x80 ){
			if( IS_EUC_HANKAKU(sp[0],sp[1]) ){
				*dp++ = sp[1];
				sp += 2;
			}else
			if( sp[1] && (sp[1] & 0x80) ){
				JIS_TO_SJIS1(sp[0]&0x7F,sp[1]&0x7F,dp);
				dp += 2; sp += 2;
			}else	sp++;
		}else	*dp++ = *sp++;
	}
	*dp = 0;
	return dst;
}
TO_SJIS(any,sjis,ctype)
	char *any,*sjis;
	char *ctype;
{	char *euc;

	euc = malloc(strlen(any)*2+1);
	TO_EUC(any,euc,ctype);
	if( is_EUC_JP(euc) )
		EUC_TO_SJIS(euc,sjis);
	else	strcpy(sjis,any);
	free(euc);
}
TO_JIS(any,jis,ctype)
	char *any;
	unsigned char *jis;
	char *ctype;
{	char *euc;

	if( any[0] == 0 ){
		jis[0] = 0;
		return;
	}

	euc = malloc(strlen(any)*2+1);
	TO_EUC(any,euc,ctype);

	/* if NOT old-Netscape ??? DeleGate/2.1.5-DeleGate/2.1.9 */
	if( is_EUC_JP(euc) )
		EUC_TO_JIS(euc,jis,TO_KANJI,TO_ASCII);
	else	strcpy(jis,any);
	free(euc);
}
#define Strcpy(a,b)	(strcpy((char*)a,(char*)b),&a[strlen((char*)a)])
unsigned char *
EUC_TO_JIS(src,dst,toK,toA)
	unsigned char *src,*dst;
	char *toK,*toA;
{	register unsigned char kana_mode = 0;
	register unsigned char cch;
	register unsigned char *sp = src;
	register unsigned char *dp = dst;
	int is_JIS = 0;

	while( cch = *sp++ ){
		if( cch & 0x80 ){
			if( !IS_EUC(cch,*sp) ){
				if( cch == 0xA0 && is_JIS ) /* ignore NBSP */
					continue;
				is_JIS++;
				*dp++ = cch;
				continue;
			}
			if(!kana_mode){
				kana_mode = ~kana_mode;
				dp = Strcpy(dp,toK);
			}
			if( IS_EUC_HANKAKU(cch,*sp) ){
				unsigned char *sp1,*dp1;
				sp1 = sp - 1;
				dp1 = dp;
				EUC_hankaku_TO_JIS(&sp1,&dp1);
				sp = sp1;
				dp = dp1;
			}else
			if( *sp & 0x80 ){
				*dp++ = cch & ~0x80;
				*dp++ = *sp++ & ~0x80;
			}
		}else{
			if(kana_mode){
				kana_mode = ~kana_mode;
				dp = Strcpy(dp,toA);
			}
			*dp++ = cch;
		}
	}
	if(kana_mode)
		dp = Strcpy(dp,toA);

	if(dp)	*dp = 0;
	return dst;
}
unsigned char *
SJIS_TO_EUC1(HI,LO,EUC)
	unsigned char HI,LO,*EUC;
{
	SJIS_TO_JIS1(HI,LO,EUC);
	EUC[0] |= 0x80;
	EUC[1] |= 0x80;
	return EUC;
}
unsigned char*
TO_EUC(jis,euc,ctype)
	unsigned char *jis,*euc;
	char *ctype;
{	register unsigned char *s,c,jis_stat;
	register to1B,to2B;
	register int in_sjis = 0;
	int in_1Bkana = 0;
	unsigned char *d;
	static int nje;
	int n8bits;
	int is_JIS;

	nje++;
	n8bits = 0;
	s = jis;
	d = euc;
	jis_stat = 0;
	to2B = TO_2BCODE;
	to1B = TO_1BCODE;
	in_sjis = IS_SJIS_STR(jis);
	is_JIS = 0;

	while( c = *s++ ){
		if( c == 0x80 )
			continue; /* ignore it */
		if( c == 0xA0 && is_JIS )
			continue; /* ignore Non-breaking space */

		if( c == to2B && jis_stat == 0 && repair_JIS ){
			if(*s == 'B' || *s == '@'){
				unsigned char *ts;
				if( ts = repairJIStoEUC(s+1,&d) ){
					s = ts;
					continue;
				}
			}
		}
		if( c == ESC ){
			in_1Bkana = 0;
			if(*s == to2B){
				if((s[1] == 'B') || (s[1] == '@')){
					jis_stat = 0x80; s += 2;
					is_JIS++;
					continue;
				}
				jis_stat = 0;
			}else
			if(*s == to1B){
				jis_stat = 0;
				if((s[1]=='B') || (s[1]=='J') || (s[1]=='H')){
					s += 2;
					continue;
				}
				if( s[1]=='I' ){
					s += 2;
					is_JIS++;
					in_1Bkana = 1;
					continue;
				}
			}else
			if(*s == ',') /* MULE */ {
				jis_stat = 0;
			}
		}
		if( is_JIS ){
			if( c == SO ){
				in_1Bkana = 1;
				continue;
			}
			if( c == SI ){
				in_1Bkana = 0;
				continue;
			}
		}

		if( c & 0x80 )
			n8bits++;

		if( !in_sjis && c == EUC_HANKAKU_HI && IS_SJIS_1B(*s) ){
			*d++ = c;
			*d++ = *s++;
		}else
		if( in_sjis&&IS_SJIS_1B(c) || in_1Bkana&&IS_SJIS_1B(0x80|c) ){
			d[0] = EUC_HANKAKU_HI;
			d[1] = c | 0x80;
			d += 2;
			is_JIS++;
		}else
		if( IS_SJIS_2B(c,*s,in_sjis) ){
			SJIS_TO_EUC1(c,*s,d);
			d += 2;
			s++;
			is_JIS++;
		}else
		if( jis_stat ){
			if( c <= 0x20 || 0x7F <= c ){
				*d++ = c;
				if( c == '\n' )
					jis_stat = 0;
			}else{
				if( IS_JIS7(c,*s)){
					*d++ = jis_stat | c;
					*d++ = jis_stat | *s++;
				}else	*d++ = c;
			}
		}else{
			if( n8bits == 0 && (c == SI || c == SO) ){
			}else{
				*d++ = c;
			}
		}
	}
	*d = 0;
	return euc;
}
#endif

CCX_TOXX(src,slen,dst,dsiz,ctype,chset)
	unsigned char *src,*dst;
	char *ctype,*chset;
{	CCX ccx;

	CCXcreate("*",chset,&ccx);
	CCXexec(&ccx,src,slen,dst,dsiz);
}
CCX_TOX(src,dst,ctype,chset)
	unsigned char *src,*dst;
	char *ctype,*chset;
{	int len;

	len = strlen(src);
	CCX_TOXX(src,len,dst,1024+len*2,ctype,chset);
}
TO_EUC(any,euc,ctype)
	char *any,*euc,*ctype;
{
	CCX_TOX(any,euc,ctype,"euc");
}
TO_JIS(any,jis,ctype)
	char *any,*jis,*ctype;
{
	CCX_TOX(any,jis,ctype,"jis");
}
TO_SJIS(any,sjis,ctype)
	char *any,*sjis,*ctype;
{
	CCX_TOX(any,sjis,ctype,"sjis");
}
TO_UTF8(any,utf8,ctype)
	char *any,*utf8,*ctype;
{
	CCX_TOX(any,utf8,ctype,"utf8");
}
