/*////////////////////////////////////////////////////////////////////////
Copyright (c) 1998 Electrotechnical Laboratry (ETL), AIST, MITI
Copyright (c) 1998 Yutaka Sato

Permission to use, copy, modify, and distribute this material for any
purpose and without fee is hereby granted, provided that the above
copyright notice and this permission notice appear in all copies.
ETL MAKES NO REPRESENTATIONS ABOUT THE ACCURACY OR SUITABILITY OF THIS
MATERIAL FOR ANY PURPOSE.  IT IS PROVIDED "AS IS", WITHOUT ANY EXPRESS
OR IMPLIED WARRANTIES.
/////////////////////////////////////////////////////////////////////////
Content-Type:	program/C; charset=US-ASCII
Program:	sslway.c (SSL encoder/decoder with SSLeay/openSSL)
Author:		Yutaka Sato <ysato@etl.go.jp>
Description:

  Given environment:
    file descriptor 0 is a socket connected to a client
    file descriptor 1 is a socket connected to a server

  Commandline argument:
    -cert file -- certificate (can be with private key) of this SSLway
                  to be shown to a peer

    -CAfile file -- the name of file contains a CA's certificate
    -CApath dir -- directory contains CA's certificate files each named with
                   `X509 -hash -noout < certificate.pem`

    -Vrfy -- peer's certificate must be shown and must be authorized
    -vrfy -- peer's certificate, if shown, must be authorized
    -Auth -- peer must show its certificate, but it can be unauthorized
    -auth -- just record the peer's certificate, if exists, into log
             -- equals to obsoleted "-client_auth".

    -ht  through pass if the request is in bare HTTP protocol (GET,HEAD,POST)

    Following options can be omitted when the sslway is called from DeleGate
    with FCL, FSV or FMD since it will be detected automatically.

    -co  apply SSL for the connection to the server
    -ac  aplly SSL for the accepted connection from the client

  Usage:
    delegated FSV=sslway
    delegated FCL=sslway ...

  How to make:
    - do make at .. or ../src directory
    - edit Makefile.go to let SSLEAY points to the directory of libssl.a
    - make -f Makefile.go sslway

History:
	980412	created
	980428	renamed from "sslrelay" to "sslway"
//////////////////////////////////////////////////////////////////////#*/
#include <stdio.h>
#include <string.h>
extern char *getenv();

static int debugging = 0;
static char *client_host;
static int PID;
static ERROR(fmt,a,b,c,d,e,f,g)
	char *fmt,*a,*b,*c,*d,*e,*f,*g;
{	char msg[0x4000];

	sprintf(msg,"## SSLway[%d](%s) ",PID,client_host);
	sprintf(msg+strlen(msg),fmt,a,b,c,d,e,f,g);
	syslog_ERROR("%s\n",msg);
}
#define DEBUG	(debugging==0)?0:ERROR

/*
#include "ssl.h"
*/
#define SSL_FILETYPE_PEM 1
#define SSL_VERIFY_NONE			0x00
#define SSL_VERIFY_PEER			0x01
#define SSL_VERIFY_FAIL_IF_NO_PEER_CERT	0x02
#define SSL_VERIFY_CLIENT_ONCE		0x04

typedef void SSL_CTX;
typedef void SSL_METHOD;
typedef void SSL;
typedef void X509;
#define BIO_NOCLOSE 0
X509 *PEM_read_bio_X509();
typedef void BIO;
BIO *BIO_new_fp();
X509 *SSL_get_peer_certificate();
SSL_METHOD *SSLv2_method();
SSL_METHOD *SSLv3_method();
SSL_METHOD *SSLv3_server_method();
SSL_METHOD *SSLv3_client_method();
SSL_METHOD *SSLv23_server_method();
SSL_METHOD *SSLv23_client_method();
SSL_CTX *SSL_CTX_new();
SSL *SSL_new();
char *X509_NAME_oneline();
void *X509_verify_cert_error_string();
void *X509_STORE_CTX_get_current_cert();
void *X509_get_subject_name();

static SSL_CTX *ssl_new(serv)
{	SSL_CTX *ctx;
	SSL_METHOD *meth;

	SSL_library_init();
	SSL_load_error_strings();
	if( serv )
		meth = SSLv23_server_method();
	else	meth = SSLv23_client_method();
	ctx = SSL_CTX_new(meth);
	return ctx;
}
static ssl_printf(ssl,fd,fmt,a,b,c,d,e,f,g)
	SSL *ssl;
	char *fmt,*a,*b,*c,*d,*e,*f,*g;
{	char buf[0x4000];

	sprintf(buf,fmt,a,b,c,d,e,f,g);
	if( ssl )
		SSL_write(ssl,buf,strlen(buf));
	else	write(fd,buf,strlen(buf));
}
static ssl_prcert(ssl,show,outssl,outfd,what)
	SSL *ssl;
	char *what;
{	X509 *peer;
	char subjb[256],*sb,issrb[256],*is;

	if( peer = SSL_get_peer_certificate(ssl) ){
		sb = X509_NAME_oneline(X509_get_subject_name(peer),subjb,1024);
		is = X509_NAME_oneline(X509_get_issuer_name(peer),issrb,1024);
		if( show ){
			ssl_printf(outssl,outfd,
				"##[SSLway: %s's certificate]\r\n",what);
			ssl_printf(outssl,outfd,"## Subject: %s\r\n",sb);
			ssl_printf(outssl,outfd,"## Issuer: %s\r\n",is);
		}
		ERROR("%s's cert. = **subject<<%s>> **issuer<<%s>>",what,sb,is);
		X509_free(peer);
	}else{
		ERROR("%s's cert. = NONE",what);
	}
}
static SSL *ssl_conn(ctx,confd)
	SSL_CTX *ctx;
{	SSL *conSSL;

	conSSL = SSL_new(ctx);
	SSL_set_connect_state(conSSL);
	SSL_set_fd(conSSL,SocketOf(confd));
	if( SSL_connect(conSSL) < 0 ){
		ERROR("connect failed");
		ERR_print_errors_fp(stderr);
		return NULL;
	}else{
		ERROR("connected");
		return conSSL;
	}
}
static SSL *ssl_acc(ctx,accfd)
	SSL_CTX *ctx;
{	SSL *accSSL;

	accSSL = SSL_new(ctx);
	SSL_set_accept_state(accSSL);
	SSL_set_fd(accSSL,SocketOf(accfd));
	if( SSL_accept(accSSL) < 0 ){
		ERROR("accept failed");
		ERR_print_errors_fp(stderr);
		ssl_printf(accSSL,0,"SSLway: accept failed\n");
		return NULL;
	}else{
		ERROR("accepted");
		return accSSL;
	}
}
static ssl_setCAs(ctx,file,dir)
	SSL_CTX *ctx;
	char *file,*dir;
{	char xfile[1024],xdir[1024];

	LIBFILE_IS(file,xfile);
	LIBFILE_IS(dir, xdir);

	if( !SSL_CTX_load_verify_locations(ctx,xfile,xdir)
	 || !SSL_CTX_set_default_verify_paths(ctx) ){
		ERROR("CAs not found or wrong: [%s][%s]",xfile,xdir);
	}else{
		ERROR("CAs = [%s][%s]",xfile,xdir);
	}
}

typedef struct {
	char	*x_cert;	/* cetificate file */
	char	*x_key;		/* private key */
	char	*x_pass;	/* to decrypt the cert */
	char	*x_CApath;	/* CA's certificates */
	char	*x_CAfile;	/* A CA's certificate */
	int	 x_do_SSL;	/* use SSL */
	int	 x_verify;
} SSLContext;

static char cl_cert_default[] = "client-cert.pem";
static char cl_key_default[] = "client-key.pem";

static SSLContext sslctx[2] = {
	{ "server-cert.pem", "server-key.pem" },
	{ cl_cert_default, cl_key_default },
};
static int   acc_bareHTTP = 0;
static int   verify_depth = -1;
static int   do_showCERT = 0;
static char *cipher_list = NULL;

#define XACC	0
#define XCON	1

#define sv_cert		sslctx[XACC].x_cert
#define sv_key		sslctx[XACC].x_key
#define sv_pass		sslctx[XACC].x_pass
#define cl_CApath	sslctx[XACC].x_CApath
#define cl_CAfile	sslctx[XACC].x_CAfile
#define do_accSSL	sslctx[XACC].x_do_SSL
#define cl_vrfy		sslctx[XACC].x_verify

#define cl_cert		sslctx[XCON].x_cert
#define cl_key		sslctx[XCON].x_key
#define cl_pass		sslctx[XCON].x_pass
#define sv_CApath	sslctx[XCON].x_CApath
#define sv_CAfile	sslctx[XCON].x_CAfile
#define do_conSSL	sslctx[XCON].x_do_SSL
#define sv_vrfy		sslctx[XCON].x_verify

static ssl_keycert(ctx,keyfile,certfile,clnt)
	SSL_CTX *ctx;
	char *keyfile,*certfile;
{	int code = 0;
	char cwd[1024];
	char xkeyfile[1024],xcertfile[1024];
	char *dp,passfile[1024],pass[128];
	FILE *fp;

	if( LIBFILE_IS(keyfile,xkeyfile) )
		keyfile = xkeyfile;
	if( LIBFILE_IS(certfile,xcertfile) )
		certfile = xcertfile;

	if( dp = strrchr(keyfile,'.') ){
		strcpy(passfile,keyfile);
		dp = strrchr(passfile,'.');
		strcpy(dp,".pas");
		if( fp = fopen(passfile,"r") ){
			fgets(pass,sizeof(pass),fp);
			fclose(fp);
			if( dp = strpbrk(pass,"\r\n") )
				*dp = 0;
			if( clnt )
				cl_pass = strdup(pass);
			else	sv_pass = strdup(pass);
		}
	}

	getcwd(cwd,sizeof(cwd));
	if( SSL_CTX_use_RSAPrivateKey_file(ctx,keyfile,SSL_FILETYPE_PEM) ){
		DEBUG("keyfile loaded: %s",keyfile);
	}else{
		ERROR("keyfile not found or wrong: %s [at %s]",keyfile,cwd);
		code = -1;
	}
	if( SSL_CTX_use_certificate_file(ctx,certfile,SSL_FILETYPE_PEM) ){
		DEBUG("certfile loaded: %s",keyfile);
	}else{
		ERROR("certfile not found or wrong: %s [at %s]",certfile,cwd);
		code = -1;
	}
	return code;
}
static verify_callback(ok,ctx)
	void *ctx;
{	int err,depth;
	char *errsym;
	void *cert;
	char subjb[256];

	cert =   X509_STORE_CTX_get_current_cert(ctx);
	err =    X509_STORE_CTX_get_error(ctx);
	depth =  X509_STORE_CTX_get_error_depth(ctx);
	X509_NAME_oneline(X509_get_subject_name(cert),subjb,sizeof(subjb));
	errsym = X509_verify_cert_error_string(err);
	ERROR("depth=%d/%d %d:\"%s\" %s",depth,verify_depth,err,errsym,subjb);

	if( !ok ){
		if( depth <= verify_depth )
			ok = 1;
	}
	return ok;
}
static ssl_relay(accSSL,accfd,conSSL,confd)
	SSL *accSSL,*conSSL;
{	int fdv[2],rfdv[2],nready,rcc,wcc;
	char buf[8*1024];

	fdv[0] = accfd;
	fdv[1] = confd;

	for(;;){
		nready = 0;
		rfdv[0] = rfdv[1] = 0;
		if( accSSL && SSL_pending(accSSL) ){
			rfdv[0] = 1;
			nready++;
		}
		if( conSSL && SSL_pending(conSSL) ){
			rfdv[1] = 1;
			nready++;
		}
		if( nready == 0 ){
			nready = PollIns(0,2,fdv,rfdv);
			if( nready <= 0 )
				break;
		}

		if( rfdv[0] ){
			if( accSSL )
				rcc = SSL_read(accSSL,buf,sizeof(buf));
			else	rcc = read(accfd,buf,sizeof(buf));
			if( rcc <= 0 )
				break;
			if( conSSL )
				wcc = SSL_write(conSSL,buf,rcc);
			else	wcc = write(confd,buf,rcc);
		}
		if( rfdv[1] ){
			if( conSSL )
				rcc = SSL_read(conSSL,buf,sizeof(buf));
			else	rcc = read(confd,buf,sizeof(buf));
			if( rcc <= 0 )
				break;

			if( accSSL )
				wcc = SSL_write(accSSL,buf,rcc);
			else	wcc = write(accfd,buf,rcc);
		}
	}
}
static HTTP_CAresp(fd,certfile)
	char *certfile;
{	FILE *tc,*cfp;
	X509 *cert;
	BIO *in,*out;

	tc = fdopen(fd,"w");
	cfp = fopen(certfile,"r");
	if( cfp == NULL )
		return -1;

	fprintf(tc,"HTTP/1.0 200 ok\r\n");
	fprintf(tc,"MIME-Version: 1.0\r\n");
	fprintf(tc,"Content-Type: application/x-x509-ca-cert\r\n");
	fprintf(tc,"\r\n");

	in = BIO_new_fp(cfp,BIO_NOCLOSE);
	cert = PEM_read_bio_X509(in,NULL,NULL);
	out = BIO_new_fp(tc,BIO_NOCLOSE);
	i2d_X509_bio(out,cert);

	BIO_free(in);
	BIO_free(out);
	fclose(tc);
	return 0;
}
static CArequest(accfd,isHTTP,certfile)
	int *isHTTP;
	char *certfile;
{	char method[8],line[1024],url[1024];
	int rcc;

	setNonblockingIO(accfd,1);
	rcc = RecvPeek(accfd,method,6);
	setNonblockingIO(accfd,0);

	if( rcc <= 0 )
		return 0;

	method[rcc] = 0;

	if( strncmp(method,"GET ",4) == 0 ){
		setNonblockingIO(accfd,1);
		rcc = RecvPeek(accfd,line,16);
		setNonblockingIO(accfd,0);
		line[rcc] = 0;

		wordscanX(line+4,url,sizeof(url));
		if( strcmp(url,"/-/ca.der") == 0 ){
			HTTP_CAresp(0,certfile);
			ERROR("sent cert");
			return 1;
		}
		*isHTTP = 1;
	}else
	if( strncmp(method,"HEAD ",5) == 0 || strncmp(method,"POST ",5) == 0 )
		*isHTTP = 1;
	return 0;
}
static _passwd(what,pass,buf,siz,vrfy)
	char *what,*pass;
	char *buf;
{
	if( pass ){
		ERROR("%s_passwd(%x,%d,%d)",what,buf,siz,vrfy);
		strcpy(buf,pass);
		return strlen(pass);
	}else{
		ERROR("%s_passwd(%x,%d,%d) -- SSL_%s_KEY_PASSWD undefined",
			what,buf,siz,vrfy,what);
		return -1;
	}
}
static sv_passwd(buf,siz,vrfy)
	char *buf;
{
	return _passwd("SERVER",sv_pass,buf,siz,vrfy);
}
static cl_passwd(buf,siz,vrfy)
	char *buf;
{
	return _passwd("CLIENT",cl_pass,buf,siz,vrfy);
}

static put_help()
{
	syslog_ERROR("SSLway 2000-09-06 <ysato@delegate.org>\r\n");
}
main(ac,av)
	char *av[];
{	int ai;
	char *arg;
	int accfd,confd;
	SSL_CTX *ctx;
	SSL *accSSL,*conSSL;
	char *env;
	int fdv[2],rfdv[2];
	int vflag;

	CFI_init(ac,av);
	if( env = getenv("CFI_TYPE") ){
		if( strcmp(env,"FCL") == 0 ){
			DEBUG("CFI_TYPE=%s: -ac is assumed\n",env);
			do_accSSL = 1;
		}else
		if( strcmp(env,"FSV") == 0 || strcmp(env,"FMD") == 0 ){
			DEBUG("CFI_TYPE=%s: -co is assumed\n",env);
			do_conSSL = 1;
		}
	}
	if( env = getenv("SSL_KEY_PASSWD") )
		sv_pass = cl_pass = env;
	if( env = getenv("SSL_CLIENT_KEY_PASSWD") )
		cl_pass = env;
	if( env = getenv("SSL_SERVER_KEY_PASSWD") )
		sv_pass = env;

	PID = getpid();
	if( (client_host = getenv("REMOTE_HOST")) == 0 )
		client_host = "?";

	accfd = dup(0);
	confd = dup(1);

	if( env = getenv("SSL_CIPHER") ) cipher_list = env;

	if( env = getenv("SSL_CERT_FILE") )
		sv_cert = sv_key = cl_cert = cl_key = env;

	if( env = getenv("SSL_SERVER_KEY_FILE" ) ) sv_key  = env;
	if( env = getenv("SSL_SERVER_CERT_FILE") ) sv_cert = env;

	if( env = getenv("SSL_CLIENT_KEY_FILE" ) ) cl_key  = env;
	if( env = getenv("SSL_CLIENT_CERT_FILE") ) cl_cert = env;

	for( ai = 1; ai < ac; ai++ ){
		arg = av[ai];
		if( strcmp(arg,"-help") == 0 || strcmp(arg,"-v") == 0 ){
			put_help();
			exit(0);
		}else 
		if( strncmp(arg,"-ac",3) == 0 ){
			do_accSSL = 1;
		}else
		if( strncmp(arg,"-co",3) == 0 ){
			do_conSSL = 1;
		}else
		if( strncmp(arg,"-ht",3) == 0 ){
			acc_bareHTTP = 1;
		}else
		if( strncmp(arg,"-show",3) == 0 ){
			do_showCERT = 1;
		}else
		if( strcmp(arg,"-CApath") == 0 ){
			if( ac <= ai + 1 ){
				ERROR("Usage: %s directory-name",arg);
				return -1;
			}
			cl_CApath = sv_CApath = av[++ai];
		}else
		if( strcmp(arg,"-CAfile") == 0 ){
			if( ac <= ai + 1 ){
				ERROR("Usage: %s file-name",arg);
				return -1;
			}
			cl_CAfile = sv_CAfile = av[++ai];
		}else
		if( strcasecmp(arg,"-vrfy")==0 || strcasecmp(arg,"-auth")==0 ){
			vflag = SSL_VERIFY_PEER
				| SSL_VERIFY_CLIENT_ONCE;
			if( arg[1] == 'V' || arg[1] == 'A' )
				vflag |= SSL_VERIFY_FAIL_IF_NO_PEER_CERT;
			if( arg[1] == 'V' || arg[1] == 'v' )
				verify_depth = -1;
			else	verify_depth = 10;
			cl_vrfy = vflag;
			sv_vrfy = vflag;
		}else
		if( strcasecmp(arg,"-verify") == 0 ){
			if( ac <= ai + 1 ){
				ERROR("Usage: %s max-depth",arg);
				return -1;
			}
			verify_depth = atoi(av[++ai]);
			vflag = SSL_VERIFY_PEER | SSL_VERIFY_CLIENT_ONCE;
			if( arg[1] == 'V' )
				vflag |= SSL_VERIFY_FAIL_IF_NO_PEER_CERT;
			cl_vrfy = vflag;
			sv_vrfy = vflag;
		}else
		if( strcmp(arg,"-client_auth") == 0 ){
			verify_depth = 10;
			cl_vrfy = SSL_VERIFY_PEER
				| SSL_VERIFY_CLIENT_ONCE;
		}else
		if( strcmp(arg,"-cipher") == 0 ){
			if( ac <= ai + 1 ){
				ERROR("Usage: %s cipher-list",arg);
				return -1;
			}
			cipher_list = av[++ai];
		}else
		if( strcmp(arg,"-certkey") == 0
		 || strcmp(arg,"-cert") == 0 ){
			if( ac <= ai + 1 ){
				ERROR("Usage: %s cert-key-file-name",arg);
				return -1;
			}
			sv_cert = sv_key = cl_cert = cl_key = av[++ai];
		}
	}

	accSSL = NULL;
	conSSL = NULL;

	if( do_conSSL || do_accSSL )
		ERROR("start");

	fdv[0] = accfd;
	fdv[1] = confd;

	if( acc_bareHTTP ){
		int isHTTP;
		if( 0 < PollIns(100,2,fdv,rfdv) && 0<rfdv[0] && rfdv[1] <= 0 ){
			isHTTP = 0;
			if( CArequest(accfd,&isHTTP,sv_cert) )
				return 0;
			if( isHTTP ){
				 /* ... through raw HTTP request ...
				do_accSSL = 0;
				 */
			}
		}
	}

	if( do_conSSL ){
		ctx = ssl_new(0);
		SSL_CTX_set_default_passwd_cb(ctx,cl_passwd);
		if( cipher_list )
			SSL_CTX_set_cipher_list(ctx,cipher_list);
		if( cl_cert != cl_cert_default || LIBFILE_IS(cl_cert,NULL) )
			ssl_keycert(ctx,cl_key,cl_cert,1);

		if( sv_CAfile || sv_CApath )
			ssl_setCAs(ctx,sv_CAfile,sv_CApath);
		if( sv_vrfy )
			SSL_CTX_set_verify(ctx,sv_vrfy,verify_callback);

		conSSL = ssl_conn(ctx,confd);
		if( conSSL == NULL )
			return -1;
	}

	if( do_accSSL ){
		ctx = ssl_new(1);
		SSL_CTX_set_default_passwd_cb(ctx,sv_passwd);
		if( cipher_list )
			SSL_CTX_set_cipher_list(ctx,cipher_list);
		if( ssl_keycert(ctx,sv_key,sv_cert,0) < 0 )
			return -1;

		if( cl_CAfile || cl_CApath )
			ssl_setCAs(ctx,cl_CAfile,cl_CApath);
		if( cl_vrfy )
			SSL_CTX_set_verify(ctx,cl_vrfy,verify_callback);

		accSSL = ssl_acc(ctx,accfd);
		if( accSSL == NULL )
			return -1;
	}

	if( conSSL )
		ssl_prcert(conSSL,do_showCERT,NULL,  accfd,"server");
	if( accSSL )
		ssl_prcert(accSSL,do_showCERT,accSSL,accfd,"client");

	ssl_relay(accSSL,accfd,conSSL,confd);

	if( do_conSSL || do_accSSL )
		ERROR("done");
	return 0;
}
