/*////////////////////////////////////////////////////////////////////////
Copyright (c) 1997 Electrotechnical Laboratry (ETL), AIST, MITI

Permission to use, copy, modify, and distribute this material for any
purpose and without fee is hereby granted, provided that the above
copyright notice and this permission notice appear in all copies, and
that the name of ETL not be used in advertising or publicity pertaining
to this material without the specific, prior written permission of an
authorized representative of ETL.
ETL MAKES NO REPRESENTATIONS ABOUT THE ACCURACY OR SUITABILITY OF THIS
MATERIAL FOR ANY PURPOSE.  IT IS PROVIDED "AS IS", WITHOUT ANY EXPRESS
OR IMPLIED WARRANTIES.
/////////////////////////////////////////////////////////////////////////
Content-Type:	program/C; charset=US-ASCII
Program:	cafe.c (Cache file expirer)
Author:		Yutaka Sato <ysato@etl.go.jp>
Description:
History:
	970429	created
//////////////////////////////////////////////////////////////////////#*/

#include <fcntl.h>
#include <stdio.h>
#include <ctype.h>
#include <string.h>
extern void *malloc();

#include <sys/types.h>
#include <sys/stat.h>

#ifndef S_ISDIR
#define S_ISDIR(m)  (((m)&S_IFMT) == S_IFDIR)
#endif

#include "../include/vsocket.h"

static char *USAGE = "\
Cafe -- cache file expirer\n\
-----------------------------------------------------------------------\n\
Usage: %s [options] directories\n\
  -ic[DIR]      keep inode cache at dir for speed up\n\
  -du[OPT]      ex. -dua generates output like `du -a'\n\
  -atime N[m|h] select last accessed before than N days [min|hour]\n\
  -mtime N[m|h] select last modified before than N days [min|hour]\n\
  -asize N      total size (bytes) to be remained\n\
  -h            follow symbolic link\n\
  -m            follow mounted file system\n\
  -rm           remove selected file\n\
  -mvDIR        move slected file under DIR\n\
  -ls[OPT]      list selected file in `ls' like format (ex. -lslsu)\n\
  -print        print selected file name\n\
  -s            work silently reporting the total only (with -du)\n\
  dir           directory to be scanned\n\
Exmaple:\n\
  %s DIR -atime 12h -lslsu  -- list files not acceessed within 12 hours\n\
  %s DIR -atime 12h -rm  -- remove files not accessed within 12 hours\n\
-----------------------------------------------------------------------\n\
<ysato@etl.go.jp>\n\
";
static put_usage(out,me)
	FILE *out;
	char *me;
{
	fprintf(out,USAGE,me,me,me,me);
}

static int ROOT_DEV;
static int DIRLEV;
static char *CWD;

static int F_FOLLOW_MOUNT;
static int F_FOLLOW_SYMLINK;
static char *ls_opts;
static int F_LSFORM;
static int F_UATIME; /* update the atime of directory after scanning */
static int F_VERBOSE;
static int F_DU;
static int F_DIR;
static int F_PRINT;
static int N_LIST;
static int F_REMOVE;
static int F_MOVE;
static int F_SUMMARY;
static int REM_SIZE;

static int A_FROM;
static int A_TILL;
static int M_FROM;
static int M_TILL;
static int NOW;
static int CNT;

static int N_DIR;
static int N_REG;
static int N_REM;
static int N_ERR;
static int N_BLK;
static int N_BLKREM;

static loghead(out)
	FILE *out;
{	char stime[128];

	StrftimeLocal(stime,sizeof(stime),"%Y/%m/%d-%H:%M:%S",time(0));
	fseek(out,0,2);
	fprintf(out,"%s [%d] ",stime,getpid());
}

static FILE *LOG;
static errlog(fmt,a,b,c,d,e,f,g,h,i,j,k,l,m,n)
	char *fmt,*a,*b,*c,*d,*e,*f,*g,*h,*i,*j,*k,*l,*m,*n;
{
	loghead(LOG);
	fprintf(LOG,fmt,a,b,c,d,e,f,g,h,i,j,k,l,m,n);
	fflush(LOG);
}
static summary(out,root)
	FILE *out;
	char *root;
{
	loghead(out);
	fprintf(out,"%s: %d blk, %d dir, %d reg",root,N_BLK,N_DIR,N_REG);

	if( F_REMOVE )
		fprintf(out,", %d rem (%d blk), %d err",
			N_REM,N_BLKREM,N_ERR);

	fprintf(out,", %d sec\n",time(0)-NOW);
}

typedef struct {
	int	d_top;
	FILE   *d_out;
	char   *d_cwd;
	int	d_dev;
	int	d_ino;
	int	d_nch;
	int	d_nrm;
	int	d_lev;
	int	d_blocks;
} DA;

static mkabspath(path,dir,file)
	char *path,*dir,*file;
{	char *dp,dc,*pp;

	pp = path;
	if( !isFullpath(file) ){
		for( dp = dir; dc = *dp; dp++ )
			*pp++ = dc;
		if( path < pp &&  pp[-1] != '/' )
			*pp++ = '/';
	}
	strcpy(pp,file);
}

static Stat(path,stp)
	char *path;
	struct stat *stp;
{
}

static char *filesys(path,root,parent)
	char *path,*root,*parent;
{	int dev;
	struct stat st;
	char *dp,dc;
	char cwd[1024],xpath[1024];

	if( stat(path,&st) != 0 ){
		errlog("can't stat: %s\n",path);
		exit(1);
	}
	dev = st.st_dev;
	getcwd(cwd,sizeof(cwd));
	strcpy(xpath,path);
	if( !fileIsdir(xpath) ){
		if( dp = strrchr(xpath,'/') )
			*dp = 0;
		else	strcpy(xpath,".");
	}
	chdir(xpath);
	getcwd(xpath,sizeof(xpath));
	strcpy(parent,xpath);

	for( dp = strchr(xpath,'/'); dp; dp = strchr(dp+1,'/') ){
		dc = dp[1];
		dp[1] = 0;
		if( stat(xpath,&st) == 0 ){
			if( st.st_dev == dev ){
				if( dp != xpath )
					*dp = 0;
				strcpy(root,xpath);
				break;
			}
		}
		dp[1] = dc;
	}

	chdir(cwd);
	return root;
}
static put_icacheR(path)
	char *path;
{	char *dp,dc,*np;
	int pdev,pino;
	struct stat st;
	char xpath[1024];

	strcpy(xpath,path);
	pdev = pino = 0;

	for( dp = strchr(xpath,'/'); dp; dp = strchr(dp+1,'/') ){
		dc = dp[1];
		dp[1] = 0;
		if( file_stats(pdev,pino,xpath,xpath,&st,NULL) < 0 ){
			dp[1] = dc;
			break;
		}
		dp[1] = dc;
		pdev = st.st_dev;
		pino = st.st_ino;
	}
	file_stats(pdev,pino,xpath,xpath,&st,NULL);
}

/*
 *	INODE CACHE
 */
static char *icachedir = NULL;

typedef struct {
	int	 ic_dev;
	char	*ic_rootpath;
	FILE	*ic_file;
} IC;
static IC ICs[32];

static FILE *fopen_rw(path)
	char *path;
{	FILE *fp;

	if( (fp = fopen(path,"r+")) == NULL )
		fp = fopen(path,"w+");
	return fp;
}

static int put_force;
static int inoc;

static FILE *ino_cache_sort(dev,key)
	char *key;
{	char path[1024];
	FILE *ics;

	sprintf(path,"%s/dev-%08x/ino-sort-by-%s",icachedir,dev,key);
	ics = fopen(path,"r+");
	if( ics == NULL )
		ics = fopen(path,"w+");
	return ics;
}
static FILE *ino_cache(rootpath,dev,ixp)
	char *rootpath;
	int *ixp;
{	char icache[1024],path[1024],rootpathbuf[1024];
	char parent[1024];
	FILE *ic;
	int ix,dev1;

	if( icachedir == NULL )
		return NULL;

	for( ix = 0; dev1 = ICs[ix].ic_dev; ix++ ){
		if( dev1 == dev && ICs[ix].ic_file ){
			if( ixp ) *ixp = ix;
			if( rootpath && *rootpath == 0 )
				strcpy(rootpath,ICs[ix].ic_rootpath);
			return ICs[ix].ic_file;
		}
	}

	sprintf(icache,"%s/dev-%08x",icachedir,dev);
	if( !fileIsdir(icache) ){
		if( mkdir(icache,0755) != 0 ){
			errlog("can't make inode cache directory: %s\n",
				icache);
			exit(1);
		}
	}

	sprintf(path,"%s/rootpath",icache);
	if( rootpath && *rootpath ){
		ic = fopen_rw(path);
		if( ic == NULL ){
			errlog("can't make inode cache file: %s\n",path);
			return NULL;
		}
		rootpath = filesys(rootpath,rootpathbuf,parent);
		fputs(rootpath,ic);
		fclose(ic);

		if( inoc == 0 ){
			inoc++;
			put_force = 1;
			put_icacheR(parent);
			put_force = 0;
			inoc--;
		}

	}else{
		ic = fopen(path,"r");
		if( ic == NULL ){
			errlog("can't open inode cache file: %s\n",path);
			return NULL;
		}
		if( rootpath == NULL )
			rootpath = rootpathbuf;
		fscanf(ic,"%s",rootpath);
		fclose(ic);
	}

	sprintf(path,"%s/ino",icache);
	ic = fopen_rw(path);
	if( ic == NULL ){
		errlog("can't make inode cache file: %s\n",path);
		exit(1);
	}

	ICs[ix].ic_dev = dev;
	ICs[ix].ic_file = ic;
	ICs[ix].ic_rootpath = strcpy(malloc(strlen(rootpath)+1),rootpath);
	if( ixp ) *ixp = ix;

	return ic;
}

typedef struct {
	int	i_ino;
	int	i_pino;
	int	i_blks;
	int	i_atime;
} INO;

static put_inocache(pdev,pino,path,stp,blocks)
	char *path;
	struct stat *stp;
{	FILE *ic;
	int ix;
	INO inoc;
	int dev = stp->st_dev;
	int ino = stp->st_ino;
	int atime = stp->st_atime;

	if( pino == 0 )
		return;

	if( !put_force )
	if( pdev != dev )
		return;

	if( (ic = ino_cache(path,dev,&ix)) == NULL )
		return;

	inoc.i_ino   = ino;
	inoc.i_pino  = pino;
	inoc.i_blks  = blocks;
	inoc.i_atime = atime;
	fseek(ic,ino*sizeof(INO),0);
	fwrite(&inoc,sizeof(INO),1,ic);

	if( put_force ){
		fflush(ic);
		errlog("FLUSH: %d/%x %s\n",fileno(ic),ic,path);
	}
}

static int file_stats(pdev,pino,path,file,stp,blocksp)
	char *path,*file;
	struct stat *stp;
	int *blocksp;
{       int rcode,blocks;

	if( FS_withSymlink() ){
		if( F_FOLLOW_SYMLINK )
			rcode = stat(file,stp);
		else	rcode = lstat(file,stp);
	}else{
		rcode = stat(file,stp);
	}

        if( rcode == 0 ){
		blocks = stat_blocks(stp);
		if( blocksp ) *blocksp = blocks;
		put_inocache(pdev,pino,path,stp,blocks);
		return S_ISDIR(stp->st_mode);
        }else	return -1;
}


static cmpino(i1,i2)
	INO **i1,**i2;
{	int diff;

	if( diff = (*i1)->i_atime - (*i2)->i_atime )
		return diff;
	if( diff = (*i2)->i_blks - (*i1)->i_blks )
		return diff;
	return (*i2)->i_ino - (*i1)->i_ino;
}

extern int scan_ino;
static inofind1(file,ino,name)
	char *file,*name;
{
	if( scan_ino == ino ){
		strcpy(name,file);
		return ino;
	}
	return 0;
}
static ino_name(dir,ino,name)
	char *dir,*name;
{
	name[0] = 0;
	Scandir(dir,inofind1,ino,name);
}

static ino2path(icx,root,ic,ino,path)
	char *root;
	FILE *ic;
	char *path;
{	int inov[256],lev,li;
	INO inob;
	char *pp;
	int cx;

	strcpy(path,root);
	for( lev = 0; lev < 256; lev++ ){
		inov[lev] = ino;
		fseek(ic,ino*sizeof(INO),0);
		fread(&inob,sizeof(INO),1,ic);
		if( inob.i_pino == 0 || inob.i_pino == ino )
			break;
		ino = inob.i_pino;
	}

	pp = path + strlen(path);
	for( li = lev-1; 0 <= li; li-- ){
		if( path < pp && pp[-1] != '/' )
			*pp++ = '/';
		ino_name(path,inov[li],pp);
		pp += strlen(pp);
	}
}

static find_icache1(out,dev,nlist)
	FILE *out;
{	int imax;
	INO **inos,*inov;
	int ino,ngot,icx,ix,blks,rblks;
	FILE *ic,*ics;
	char root[1024],path[2048];
	char ctime[64];

	*root = 0;
	ic = ino_cache(root,dev,&icx);
	if( ic == NULL )
		return;

	fflush(ic);
	fseek(ic,0,0);

	imax = file_size(fileno(ic)) / sizeof(INO);
	inos = (INO**)malloc(imax*sizeof(INO*));
	inov = (INO*)malloc(imax*sizeof(INO));

	ngot = 0;
	blks = 0;

	for( ino = 0; ino < imax && ngot < imax; ino++ ){
		if( fread(&inov[ngot],sizeof(INO),1,ic) == 0 ){
			errlog("premature EOF of inode-cache: %s\n",
				root);
			break;
		}
		if( inov[ngot].i_atime != 0 ){
			inos[ngot] = &inov[ngot];
			blks += inos[ngot]->i_blks;
			ngot++;
		}
	}

	qsort(inos,ngot,sizeof(INO*),cmpino);
	ics = ino_cache_sort(dev,"atime");
	for( ix = 0; ix < ngot; ix++ )
		fwrite(inos[ix],sizeof(INO),1,ics);
	fclose(ics);

	printf("%s: %d inodes cached...\n",root,ngot);

	if( nlist == 0 )
		nlist = imax;
	rblks = 0;
	for( ix = 0; ix < ngot && ix < nlist; ix++ ){
		ino2path(icx,root,ic,inos[ix]->i_ino,path);
		rblks += inos[ix]->i_blks;
		blks -= inos[ix]->i_blks;

		if( F_LSFORM ){
			struct stat st;
			if( stat(path,&st) == 0 ){
				printf("%8d ",st.st_ino);
				ls_unix(out,ls_opts,NULL,path,&st);
			}else{
				/* stale cache */
				/* remove it */
			}
		}else{
			rsctime(inos[ix]->i_atime,ctime);
			printf("%8d %8d - %8d %6d %6d %s %s\n",
				rblks,blks,inos[ix]->i_blks,
				inos[ix]->i_ino,inos[ix]->i_pino,
				ctime,path);
		}

		if( REM_SIZE && REM_SIZE <= rblks )
			break;
	}

	free(inos);
	free(inov);
}

static find_icache(out,path,nlist)
	FILE *out;
	char *path;
{	int dev;
	struct stat st;

	if( stat(path,&st) == 0 )
		find_icache1(out,st.st_dev,nlist);
	else	errlog("unknown path: %s\n",path);
}

static find1(file,da)
	char *file;
	DA *da;
{	int isdir,mtime,atime,size,blocks,dev,ino;
	int cnt;
	int match,rcode,cblocks;
	char path[1024];
	FILE *out = da->d_out;
	struct stat st;
	char *ftype;

	if( !da->d_top && file[0] == '.' )
		if( file[1] == 0 || file[1] == '.' && file[2] == 0 )
			return 0;
	da->d_nch++;
	mkabspath(path,da->d_cwd,file);

	CNT++;
	match = 0;
	rcode = 0;

	isdir = file_stats(da->d_dev,da->d_ino,path,file,&st,&blocks);
	dev = st.st_dev;
	ino = st.st_ino;
	mtime = st.st_mtime;
	atime = st.st_atime;
	size = st.st_size;
	
	if( isdir < 0 ){
		errlog("can't access: %s\n",path);
		return 0;
	}

	if( !F_FOLLOW_MOUNT && dev != ROOT_DEV ){
		if( ROOT_DEV == 0 )
			ROOT_DEV = dev;
		else	return 0;
	}

	N_BLK += blocks;

	if( isdir ){
		N_DIR++;
		match = 0 < dir1(da->d_cwd,path,file,dev,ino,out,&cblocks);
		if( F_REMOVE ){
			match = match && (0 < N_REM);
		}else
		if( (A_FROM==0||A_FROM<=atime) && (A_TILL==0||atime<=A_TILL) )
		if( (M_FROM==0||M_FROM<=mtime) && (M_TILL==0||mtime<=M_TILL) )
			match = 1;
		else	match = 0;
	}else{
		N_REG++;
		if( (A_FROM==0||A_FROM<=atime) && (A_TILL==0||atime<=A_TILL) )
		if( (M_FROM==0||M_FROM<=mtime) && (M_TILL==0||mtime<=M_TILL) )
			match = 1;
	}

	if( match && F_REMOVE && !da->d_top ){
		if( isdir )
			rcode = rmdir(file);
		else	rcode = unlink(file);
		if( rcode == 0 ){
			da->d_nrm++;
			N_BLKREM += blocks;
			N_REM++;
		}else	N_ERR++;
	}
	if( match || F_DU ){
		da->d_blocks += blocks;
		if( isdir ) da->d_blocks += cblocks;
	}
	if( match && (F_PRINT || F_LSFORM)
	 || da->d_top && !F_REMOVE
	 || isdir && F_DIR
	/* || isdir && match_in_child */
	){
		fseek(out,0,2);
		if( F_DU ){
			if( isdir )
				fprintf(out,"%-7d ",(blocks+cblocks)/2);
			else	fprintf(out,"%-7d ",blocks/2);
		}
/*
		{
			fprintf(out,"%d %6d %9d %9d %4d %4d ",
				rcode,ino,NOW-mtime,NOW-atime,blocks,size);
		}
*/
		if( F_LSFORM ){
			ls_unix(out,ls_opts,NULL,path,&st);
		}else{
			ftype = "";
			if( isdir && path[strlen(path)-1] != '/' )
				ftype = "/";
			fprintf(out,"%s%s\n",path,ftype);
		}
	}
	return 0;
}
static find0(out,top)
	FILE *out;
	char *top;
{	DA dab;
	struct stat st;

	dab.d_top = 1;
	dab.d_out = out;
	dab.d_cwd = "";
	dab.d_dev = 0;
	dab.d_ino = 0;
	dab.d_blocks = 0;
	dab.d_nch = 0;
	dab.d_nrm = 0;
	dab.d_lev = DIRLEV;
	find1(top,&dab);
}

ScandirNCAT(path,rdir,func,arg)
	char *path,*rdir;
	int (*func)();
	char *arg;
{	struct stat st;

	stat(path,&st);
	Scandir(rdir,func,arg);
	set_utimes(path,st.st_atime,st.st_mtime);
}

static dir1(cwd,dir,rdir,dev,ino,out,blocks)
	char *cwd,*dir,*rdir;
	FILE *out;
	int *blocks;
{	DA dab;
	char pcwd[1024];
	char dirbuf[1024];

	*blocks = 0;
	if( chdir(rdir) != 0 ){
		errlog("can't chdir: %s\n",dir);
		return 0;
	}
	if( !isFullpath(cwd) ){
		mkabspath(pcwd,CWD,cwd);
		cwd = pcwd;
	}

	dab.d_top = 0;
	dab.d_out = out;
	dab.d_cwd = dir;
	dab.d_dev = dev;
	dab.d_ino = ino;
	dab.d_blocks = 0;
	dab.d_nch = 0;
	dab.d_nrm = 0;
	dab.d_lev = DIRLEV;
	DIRLEV++;
	if( F_UATIME )
		Scandir(".",find1,&dab);
	else{
		if( !isFullpath(dir) ){
			mkabspath(dirbuf,CWD,dir);
			dir = dirbuf;
		}
		ScandirNCAT(dir,".",find1,&dab);
	}
	DIRLEV--;

	if( chdir(cwd) != 0 ){
		errlog("can't return to: %s\n",cwd);
		exit(-1);
	}
	*blocks = dab.d_blocks;
	return dab.d_nch == dab.d_nrm;
}

static get_period(pspec)
	char *pspec;
{	int period;
	int type;

	type = pspec[0];
	if( type == '-' || type == '+' )
		pspec++;

	period = scan_period(pspec,'d',1);

	if( type == '-' )
		return -period;
	else	return  period;
}

static FILE *setlog(path)
	char *path;
{	FILE *fp;

	if( path[0] == '-' )
		fp = fdopen(atoi(&path[1]),"a");
	else	fp = fopen(path,"a");

	if( fp == NULL ){
		syslog_ERROR("cannot open expire log: %s\n",path);
		exit(1);
	}
	if( lock_exclusiveTO(fileno(fp),1000,NULL) != 0 ){
		syslog_ERROR("cannot lock expire log: %s\n",path);
		exit(1);
	}
	return fp;
}
static char *catarg1(args,ap,arg)
	char *args,*ap,*arg;
{
	if( ap != args )
		*ap++ = ' ';
	strcpy(ap,arg);
	ap += strlen(ap);
	return ap;
}
cafe_main(ac,av)
	char *av[];
{	char *arg;
	int ai;
	int period;
	char cwd[1024];
	char *tops[128],*top;
	int topx,tx;
	int cacheonly = 0,wonly = 0;
	char ls_optsb[128];
	char args[1024],*ap=args,*pp;

	topx = 0;
	LOG = stdout;

	if( ac == 1 ){
		put_usage(LOG,av[0]);
		exit(0);
	}

	NOW = time(0);

	for( ai = 1; ai < ac; ai++ ){
		arg = av[ai];
		pp = ap;
		ap = catarg1(args,ap,arg);

		if( strcmp(arg,"-ign") == 0 ){
			++ai;
			ap = pp;
		}else
		if( strcmp(arg,"-log") == 0 || strcmp(arg,"-err") == 0 ){
			if( ++ai < ac ){
				LOG = setlog(av[ai]);
				ap = pp;
			}
		}else
		if( strcmp(arg,"-utime") == 0 ){
			F_UATIME = 1;
		}else
		if( strcmp(arg,"-atime") == 0 ){
			if( ++ai < ac ){
				ap = catarg1(args,ap,av[ai]);
				period = get_period(av[ai]);
				if( period < 0 )
					A_FROM = NOW + period;
				else	A_TILL = NOW - period;
			}
		}else
		if( strcmp(arg,"-mtime") == 0 ){
			if( ++ai < ac ){
				ap = catarg1(args,ap,av[ai]);
				period = get_period(av[ai]);
				if( period < 0 )
					M_FROM = NOW + period;
				else	M_TILL = NOW - period;
			}
		}else
		if( strcmp(arg,"-asize") == 0 ){
			if( ++ai < ac ){
				ap = catarg1(args,ap,av[ai]);
				REM_SIZE = atoi(av[ai]);
			}
		}else
		if( strncmp(arg,"-ls",3) == 0 ){
			F_LSFORM = 1;
			if( arg[3] == 0 )
				strcpy(ls_optsb,"dl");
			else	sprintf(ls_optsb,"d%s",&arg[3]);
			ls_opts = ls_optsb;
		}else
		if( strncmp(arg,"-du",3) == 0 ){
			F_DU = 1;
			if( strchr(&arg[3],'s') == 0 )
				F_DIR = 1;
			if( strchr(&arg[3],'a') != 0 )
				F_PRINT = 1;
		}else
		if( strcmp(arg,"-s") == 0 ){
			F_DIR = 0;
		}else
		if( strncmp(arg,"-ic",3) == 0
		 || strncmp(arg,"-iw",3) == 0
		 || strncmp(arg,"-ir",3) == 0
		){
			if( arg[3] )
				icachedir = &arg[3];
			else	icachedir = "/tmp/cafe";
			if( strncmp(arg,"-ir",3) == 0 )
				cacheonly = 1;
			if( strncmp(arg,"-iw",3) == 0 )
				wonly = 1;
		}else
		if( strncmp(arg,"-mv",3) == 0 ){
			F_MOVE = 1;
		}else
		if( strcmp(arg,"-h") == 0 ){ F_FOLLOW_SYMLINK = 1; }else
		if( strcmp(arg,"-m") == 0 ){ F_FOLLOW_MOUNT = 1; }else
		if( strcmp(arg,"-v") == 0 ){ F_VERBOSE = 1; }else
		if( strcmp(arg,"-a") == 0 ){ /* ignore */ }else
		if( strncmp(arg,"-p",2) == 0 ){ F_PRINT = 1; }else
		if( strcmp(arg,"-sum") == 0 ){ F_SUMMARY = 1; }else
		if( strcmp(arg,"-rm") == 0 ){ F_REMOVE = 1; }else
		if( arg[0] == '-' ){
			if( atoi(&arg[1]) )
				N_LIST = atoi(&arg[1]);
		}else{
			tops[topx++] = arg;
			tops[topx] = 0;
			ap = pp;
		}
	}
	*ap = 0;

	if( topx == 0 ){
		tops[topx++] = ".";
		tops[topx] = 0;
	}

	if( !F_PRINT && !F_LSFORM && !F_DU && !F_SUMMARY ){
		F_DU = 1;
	}

	if( icachedir != NULL && !fileIsdir(icachedir) ){
		if( mkdir(icachedir,0755) != 0 ){
			errlog("%s: can't make inode cache directory: %s\n",
				av[0],icachedir);
			exit(1);
		}
	}

	getcwd(cwd,sizeof(cwd));
	CWD = cwd;

	for( tx = 0; tx < topx; tx++ ){
		top = tops[tx];

		if( LOG != stdout )
		errlog("%s %s\n",top,args);

		if( !icachedir || !cacheonly )
			find0(LOG,top);

		if( icachedir != NULL )
		if( !wonly )
			find_icache(LOG,top,N_LIST);

		if( F_SUMMARY )
			summary(LOG,top);
	}

	fclose(LOG);
	exit(0);
}
