require_relative '../../spec_helper'
require_relative '../rational/fixtures/rational'

describe "Kernel.Rational" do
  describe "passed Integer" do
    # Guard against the Mathn library
    guard -> { !defined?(Math.rsqrt) } do
      it "returns a new Rational number with 1 as the denominator" do
        Rational(1).should eql(Rational(1, 1))
        Rational(-3).should eql(Rational(-3, 1))
        Rational(bignum_value).should eql(Rational(bignum_value, 1))
      end
    end
  end

  describe "passed two integers" do
    it "returns a new Rational number" do
      rat = Rational(1, 2)
      rat.numerator.should == 1
      rat.denominator.should == 2
      rat.should be_an_instance_of(Rational)

      rat = Rational(-3, -5)
      rat.numerator.should == 3
      rat.denominator.should == 5
      rat.should be_an_instance_of(Rational)

      rat = Rational(bignum_value, 3)
      rat.numerator.should == bignum_value
      rat.denominator.should == 3
      rat.should be_an_instance_of(Rational)
    end

    it "reduces the Rational" do
      rat = Rational(2, 4)
      rat.numerator.should == 1
      rat.denominator.should == 2

      rat = Rational(3, 9)
      rat.numerator.should == 1
      rat.denominator.should == 3
    end
  end

  describe "when passed a String" do
    it "converts the String to a Rational using the same method as String#to_r" do
      r = Rational(13, 25)
      s_r = ".52".to_r
      r_s = Rational(".52")

      r_s.should == r
      r_s.should == s_r
    end

    it "scales the Rational value of the first argument by the Rational value of the second" do
      Rational(".52", ".6").should == Rational(13, 15)
      Rational(".52", "1.6").should == Rational(13, 40)
    end

    it "does not use the same method as Float#to_r" do
      r = Rational(3, 5)
      f_r = 0.6.to_r
      r_s = Rational("0.6")

      r_s.should == r
      r_s.should_not == f_r
    end
  end

  describe "when passed a Numeric" do
    it "calls #to_r to convert the first argument to a Rational" do
      num = RationalSpecs::SubNumeric.new(2)

      Rational(num).should == Rational(2)
    end
  end

  describe "when passed a Complex" do
    context "[Complex]" do
      it "returns a Rational from the real part if the imaginary part is 0" do
        Rational(Complex(1, 0)).should == Rational(1)
      end

      it "raises a RangeError if the imaginary part is not 0" do
        -> { Rational(Complex(1, 2)) }.should raise_error(RangeError, "can't convert 1+2i into Rational")
      end
    end

    context "[Numeric, Complex]" do
      it "uses the real part if the imaginary part is 0" do
        Rational(1, Complex(2, 0)).should == Rational(1, 2)
      end

      it "divides a numerator by the Complex denominator if the imaginary part is not 0" do
        Rational(1, Complex(2, 1)).should == Complex(2/5r, -1/5r)
      end
    end
  end

  context "when passed neither a Numeric nor a String" do
    it "converts to Rational with #to_r method" do
      obj = Object.new
      def obj.to_r; 1/2r; end

      Rational(obj).should == 1/2r
    end

    it "tries to convert to Integer with #to_int method if it does not respond to #to_r" do
      obj = Object.new
      def obj.to_int; 1; end

      Rational(obj).should == 1r
    end

    it "raises TypeError if it neither responds to #to_r nor #to_int method" do
      -> { Rational([]) }.should raise_error(TypeError, "can't convert Array into Rational")
      -> { Rational({}) }.should raise_error(TypeError, "can't convert Hash into Rational")
      -> { Rational(nil) }.should raise_error(TypeError, "can't convert nil into Rational")
    end

    it "swallows exception raised in #to_int method" do
      object = Object.new
      def object.to_int() raise NoMethodError; end

      -> { Rational(object) }.should raise_error(TypeError)
      -> { Rational(object, 1) }.should raise_error(TypeError)
      -> { Rational(1, object) }.should raise_error(TypeError)
    end

    it "raises TypeError if #to_r does not return Rational" do
      obj = Object.new
      def obj.to_r; []; end

      -> { Rational(obj) }.should raise_error(TypeError, "can't convert Object to Rational (Object#to_r gives Array)")
    end
  end

  it "raises a ZeroDivisionError if the second argument is 0" do
    -> { Rational(1, 0) }.should raise_error(ZeroDivisionError, "divided by 0")
    -> { Rational(1, 0.0) }.should raise_error(ZeroDivisionError, "divided by 0")
  end

  it "raises a TypeError if the first argument is nil" do
    -> { Rational(nil) }.should raise_error(TypeError, "can't convert nil into Rational")
  end

  it "raises a TypeError if the second argument is nil" do
    -> { Rational(1, nil) }.should raise_error(TypeError, "can't convert nil into Rational")
  end

  it "raises a TypeError if the first argument is a Symbol" do
    -> { Rational(:sym) }.should raise_error(TypeError)
  end

  it "raises a TypeError if the second argument is a Symbol" do
    -> { Rational(1, :sym) }.should raise_error(TypeError)
  end

  describe "when passed exception: false" do
    describe "and [non-Numeric]" do
      it "swallows an error" do
        Rational(:sym, exception: false).should == nil
        Rational("abc", exception: false).should == nil
      end

      it "swallows an exception raised in #to_r" do
        obj = Object.new
        def obj.to_r; raise; end
        Rational(obj, exception: false).should == nil
      end

      it "swallows an exception raised in #to_int" do
        obj = Object.new
        def obj.to_int; raise; end
        Rational(obj, exception: false).should == nil
      end
    end

    describe "and [non-Numeric, Numeric]" do
      it "swallows an error" do
        Rational(:sym, 1, exception: false).should == nil
        Rational("abc", 1, exception: false).should == nil
      end

      it "swallows an exception raised in #to_r" do
        obj = Object.new
        def obj.to_r; raise; end
        Rational(obj, 1, exception: false).should == nil
      end

      it "swallows an exception raised in #to_int" do
        obj = Object.new
        def obj.to_int; raise; end
        Rational(obj, 1, exception: false).should == nil
      end
    end

    describe "and [anything, non-Numeric]" do
      it "swallows an error" do
        Rational(:sym, :sym, exception: false).should == nil
        Rational("abc", :sym, exception: false).should == nil
      end

      it "swallows an exception raised in #to_r" do
        obj = Object.new
        def obj.to_r; raise; end
        Rational(obj, obj, exception: false).should == nil
      end

      it "swallows an exception raised in #to_int" do
        obj = Object.new
        def obj.to_int; raise; end
        Rational(obj, obj, exception: false).should == nil
      end
    end

    describe "and non-Numeric String arguments" do
      it "swallows an error" do
        Rational("a", "b", exception: false).should == nil
        Rational("a", 0, exception: false).should == nil
        Rational(0, "b", exception: false).should == nil
      end
    end

    describe "and nil arguments" do
      it "swallows an error" do
        Rational(nil, exception: false).should == nil
        Rational(nil, nil, exception: false).should == nil
      end
    end
  end

  it "freezes its result" do
    Rational(1).frozen?.should == true
  end
end
