#
# pop.rb
#
#   author: Minero Aoki <aamine@dp.u-netsurf.ne.jp>
#

require 'Mail/session'
require 'md5'


module Mail

  class POPSession < Session

    attr :mails

    def each() @mails.each{|m| yield m} end


    private


    def proto_initialize
      @proto_type = POP3
      @port       = 110
      @mails      = [].freeze
    end


    def do_start( acnt, pwd )
      @proto.auth( acnt, pwd )
      @mails = []
      arr = @proto.list
      arr.each_index do |idx|
        if (item = arr[idx]) then
          @mails.push POPMail.new( idx, item, @proto )
        end
      end
      @mails.freeze
    end


    def do_finish
      @proto.quit
    end



    class POPMail

      def initialize( idx, siz, pro )
        @num     = idx
        @size    = siz
        @proto   = pro

        @deleted = false
      end

      attr :size

      def all
        @proto.retr( @num )
      end
      alias pop all
      alias mail all

      def top( lines )
        @proto.top( @num, lines )
      end

      def header
        top 0
      end

      def delete
        @proto.dele( @num )
        @deleted = true
      end
      alias delete! delete

      def deleted?
        @deleted
      end

    end

  end


  class APOPSession < POPSession

    def proto_initialize
      super
      @proto_type = APOP
    end

  end



  class POP3 < Protocol

    def auth( acnt, pass )
      @socket.writeline( 'USER ' + acnt )
      check_reply_auth

      @socket.writeline( 'PASS ' + pass )
      ret = check_reply_auth

      return ret
    end


    def list
      @socket.writeline( 'LIST' )
      check_reply( SuccessCode )
      
      arr = []
      @socket.read_pendlist do |line|
        num, siz = line.split( / +/o )
        arr[ num.to_i ] = siz.to_i
      end

      return arr
    end


    def rset
      @socket.writeline( 'RSET' )
      check_reply( SuccessCode )
    end


    def top( num, lines = 0 )
      @socket.writeline( sprintf( 'TOP %d %d', num, lines ) )
      check_reply( SuccessCode )

      return @socket.read_pendstr
    end


    def retr( num )
      @socket.writeline( sprintf( 'RETR %d', num ) )
      check_reply( SuccessCode )

      return @socket.read_pendstr
    end

    
    def dele( num )
      @socket.writeline( sprintf( 'DELE %s', num ) )
      check_reply( SuccessCode )
    end



    private


    def do_quit
      @socket.writeline( 'QUIT' )
      check_reply( SuccessCode )
    end


    def check_reply_auth
      begin
        cod = check_reply( SuccessCode )
      rescue ProtocolError
        raise ProtoAuthError, 'Fail to POP authentication'
      end

      return cod
    end


    def get_reply
      str = @socket.readline

      if /\A\+/ === str then
        return SuccessCode.new( str[0,3], str[3, str.size - 3].strip )
      else
        return ErrorCode.new( str[0,4], str[4, str.size - 4].strip )
      end
    end

  end



  class APOP < POP3

    def initialize( adr, po )
      rep = super( adr, po )

      /<[^@]+@[^@>]+>/o === rep.msg
      @stamp = $&
      unless @stamp then
        raise ProtoAuthError, "This is not APOP server: can't login"
      end
    end


    def auth( acnt, pass )
      @socket.writeline( "APOP #{acnt} #{digest(@stamp + pass)}" )
      return check_reply_auth
    end


    def digest( str )
      temp = MD5.new( str ).digest

      ret = ''
      temp.each_byte do |i|
        if i < 17 then ret.concat '0' end
        ret.concat( '%x' % i )
      end
      return ret
    end
      
  end

end


unless Mail::Session::Version == '0.3.0' then
  $stderr.puts "WARNING: wrong version of session.rb & pop.rb"
end
