/*
 * readline.c
 * module for GNU Readline Library
 *
 * Author: maeda shugo <shugo@po.aianet.ne.jp>
 * Version: $Id: readline.c,v 1.2 1997/08/21 10:41:33 shugo Exp $
 *
 * Usage:
 *
 *   require "readline"
 *   include Readline
 *
 *   line = readline("Prompt> ", TRUE)
 */

#include <stdio.h>
#include <readline/readline.h>
#include <readline/history.h>

#include "ruby.h"

extern VALUE cProc;
extern VALUE cArray;
extern VALUE cString;

VALUE mReadline;
static VALUE histary;  /* HISTory ARraY */

static ID id_completion_proc;
static ID id_completion_case_fold;
static ID id_call;

static VALUE readline_readline(int argc, VALUE *argv,
			  VALUE self)
{
    VALUE tmp, add_hist, result;
    char *prompt = NULL;
    char *buff;

    if (rb_scan_args(argc, argv, "02", &tmp, &add_hist) > 0) {
	Check_Type(tmp, T_STRING);
	prompt = RSTRING(tmp)->ptr;
    }
    buff = readline(prompt);
    if (RTEST(add_hist) && buff) {
	add_history(buff);
    }
    if (buff)
	result = str_new2(buff);
    else
	result = Qnil;
    if (buff) free(buff);
    return result;
}

static VALUE readline_s_set_completion_proc(VALUE self, VALUE proc)
{
    if (proc != Qnil && !obj_is_instance_of(proc, cProc))
	ArgError("argument needs to be proc or nil");
    return rb_ivar_set(mReadline, id_completion_proc, proc);
}

static VALUE readline_s_get_completion_proc(VALUE self)
{
    return rb_ivar_get(mReadline, id_completion_proc);
}

static VALUE readline_s_set_completion_case_fold(VALUE self, VALUE val)
{
    return rb_ivar_set(mReadline, id_completion_case_fold, val);
}

static VALUE readline_s_get_completion_case_fold(VALUE self)
{
    return rb_ivar_get(mReadline, id_completion_case_fold);
}

static char **readline_attempted_completion_function(char *text,
						     int start,
						     int end)
{
    VALUE proc, ary, temp;
    char **result;
    int case_fold;
    int i, matches;

    proc = rb_ivar_get(mReadline, id_completion_proc);
    if (!obj_is_instance_of(proc, cProc))
	return NULL;
    rl_attempted_completion_over = 1;
    case_fold = RTEST(rb_ivar_get(mReadline, id_completion_case_fold));
    ary = rb_funcall(proc, id_call, 1, str_new2(text));
    if (!obj_is_instance_of(ary, cArray))
	return NULL;
    matches = RARRAY(ary)->len;
    if (matches == 0)
	return NULL;
    result = (char **) xmalloc(sizeof(char *) * (matches + 2));
    for (i = 0; i < matches; i++) {
	temp = RARRAY(ary)->ptr[i];
	if (!obj_is_instance_of(temp, cString)) {
	    if (result) free(result);
	    return NULL;
	}
	result[i + 1]
	    = (char *) xmalloc(sizeof(char) * (RSTRING(temp)->len + 1));
	strcpy(result[i + 1], RSTRING(temp)->ptr);
    }
    result[matches + 1] = NULL;

    if (matches == 1) {
	result[0] = result[1];
	result[1] = NULL;
    } else {
	register int i = 1;
	int low = 100000;

	while (i < matches) {
	    register int c1, c2, si;

	    if (case_fold) {
		for (si = 0;
		     (c1 = to_lower(result[i][si])) &&
			 (c2 = to_lower(result[i + 1][si]));
		     si++)
		    if (c1 != c2) break;
	    } else {
		for (si = 0;
		     (c1 = result[i][si]) &&
			 (c2 = result[i + 1][si]);
		     si++)
		    if (c1 != c2) break;
	    }

	    if (low > si) low = si;
	    i++;
	}
	result[0] = xmalloc (low + 1);
	strncpy (result[0], result[1], low);
	result[0][low] = '\0';
    }

    return result;
}

static VALUE hist_to_s(VALUE self)
{
    return str_new2("HISTORY");
}

static VALUE hist_get(VALUE self, VALUE index)
{
    HISTORY_STATE *state;
    int i;

    state = history_get_history_state();
    i = NUM2INT(index);
    if (i < 0 || i > state->length - 1) {
	IndexError("Invalid index");
    }
    return str_new2(state->entries[i]->line);
}

static VALUE hist_set(VALUE self, VALUE index, VALUE str)
{
    HISTORY_STATE *state;
    int i;

    state = history_get_history_state();
    i = NUM2INT(index);
    if (i < 0 || i > state->length - 1) {
	IndexError("Invalid index");
    }
    Check_Type(str, T_STRING);
    replace_history_entry(i, RSTRING(str)->ptr, NULL);
    return str;
}

static VALUE hist_push(VALUE self, VALUE str)
{
    Check_Type(str, T_STRING);
    add_history(RSTRING(str)->ptr);
    return self;
}

static VALUE hist_push_method(int argc, VALUE *argv,
			      VALUE self)
{
    VALUE str;
    
    while (argc--) {
	str = *argv++;
	Check_Type(str, T_STRING);
	add_history(RSTRING(str)->ptr);
    }
    return self;
}

static VALUE hist_pop(VALUE self)
{
    HISTORY_STATE *state;
    HIST_ENTRY *entry;

    state = history_get_history_state();
    if (state->length > 0) {
	entry = remove_history(state->length - 1);
	return str_new2(entry->line);
    } else {
	return Qnil;
    }
}

static VALUE hist_shift(VALUE self)
{
    HISTORY_STATE *state;
    HIST_ENTRY *entry;

    state = history_get_history_state();
    if (state->length > 0) {
	entry = remove_history(0);
	return str_new2(entry->line);
    } else {
	return Qnil;
    }
}

static VALUE hist_each(VALUE self)
{
    HISTORY_STATE *state;
    int i;

    state = history_get_history_state();
    for (i = 0; i < state->length; i++) {
	rb_yield(state->entries[i]);
    }
    return Qnil;
}

static VALUE hist_length(VALUE self)
{
    HISTORY_STATE *state;

    state = history_get_history_state();
    return INT2NUM(state->length);
}

static VALUE hist_empty_p(VALUE self)
{
    HISTORY_STATE *state;

    state = history_get_history_state();
    if (state->length == 0)
	return TRUE;
    else
	return FALSE;
}

static VALUE hist_delete_at(VALUE self, VALUE index)
{
    HISTORY_STATE *state;
    HIST_ENTRY *entry;
    int i;

    state = history_get_history_state();
    i = NUM2INT(index);
    if (i < 0 || i > state->length - 1) {
	IndexError("Invalid index");
    }
    entry = remove_history(NUM2INT(index));
    return str_new2(entry->line);
}

void Init_readline(void)
{
    extern VALUE mEnumerable;

    id_completion_proc = rb_intern("completion_proc");
    id_completion_case_fold = rb_intern("completion_case_fold");
    id_call = rb_intern("call");

    using_history();

    mReadline = rb_define_module("Readline");
    rb_define_module_function(mReadline, "readline",
			      readline_readline, -1);
    rb_define_singleton_method(mReadline, "completion_proc=",
			       readline_s_set_completion_proc, 1);
    rb_define_singleton_method(mReadline, "completion_proc",
			       readline_s_get_completion_proc, 0);
    rb_define_singleton_method(mReadline, "completion_case_fold=",
			       readline_s_set_completion_case_fold, 1);
    rb_define_singleton_method(mReadline, "completion_case_fold",
			       readline_s_get_completion_case_fold, 0);

    histary = obj_alloc(cObject);
    rb_extend_object(histary, mEnumerable);
    rb_define_singleton_method(histary,"to_s", hist_to_s, 0);
    rb_define_singleton_method(histary,"[]", hist_get, 1);
    rb_define_singleton_method(histary,"[]=", hist_set, 2);
    rb_define_singleton_method(histary,"<<", hist_push, 1);
    rb_define_singleton_method(histary,"push", hist_push_method, -1);
    rb_define_singleton_method(histary,"pop", hist_pop, 0);
    rb_define_singleton_method(histary,"shift", hist_shift, 0);
    rb_define_singleton_method(histary,"each", hist_each, 0);
    rb_define_singleton_method(histary,"length", hist_length, 0);
    rb_define_singleton_method(histary,"empty?", hist_empty_p, 0);
    rb_define_singleton_method(histary,"delete_at", hist_delete_at, 1);
    rb_define_const(mReadline, "HISTORY", histary);

    rl_attempted_completion_function
	= (CPPFunction *) readline_attempted_completion_function;
}
