#!/usr/bin/env perl
use strict;
use warnings;

use Test::More tests => 5;
use Test::MockModule;
use Test::Exception;

use Git::Wrapper::More;

# # hides:
# [warn] No git remote found, using origin. at /home/store/git/idev-coverage/lib/IdevCover/Git::Wrapper::More.pm line 79.
close(STDERR);
my $hide_stderr;
open( STDERR, ">>", \$hide_stderr );

{
    my $label = 'Happy Path';

    my $mgw = Test::MockModule->new('Git::Wrapper');
    $mgw->mock(
        'config',
        sub {
            my $self = shift;
            return 'origin';
        },
    );
    $mgw->mock(
        'merge_base',
        sub {
            my $self = shift;
            return '73603e09df6d97e58544068d93e5d9185e3548fc';
        },
    );

    my $gwm = Git::Wrapper::More->new( dir => '/dummy/dir' );

    my $ret = $gwm->mergebase('dummy_branch');
    like(
        $ret,
        qr/^73603e09df6d97e58544068d93e5d9185e3548fc$/,
        "$label: mergebase returns the commit hash from Git::Wrapper merge_base"
    );
}

{
    my $label = 'Empty Remote';

    my $mgw = Test::MockModule->new('Git::Wrapper');

    $mgw->mock(
        'merge_base',
        sub {
            my $self = shift;
            return '73603e09df6d97e58544068d93e5d9185e3548fc';
        },
    );

    $mgw->mock(
        'error',
        sub {
            my $self = shift;
            return 'git exited non-zero but had no output to stderr';
        },
    );

    # TODO this doesn't mimic G:W accurately: config should die,
    #  but _run_cmd prints $_->error, which breaks this scaffolding
    $mgw->mock(
        'config',
        sub {
            my $self = shift;
            warn "git exited non-zero but had no output to stderr\n";
            return $self->error;
        },
    );

    my $gwm = Git::Wrapper::More->new( dir => '/dummy/dir' );

    lives_ok { $gwm->mergebase('dummy_branch') } "$label: mergebase lives";
}

{
    my $label = 'Remote is defined';

    my $mgw = Test::MockModule->new('Git::Wrapper');
    $mgw->mock(
        'config',
        sub {
            my $self = shift;
            return 'origin';
        },
    );
    $mgw->mock(
        'merge_base',
        sub {
            my $self = shift;
            return '73603e09df6d97e58544068d93e5d9185e3548fc';
        },
    );

    my $gwm = Git::Wrapper::More->new( dir => '/dummy/dir' );

    my $ret = $gwm->mergebase('dummy_branch', 'master', 'origin');
    like(
        $ret,
        qr/^73603e09df6d97e58544068d93e5d9185e3548fc$/,
        "$label: mergebase returns the commit hash from Git::Wrapper merge_base"
    );
}

{
    my $label = 'GW merge_base never works';

    my $mgw = Test::MockModule->new('Git::Wrapper');

    $mgw->mock(
        'merge_base',
        sub {
            my $self = shift;
            return undef;
        },
    );

    my $gwm = Git::Wrapper::More->new( dir => '/dummy/dir' );

    # clear stderr and temp redirect stdout for inspection
    $hide_stderr = '';
    my $hide_stdout;
    open( my $fh, ">>", \$hide_stdout ) or die "Couldn't open \$hide_stdout: $!";
    select($fh);

    # main part under test
    $gwm->mergebase('dummy_branch');

    # reopen STDOUT as normal
    select(STDOUT);

    like( $hide_stdout, qr/No common commit/, "$label: user is informed of no result" );
    like( $hide_stderr, qr/called at/,        "$label: user gets stacktrace in STDERR" );
}

exit;

__END__
