// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

//-----------------------------------------------------------------------------

// include files
#include "Utilities/Pool.h"
#include "Utilities/PAssert.h"


//----------------------------------------------------------------------
//
// Make a new pool with a given size block.
//
//----------------------------------------------------------------------

Pool::Pool(size_t sz)
  :
  // The number of outstanding allocs.
  outstandingAllocs_m(0),
  // The size of each block
  bsize_m(roundToAlign(sz)),     
  // Number of blocks
  nblock_m(blocksInPage(bsize_m)), 
  // The first one. Start out with nothing there.
  head_m(0)			     
{
}

//----------------------------------------------------------------------
//
// Make in invalid pool using the null ctor.
//
//----------------------------------------------------------------------

Pool::Pool()
  :
  // The number of outstanding allocs.
  outstandingAllocs_m(0),
  // The size of each block
  bsize_m(0),
  // Number of blocks
  nblock_m(0), 
  // The first one. Start out with nothing there.
  head_m(0)			     
{
}

//----------------------------------------------------------------------
//
// Delete a pool.
//
//----------------------------------------------------------------------

Pool::~Pool()
{
  PInsist(outstandingAllocs_m==0,"Not all of the pooled memory was freed!");

  // Loop over the allocated chunks.
  for (std::vector<char*>::iterator p=chunks_m.begin(); p!=chunks_m.end(); ++p)
    // Delete it.
    delete [] *p;
}

//----------------------------------------------------------------------
//
// Grow a Pool.
//
//----------------------------------------------------------------------

void Pool::grow()
{
  // Calculate the size to allocate.
  size_t alloc_this;

  // If we need more than a page, allocate that
  if ( bsize_m>page )
    alloc_this = bsize_m;
  else
    // Otherwise allocate a page.
    alloc_this = page;

  // Allocate a chunk.
  char *start = new char[alloc_this]; 

  // Put it in the list of things to delete.
  chunks_m.push_back(start);

  // Get a pointer to the last one in the chunk.
  char *last  = start + (nblock_m-1)*bsize_m; 

  // For all but the last one
  for (char *p=start; p!=last; p+=bsize_m)  
    // point to the next.
    ((Link*)p)->next_m = (Link*)(p+bsize_m);  

  // The last points to the current head of the list.
  ((Link*)last)->next_m = head_m;		  

  // Reset the head to the first in this chunk.
  head_m = (Link*)start;			  
}

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: Pool.cmpl.cpp,v $   $Author: swhaney $
// $Revision: 1.12 $   $Date: 2000/03/07 13:18:27 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
