// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

//-----------------------------------------------------------------------------
// Classes: 
//   UniformRectilinear<Dim, TM, CoordinatSystem>
//   FieldEngine<UniformRecilinear, T, EngineTag>
//-----------------------------------------------------------------------------

#ifndef POOMA_NEWFIELD_FIELDENGINE_UR_H
#define POOMA_NEWFIELD_FIELDENGINE_UR_H

//-----------------------------------------------------------------------------
// Overview: 
// 
// FieldEngine<UniformRectilinear, T, EngineTag>
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Includes:
//-----------------------------------------------------------------------------

#include "NewField/FieldEngine/FieldEngineBase.h"

#include "Array/Array.h"
#include "CoordinateSystems/Cartesian.h"
#include "Domain/Interval.h"
#include "Engine/ConstantFunctionEngine.h"
#include "Engine/IndexFunctionEngine.h"
#include "Layout/INode.h"
#include "Tiny/Vector.h"
#include "Utilities/PAssert.h"


//-----------------------------------------------------------------------------
// Forward Declarations:
//-----------------------------------------------------------------------------

template<class GeometryTag, class T, class EngineTag> class FieldEngine;
template<class Components> class ComponentWrapper;


//-----------------------------------------------------------------------------
// Full Description:
// 
// UniformRectilinear tag class.
//-----------------------------------------------------------------------------

template<int Dim, class T = POOMA_DEFAULT_POSITION_TYPE,
  class CoordinateSystem = Cartesian<Dim> >
struct UniformRectilinear;


//-----------------------------------------------------------------------------
// Full Description:
// 
// FieldEngine, specialized for uniform rectilinear meshes.
//-----------------------------------------------------------------------------

template<int Dim, class TM, class CoordinateSystem, class T, class EngineTag>
class FieldEngine<UniformRectilinear<Dim, TM, CoordinateSystem>, T, EngineTag>
: public FieldEngineBase<Dim, T, EngineTag>
{
public:

  //---------------------------------------------------------------------------
  // Exported typedefs and enumerations.

  // This class.
  
  typedef 
    FieldEngine<UniformRectilinear<Dim, TM, CoordinateSystem>, T, EngineTag>
      This_t;
    
  // Our base class.
  
  typedef FieldEngineBase<Dim, T, EngineTag> Base_t;

  // The geometry tag.
  
  typedef UniformRectilinear<Dim, TM, CoordinateSystem> GeometryTag_t;
  
  // The engine tag.
  
  typedef EngineTag EngineTag_t;
  
  // The coordinate system.
  
  typedef CoordinateSystem CoordinateSystem_t;
  
  // The domain type.
  
  typedef typename Base_t::Domain_t Domain_t;
  
  // The number of indices required to select a point in this mesh.
  
  enum { dimensions = Dim };
  
  // The type used to represent a point in the mesh.
  
  typedef Vector<Dim, TM> PointType_t;
  
  // Our engine type.
  
  typedef Engine<Dim, T, EngineTag> Engine_t;
  
  // Our layout type.
  
  typedef typename Engine_t::Layout_t Layout_t;


  //---------------------------------------------------------------------------
  // Constructors. We don't use default arguments for the origin and spacings
  // because that produces ambiguities.

  FieldEngine()
  : Base_t()
    { }

  template<class BaseInitializer>  
  FieldEngine(const BaseInitializer &baseInit, 
    const Layout_t &layout,
    const PointType_t &origin,
    const PointType_t &spacings)
  : Base_t(baseInit, layout),
    origin_m(origin),
    spacings_m(spacings)
    { }

  // Version that constructs a field with a new centering but based on the
  // same geometry.

  template<class BaseInitializer, class T2, class ET2>  
  FieldEngine(const BaseInitializer &baseInit,
	      const FieldEngine<GeometryTag_t, T2, ET2> &model)
  : Base_t(baseInit, model),
    origin_m(model.origin()),
    spacings_m(model.spacings())
    {
    }
    
  // Copy constructor.
  
  FieldEngine(const This_t &model)
  : Base_t(model),
    origin_m(model.origin_m),
    spacings_m(model.spacings_m)
    { }
    
  // Sub-field constructor.

  template<class ET2>  
  FieldEngine(const FieldEngine<GeometryTag_t, T, ET2> &model, 
    const int &iSubField)
  : Base_t(model, iSubField),
    origin_m(model.origin()),
    spacings_m(model.spacings())
    { }

  // Domain view constructor.

  template<class T2, class ET2>  
  FieldEngine(const FieldEngine<GeometryTag_t, T2, ET2> &model, 
    const Domain_t &d)
  : Base_t(model, d),
    origin_m(model.origin()),
    spacings_m(model.spacings())
    {
      for (int i = 0; i < Dim; i++)
        origin_m(i) += 
          (d[i].first() - model.physicalCellDomain()[i].first()) *
          spacings_m(i);
    }

  template<class T2, class ET2>  
  FieldEngine(const FieldEngine<GeometryTag_t, T2, ET2> &model, 
    const INode<Dim> &inode)
  : Base_t(model, inode),
    origin_m(model.origin()),
    spacings_m(model.spacings())
    {
      for (int i = 0; i < Dim; i++)
        origin_m(i) += 
          (inode.domain()[i].first() - model.physicalCellDomain()[i].first()) *
          spacings_m(i);
    }

  // Expression-engine constructor.    

  FieldEngine(const Engine<Dim, T, EngineTag> &e)
  : Base_t(e)
    {
      origin_m = referenceField().fieldEngine().origin(); 
      spacings_m = referenceField().fieldEngine().spacings(); 
    }

  // Patch constructor.    

  template<class ET2>
  FieldEngine(const FieldEngine<GeometryTag_t, T, ET2> &model, 
    const EnginePatch &p)
  : Base_t(model, p),
    spacings_m(model.spacings())
    { 
      for (int i = 0; i < Dim; i++)
        origin_m(i) += 
          (physicalCellDomain()[i].first() - 
          model.physicalCellDomain()[i].first()) *
          spacings_m(i);
    }
    
  // ComponentView constructor. The geometry for the model should be the
  // same since a component view simply mutates the elements.

  template<class T2, class ET2, class Components>  
  FieldEngine(const FieldEngine<GeometryTag_t, T2, ET2> &model, 
    const ComponentWrapper<Components> &c)
  : Base_t(model, c),
    origin_m(model.origin()),
    spacings_m(model.spacings())
    { }


  //---------------------------------------------------------------------------
  // Copy assignment operator (shallow).
  
  This_t &operator=(const This_t &rhs)
  {
    origin_m = rhs.origin();
    spacings_m = rhs.spacings();
    initialize(rhs);
    return *this;
  }

  //---------------------------------------------------------------------------
  // Empty destructor is fine.
  
  ~FieldEngine() { }


  //---------------------------------------------------------------------------
  // Accessors.
  
  inline const PointType_t &origin() const { return origin_m; }
  inline const PointType_t &spacings() const { return spacings_m; }


  //---------------------------------------------------------------------------
  // Return an index-function-engine-based array that can return the positions
  // of the geometric center of a cell's faces. 
  
  class FaceCenterFunctor
  {
    public:
    
      FaceCenterFunctor() { }
      
      FaceCenterFunctor(const PointType_t &origin, const PointType_t &spacings,
                        const Domain_t &pcd, int orientation)
      : origin_m(origin), spacings_m(spacings)
        {
          for (int i = 0; i < PointType_t::dimensions; i++)
            {
              if (i != orientation)
                origin_m(i) += 0.5 * spacings_m(i);
              origin_m(i) -= pcd[i].first() * spacings_m(i);
            }
        }
      
      PointType_t operator()(int i0) const
        {
          return origin + PointType_t(i0) * spacings;
        }
      
      PointType_t operator()(int i0, int i1) const
        {
          return origin + PointType_t(i0, i1) * spacings;
        }

        PointType_t operator()(int i0, int i1, int i2) const
        {
          return origin + PointType_t(i0, i1, i3) * spacings;
        }

    private:
    
      PointType_t origin, spacings;
  };
    
  typedef Array<Dim, PointType_t, IndexFunction<FaceCenterFunctor> > 
    FaceCentersArray_t;

  FaceCentersArray_t faceCenters(int orientation, const Domain_t &d) const;
  
private:

  PointType_t origin_m, spacings_m;

};


///////////////////////////////////////////////////////////////////////////////
//
// Non-inline members.
//
///////////////////////////////////////////////////////////////////////////////

//-----------------------------------------------------------------------------
// Return the face centers.
//-----------------------------------------------------------------------------

template <int Dim, class TM, class CoordinateSystem, class T, class EngineTag>
typename 
  FieldEngine<UniformRectilinear<Dim, TM, CoordinateSystem>, T, EngineTag>::
    FaceCentersArray_t 
FieldEngine<UniformRectilinear<Dim, TM, CoordinateSystem>, T, EngineTag>::
faceCenters(int orientation, const Domain_t &d) const
{
  FaceCentersArray_t array(d);
  FaceCenterFunctor functor(origin_m, spacings_m, physicalCellDomain(),
    orientation);
  array.engine().setFunctor(functor);
  return array;
}

#endif // POOMA_NEWFIELD_FIELDENGINE_UR_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: FieldEngine.UR.h,v $   $Author: sa_smith $
// $Revision: 1.3 $   $Date: 2000/07/26 16:51:38 $
// ----------------------------------------------------------------------
// ACL:rcsinfo

