// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

//-----------------------------------------------------------------------------
// Classes: 
//   DiscreteGeometry<RTComponent<NumComp,Centering>, Mesh>
//-----------------------------------------------------------------------------

#ifndef POOMA_GEOMETRY_DISCRETEGEOMETRY_RTCOMP_H
#define POOMA_GEOMETRY_DISCRETEGEOMETRY_RTCOMP_H

//-----------------------------------------------------------------------------
// Overview: 
// 
// DiscreteGeometry<RTComponent<Dim,Centering>, Mesh>:
//
// DiscreteGeometry specialization for POOMA-generated geometries that result from
// taking a component view of a multi-centered field.
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Includes:
//-----------------------------------------------------------------------------

#include "Domain/Loc.h"
#include "Domain/Region.h"
#include "Geometry/CenteringTags.h"
#include "Geometry/DiscreteGeometry.h"
#include "Pooma/View.h"

//-----------------------------------------------------------------------------
// Full Description:
// 
// DiscreteGeometry provides geometry services to discrete field by tying
// together a centering and a mesh, both of which are specified in the template
// parameters.
//
// The version in this file wraps another geometry that has different
// centerings for different components. If is generated by POOMA for
// the comp() and compRead() member functions for Field.
//
// Exported typedefs:
//
//   Centering_t - the centering tag.
//   CoordinateSystem_t - the coordinate system (obtained from the mesh).
//   Domain_t - the types of the physical/total domains (obtained from the 
//     mesh).
//   GuardLayers_t - the type of GuardLayers object for this geometry.
//   PointType_t - the type representing points in the geometry.
//   PositionArray_t - the type of constArray returned from x().
//   This_t - this class.
//
// Exported enumerations and constants:
//
//   dimension - the dimensionality of the mesh.
//   coordinateDimensions - the dimension of the coordinate system.
//
// Constructors:
//
//   DiscreteGeometry(OrigGeometry_t,Centering_t) - sets up this geometry using
//     old one and the selected componenents.
//   DiscreteGeometry(This_t) - copy constructor.
//
// Accessors:
//
//   centering() - returns the centering object for this geometry.
//   guardLayers() - returns the GuardLayers object for this geometry.
//   initialized() - returns whether or not a mesh has been initialized.
//   mesh() - returns the mesh.
//   physicalDomain() - returns this geometry's physical domain; that is, the
//     domain that is used to define a field (excludes global guard layers).
//   totalDomain() - returns this geometry's total domain (includes global 
//     guard layers).
//   x() - returns an array of centering positions corresponding to the
//     total domain.
//-----------------------------------------------------------------------------

template<int NumComp, class Centering, class Mesh>
class DiscreteGeometry<RTComponent<NumComp, Centering>, Mesh>
{
public:

  //---------------------------------------------------------------------------
  // Exported typedefs and enumerations.

  // The centering type.
  
  typedef RTComponent<NumComp, Centering> Centering_t;
  
  // The original centering type.
  
  typedef Centering OrigCentering_t;
  
  // The mesh type.
  
  typedef Mesh Mesh_t;
  
  // This class.
  
  typedef DiscreteGeometry<Centering_t, Mesh_t> This_t;
  
  // The class that is used to construct this class.
  
  typedef DiscreteGeometry<OrigCentering_t, Mesh_t> OrigGeometry_t;
  
  // The coordinate point type.
  
  typedef typename OrigGeometry_t::PointType_t PointType_t;
  
  // The coordinate system.
  
  typedef typename OrigGeometry_t::CoordinateSystem_t CoordinateSystem_t;
  
  // The domain type.
  
  typedef typename OrigGeometry_t::Domain_t Domain_t;
  
  // The guard layers type.
  
  typedef typename OrigGeometry_t::GuardLayers_t GuardLayers_t;
  
  // Dimensions.
  
  enum { dimensions = OrigGeometry_t::dimensions };
  enum { coordinateDimensions = OrigGeometry_t::coordinateDimensions };

  // Are we component-centered?
  
  static const bool componentCentered = false;
  
  // Array types
  
  typedef typename OrigGeometry_t::PositionArray_t OrigPositionArray_t;
  typedef typename ComponentView<Loc<NumComp>, OrigPositionArray_t>::Type_t 
    PositionArray_t;


  //---------------------------------------------------------------------------
  // Constructor using another geometry.
  
  DiscreteGeometry(const OrigGeometry_t &geom, const Loc<NumComp> &comps)
    : geom_m(geom), centering_m(geom.centering(), comps)
  {
  }

  //---------------------------------------------------------------------------
  // Copy constructor.
  
  DiscreteGeometry(const This_t &model)
    : geom_m(model.geom_m), centering_m(model.centering())
  {
  }
  
  //---------------------------------------------------------------------------
  // Constructor using the viewed geometry and a new domain.

  template<class OtherGeometry>
  DiscreteGeometry(const OtherGeometry &vg, const Domain_t &dom)
    : geom_m(vg, dom), centering_m(vg.centering())
  { }

  //---------------------------------------------------------------------------
  // Accessors.

  const Centering_t &centering() const
  {
    return centering_m;
  }
    
  const GuardLayers_t &guardLayers() const
  {
    return geom_m.guardLayers();
  }
    
  bool initialized() const
  {
    return geom_m.initialized();
  }

  Domain_t physicalDomain() const 
  {
    return geom_m.physicalDomain(); 
  }

  const Domain_t totalDomain() const 
  {
    return geom_m.totalDomain(); 
  }
  
  const Mesh_t &mesh() const 
  { 
    return geom_m.mesh(); 
  }
  
  PositionArray_t x() const
  {
    return geom_m.x().comp(centering().components());
  }

private:

  const OrigGeometry_t geom_m;
  Centering_t centering_m;
};

//-----------------------------------------------------------------------------
// Full Description:
// 
// A traits class that selects a new geometry when taking a component view.
//-----------------------------------------------------------------------------

template<class Components, class Geom, bool MultiCentered>
struct NewComponentGeometryImplementation { };

// The multi-centered version is only specialized for component views
// that are Loc's.  We would need to generalize the centering mechanism
// to recognize named components for this to work in general.

template<int NumComp, class Geom>
struct NewComponentGeometryImplementation<Loc<NumComp>, Geom, true>
{
  // Multi-centered version.
  
  typedef DiscreteGeometry<RTComponent<NumComp, typename Geom::Centering_t>,
    typename Geom::Mesh_t> Type_t;
  typedef Type_t Ret_t;
  static inline Ret_t make(const Geom &g, const Loc<NumComp> &l)
  {
    return Type_t(g, l);
  }
};

template<class Components, class Geom>
struct NewComponentGeometryImplementation<Components, Geom, false>
{
  // Single-centered version.
  
  typedef Geom Type_t;
  typedef const Type_t &Ret_t;
  static inline Ret_t make(const Geom &g, const Components &)
  {
    return g;
  }
};

template<class Components, class Geom>
struct NewComponentGeometry
{
  typedef typename NewComponentGeometryImplementation<Components,
    Geom, Geom::componentCentered>::Type_t Type_t;
  typedef typename NewComponentGeometryImplementation<Components,
    Geom, Geom::componentCentered>::Ret_t Ret_t;
  
  static inline Ret_t make(const Geom &g, const Components &l)
  {
    return NewComponentGeometryImplementation<Components,
      Geom, Geom::componentCentered>::make(g, l);
  }
};

#endif // POOMA_GEOMETRY_DISCRETEGEOMETRY_RTCOMP_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: DiscreteGeometry.RTComp.h,v $   $Author: swhaney $
// $Revision: 1.5 $   $Date: 2000/03/07 13:17:22 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
