// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

//-----------------------------------------------------------------------------
// Class:
// ComponentAccess
//-----------------------------------------------------------------------------

#ifndef POOMA_FUNCTIONS_COMPONENT_ACCESS_H
#define POOMA_FUNCTIONS_COMPONENT_ACCESS_H

//////////////////////////////////////////////////////////////////////

//-----------------------------------------------------------------------------
// Overview: 
//
// ComponentAccess<Container, Comp> is a general functor class that users can
// specialize to tell POOMA how to access components inside an object used as
// an element in expressions.  (For example, Vector, Tensor etc.)
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Typedefs:
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Includes:
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Forward Declarations:
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// ComponentAccess is an interface class that is used to provide an API for
// accessing components of a composite type. The default version handles
// scalars.
//-----------------------------------------------------------------------------

template<class T, class Components>
struct ComponentAccess
{
  typedef T Element_t;
  typedef T &ElementRef_t;
  
  static inline ElementRef_t indexRef(T &v, const Components &)
  {
    return v;
  }
  
  static inline Element_t index(const T &v, const Components &)
  {
    return v;
  }
};

//-----------------------------------------------------------------------------
//
// POOMA_COMPONENT_ACCESS macro:
//
// This macro simplifies writing component accessors for user defined structs
// where you can safely just return references to the components.  For example:
//
// struct Bob
// {
//   double density;
//   Vector<2> velocity;
// }
//
// POOMA_COMPONENT_ACCESS(Bob,Density,double,density)
// POOMA_COMPONENT_ACCESS(Bob,Velocity,Vector<2>,velocity)
//
// Array<1,Bob> a;
//
// a.comp(Density()) = ...
// a.comp(Velocity()) = ...
//
// The four parameters to POOMA_COMPONENT_ACCESS are:
//   1) name of the user's struct or class
//   2) name of a tag class (which the macro defines for you)
//   3) type of the component
//   4) access method for the component.
//-----------------------------------------------------------------------------

#define POOMA_COMPONENT_ACCESS(IN, TAG, TYPE, MEMBER)                   \
                                                                        \
struct TAG                                                              \
{                                                                       \
  TAG() { }                                                             \
  TAG(const TAG &) { }                                                  \
};                                                                      \
                                                                        \
template<>                                                              \
struct ComponentAccess<IN, TAG>                                         \
{                                                                       \
  typedef TYPE Element_t;                                               \
  typedef TYPE &ElementRef_t;                                           \
                                                                        \
  static inline ElementRef_t indexRef(IN &in, const TAG &)              \
  {                                                                     \
    return in.MEMBER;                                                   \
  }                                                                     \
                                                                        \
  static inline Element_t index(const IN &in, const TAG &)              \
  {                                                                     \
    return in.MEMBER;                                                   \
  }                                                                     \
};

//////////////////////////////////////////////////////////////////////

#endif     // POOMA_FUNCTIONS_COMPONENT_ACCESS_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: ComponentAccess.h,v $   $Author: sa_smith $
// $Revision: 1.4 $   $Date: 2000/05/04 19:30:46 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
