// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

// ----------------------------------------------------------------------------
// TestSumIndexedField.cpp , Tim Williams 6/23/1999
// 
// Tests slice-type subsetting of multidimensional Fields, including global
// reduction operations like sum() on slices.
// ----------------------------------------------------------------------------

// include files
#include "Pooma/Fields.h"
#include "Utilities/Tester.h"

int main(int argc, char* argv[])
{
  // Initialize POOMA and Tester class.
  Pooma::initialize(argc, argv);
  Pooma::Tester tester(argc, argv);

  tester.out() << argv[0] << ": " << std::endl;
  tester.out() << "------------------------------------------------"
               << std::endl;

  const int Dim = 3; // Dimensionality

  // Vertex and cell domains:
  int nVerts[3] = {5, 3, 4};
  Interval<Dim> vertDomain;
  for (int d = 0; d < Dim; d++) {
    vertDomain[d] = Interval<1>(nVerts[d]);
  }
  Interval<1> I = vertDomain[0];
  Interval<1> J = vertDomain[1];
  Interval<1> K = vertDomain[2];

  // Create the mesh; default origin and spacings:
  typedef UniformRectilinearMesh<Dim> Mesh_t;
  Mesh_t mesh(vertDomain);
  
  // Create the geometry:
  DiscreteGeometry<Vert, Mesh_t> geomc(mesh, GuardLayers<Dim>(2));
  
  // Make a 3D Field:
  typedef Field<DiscreteGeometry<Vert, Mesh_t>, double> ScalarField_t;
  ScalarField_t a(geomc);

  int i, j, k;

  // Assign values to 3D field
  //WCY  a(I,J,K) = I + J*K;
  a(I,J,K) = a.x().comp(0)(I,J,K) + (a.x().comp(1) * a.x().comp(2))(I,J,K);

  // 2D yz slices
  tester.out() << "----------------------------------" << std::endl;
  for (i = 0; i < I.size(); ++i) {
    Interval<Dim> yzSlice = vertDomain;
    yzSlice[0] = Interval<1>(i,i);
    double s = sum(a(yzSlice));
    double comps = 0;
    for (j = 0; j < J.size(); ++j)
      for (k = 0; k < K.size(); ++k)
	comps += (i + j*k);
    tester.out() << "Sum of a(" << i << ",J,K) = " << s
		 << "  (should be " << comps << ") ... " << std::endl;
    tester.check(s == comps);
  }

  // 2D xz slices
  tester.out() << "----------------------------------" << std::endl;
  for (j=0; j < J.size(); ++j) {
    Interval<Dim> xzSlice = vertDomain;
    xzSlice[1] = Interval<1>(j,j);
    double s = sum(a(xzSlice));
    double comps = 0;
    for (i=0; i < I.size(); ++i)
      for (k=0; k < K.size(); ++k)
	comps += (i + j*k);
    tester.out() << "Sum of a(I," << j << ",K) = " << s
		 << "  (should be " << comps << ") ... " << std::endl;
    tester.check(s == comps);
  }

  // 2D xy slices
  tester.out() << "----------------------------------" << std::endl;
  for (k=0; k < K.size(); ++k) {
    Interval<Dim> xySlice = vertDomain;
    xySlice[2] = Interval<1>(k,k);
    double s = sum(a(xySlice) * a(xySlice));
    double comps = 0;
    for (i=0; i < I.size(); ++i)
      for (j=0; j < J.size(); ++j)
	comps += (i + j*k)*(i + j*k);
    tester.out() << "Sum of a(I,J," << k << ") = " << s
		 << "  (should be " << comps << ") ... " << std::endl;
    tester.check(s == comps);
  }

  // 1D z slices
  tester.out() << "----------------------------------" << std::endl;
  for (i=0; i < I.size(); ++i) {
    for (j=0; j < J.size(); ++j) {
      Interval<Dim> zSlice = vertDomain;
      zSlice[0] = Interval<1>(i,i);
      zSlice[1] = Interval<1>(j,j);
      double s = sum(a(zSlice));
      double comps = 0;
      for (k=0; k < K.size(); ++k)
	comps += (i + j*k);
      tester.out() << "Sum of a(" << i << "," << j << ",K) = " << s
		   << "  (should be " << comps << ") ... " << std::endl;
      tester.check(s == comps);
    }
  }

  tester.out() << "------------------------------------------------"
               << std::endl;
  int retval = tester.results("TestFieldSlicing");
  Pooma::finalize();
  return retval;
}

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: TestFieldSlicing.cpp,v $   $Author: swhaney $
// $Revision: 1.7 $   $Date: 2000/03/07 13:17:15 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
