// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

// -----------------------------------------------------------------------------
// Scalar advection example, hardwired to 1D. Illustrates use of Field, mesh,
// and geometry classes.
// -----------------------------------------------------------------------------

#include "Pooma/Fields.h"
#include <iostream>

int main(
    int                 argc,
    char*               argv[]
){
    // Set up library.
    Pooma::initialize(argc, argv);

    // Create the 1D physical domain (number of vertices that the mesh will have):
    const int nVerts = 129;
    const int nCells = nVerts - 1;
    Interval<1> vertexDomain(nVerts);

    // Create the (uniform, logically rectilinear) mesh:
    Vector<1> origin(0.0), spacings(0.2);
    typedef UniformRectilinearMesh<1> Mesh_t;
    Mesh_t mesh(vertexDomain, origin, spacings);
  
    // Create two geometry objects - one allowing 1 guard layer to
    // account for the finite-difference stencil widths and another
    // with no guard layers to support temporaries:
    typedef DiscreteGeometry<Cell, UniformRectilinearMesh<1> > Geometry_t;
    Geometry_t geom1c(mesh, GuardLayers<1>(1));
    Geometry_t geom1ng(mesh);
  
    // Create the Fields:

    // The scalar flow Field u(x,t):
    Field<Geometry_t, double> u(geom1c);

    // The same, stored at the previous timestep for staggered
    // leapfrog, plus a useful temporary:
    Field<Geometry_t, double> uPrev(geom1ng), uTemp(geom1ng);

    // Initialize flow Field to zero everywhere, even global guard
    // layers:
    u(u.totalDomain()) = 0.0;

    // Set up periodic boundary conditions:
    u.addBoundaryCondition(PeriodicFaceBC(0));          // Low X face
    u.addBoundaryCondition(PeriodicFaceBC(1));          // High X face
    
    // Used various places below:
    Range<1> tr(u.totalDomain().first(), u.totalDomain().last(), 1);
    Interval<1> pd = u.physicalDomain();
    const int ib = pd.first();
    const int il = pd.last();
    int i;

    // Load initial condition u(x,0), a symmetric pulse centered
    // around nCells/4 and decaying to zero away from nCells/4 both
    // directions, with a height of 1.0, with a half-width of
    // nCells/8:
    double pulseWidth = spacings(0)*nCells/8;
    double u0 = u.x(nCells/4)(0);
    u = 1.0*exp(-((u.x().comp(0) - u0)*(u.x().comp(0) - u0)) / (2.0*pulseWidth));

    // Be sure to apply the boundary conditions after scalar code.
    u.applyBoundaryConditions();

    // Output the initial field:
    // First, the position (cell-center) values:
    std::cout << "x ";
    for (i = ib; i <= il; i++)
    {
        std::cout << u.x(i)(0) << " ";
    }
    std::cout << std::endl;

    // Next, the Field values:
    std::cout << 0.0 << " ";
    for (i = ib; i <= il; i++)
    {
        std::cout << u(i) << " ";
    }
    std::cout << std::endl;

    double v = 0.2;                                     // Propagation velocity
    double dt = 0.1;                                    // Timestep

    // Prime the leapfrog by setting the field at the previous timestep using the
    // initial conditions:
    uPrev = u;

    // Do a preliminary timestep using forward Euler, coded directly
    // using data-parallel syntax:
    Interval<1> pCells = u.geometry().mesh().physicalCellDomain();
    u(pCells) = u(pCells) - 0.5*v*dt*(u(pCells + 1) - u(pCells - 1))/spacings(0);

    // Now use staggered leapfrog (second-order) for the remaining
    // timesteps:

    for (int timestep = 2; timestep <= 1000; timestep++)
    {
        uTemp = u;
        u(pCells) = uPrev(pCells) - v*dt*(u(pCells+ 1) - u(pCells - 1))/spacings(0);

        if ((timestep % 200) == 0)
        {
            // Update boundary conditions prior to output.
            u.applyBoundaryConditions();

            // Output
            std::cout << "\nt = " << timestep*dt << std::endl;
            for (i = ib; i <= il; i++)
            {
                std::cout << u.comp(0)(i) << " ";
            }
            std::cout << std::endl;
        }
        uPrev = uTemp;
    }

    // Tidy up and exit.
    Pooma::finalize();
    return 0;
}

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: ScalarAdvection1D.cpp,v $   $Author: swhaney $
// $Revision: 1.6 $   $Date: 2000/03/07 13:17:14 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
