// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

//-----------------------------------------------------------------------------
// Class:
// OpMask<Op>
// WhereMask
// MaskAssign<T>
// ForEach<BinaryNode<OpMask>,FTag,OpCombine>
//-----------------------------------------------------------------------------

#ifndef POOMA_EVALUATOR_OPMASK_H
#define POOMA_EVALUATOR_OPMASK_H

//-----------------------------------------------------------------------------
// Overview:
// These classes implement the two argument where.
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Typedefs:
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Includes:
//-----------------------------------------------------------------------------

#include "PETE/PETE.h"

//-----------------------------------------------------------------------------
// Forward Declarations:
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
//
// Full Description:
//
// The expression,
//
// a += where(f, b);
//
// implements the following code:
//
// for (loc in domain)
// {
//   if (f(loc))
//     a(loc) += b(loc);
// }
//
// To implement this behaviour, the expression is translated to the tree:
//
//        OpMask<OpAddAssign>
//           /        \
//          A        WhereMask
//                     /    \
//                    F      B
//
// ForEach is specialized for WhereMask to evaluate B only if F is true.
// The result is returned in a MaskAssign<B::Element_t>, which contains
// the bool from F and the value from B if F is true.  OpMask applies the
// operator Op to the result from A and B if F is true.
//
// This design has the advantage that A is on the LHS and F and B are on the
// RHS, so we apply write locks to A and read locks to F and B.  Another
// approach is the tree OpMask(F, OpAddAssign(A,B)), which would lead to
// complicated interpretation of expressions.
// Pooma r1 used the implementation OpAddAssign(A, OpWhere(F, B)), which meant
// that operators needed specializations to deal with the MaskAssign<T>
// object. 
//
// WhereProxy wraps f and b so that the assignment operator can convert it 
// to the WhereMask object that goes on the RHS and convert the operator
// into OpMask.
// 
//-----------------------------------------------------------------------------


//-----------------------------------------------------------------------------
//
// Conditional Assign wraps the result of WhereMask(F,B);
// It contains the value of B and a bool which is true if the value is defined.
//
//-----------------------------------------------------------------------------

template<class T>
struct MaskAssign
{
  MaskAssign() { }
  MaskAssign(bool q) : cond_m(q) { }
  MaskAssign(bool q, const T& v) : cond_m(q), value_m(v) { }
  ~MaskAssign() { }

  inline bool defined() const { return cond_m; }
  inline const T &value() const { return value_m; }

  inline bool operator!=(const MaskAssign<T> &other) const
  {
    if (defined())
    {
      return ((other.defined() != defined()) || (other.value() != value()));
    }
    else
    {
      return other.defined();
    }
  }

  // To make purify happy:

  MaskAssign(const MaskAssign<T> &) { }
  MaskAssign<T> &operator=(const MaskAssign<T> &) { return *this; }

  bool cond_m;
  T value_m;
};

//-----------------------------------------------------------------------------
//
//-----------------------------------------------------------------------------

template<class Op>
struct OpMask
{
  OpMask() { }
  OpMask(const Op &op) : op_m(op) { }
  ~OpMask() { }

  template<class T1, class T2>
  inline T1&
  operator()(T1 &a, const T2 &b) const
  {
    if (b.defined())
    {
      op_m(a, b.value());
    }
    return a;
  }
  Op op_m;
};

template<class T1, class T2, class Op>
struct BinaryReturn<T1, T2, OpMask<Op> >
{
  typedef T1 &Type_t;
};

//-----------------------------------------------------------------------------
//
//-----------------------------------------------------------------------------

struct WhereMask
{
  WhereMask() { }
  WhereMask(const WhereMask &) { }
  WhereMask &operator=(const WhereMask &) { return *this; }

  ~WhereMask() { }
  // Should never explitly evaluate WhereMask
};

//-----------------------------------------------------------------------------
//
//-----------------------------------------------------------------------------

template<class T1, class T2>
struct BinaryReturn<T1, T2, WhereMask>
{
  typedef MaskAssign<T2> Type_t;
};

//-----------------------------------------------------------------------------
//
// ForEach
//
// To evaluate a WhereMask, we have to specialize ForEach.  The right hand
// expression should not be evaluated at all if the mask isn't true, so
// the stucture of the member function apply() has to be different from
// the generic ForEach.
//
//-----------------------------------------------------------------------------

template<class A, class B, class FTag>
struct ForEach< BinaryNode<WhereMask, A, B>, FTag, OpCombine >
{
  // The return type for the left tree.
  // really should be a bool
  typedef typename ForEach<A,FTag,OpCombine>::Type_t TypeA_t;

  // The return type for the right tree.
  typedef typename ForEach<B,FTag,OpCombine>::Type_t TypeB_t;

  // The return type for the expression.
  typedef MaskAssign<TypeB_t> Type_t;

  // How to evaluate the expression.
  inline
  static Type_t
  apply(const BinaryNode<WhereMask,A,B>& expr, 
	const FTag &f, const OpCombine &c) 
  {
    // Evaluate the left.
    bool mask = forEach(expr.left(), f, c);

    if ( mask )
    {
      return Type_t(mask, forEach(expr.right(), f, c));
    }
    else
    {
      return Type_t(mask);
    }
  }
};

#endif

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: OpMask.h,v $   $Author: swhaney $
// $Revision: 1.18 $   $Date: 2000/03/07 13:17:03 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
