// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

#ifndef POOMA_DOMAIN_DOMAIN_TRAITS_SLICE_RANGE_H
#define POOMA_DOMAIN_DOMAIN_TRAITS_SLICE_RANGE_H

//-----------------------------------------------------------------------------
// Class:
// DomainTraits<SliceRange<Dim,SliceDim>>
//-----------------------------------------------------------------------------

//////////////////////////////////////////////////////////////////////

//-----------------------------------------------------------------------------
// Overview:
// DomainTraits<SliceRange<Dim,SliceDim>> is a specialization of the
// general DomainTraits class, for the case of SliceRange domain objects.
// It defines the general behavior of SliceRange, including its typedefed
// and enumerated characteristics, how to store data for a SliceRange, etc.
// It is used by the SliceDomain base class of SliceRange to implement most
// of the public interface.
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Typedefs:
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Includes:
//-----------------------------------------------------------------------------

#include "Domain/DomainTraits.h"
#include "Domain/Range.h"


//-----------------------------------------------------------------------------
// Forward Declarations:
//-----------------------------------------------------------------------------

template<int TotalDim, int SliceDim> class SliceRange;


//-----------------------------------------------------------------------------
//
// Full Description:
//
// DomainTraits<SliceRange<Dim,SliceDim>> stores the characteristics and
// much of the implementation details for SliceRange domain objects.
// A SliceRange represents a set of two domain objects, one a "total"
// domain, and the other a "slice" domain which is a subset of the total.
// SliceRange stores these two domains as Range<> objects.
//
// The DomainTraits for slice domains is quite a bit simpler than the
// DomainTraits for regular domains.  This is because SliceDomains have
// a much simpler interface than regular domains, and are not intended for
// direct user manipulation.  The DomainTraits for SliceDomain subclasses
// like SliceRange includes the following interface:
//
// enum { dimensions } = # of total dimensions
// enum { sliceDimensions } = # of slice dimensions
//
// typedef SliceRange<TotalDim,SliceDim> Domain_t;
// typedef Range<SliceDim>               SliceDomain_t;
// typedef Range<TotalDim>               TotalDomain_t;
// typedef Range<1>                      OneDomain_t;
//
// static OneDomain_t &getDomain(Domain_t &d, int n);
// static OneDomain_t &getSliceDomain(Domain_t &d, int n);
//
//-----------------------------------------------------------------------------

template<int TotalDim, int SliceDim>
struct DomainTraits< SliceRange<TotalDim,SliceDim> >
{
  // necessary enums
  enum { domain          = 1 };
  enum { dimensions      = TotalDim };
  enum { sliceDimensions = SliceDim };
  enum { unitStride      = 0 };
  enum { singleValued    = 0 };
  enum { wildcard        = 0 };

  // necessary typedefs
  typedef SliceRange<TotalDim,SliceDim> Domain_t;
  typedef SliceRange<TotalDim,SliceDim> NewDomain1_t;
  typedef Range<SliceDim>               SliceDomain_t;
  typedef Range<TotalDim>               TotalDomain_t;
  typedef Range<1>                      OneDomain_t;
  typedef Range<1>                      PointDomain_t;

  // get the Nth element of the total domain, and return a OneDomain_t
  // object with it.
  static OneDomain_t &getDomain(Domain_t &d, int n) {
    return d.totalDomain()[n];
  }
  static const OneDomain_t &getDomain(const Domain_t &d,int n) {
    return d.totalDomain()[n];
  }

  // get the Nth element of the sliced domain, and return a OneDomain_t
  // object with it
  static OneDomain_t &getSliceDomain(Domain_t &d, int n) {
    return d.sliceDomain()[n];
  }
  static const OneDomain_t &getSliceDomain(const Domain_t &d, int n) {
    return d.sliceDomain()[n];
  }
  
  // convert from the Nth element of the domain to a single point, if
  // possible, and return a PointDomain_t.  Here, we just return a OneDomain_t,
  // since this is not a single-valued domain.
  static PointDomain_t &getPointDomain(Domain_t &d, int n) {
    return getDomain(d, n);
  }
  static const PointDomain_t &getPointDomain(const Domain_t &d, int n) {
    return getDomain(d, n);
  }

  // set the given dimension as ignorable
  static void cantIgnoreDomain(Domain_t &d, int n) {
    d.cantIgnoreDomain(n);
  }

  // get the ignore status of the given dimension
  static bool getIgnorable(const Domain_t &d, int n) {
    return d.ignorable(n);
  }

  // set the ignore status of the given dimension
  static void setIgnorable(Domain_t &d, int n, bool i) {
    d.ignorable(n) = i;
  }
};


//////////////////////////////////////////////////////////////////////

#endif     // POOMA_DOMAIN_DOMAIN_TRAITS_SLICE_RANGE_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: DomainTraits.SliceRange.h,v $   $Author: swhaney $
// $Revision: 1.11 $   $Date: 2000/03/07 13:16:36 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
