// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

// -----------------------------------------------------------------------------
// BCond<Field<Geom, T, EngineTag>, LinearExtrapolateFaceBC> template
// definitions.
// -----------------------------------------------------------------------------

// Include files.

#include "Geometry/CenteringTags.h"
#include "Meshes/MeshTraits.h"
#include "Utilities/PAssert.h"

//-----------------------------------------------------------------------------
// BCond<Field<Geom, T, EngineTag>, LinearExtrapolateFaceBC>()
// 
// Constructor.
//-----------------------------------------------------------------------------

template<class Geom, class T, class EngineTag>
BCond<Field<Geom, T, EngineTag>, LinearExtrapolateFaceBC>::
BCond(const Field<Geom, T, EngineTag> &f, const LinearExtrapolateFaceBC &bc)
  : Base_t(f, f.totalDomain(), f.totalDomain()), bc_m(bc)
{
  // This BC only makes sense for geometries on logically rectilinear meshes
  //  PAssert(MeshTraits<Subject::Mesh_t>::isLogicallyRectilinear);
  CTAssert(MeshTraits<typename Geom::Mesh_t>::isLogicallyRectilinear);

  // Note: a convertor from Interval<Dim> Range<Dim> would be handy here:
  for (int dd = 0; dd < Geom::dimensions; dd++) {
    destRange_m[dd] = 
      Range<1>(destDomain()[dd].min(), destDomain()[dd].max(), 1);
    srcLayer1_m[dd] = Range<1>(srcDomain()[dd].min(), 
                               srcDomain()[dd].max(), 1);
  }
  srcLayer2_m = srcLayer1_m;

  // Note: may need to replace some .min() and .max() with .first() and .last()
  // to make this all correct for Fields defined on domains with negative
  // strides.

  typedef typename Geom::Centering_t Centering_t;
      
  // Get the direction.
  int d = bc.face() / 2;

  // Select the high or low face:

  if (bc.face() & 1) {

    // High face.

    // Get the number of guard layers in the upper direction.
    int nGuards = subject().geometry().guardLayers().upper(d);
	
    // The other directions span the subject's total domain. Therefore, we just
    // chop out the guard layers. This BC is the same for cell or vert
    // centering.

    destRange_m[d] = Range<1>(destRange_m[d].max() - (nGuards - 1), 
			      destRange_m[d].max(),
			      1);

    srcLayer2_m[d] = Range<1>(srcDomain()[d].max() - nGuards,
			      srcDomain()[d].max() - nGuards, 1);
    srcLayer1_m[d] = Range<1>(srcDomain()[d].max() - (nGuards + 1),
			      srcDomain()[d].max() - (nGuards + 1), 1);

  } else {

    // Low face.
        
    // Get the number of guard layers in the lower direction.
    int nGuards = subject().geometry().guardLayers().lower(d);

    // The other directions span the subject's total domain. Therefore, we just
    // chop out the guard layers. This BC is the same for cell or vert
    // centering.

    destRange_m[d] = Range<1>(destRange_m[d].min() + (nGuards - 1), 
			      destRange_m[d].min(),
			      -1);

    srcLayer2_m[d] = Range<1>(srcDomain()[d].min() + nGuards,
			      srcDomain()[d].min() + nGuards);
    srcLayer1_m[d] = Range<1>(srcDomain()[d].min() + (nGuards + 1),
			      srcDomain()[d].min() + (nGuards + 1), 1);

  }

  destLayer_m = destRange_m;
}


//-----------------------------------------------------------------------------
// void applyBoundaryCondition()
// 
// The apply method for this boundary condition. (Kind of self-describing.)
//-----------------------------------------------------------------------------

template<class Geom, class T, class EngineTag>
void BCond<Field<Geom, T, EngineTag>, LinearExtrapolateFaceBC>::
applyBoundaryCondition()
{
  int d = bc_m.face() / 2; // Direction perpendicular to face
  int nGuards = destLayer_m[d].length();

  // The slope comes from subtracting the two source layers' values.

  // Now, use these to compute the slope and fill in the destination domain,
  // one layer at a time:
  int minGuard = destRange_m[d].first();
  int destRangeStride = destRange_m[d].stride();
  for (int gl = 0; gl < nGuards; gl++) {
    destLayer_m[d] = Range<1>(minGuard + gl*destRangeStride,
			      minGuard + gl*destRangeStride, 1);
    subject()(destLayer_m) = subject()(srcLayer2_m) + 
      (subject()(srcLayer2_m) - subject()(srcLayer1_m)) * (gl + 1);
  }      
      
}


//-----------------------------------------------------------------------------
// Base_t *retarget()
// 
// This boundary condition's retarget function.
//-----------------------------------------------------------------------------

template<class Geom, class T, class EngineTag>
FieldBCondBase< Field<Geom, T, EngineTag> > *
BCond<Field<Geom, T, EngineTag>, LinearExtrapolateFaceBC>::
retarget(const Field<Geom, T, EngineTag> &f) const
{
  return new This_t(f, bc_m);
}

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: LinExtrapFaceBC.cpp,v $   $Author: swhaney $
// $Revision: 1.9 $   $Date: 2000/03/07 13:16:12 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
