// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license
//-----------------------------------------------------------------------------
// DataBlockPtr test code.
//-----------------------------------------------------------------------------

#include "Pooma/Pooma.h"
#include "Utilities/DataBlockPtr.h"
#include "Utilities/PAssert.h"
#include "Utilities/Tester.h"

#include <iostream>

class SharedInt : public RefCounted
{
public:
  SharedInt(int i) : d_m(i) {};
  SharedInt(const SharedInt &model) : d_m(model.d_m) { }

  SharedInt & operator=(const SharedInt &model)
  { 
    if (&model == this) return *this;
    d_m = model.d_m;
    return *this;
  }

  SharedInt & operator=(int i) { d_m = i; return *this; }

  bool operator==(const SharedInt &rhs) const
  { return d_m == rhs.d_m; }

  bool operator!=(const SharedInt &rhs) const
  { return d_m != rhs.d_m; }

  int val() const {return d_m;}

private:

  int d_m;
};


typedef DataBlockPtr<SharedInt,true> SBlock_t;


int main(int argc, char* argv[])
{
  Pooma::initialize(argc,argv);
  Pooma::Tester tester(argc, argv);

  int test_number = 0;
#if POOMA_EXCEPTIONS
  try {
#endif
    tester.out() << "\nTest that blocks work if T has no T()" 
		 << std::endl;

    SBlock_t foo(10,SharedInt(3));

    foo[2] = 2;
    foo[6] = 8;

#ifdef TEST_3B
    {
      SBlock_t bar(2); // illegal - SharedInt has no default constructor.
    
      foo[8] = bar[0];
      foo[9] = bar[1];
    }
#endif
    
    for(int i = 0; i < 10; i++)
      tester.out() << "Value is " << foo[i].val()<<std::endl;;

    SBlock_t bar = foo;
    
    PAssert(foo.isShared());
    PAssert(bar.isShared());
    
    for(int i = 0; i < 10; i++)
      tester.out() << "Value is " << bar[i].val() << std::endl;

    bar.makeOwnCopy();
	
    PAssert(!foo.isShared());
    PAssert(!bar.isShared());
    tester.check("!foo.isShared()",!foo.isShared());
    tester.check("!bar.isShared()",!bar.isShared());

    bar[0] = -111;
    bar[1] = -222;

    for(int i = 0; i < 10; i++)
      tester.out() << "Value is " << bar[i].val() << std::endl;
    
    for(int i = 0; i < 10; i++)
      tester.out() << "Value is " <<  foo[i].val()<< std::endl;
    
    bar.invalidate();
    foo.invalidate();
#if POOMA_EXCEPTIONS
  }
  catch(const char *err) 
    { 
      tester.exceptionHandler( err );
      tester.set( false );
    }
  catch(const Pooma::Assertion &err)
    { 
      tester.exceptionHandler( err );
      tester.set( false );
    }
#endif 

  tester.out() << "All Done!" << std::endl;
  int res = tester.results(" dbptr_test2 " );
  Pooma::finalize();  
  return res;
}

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: dbptr_test2.cpp,v $   $Author: swhaney $
// $Revision: 1.9 $   $Date: 2000/03/07 13:18:31 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
