// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license
//-----------------------------------------------------------------------------
// Overview:
// A tag for checking whether the terms in an expression have
// conformant domains.
//-----------------------------------------------------------------------------

#ifndef POOMA_UTILITIES_CONFORM_H
#define POOMA_UTILITIES_CONFORM_H

//-----------------------------------------------------------------------------
// Typedefs:
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Includes:
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Forward Declarations:
//-----------------------------------------------------------------------------

template<class T, class A> struct LeafFunctor;
template<class T> class Scalar;

//-----------------------------------------------------------------------------
// class ConformTag<D>
//
// When evaluating expressions, we need to check to make sure that the
// left hand side has the same domain as the right.  To do that, we
// build a ConformTag functor with the domain from the lhs, and
// drop it down the rhs tree.  It makes sure that the lengths of the
// domains are all the same.
//-----------------------------------------------------------------------------

template<int D>
class ConformTag
{
public:
  template<class Domain>
  ConformTag(const Domain& domain) 
  {
    for (int i=0; i<D; ++i)
      lengths_m[i] = domain[i].length();
  }
  int length(int i) const { return lengths_m[i]; }
private:
  int lengths_m[D];
};

//----------------------------------------------------------------------
// conforms(Domain,ConformTag)
// Check to see whether a given domain conforms with the given
// ConformTag.  
// 
// We just check to see if the length of each dimension is the same.
//
// We specialize this for ranks 1 thru 7 for efficiency.
//----------------------------------------------------------------------

template<class Domain>
bool conforms(const Domain &d, const ConformTag<1> &ct)
{
  return d.length() == ct.length(0);
}

template<class Domain>
bool conforms(const Domain &d, const ConformTag<2> &ct)
{
  return (d[0].length() == ct.length(0))
      && (d[1].length() == ct.length(1));
}

template<class Domain>
bool conforms(const Domain &d, const ConformTag<3> &ct)
{
  return (d[0].length() == ct.length(0)) 
      && (d[1].length() == ct.length(1)) 
      && (d[2].length() == ct.length(2));
}

template<class Domain>
bool conforms(const Domain &d, const ConformTag<4> &ct)
{
  return (d[0].length() == ct.length(0)) 
      && (d[1].length() == ct.length(1)) 
      && (d[2].length() == ct.length(2)) 
      && (d[3].length() == ct.length(3));
}

template<class Domain>
bool conforms(const Domain &d, const ConformTag<5> &ct)
{
  return (d[0].length() == ct.length(0)) 
      && (d[1].length() == ct.length(1))
      && (d[2].length() == ct.length(2))
      && (d[3].length() == ct.length(3))
      && (d[4].length() == ct.length(4));
}

template<class Domain>
bool conforms(const Domain &d, const ConformTag<6> &ct)
{
  return (d[0].length() == ct.length(0)) 
      && (d[1].length() == ct.length(1))
      && (d[2].length() == ct.length(2))
      && (d[3].length() == ct.length(3))
      && (d[3].length() == ct.length(4))
      && (d[5].length() == ct.length(5));
}

template<class Domain>
bool conforms(const Domain &d, const ConformTag<7> &ct)
{
  return (d[0].length() == ct.length(0)) 
      && (d[1].length() == ct.length(1))
      && (d[2].length() == ct.length(2))
      && (d[3].length() == ct.length(3))
      && (d[3].length() == ct.length(4))
      && (d[5].length() == ct.length(5))
      && (d[6].length() == ct.length(6));
}

//----------------------------------------------------------------------
// Scalars conform with anything, so always return true.
//----------------------------------------------------------------------

template<int D, class T>
struct LeafFunctor<Scalar<T>, ConformTag<D> >
{
  typedef bool Type_t;
  static Type_t apply(const Scalar<T> &, const ConformTag<D> &)
  {
    return true;
  }
};

#endif // POOMA_UTILITIES_CONFORM_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: Conform.h,v $   $Author: swhaney $
// $Revision: 1.2 $   $Date: 2000/03/07 13:18:23 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
