// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license
//-----------------------------------------------------------------------------
// Tiny operations test
//-----------------------------------------------------------------------------

#include <stdio.h>
#include <stdlib.h>

#include "Tiny/TinyMatrix.h"
#include "Tiny/Vector.h"
#include "Tiny/VectorTinyMatrix.h"

const int D=3;
const int N=100;

template<int D1, int D2, class T, class E>
void fill( TinyMatrix<D1,D2,T,E>& x )
{
  for (int j=0; j<D1; ++j)
    for (int k=0; k<D2; ++k)
      x(j,k) = rand() % 1000000;
}

template<int D1, class T, class E>
void fill( Vector<D1,T,E>& x)
{
  for (int j=0; j<D1; ++j)
    x(j) = rand() % 1000000;
}

void testTinyMatrixDot()
{
  TinyMatrix<D,D> x,y;
  int i,j,k,n;

  double a = 0;
  for (n=0; n<N; ++n)
    {
      fill(x);
      fill(y);
      TinyMatrix<D,D> z = dot(x,y);
      for (i=0; i<D; ++i)
        for (j=0; j<D; ++j)
          for (k=0; k<D; ++k)
            z(i,k) -= x(i,j)*y(j,k);
      z = z*z;
      for (i=0; i<D; ++i)
        for (j=0; j<D; ++j)
          a += z(i,j);
    }

  if ( a==0 )
    printf("PASSED");
  else
    printf("FAILED");
  printf(" TinyMatrix dot\n");
}

void testVectorDot()
{
  Vector<D> x,y;

  double a = 0;
  for (int n=0; n<N; ++n)
    {
      fill(x);
      fill(y);
      a += dot(x,y);
      for (int i=0; i<D; ++i)
        a -= x(i)*y(i);
    }

  if ( a==0 )
    printf("PASSED");
  else
    printf("FAILED");
  printf(" Vector dot\n");
}

void testTinyMatrixEquality()
{
  TinyMatrix<D,D> x,y;

  bool eq, ans;  
  for (int n=0; n<N; ++n)
    {
      fill(x);
      fill(y);
      eq = x == y;
      ans = true;
      for (int j=0; j<D; ++j)
        for (int i=0; i<D; ++i)
        ans = ans && (x(i,j) == y(i,j));
    }

  if ( ans == eq)
    printf("PASSED");
  else
    printf("FAILED");
  printf(" TinyMatrix equality\n");
}

void testVectorEquality()
{
  Vector<D> x,y;

  bool eq, ans;  
  for (int n=0; n<N; ++n)
    {
      fill(x);
      fill(y);
      eq = x == y;
      ans = true;
      for (int i=0; i<D; ++i)
        ans = ans && (x(i) == y(i));
    }

  if ( ans == eq)
    printf("PASSED");
  else
    printf("FAILED");
  printf(" Vector equality\n");
}

void testVectorAdd()
{
  Vector<D> x,y;
  double a=0;

  for (int n=0; n<N; ++n)
    {
      fill(x);
      fill(y);
      Vector<D> b = x + y;
      for (int i=0; i<D; ++i)
        a += b(i) - (x(i)+y(i));
    }

  if ( a==0 )
    printf("PASSED");
  else
    printf("FAILED");
  printf(" Vector add\n");
}

void testVectorScalar()
{
  Vector<D> x;
  double a=0;
  int i;

  for (int n=0; n<N; ++n)
    {
      fill(x);
      Vector<D> b = x + 1.0;
      for (i=0; i<D; ++i)
        a += b(i) - (x(i)+1.0);
      b = 1.0 + x;
      for (i=0; i<D; ++i)
        a += b(i) - (1.0+x(i));
    }

  if ( a==0 )
    printf("PASSED");
  else
    printf("FAILED");
  printf(" Vector scalar\n");
}

void testTinyMatrixAdd()
{
  TinyMatrix<D,D> x,y;
  double a=0;

  for (int n=0; n<N; ++n)
    {
      fill(x);
      fill(y);
      TinyMatrix<D,D> b = x + y;
      for (int i=0; i<D; ++i)
        for (int j=0; j<D; ++j)
          a += b(i,j) - (x(i,j)+y(i,j));
    }

  if ( a==0 )
    printf("PASSED");
  else
    printf("FAILED");
  printf(" TinyMatrix add\n");
}

void testTinyMatrixScalar()
{
  TinyMatrix<D,D> x;
  double a=0;
  int i,j;

  for (int n=0; n<N; ++n)
    {
      fill(x);
      TinyMatrix<D,D> b = x + 1.0;
      for (i=0; i<D; ++i)
        for (j=0; j<D; ++j)
          a += b(i,j) - (x(i,j)+1.0);
      b = 1.0 + x;
      for (i=0; i<D; ++i)
        for (j=0; j<D; ++j)
          a += b(i,j) - (1.0+x(i,j));
    }

  if ( a==0 )
    printf("PASSED");
  else
    printf("FAILED");
  printf(" TinyMatrix scalar\n");
}

void testVectorNegate()
{
  Vector<D> x;
  double a=0;

  for (int n=0; n<N; ++n)
    {
      fill(x);
      Vector<D> b = -x;
      for (int i=0; i<D; ++i)
        a += b(i) + x(i);
    }

  if ( a==0 )
    printf("PASSED");
  else
    printf("FAILED");
  printf(" Vector negate\n");
}

void testTinyMatrixNegate()
{
  TinyMatrix<D,D> x;
  double a=0;

  for (int n=0; n<N; ++n)
    {
      fill(x);
      TinyMatrix<D,D> b = -x;
      for (int i=0; i<D; ++i)
        for (int j=0; j<D; ++j)
          a += b(i,j) + x(i,j);
    }

  if ( a==0 )
    printf("PASSED");
  else
    printf("FAILED");
  printf(" TinyMatrix negate\n");
}

void testVectorDotTinyMatrix()
{
  Vector<D> x;
  TinyMatrix<D,D> y;
  int i,j,n;

  double a = 0;
  for (n=0; n<N; ++n)
    {
      fill(x);
      fill(y);
      Vector<D> z = dot(x,y);
      for (i=0; i<D; ++i)
        for (j=0; j<D; ++j)
          z(j) -= x(i)*y(i,j);
      for (i=0; i<D; ++i)
        a += z(i)*z(i);
    }

  if ( a==0 )
    printf("PASSED");
  else
    printf("FAILED");
  printf(" Vector dot TinyMatrix\n");
}

void testTinyMatrixDotVector()
{
  Vector<D> x;
  TinyMatrix<D,D> y;
  int i,j,n;

  double a = 0;
  for (n=0; n<N; ++n)
    {
      fill(x);
      fill(y);
      Vector<D> z = dot(y,x);
      for (i=0; i<D; ++i)
        for (j=0; j<D; ++j)
          z(i) -= y(i,j)*x(j);
      for (i=0; i<D; ++i)
        a += z(i)*z(i);
    }

  if ( a==0 )
    printf("PASSED");
  else
    printf("FAILED");
  printf(" TinyMatrix dot Vector\n");
}

void testTinyMatrixDot2()
{
  const int D1 = 3;
  const int D2 = 2;
  const int D3 = 4;
  TinyMatrix<D1,D2> t1;
  TinyMatrix<D2,D3> t2;
  TinyMatrix<D1,D3> t3;

  double a = 0;
  for (int i=0; i<N; ++i)
    {
      fill(t1);
      fill(t2);
      t3 = dot(t1,t2);
      for (int i1=0; i1<D1; ++i1)
        for (int i3=0; i3<D3; ++i3)
          {
            double x = t3(i1,i3);
            for (int i2=0; i2<D2; ++i2)
              x -= t1(i1,i2)*t2(i2,i3);
            a += x*x;
          }
    }

  if ( a==0 )
    printf("PASSED");
  else 
    printf("FAILED");
  printf(" TinyMatrix<%d,%d> dot TinyMatrix<%d,%d>\n",D1,D2,D2,D3);
}

void testTinyMatrixDotVector2()
{
  const int D1 = 3;
  const int D2 = 2;
  TinyMatrix<D1,D2> t1;
  Vector<D1> v1;
  Vector<D2> v2;

  double a = 0;
  for (int i=0; i<N; ++i)
    {
      fill(t1);
      fill(v2);
      v1 = dot(t1,v2);
      for (int i1=0; i1<D1; ++i1)
        {
          double x = v1(i1);
          for (int i2=0; i2<D2; ++i2)
            x -= t1(i1,i2)*v2(i2);
          a += x*x;
        }
    }

  if ( a==0 )
    printf("PASSED");
  else 
    printf("FAILED");
  printf(" TinyMatrix<%d,%d> dot Vector<%d>\n",D1,D2,D2);
}

void testVectorDotTinyMatrix2()
{
  const int D1 = 3;
  const int D2 = 2;
  TinyMatrix<D1,D2> t1;
  Vector<D1> v1;
  Vector<D2> v2;

  double a = 0;
  for (int i=0; i<N; ++i)
    {
      fill(t1);
      fill(v1);
      v2 = dot(v1,t1);
      for (int i2=0; i2<D2; ++i2)
        {
          double x = v2(i2);
          for (int i1=0; i1<D1; ++i1)
            x -= v1(i1)*t1(i1,i2);
          a += x*x;
        }
    }

  if ( a==0 )
    printf("PASSED");
  else 
    printf("FAILED");
  printf(" Vector<%d> dot TinyMatrix<%d,%d>\n",D1,D1,D2);
}

void testVectorAccum()
{
  Vector<D> v1,v2,v3;

  double a = 0;
  for (int i=0; i<N; ++i)
    {
      fill(v1);
      fill(v2);
      v3 = v1;
      v1 += v2;
      for (int j=0; j<D; ++j)
        {
          double x = v1(j) - ( v3(j) + v2(j) );
          a += x*x;
        }

      fill(v1);
      v2 = v1;
      v1 += 73;
      for (int k=0; k<D; ++k)
        {
          double x = v1(k) - ( v2(k) + 73 );
          a += x*x;
        }
    }

  if ( a==0 )
    printf("PASSED");
  else 
    printf("FAILED");
  printf(" Vector accum\n");
}

void testTinyMatrixAccum()
{
  TinyMatrix<D,D> v1,v2,v3;
  int i,j,k;

  double a = 0;
  for (i=0; i<N; ++i)
    {
      fill(v1);
      fill(v2);
      v3 = v1;
      v1 += v2;
      for (j=0; j<D; ++j)
        for (k=0; k<D; ++k)
          {
            double x = v1(j,k) - ( v3(j,k) + v2(j,k) );
            a += x*x;
          }

      fill(v1);
      v2 = v1;
      v1 += 73;
      for (j=0; j<D; ++j)
        for (k=0; k<D; ++k)
          {
            double x = v1(j,k) - ( v2(j,k) + 73 );
            a += x*x;
          }
    }

  if ( a==0 )
    printf("PASSED");
  else 
    printf("FAILED");
  printf(" TinyMatrix accum\n");
}

int main()
{
  testTinyMatrixDot();
  testVectorDot();
  testTinyMatrixEquality();
  testVectorEquality();
  testTinyMatrixAdd();
  testVectorAdd();
  testTinyMatrixNegate();
  testVectorNegate();
  testVectorScalar();
  testTinyMatrixScalar();
  testVectorDotTinyMatrix();
  testTinyMatrixDotVector();
  testTinyMatrixDot2();
  testTinyMatrixDotVector2();
  testVectorDotTinyMatrix2();
  testVectorAccum();
  testTinyMatrixAccum();
  return 0;
}

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: t1.cpp,v $   $Author: swhaney $
// $Revision: 1.9 $   $Date: 2000/03/07 13:18:19 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
