// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

// ----------------------------------------------------------------------------
// Various tests of det(Tensor<>) global function (determinant).
// ----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Includes:
//-----------------------------------------------------------------------------

#include "Pooma/Fields.h"
#include "Utilities/Tester.h"

// Forward declarations:
template<int Dim>
void testDet(Pooma::Tester &tester);


//-----------------------------------------------------------------------------
// Main program:
//-----------------------------------------------------------------------------

int main(int argc, char *argv[])
{
  Pooma::initialize(argc,argv);
  Pooma::Tester tester(argc, argv);

  testDet<1>(tester);
  testDet<2>(tester);
  testDet<3>(tester);

  int ret = tester.results("TestDet");
  Pooma::finalize();
  return ret;
}

template<int Dim>
void testDet(Pooma::Tester &tester)
{
  // Create the physical Domains:
  const int nVerts = 6;
  const int nCells = nVerts - 1;
  int nCellsTot = 1;
  Interval<Dim> vertexDomain;
  for (int d = 0; d < Dim; d++) {
    vertexDomain[d] = Interval<1>(nVerts);
    nCellsTot *= nCells;
  }

  // Create the (uniform, logically rectilinear) mesh.
  Vector<Dim> origin(0.0), spacings(0.2);
  typedef UniformRectilinearMesh<Dim> Mesh_t;
  Mesh_t mesh(vertexDomain, origin, spacings);

  // Create geometry object:
  typedef DiscreteGeometry<Cell, Mesh_t> Geometry_t;
  Geometry_t geom(mesh);

  // Create the Fields:

  // Full, Antisymmetric, Symmetric, Diagonal Tensor Fields:
  Field<Geometry_t,Tensor<Dim,double,Full> > tff(geom);
  Field<Geometry_t,Tensor<Dim,double,Symmetric> > tfs(geom);
  Field<Geometry_t,Tensor<Dim,double,Antisymmetric> > tfa(geom);
  Field<Geometry_t,Tensor<Dim,double,Diagonal> > tfd(geom);

  // Assign values:
  Tensor<Dim,double,Full> tf(0.0);
  Tensor<Dim,double,Symmetric> ts(0.0);
  Tensor<Dim,double,Antisymmetric> ta(0.0);
  Tensor<Dim,double,Diagonal> td(0.0);
  double diagDet = 1.0;
  for (int i = 0; i < Dim; i++) {
    for (int j = 0; j < Dim; j++) {
      tf(i,j) = (i+1)*(i+1) + (j+1)*(j+1) + (i+4)*(j+4) + i;
      if (i == j) diagDet *= tf(i,j);
    }
  }
  ts = symmetrize<Symmetric>(tf);
  ta = symmetrize<Antisymmetric>(tf);
  td = symmetrize<Diagonal>(tf);
  tff = tf;
  tfs = ts;
  tfa = ta;
  tfd = td;

  // Test det of Full Tensor:
  double detValue;
  detValue = sum(det(tff));
  switch (Dim) {
  case 1:
    if (!tester.check("detValue", detValue, 18.0*nCellsTot)) {
      tester.out() << Dim << "D, sum(det(tff)) = " << detValue
                   << " != 18.0*nCellsTot = " << 18.0*nCellsTot << std::endl;
    }
    break;
  case 2:
    if (!tester.check("detValue", detValue, -38.0*nCellsTot)) {
      tester.out() << Dim << "D, sum(det(tff)) = " << detValue
                   << " != -38.0*nCellsTot = " << -38.0*nCellsTot << std::endl;
    }
    break;
  case 3:
    if (!tester.check("detValue", detValue, -4.0*nCellsTot)) {
      tester.out() << Dim << "D, sum(det(tff)) = " << detValue
                   << " != -4.0*nCellsTot = " << -4.0*nCellsTot << std::endl;
    }
    break;
  default:
    PInsist(Dim<4, "Tensor det() function not implemented for D>3!");
    break;
  }

  // Test det of Symmetric Tensor:
  detValue = sum(det(tfs));
  switch (Dim) {
  case 1:
    if (!tester.check("detValue", detValue, 18.0*nCellsTot)) {
      tester.out() << Dim << "D, sum(det(tfs)) = " << detValue
                   << " != 18.0*nCellsTot = " << 18.0*nCellsTot << std::endl;
    }
    break;
  case 2:
    if (!tester.check("detValue", detValue, -38.25*nCellsTot)) {
      tester.out() << Dim << "D, sum(det(tfs)) = " << detValue
                   << " != -38.25*nCellsTot = " 
                   << -38.25*nCellsTot << std::endl;
    }
    break;
  case 3:
    if (!tester.check("detValue", detValue, -4.0*nCellsTot)) {
      tester.out() << Dim << "D, sum(det(tfs)) = " << detValue
                   << " != -4.0*nCellsTot = " << -4.0*nCellsTot << std::endl;
    }
    break;
  default:
    PInsist(Dim<4, "Tensor det() function not implemented for D>3!");
    break;
  }

  // Test det of Antiymmetric Tensor:
  detValue = sum(det(tfa));
  switch (Dim) {
  case 1:
    if (!tester.check("detValue", detValue, 0.0)) {
      tester.out() << Dim << "D, sum(det(tfa)) = " << detValue
                   << " != 0.0 ." << std::endl;
    }
    break;
  case 2:
    if (!tester.check("detValue", detValue, ta(1,0)*ta(1,0)*nCellsTot)) {
      tester.out() << Dim << "D, sum(det(tfa)) = " << detValue
                   << " != ta(1,0)*ta(1,0)*nCellsTot = " 
                   << ta(1,0)*ta(1,0)*nCellsTot << std::endl;
    }
    break;
  case 3:
    if (!tester.check("detValue", detValue, 0.0)) {
      tester.out() << Dim << "D, sum(det(tfa)) = " << detValue
                   << " != 0.0 ." << std::endl;
    }
    break;
  default:
    PInsist(Dim<4, "Tensor det() function not implemented for D>3!");
    break;
  }

  // Test det of Diagonal Tensor:
  detValue = sum(det(tfd));
  if (Dim <= 3) {
    if (!tester.check("detValue", detValue, diagDet*nCellsTot)) {
      tester.out() << Dim << "D, sum(det(tfd)): " << detValue 
                   << " != diagDet*nCellsTot = " 
                   << diagDet*nCellsTot << std::endl;
    }
  } else {
    PInsist(Dim<4, "Tensor det() function not implemented for D>3!");
  }

}

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: TestDet.cpp,v $   $Author: swhaney $
// $Revision: 1.2 $   $Date: 2000/03/07 13:18:17 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
