// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

//-----------------------------------------------------------------------------
// Classes: 
//   Cell
//   Vert
//   Edge
//   Face
//   DistributeSubFields
//   ReplicateSubFields
//-----------------------------------------------------------------------------

#ifndef POOMA_NEWFIELD_FIELDINITIALIZERS_H
#define POOMA_NEWFIELD_FIELDINITIALIZERS_H

//-----------------------------------------------------------------------------
// Overview: 
// 
// POOMA supports a hierarchy of multiple centering points per 
// cell. The centering information, manageed by the FieldEngineBase
// class, is initialized using a flexible set of functors. Below are some
// pre-defined functors that set up some common centerings along with some
// versions for building multi-material and multi-centered fields.
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Includes:
//-----------------------------------------------------------------------------

#include "Domain/Loc.h"
#include "Utilities/PAssert.h"

#include <iostream>

//-----------------------------------------------------------------------------
// Forward declarations:
//-----------------------------------------------------------------------------

/* None for now */

// ----------------------------------------------------------------------------
// Full description:
//
// Initialization functors for the Field class:
//   o Cell:                 cell-centered in every direction
//   o Vert:                 vertex-centered in every direction
//   o Edge                  cell-centered in specified direction, 
//                           vertex-centered otherwise
//   o Face:                 vertex-centered in specified direction, 
//                           cell-centered otherwise
//   o DistributeSubFields: distributes a centering along orientations
//                           corresponding to coordinate directions
//                           Example: AllEdge..edge-centered in every direction
//                           Example: AllFace..face-centered in every direction
//   o ReplicateSubFields:  replicates a centering N times
//                           Example: multi-material fields
//
// Usage Examples:
//
//  Interval<3> I(10, 10, 10), J(5, 5, 5);
//  FieldEngineBase<3, double, Brick> 
//    c1(Cell(), I),
//    c2(Edge(1), I),                             
//    c3(DistributeSubFields<Face>(), I),      
//    c4(ReplicateSubFields<DistributeSubFields<Face> >
//      (DistributeSubFields<Face>(), 5), I); 
//
// Declares a cell-centering, an edge-centering aligned with the Y axis, an
// all-face centering, and a replicated all-face centering.
// ----------------------------------------------------------------------------

class Cell {
public:

  inline Cell() { }
  
  template<class Subject, class Initializer>
  inline void initialize(Subject &s, const Initializer &init) const
    {
      Loc<Subject::dimensions> l(0);
      s.initialize(l, init);
    }
};

class Vert {
public:

  inline Vert() { }
  
  template<class Subject, class Initializer>
  inline void initialize(Subject &s, const Initializer &init) const
    {
      Loc<Subject::dimensions> l(1);
      s.initialize(l, init);
    }
};

class Edge {
public:

  Edge(int orientation) : orientation_m(orientation) { }
  
  template<class Subject, class Initializer>
  inline void initialize(Subject &s, const Initializer &init) const
    {
      Loc<Subject::dimensions> l(1); l[orientation_m] = Loc<1>(0);
      s.initialize(l, init);
    }

private:

  int orientation_m;  
};

class Face {
public:

  Face(int orientation) : orientation_m(orientation) { }
  
  template<class Subject, class Initializer>
  inline void initialize(Subject &s, const Initializer &init) const
    {
      Loc<Subject::dimensions> l(0); l[orientation_m] = Loc<1>(1);
      s.initialize(l, init);
    }

private:

  int orientation_m;  
};

template<class S>
class DistributeSubFields {
public:

  DistributeSubFields() { }
  
  template<class Subject, class Initializer>
  inline void initialize(Subject &s, const Initializer &init) const
    {
      s.addSubFields(Subject::dimensions);
      for (int i = 0; i < Subject::dimensions; i++)
        S(i).initialize(s.subField(i), init);
    }
};

typedef DistributeSubFields<Edge> AllEdge;
typedef DistributeSubFields<Face> AllFace;

template<class S>
class ReplicateSubFields {
public:

  explicit ReplicateSubFields(int n) : n_m(n) { }
  ReplicateSubFields(const S &c, int n) : subField_m(c), n_m(n) { }
  
  template<class Subject, class Initializer>
  inline void initialize(Subject &s, const Initializer &init) const
    {
      s.addSubFields(n_m);
      for (int i = 0; i < n_m; i++)
        subField_m.initialize(s.subField(i), init);
    }
    
private:

  S subField_m;
  int n_m;
};

#endif // POOMA_NEWFIELD_FIELDINITIALIZERS_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: FieldInitializers.h,v $   $Author: swhaney $
// $Revision: 1.3 $   $Date: 2000/07/24 23:36:22 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
