// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license
//-----------------------------------------------------------------------------
// SparseTileLayout test: Create and use SparseTileLayout objects
//-----------------------------------------------------------------------------
#include "Pooma/Pooma.h"
#include "Pooma/Domains.h"
#include "Pooma/GMPArrays.h"
#include "Layout/SparseTileLayout.h"
#include "Partition/TilePartition.h"
#include "Utilities/Tester.h"
#include "Pooma/Arrays.h"
#include <iostream>
#include <iterator>

typedef Brick                                       PatchEngineTag_t;
typedef  MultiPatch<SparseTileTag,PatchEngineTag_t> mp_t;
typedef  Array<2,double,mp_t>                       Array_t;


int main(int argc, char *argv[]) 
{
  Pooma::initialize(argc, argv);


  // Initialize POOMA and output stream, using Tester class


  Pooma::Tester tester(argc, argv); 
  tester.out() << argv[0] << ": SparseTileLayout operations." << std::endl;
  tester.out() << "----------------------------------------" << std::endl;

  typedef SparseTileLayout<2>::Domain_t Domain_t;
  Domain_t f(Interval<1>(0,9),Interval<1>(0,9));

  typedef SparseTileLayout<2>::PatchList_t  PatchList_t;
  PatchList_t plist;

  plist.push_back(Interval<2>(Interval<1>(0,4),Interval<1>(0,4)));
  plist.push_back(Interval<2>(Interval<1>(5,9),Interval<1>(0,4)));
  plist.push_back(Interval<2>(Interval<1>(0,4),Interval<1>(5,9)));
  plist.push_back(Interval<2>(Interval<1>(5,9),Interval<1>(5,9)));

  GuardLayers<2> igl(2),egl(2);

  TilePartition<2> tp(plist,igl,egl);

  SparseTileLayout<2> stl_pl(f,plist,ReplicatedTag());

  stl_pl.print(tester.out());

  stl_pl.syncPatch();

  SparseTileLayout<2> pp(f,tp,ReplicatedTag());

  pp.syncPatch();

  tester.out()<< std::endl;
  tester.out()<< " printing out the sparse tile layout " <<std::endl;
  tester.out()<< "   4 equal size patches tile the domain " <<std::endl;
  tester.out()<< "   this is equivalent to a "<<std::endl;
  tester.out()<< "    UGL(domain,Loc<2>(2),GuardLayers<2>(2))"<<std::endl;

  pp.print(tester.out());

  //*************
  
  PatchList_t pplist;
  
  pplist.push_back(Interval<2>(Interval<1>(0,4),Interval<1>(3,9)));
  pplist.push_back(Interval<2>(Interval<1>(5,9),Interval<1>(0,7)));

  TilePartition<2> tp2(pplist,igl,egl);
  SparseTileLayout<2> twopatch(f,tp2,ReplicatedTag());

  Array_t STa(twopatch);

  STa = 1.1;


  // this bit assigns border regions. 

  SparseTileLayout<2>::BorderFillIterator_t
    tp_b_start = twopatch.beginBorderFillList();

  SparseTileLayout<2>::BorderFillIterator_t
    tp_b_end = twopatch.endBorderFillList();
  
  tester.out() << " testing assigning into border regions " << std::endl;

  tester.out() << " layout is: " << std::endl;
  tester.out() << twopatch << std::endl;


  for ( ; tp_b_start != tp_b_end ; ++tp_b_start )
    {
      tester.out() << " domain "<< tp_b_start->domain() 
		   << " Patch id "<< tp_b_start->patchID()<<std::endl;

      tester.out() << " domain of the patch is "<<
	STa.patchLocal( tp_b_start->patchID()).domain()<<std::endl;

      STa.patchLocal( tp_b_start->patchID() ) ( tp_b_start->domain() ) = 2.2;

      tester.out() << " patch "<<tp_b_start->patchID() <<std::endl;
      tester.out() <<  STa.patchLocal( tp_b_start->patchID()) <<std::endl;
      
      tester.out() << "view of the same patch "<<tp_b_start->patchID() <<std::endl;
      tester.out() <<  STa.patchLocal( tp_b_start->patchID())() <<std::endl;
    }


  // SparseTL based arrays work in expressions...
 
  Array_t STb(twopatch);
  Array_t STc(twopatch);
 

  STa = 1.0;        
  STb = 9.9;

  STc = STa + (STb - STa);

  tester.out()<< " print out the sparse tile layout based array "<<std::endl;

  tester.out()<<STc <<std::endl;

  tester.out()<< " print an expression using a STlayout "<<std::endl;

  tester.out()<<  17 + (STc+4)*3 <<std::endl;

 
 
  // Now assign to views that include undefined areas.


  Interval<2> view(Interval<1>(2,6),Interval<1>(0,9));
  STc(view) = .5*STa(view);

  tester.out()<<STc <<std::endl;

  tester.out() << "-------------------------------------------" << std::endl;
  int retval = tester.results("SparseTileLayout operations");
  Pooma::finalize();
  return retval;
} 

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: sparsetilelayout_test.cpp,v $   $Author: luchini $
// $Revision: 1.13 $   $Date: 2000/06/27 19:11:53 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
