// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

//-----------------------------------------------------------------------------
// Classes: 
//   UniformGridLayout<Dim>
//   UniformGridLayoutView<Dim, Dim2>
//   UniformTag
//   MultiPatchLayoutTraits<UniformTag,Dim>
//-----------------------------------------------------------------------------

#ifndef POOMA_LAYOUT_UNIFORMGRIDLAYOUT_H
#define POOMA_LAYOUT_UNIFORMGRIDLAYOUT_H

//-----------------------------------------------------------------------------
// Overview:
//
//   UniformGridLayout<Dim>
//     - Layout class that breaks Dim-dimensional domain into equal
//       sized sub-domains arranged in a Dim-dimensional grid.
//   UniformGridLayoutView<Dim, Dim2>
//     - view of a UniformGridLayout
//   UniformTag
//     - tag used to specialize MultiPatchLayoutTraits
//   MultiPatchLayoutTraits<UniformTag,Dim>
//     - traits class used by MultiPatch-engine to determine layout type.
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Includes:
//-----------------------------------------------------------------------------

#include "Layout/MultiPatchLayoutTraits.h"
#include "Layout/INode.h"
#include "Layout/TouchesConstruct.h"
#include "Layout/GuardLayers.h"
#include "Domain/Interval.h"
#include "Domain/Contains.h"
#include "Domain/Intersect.h"
#include "Domain/NewDomain.h"
#include "Domain/SliceRange.h"
#include "Partition/UniformGridPartition.h"
#include "Partition/ContextMapper.h"
#include "Utilities/DerefIterator.h"
#include "Utilities/ViewIndexer.h"
#include "Utilities/Observable.h"
#include "Utilities/Observer.h"
#include "Utilities/RefCountedPtr.h"
#include "Utilities/RefCounted.h"
#include "Utilities/Unique.h"
#include "Utilities/PAssert.h"

#include "Layout/LayoutBase.h"

#include <vector>
#include <iosfwd>


///////////////////////////////////////////////////////////////////////////////
// namespace Pooma {

//============================================================
// Forward declaration
//============================================================

template <int Dim> class UniformGridLayout;
template <int Dim, int Dim2> class UniformGridLayoutView;

//-----------------------------------------------------------------------------
// Full Description:
// UniformTag
//
// Tag class.
//-----------------------------------------------------------------------------

struct UniformTag { };


//-----------------------------------------------------------------------------
// Full Description:
// MultiPatchLayoutTraits<UniformTag,Dim>
//
// Specialization of MultiPatchLayoutTraits for UniformGridLayout.
//-----------------------------------------------------------------------------

template <int Dim>
struct MultiPatchLayoutTraits<UniformTag,Dim>
{
  typedef UniformGridLayout<Dim> Layout_t;
  
  template <int ViewDim>
  struct View
  {
    typedef UniformGridLayoutView<ViewDim,Dim> Layout_t;
  };
};


//-----------------------------------------------------------------------------
//
// Full Description:
// UniformGridLayoutData<Dim>
//
// This is the actual data for the UniformGridLayout class, which is
// simply a wrapper that holds a reference counted instance of this
// data class.
//
//-----------------------------------------------------------------------------

template <int Dim>
class UniformGridLayoutData 
 : public LayoutBaseData<Dim>, 
   public RefCounted, 
   public Observable<UniformGridLayoutData<Dim> >
{
public:
  //============================================================
  // Typedefs and enumerations
  //============================================================

  // General public typedefs.

  typedef Interval<Dim>           Domain_t;
  typedef Interval<Dim>           BaseDomain_t;
  typedef int                     Context_t;
  typedef Unique::Value_t         ID_t;
  typedef Node<Domain_t>          Value_t;
  typedef std::vector<Value_t *>  List_t;           // for convenience
  typedef GuardLayers<Dim>        GuardLayers_t;    // for convenience
  
  typedef typename LayoutBaseData<Dim>::GCFillInfo GCFillInfo_t;
  
  typedef typename std::vector<GCFillInfo_t>::const_iterator FillIterator_t;
  
  // Enumerations.

  enum { dimensions = Dim };
  enum { repartitionEvent = 1 };
  enum { dynamic = false };


  //============================================================
  // Constructors
  //============================================================

  // Default version creates an empty domain.
  
  UniformGridLayoutData();
  
  // All initialization is done via the following constructor.
  // Originally we had several constructors, but once we went
  // to not storing a partitioner object, it was easier to just
  // have the Layout construct the partitioner appropriately
  // and pass it to the LayoutData constructor.
  
  template <class Partitioner>
  UniformGridLayoutData(const Domain_t &gdom, 
			const Partitioner &gpar,
			const ContextMapper<Dim> & cmap );


  //============================================================
  // Special I/O initializer
  //============================================================
  // Used by the I/O or data management system to initialize the layout based
  // on detailed state information previously stored.

  void initialize(const Domain_t& idom,
		  const List_t& nodes,
		  const Loc<Dim>& blocks,
		  bool hasIG, bool hasEG,
		  const GuardLayers_t& ig,
		  const GuardLayers_t& eg);


  //============================================================
  // Destructor
  //============================================================

  // This is only called when all references to this data go away, in
  // which case we need to delete our nodes. The Observable destructor
  // will broadcast messages up to all observers of the Layout.

  ~UniformGridLayoutData()
  {
    typename List_t::iterator a;
    for (a = all_m.begin(); a != all_m.end(); ++a)
      delete (*a);
  }


  // Find all subdomains that touch on a given domain, and insert the
  // intersection of these subdomains into the given output iterator.
  // Return the number of touching elements. This version of touches
  // can build either pointers or objects.

  template <class OtherDomain, class OutIter, class ConstructTag>
  int touches(const OtherDomain &d, OutIter o, const ConstructTag &ctag) const 
  {
    int i, count = 0;

    // Make sure we have a valid touching domain.

    PAssert(initialized());
    PAssert(contains(domain_m, d));

    // Find the starting and ending grid positions, and store as an
    // Interval.

    Interval<Dim> box = Pooma::NoInit();
    for (i = 0; i < Dim; ++i) 
      {
        int a, b;
        if (!hasExternalGuards_m)
          {
            a = (d[i].min() - firsti_m[i]) / blocksizes_m[i];
            b = (d[i].max() - firsti_m[i]) / blocksizes_m[i];
          }
        else
          {
            // If we're in the lower guards, this will fall
            // through to give zero.
            
            a = b = 0;
            int pos = d[i].min();
            int last = innerdomain_m[i].last();
            int del = pos - firsti_m[i];
            
            if (del >= 0)
              if (pos <= last)
                a = del / blocksizes_m[i];
              else
                a = allDomain_m[i].last();
            
            pos = d[i].max();
            del = pos - firsti_m[i];
            
            if (del >= 0)
              if (pos <= last)
                b = del / blocksizes_m[i];
              else
                b = allDomain_m[i].last();
          }
        box[i] = Interval<1>(a, b);
      }

    // Figure the type of the domain resulting from the intersection.

    typedef typename 
      IntersectReturnType<Domain_t,OtherDomain>::Type_t OutDomain_t;
    OutDomain_t outDomain = Pooma::NoInit();

    // Generate the type of the node pushed on the output iterator.

    typedef Node<OutDomain_t,Domain_t> OutNode_t;

    // Iterate through the Interval grid positions.

    Interval<Dim>::const_iterator boxiter = box.begin();
    
    while (boxiter != box.end()) 
      {
        // Calculate the linear position of the current node.
        
        int indx = (*boxiter)[0].first();
        for (i = 1; i < Dim; ++i)
          indx += blockstride_m[i] * (*boxiter)[i].first();

        // Get that node, intersect the domain with the requested one,
        // and write the result out to the output iterator.
        
        PAssert(indx >= 0 && indx < all_m.size());
        
        outDomain = intersect(d, all_m[indx]->domain());

        PAssert(!outDomain.empty());
        
        *o = touchesConstruct(outDomain,
                              all_m[indx]->allocated(),
                              all_m[indx]->affinity(),
                              all_m[indx]->context(),
                              all_m[indx]->globalID(),
                              all_m[indx]->localID(),
                              ctag);

        // Increment output iterator, count, and grid iterator.

        ++o;
        ++count;
        ++boxiter;
      }

      // Return the number of non-empty domains we found.
      
      return count;
  }

  template <class OtherDomain, class OutIter, class ConstructTag>
  int touchesLocal(const OtherDomain &d, 
		   OutIter o, 
		   const ConstructTag &ctag) const 
  {
    int i, count = 0;

    // Make sure we have a valid touching domain.

    PAssert(initialized());
    PAssert(contains(domain_m, d));

    // Find the starting and ending grid positions, and store as an
    // Interval.

    Interval<Dim> box = Pooma::NoInit();
    for (i = 0; i < Dim; ++i) 
      {
        int a, b;
        if (!hasExternalGuards_m)
          {
            a = (d[i].min() - firsti_m[i]) / blocksizes_m[i];
            b = (d[i].max() - firsti_m[i]) / blocksizes_m[i];
          }
        else
          {
            // If we're in the lower guards, this will fall
            // through to give zero.
            
            a = b = 0;
            int pos = d[i].min();
            int last = innerdomain_m[i].last();
            int del = pos - firsti_m[i];
            
            if (del >= 0)
              if (pos <= last)
                a = del / blocksizes_m[i];
              else
                a = allDomain_m[i].last();
            
            pos = d[i].max();
            del = pos - firsti_m[i];
            
            if (del >= 0)
              if (pos <= last)
                b = del / blocksizes_m[i];
              else
                b = allDomain_m[i].last();
          }
        box[i] = Interval<1>(a, b);
      }

    // Figure the type of the domain resulting from the intersection.

    typedef typename 
      IntersectReturnType<Domain_t,OtherDomain>::Type_t OutDomain_t;
    OutDomain_t outDomain = Pooma::NoInit();

    // Generate the type of the node pushed on the output iterator.

    typedef Node<OutDomain_t,Domain_t> OutNode_t;

    // Iterate through the Interval grid positions.

    Interval<Dim>::const_iterator boxiter = box.begin();
    
    while (boxiter != box.end()) 
      {
        // Calculate the linear position of the current node.
        
        int indx = (*boxiter)[0].first();
        for (i = 1; i < Dim; ++i)
          indx += blockstride_m[i] * (*boxiter)[i].first();

        // Get that node, intersect the domain with the requested one,
        // and write the result out to the output iterator.
        
        PAssert(indx >= 0 && indx < local_m.size());
        
        outDomain = intersect(d, local_m[indx]->domain());

        PAssert(!outDomain.empty());
        
        *o = touchesConstruct(outDomain,
                              local_m[indx]->allocated(),
                              local_m[indx]->affinity(),
                              local_m[indx]->context(),
                              local_m[indx]->globalID(),
                              local_m[indx]->localID(),
                              ctag);

        // Increment output iterator, count, and grid iterator.

        ++o;
        ++count;
        ++boxiter;
      }

      // Return the number of non-empty domains we found.
      
      return count;
  }

  // Find all Remote subdomains that touch on a given domain, and insert the
  // intersection of these subdomains into the given output iterator.
  // Return the number of touching elements. This version of touches
  // can build either pointers or objects.

  template <class OtherDomain, class OutIter, class ConstructTag>
  int touchesRemote(const OtherDomain &d, 
		    OutIter o, 
		    const ConstructTag &ctag) const 
  {
    int i, count = 0;

    // Make sure we have a valid touching domain.

    PAssert(initialized());
    PAssert(contains(domain_m, d));

    // Find the starting and ending grid positions, and store as an
    // Interval.

    Interval<Dim> box = Pooma::NoInit();
    for (i = 0; i < Dim; ++i) 
      {
        int a, b;
        if (!hasExternalGuards_m)
          {
            a = (d[i].min() - firsti_m[i]) / blocksizes_m[i];
            b = (d[i].max() - firsti_m[i]) / blocksizes_m[i];
          }
        else
          {
            // If we're in the lower guards, this will fall
            // through to give zero.
            
            a = b = 0;
            int pos = d[i].min();
            int last = innerdomain_m[i].last();
            int del = pos - firsti_m[i];
            
            if (del >= 0)
              if (pos <= last)
                a = del / blocksizes_m[i];
              else
                a = allDomain_m[i].last();
            
            pos = d[i].max();
            del = pos - firsti_m[i];
            
            if (del >= 0)
              if (pos <= last)
                b = del / blocksizes_m[i];
              else
                b = allDomain_m[i].last();
          }
        box[i] = Interval<1>(a, b);
      }

    // Figure the type of the domain resulting from the intersection.

    typedef typename 
      IntersectReturnType<Domain_t,OtherDomain>::Type_t OutDomain_t;
    OutDomain_t outDomain = Pooma::NoInit();

    // Generate the type of the node pushed on the output iterator.

    typedef Node<OutDomain_t,Domain_t> OutNode_t;

    // Iterate through the Interval grid positions.

    Interval<Dim>::const_iterator boxiter = box.begin();
    
    while (boxiter != box.end()) 
      {
        // Calculate the linear position of the current node.
        
        int indx = (*boxiter)[0].first();
        for (i = 1; i < Dim; ++i)
          indx += blockstride_m[i] * (*boxiter)[i].first();

        // Get that node, intersect the domain with the requested one,
        // and write the result out to the output iterator.
        
        PAssert(indx >= 0 && indx < remote_m.size());
        
        outDomain = intersect(d, remote_m[indx]->domain());

        PAssert(!outDomain.empty());
        
        *o = touchesConstruct(outDomain,
                              remote_m[indx]->allocated(),
                              remote_m[indx]->affinity(),
                              remote_m[indx]->context(),
                              remote_m[indx]->globalID(),
                              remote_m[indx]->localID(),
                              ctag);

        // Increment output iterator, count, and grid iterator.

        ++o;
        ++count;
        ++boxiter;
      }

      // Return the number of non-empty domains we found.
      
      return count;
  }

  template <class OtherDomain, class OutIter, class ConstructTag>
  int touchesAlloc(const OtherDomain &d, OutIter o, 
                   const ConstructTag &ctag) const 
  {
    // If there are no internal guard cells, then this calculation is the
    // same as the normal touches calculation.
    
    if (!hasInternalGuards_m) return touches(d,o,ctag);
    
    int i, count = 0;

    // Make sure we have a valid touching domain.

    PAssert(initialized());
    PAssert(contains(domain_m, d));

    // Find the starting and ending grid positions, and store as an
    // Interval.

    Interval<Dim> box = Pooma::NoInit();
    for (i = 0; i < Dim; ++i) 
      {
        int a, b;
        
        // This part is unchanged from touches. What we're going to do
	// here is simply extend the range in each direction by one block
	// and then let the intersection calculation below sort out if
	// there is actually an intersection. Otherwise we'd have to do
	// comparisons on all blocks up here and then still do the
	// intersection on the remaining blocks below. On average, this
	// should be slightly faster I think.
        
        if (!hasExternalGuards_m)
          {
            a = (d[i].min() - firsti_m[i]) / blocksizes_m[i];
            b = (d[i].max() - firsti_m[i]) / blocksizes_m[i];
          }
        else
          {
            // If we're in the lower guards, this will fall
            // through to give zero.
            
            a = b = 0;
            int pos = d[i].min();
            int last = innerdomain_m[i].last();
            int del = pos - firsti_m[i];
            
            if (del >= 0)
              if (pos <= last)
                a = del / blocksizes_m[i];
              else
                a = allDomain_m[i].last();
            
            pos = d[i].max();
            del = pos - firsti_m[i];
            
            if (del >= 0)
              if (pos <= last)
                b = del / blocksizes_m[i];
              else
                b = allDomain_m[i].last();
          }
          
        // Now we check that we're not at the ends of the domain for the
        // brick of blocks, and extend the region accordingly.
        
        if (a > 0) --a;
        if (b < allDomain_m[i].last()) ++b;
        
        box[i] = Interval<1>(a, b);
      }

    // Figure the type of the domain resulting from the intersection.

    typedef typename 
      IntersectReturnType<Domain_t,OtherDomain>::Type_t OutDomain_t;
    OutDomain_t outDomain = Pooma::NoInit();

    // Generate the type of the node pushed on the output iterator.

    typedef Node<OutDomain_t,Domain_t> OutNode_t;

    // Iterate through the Interval grid positions.

    Interval<Dim>::const_iterator boxiter = box.begin();
    
    while (boxiter != box.end()) 
      {
        // Calculate the linear position of the current node.
        
        int indx = (*boxiter)[0].first();
        for (i = 1; i < Dim; ++i)
          indx += blockstride_m[i] * (*boxiter)[i].first();

        // Get that node, intersect the *allocated* domain with the 
        // requested one, and write the result out to the output iterator.
        
        PAssert(indx >= 0 && indx < all_m.size());
        
        outDomain = intersect(d, all_m[indx]->allocated());

        // We can no longer assert that outDomain is not empty since
        // we extended the search box without checking. Thus we now 
        // have an if tests around the output.
        
        if (!outDomain.empty())
          {
            *o = touchesConstruct(outDomain,
                                  all_m[indx]->allocated(),
                                  all_m[indx]->affinity(),
                                  all_m[indx]->context(),
                                  all_m[indx]->globalID(),
                                  all_m[indx]->localID(),
                                  ctag);
          }

        // Increment output iterator, count, and grid iterator.

        ++o;
        ++count;
        ++boxiter;
      }

      // Return the number of non-empty domains we found.
      
      return count;
  }

  template <class OtherDomain, class OutIter, class ConstructTag>
  int touchesAllocLocal(const OtherDomain &d, OutIter o, 
                   const ConstructTag &ctag) const 
  {
    // If there are no internal guard cells, then this calculation is the
    // same as the normal touches calculation.
    
    if (!hasInternalGuards_m) return touches(d,o,ctag);
    
    int i, count = 0;

    // Make sure we have a valid touching domain.

    PAssert(initialized());
    PAssert(contains(domain_m, d));

    // Find the starting and ending grid positions, and store as an
    // Interval.

    Interval<Dim> box = Pooma::NoInit();
    for (i = 0; i < Dim; ++i) 
      {
        int a, b;
        
        // This part is unchanged from touches. What we're going to do
	// here is simply extend the range in each direction by one block
	// and then let the intersection calculation below sort out if
	// there is actually an intersection. Otherwise we'd have to do
	// comparisons on all blocks up here and then still do the
	// intersection on the remaining blocks below. On average, this
	// should be slightly faster I think.
        
        if (!hasExternalGuards_m)
          {
            a = (d[i].min() - firsti_m[i]) / blocksizes_m[i];
            b = (d[i].max() - firsti_m[i]) / blocksizes_m[i];
          }
        else
          {
            // If we're in the lower guards, this will fall
            // through to give zero.
            
            a = b = 0;
            int pos = d[i].min();
            int last = innerdomain_m[i].last();
            int del = pos - firsti_m[i];
            
            if (del >= 0)
              if (pos <= last)
                a = del / blocksizes_m[i];
              else
                a = allDomain_m[i].last();
            
            pos = d[i].max();
            del = pos - firsti_m[i];
            
            if (del >= 0)
              if (pos <= last)
                b = del / blocksizes_m[i];
              else
                b = allDomain_m[i].last();
          }
          
        // Now we check that we're not at the ends of the domain for the
        // brick of blocks, and extend the region accordingly.
        
        if (a > 0) --a;
        if (b < allDomain_m[i].last()) ++b;
        
        box[i] = Interval<1>(a, b);
      }

    // Figure the type of the domain resulting from the intersection.

    typedef typename 
      IntersectReturnType<Domain_t,OtherDomain>::Type_t OutDomain_t;
    OutDomain_t outDomain = Pooma::NoInit();

    // Generate the type of the node pushed on the output iterator.

    typedef Node<OutDomain_t,Domain_t> OutNode_t;

    // Iterate through the Interval grid positions.

    Interval<Dim>::const_iterator boxiter = box.begin();
    
    while (boxiter != box.end()) 
      {
        // Calculate the linear position of the current node.
        
        int indx = (*boxiter)[0].first();
        for (i = 1; i < Dim; ++i)
          indx += blockstride_m[i] * (*boxiter)[i].first();

        // Get that node, intersect the *allocated* domain with the 
        // requested one, and write the result out to the output iterator.
        
        PAssert(indx >= 0 && indx < local_m.size());
        
        outDomain = intersect(d, local_m[indx]->allocated());

        // We can no longer assert that outDomain is not empty since
        // we extended the search box without checking. Thus we now 
        // have an if tests around the output.
        
        if (!outDomain.empty())
          {
            *o = touchesConstruct(outDomain,
                                  local_m[indx]->allocated(),
                                  local_m[indx]->affinity(),
                                  local_m[indx]->context(),
                                  local_m[indx]->globalID(),
                                  local_m[indx]->localID(),
                                  ctag);
          }

        // Increment output iterator, count, and grid iterator.

        ++o;
        ++count;
        ++boxiter;
      }

      // Return the number of non-empty domains we found.
      
      return count;
  }

  template <class OtherDomain, class OutIter, class ConstructTag>
  int touchesAllocRemote(const OtherDomain &d, OutIter o, 
                   const ConstructTag &ctag) const 
  {
    // If there are no internal guard cells, then this calculation is the
    // same as the normal touches calculation.
    
    if (!hasInternalGuards_m) return touches(d,o,ctag);
    
    int i, count = 0;

    // Make sure we have a valid touching domain.

    PAssert(initialized());
    PAssert(contains(domain_m, d));

    // Find the starting and ending grid positions, and store as an
    // Interval.

    Interval<Dim> box = Pooma::NoInit();
    for (i = 0; i < Dim; ++i) 
      {
        int a, b;
        
        // This part is unchanged from touches. What we're going to do
	// here is simply extend the range in each direction by one block
	// and then let the intersection calculation below sort out if
	// there is actually an intersection. Otherwise we'd have to do
	// comparisons on all blocks up here and then still do the
	// intersection on the remaining blocks below. On average, this
	// should be slightly faster I think.
        
        if (!hasExternalGuards_m)
          {
            a = (d[i].min() - firsti_m[i]) / blocksizes_m[i];
            b = (d[i].max() - firsti_m[i]) / blocksizes_m[i];
          }
        else
          {
            // If we're in the lower guards, this will fall
            // through to give zero.
            
            a = b = 0;
            int pos = d[i].min();
            int last = innerdomain_m[i].last();
            int del = pos - firsti_m[i];
            
            if (del >= 0)
              if (pos <= last)
                a = del / blocksizes_m[i];
              else
                a = allDomain_m[i].last();
            
            pos = d[i].max();
            del = pos - firsti_m[i];
            
            if (del >= 0)
              if (pos <= last)
                b = del / blocksizes_m[i];
              else
                b = allDomain_m[i].last();
          }
          
        // Now we check that we're not at the ends of the domain for the
        // brick of blocks, and extend the region accordingly.
        
        if (a > 0) --a;
        if (b < allDomain_m[i].last()) ++b;
        
        box[i] = Interval<1>(a, b);
      }

    // Figure the type of the domain resulting from the intersection.

    typedef typename 
      IntersectReturnType<Domain_t,OtherDomain>::Type_t OutDomain_t;
    OutDomain_t outDomain = Pooma::NoInit();

    // Generate the type of the node pushed on the output iterator.

    typedef Node<OutDomain_t,Domain_t> OutNode_t;

    // Iterate through the Interval grid positions.

    Interval<Dim>::const_iterator boxiter = box.begin();
    
    while (boxiter != box.end()) 
      {
        // Calculate the linear position of the current node.
        
        int indx = (*boxiter)[0].first();
        for (i = 1; i < Dim; ++i)
          indx += blockstride_m[i] * (*boxiter)[i].first();

        // Get that node, intersect the *allocated* domain with the 
        // requested one, and write the result out to the output iterator.
        
        PAssert(indx >= 0 && indx < remote_m.size());
        
        outDomain = intersect(d, remote_m[indx]->allocated());

        // We can no longer assert that outDomain is not empty since
        // we extended the search box without checking. Thus we now 
        // have an if tests around the output.
        
        if (!outDomain.empty())
          {
            *o = touchesConstruct(outDomain,
                                  remote_m[indx]->allocated(),
                                  remote_m[indx]->affinity(),
                                  remote_m[indx]->context(),
                                  remote_m[indx]->globalID(),
                                  remote_m[indx]->localID(),
                                  ctag);
          }

        // Increment output iterator, count, and grid iterator.

        ++o;
        ++count;
        ++boxiter;
      }

      // Return the number of non-empty domains we found.
      
      return count;
  }


  //private:

  friend class UniformGridLayout<Dim>;

  //============================================================
  // Accessors
  //============================================================


  // These are deferred to UniformGridLayoutData in order to avoid a
  // bunch of indirections in their implementation.

  // Accessors for getting the global ID of the patch containing a
  // particular element.

  int globalID(const Loc<Dim> &loc) const;
  int globalID(int) const;
  int globalID(int,int) const;
  int globalID(int,int,int) const;
  int globalID(int,int,int,int) const;
  int globalID(int,int,int,int,int) const;
  int globalID(int,int,int,int,int,int) const;
  int globalID(int,int,int,int,int,int,int) const;
  
  // Mutators used for partitioning.

  template <class Partitioner>
  void partition(const Partitioner &, const ContextMapper<Dim>& cmap);

  template <class Partitioner>
  bool repartition(const Partitioner &,const ContextMapper<Dim>&);
  
  // This function calculates the cached guard-cell filling information.
  
  void calcGCFillList();
  
  
  //============================================================
  // Data
  //============================================================

  // The stride array for indexing into the 1-D list of blocks;
  // i.e. in 2D, block (i,j) is element (i + j * blockstride_m[1]) in
  // the list.
  
  int blockstride_m[Dim];

  // The patch size. Could be stored in an interval, but use an array
  // since this info is zero-based.
  
  int blocksizes_m[Dim];

  // The domain of the "brick" of patches stored in the all_m list.
  
  Interval<Dim> allDomain_m;
};


//-----------------------------------------------------------------------------
//
// Full Description:
// UniformGridLayout<Dim>
//
// UniformGridLayout is a Layout class that breaks an N-dimensional
// Interval into equal sized sub-domains arranged in an N-dimensional
// grid.
//
// This is an alternative to the more general Layout class that should
// perform somewhat faster since subdomains can be found
// arithmetically, rather than via a search.  It is only able to
// represent grid-like layouts, however.
//
// To construct a UniformGridLayout, you can do any of the following:
//
//   1) provide a global domain, and let the UniformGridLayout perform
//      its default partitioning by just using one single block;
//
//   2) provide a global domain, a Loc with the number of blocks to
//      use along each dimension
//      
//   3) provide a global domain and a UniformGridPartition object.
//      
// Alternatively, you can use the default UniformGridLayout
// constructor, and call the 'initialize' method later with the same
// possible set of arguments.
//
// You can also specify internal and external guard layers for the
// domains. See the comments in UniformGridLayout below.
//
//-----------------------------------------------------------------------------

template <int Dim>
class UniformGridLayout : public LayoutBase<Dim,UniformGridLayoutData<Dim> >,
                          public Observable<UniformGridLayout<Dim> >,
                          public Observer<UniformGridLayoutData<Dim> >
{
public:

  //============================================================
  // Typedefs and enumerations
  //============================================================

  // General public typedefs.
  
  typedef UniformGridLayoutData<Dim>           LayoutData_t; // for convenience
  typedef typename LayoutData_t::Domain_t      Domain_t;
  typedef typename LayoutData_t::BaseDomain_t  BaseDomain_t;
  typedef typename LayoutData_t::Context_t     Context_t;
  typedef typename LayoutData_t::ID_t          ID_t;
  typedef typename LayoutData_t::Value_t       Value_t;
  typedef typename LayoutData_t::List_t        List_t; // for convenience
  typedef UniformGridLayout<Dim>               This_t; // for convenience
  typedef Observable<This_t>                   Observable_t;

//   // Iterator through nodes. Basically the same as the vector iterator
//   // except it dereferences automatically.
  
  typedef DerefIterator<Value_t>               iterator;
  typedef ConstDerefIterator<Value_t>          const_iterator;
  
  // Iterator through guard-cell-fill requests. 
  
  typedef typename LayoutData_t::GCFillInfo_t GCFillInfo_t;
  
  typedef typename 
    std::vector<GCFillInfo_t>::const_iterator FillIterator_t;
  
  typedef GuardLayers<Dim> GuardLayers_t;

  // Enumerations.

  enum { dimensions = Dim };
  enum { repartitionEvent = LayoutData_t::repartitionEvent };
  enum { dynamic = false };


  //============================================================
  // Constructors
  //============================================================

  // The default constructor does not initialize the layout.  In this
  // case, layout initialization must be completed with the
  // 'initialize' method before the layout can be used.  A default
  // layout has an empty global domain, and empty subdomain lists.
  
  // This is also the only constructor that doesn't demand either
  // ReplicatedTag or DistributedTag
  
  UniformGridLayout();

  // Distributed versions


  UniformGridLayout(const Domain_t &,
		    const DistributedTag &);

  UniformGridLayout(const Domain_t &, 
                    const GuardLayers_t &,
		    const DistributedTag &);

  UniformGridLayout(const Domain_t &, 
                    const Loc<Dim> &,
		    const DistributedTag &);

  UniformGridLayout(const Domain_t &, 
                    const Loc<Dim> &, 
                    const GuardLayers_t &,
		    const DistributedTag &);

  UniformGridLayout(const Domain_t &, 
                    const Loc<Dim> &, 
                    const GuardLayers_t &, 
                    const GuardLayers_t &,
		    const DistributedTag &);
  //========================= Replicated versions ===================

  UniformGridLayout(const Domain_t &,
		    const ReplicatedTag &);

  UniformGridLayout(const Domain_t &, 
                    const GuardLayers_t &,
		    const ReplicatedTag &);

  UniformGridLayout(const Domain_t &, 
                    const Loc<Dim> &,
		    const ReplicatedTag &);

  UniformGridLayout(const Domain_t &, 
                    const Loc<Dim> &, 
                    const GuardLayers_t &,
		    const ReplicatedTag &);

  UniformGridLayout(const Domain_t &, 
                    const Loc<Dim> &, 
                    const GuardLayers_t &, 
                    const GuardLayers_t &,
		    const ReplicatedTag &);

  // Domain + partition constructor. 
  // In this case, the partitioner must be a UniformGridPartition
  // object. The partitioner must know about guard cells, so we don't
  // specify them separately here.
  
  template <class Partitioner>
  UniformGridLayout(const Domain_t &,
		    const Partitioner &,
		    const ContextMapper<Dim> & );

  template <class Partitioner>
  UniformGridLayout(const Domain_t &, 
                    const Partitioner &,
		    const DistributedTag &);

  template <class Partitioner>
  UniformGridLayout(const Domain_t &, 
                    const Partitioner &,
		    const ReplicatedTag &);

  // Copy constructor & assignment operator
  // Shallow copies with reference counting.
  
  UniformGridLayout(const This_t &);
  
  This_t &operator=(const This_t &);

  // I think we need a version of this here ... just forward to
  // the RefCountedPtr.
  
  // makeOwnCopy???


  //============================================================
  // Destructor
  //============================================================

  // The actual data will be cleaned up by the LayoutData_t destructor
  // if all references to the data go away.  If any Observers remain,
  // they will be notified by the Observable destructor.
  
  inline ~UniformGridLayout() 
  { 
    pdata_m->detach(*this);
  }


  //============================================================
  // Initialize methods
  //============================================================

  // Initialize a layout with nothing else but a global domain.  In
  // this case, a default partitioner will be used, the
  // UniformGridPartition object, which will try to make a grid with
  // one block.  
  
  void initialize(const Domain_t &,
		  const DistributedTag &);
  
  void initialize(const Domain_t &,
		  const ReplicatedTag &);
  
  void initialize(const Domain_t &, 
		  const GuardLayers_t &,
		  const DistributedTag &);
  
  void initialize(const Domain_t &, 
		  const GuardLayers_t &,
		  const ReplicatedTag &);
  
  void initialize(const Domain_t &, 
		  const Loc<Dim> &,
		  const DistributedTag &);

  void initialize(const Domain_t &, 
		  const Loc<Dim> &,
		  const ReplicatedTag &);

  void initialize(const Domain_t &, 
		  const Loc<Dim> &, 
		  const GuardLayers_t &,
		  const DistributedTag & );

  void initialize(const Domain_t &, 
		  const Loc<Dim> &, 
		  const GuardLayers_t &,
		  const ReplicatedTag & );

  void initialize(const Domain_t &, 
                  const Loc<Dim> &, 
                  const GuardLayers_t &, 
                  const GuardLayers_t &,
		  const DistributedTag &);

  void initialize(const Domain_t &, 
                  const Loc<Dim> &, 
                  const GuardLayers_t &, 
                  const GuardLayers_t &,
		  const ReplicatedTag &);

  // Domain + partition initializer.
  // In this case, the partitioner must be a UniformGridPartition
  // object.
  
  template <class Partitioner>
  void initialize(const Domain_t &, 
		  const Partitioner &,
		  const DistributedTag &);

  template <class Partitioner>
  void initialize(const Domain_t &, 
		  const Partitioner &,
		  const ReplicatedTag &);
  
  // Domain + partition + mapper  initializer.
  // In this case, the partitioner must be a UniformGridPartition
  // object.
  
  template <class Partitioner>
  void initialize(const Domain_t &, 
		  const Partitioner &,
		  const ContextMapper<Dim> &);
 
  // Used by the I/O or data management system to initialize the layout based
  // on detailed state information previously stored. Since this is specialized
  // for the I/O system, no trailing tag is used. 

  void initialize(const Domain_t& idom,
		  const List_t& nodes,
		  const Loc<Dim>& blocks,
		  bool hasIG, bool hasEG,
		  const GuardLayers_t& ig,
		  const GuardLayers_t& eg);
  //============================================================
  // Accessors
  //============================================================




  //============================================================
  // Observer methods
  //============================================================

  // Respond to events generated by the LayoutData_t.
  // These are just passed on to our observers.
  // this needs to be here, and not in the base class

  virtual void notify(LayoutData_t &d, const ObserverEvent &event)
  {
    // We should only get this message from our LayoutData_t object
    PAssert(&d == pdata_m.rawPointer());
    Observable_t::notify(event);
  }
    

  //============================================================
  // Output
  //============================================================
    
  // Print a UniformGridLayout on an output stream

  template <class Ostream>
  void print(Ostream &ostr) const {
    ostr << "UniformGridLayout " << ID() << " on global domain " 
      << domain() << ":" << '\n';
    ostr << "   Total subdomains: " << sizeGlobal() << '\n';
    ostr << "   Local subdomains: " << sizeLocal() << '\n';
    ostr << "  Remote subdomains: " << sizeRemote() << '\n';
    ostr << "        Grid blocks: " << blocks() << '\n';
    typename UniformGridLayout<Dim>::const_iterator a;
    for (a = beginGlobal(); a != endGlobal(); ++a)
      ostr << "  Global subdomain = " << *a << '\n';
    for (a = beginLocal(); a != endLocal(); ++a)
      ostr << "   Local subdomain = " << *a << '\n';
    for (a = beginRemote(); a != endRemote(); ++a)
      ostr << "  Remote subdomain = " << *a << '\n';
  }


#if !POOMA_NO_TEMPLATE_FRIENDS
  // can't seem to get specialized template friends working...
  //private:

  template <int Dim1, int Dim2>
  friend class UniformGridLayoutView;

#endif
  
  //============================================================
  // Data
  //===========================================================

  // UniformGridLayout stores its data in a LayoutBase::RefCounted class to
  // simplify memory management.
  
  friend class UniformGridLayoutData<Dim>;

};



//-----------------------------------------------------------------------------
//
// Full Description:
// UniformGridLayoutViewData<Dim>
//
// This is the actual data for the UniformGridLayoutView class, which is
// simply a wrapper that holds a reference counted instance of this
// data class.
//
//-----------------------------------------------------------------------------

template <int Dim, int Dim2>
class UniformGridLayoutViewData 
  : public LayoutBaseViewData<Dim, Dim2, UniformGridLayout<Dim2> >,
    public RefCounted
{
public:

  typedef UniformGridLayout<Dim2>              Layout_t;
  typedef UniformGridLayoutView<Dim, Dim2>     ViewLayout_t;
  typedef LayoutBaseViewData<Dim,Dim2,Layout_t> Base_t;
  typedef Interval<Dim>                        Domain_t;
  typedef Range<Dim2>                          BaseDomain_t;
  typedef int                                  Context_t;
  typedef Unique::Value_t                      ID_t;

  typedef typename Layout_t::Domain_t          AllocatedDomain_t;
  typedef ViewIndexer<Dim,Dim2>                Indexer_t;

  typedef Node<Domain_t,AllocatedDomain_t>     Value_t;
  typedef std::vector<Value_t *>               List_t;        // convenience
  typedef GuardLayers<Dim>                     GuardLayers_t; // convenience


  // Enumerations.

  enum { dim = Dim };
  enum { dim2 = Dim2};

  //============================================================
  // Constructors
  //============================================================

  UniformGridLayoutViewData() { };
  
  template <class DT>
  inline 
  UniformGridLayoutViewData(const Layout_t &layout, const Domain<Dim, DT> &dom)
  : LayoutBaseViewData<Dim,Dim2,UniformGridLayout<Dim2> >(layout,dom)
  { 
  }

  template <class DT>
  inline 
  UniformGridLayoutViewData(const Layout_t &layout, const SliceDomain<DT> &dom)
  :LayoutBaseViewData<Dim,Dim2,UniformGridLayout<Dim2> >(layout,dom)
  {  
  }

  template <class DT>
  UniformGridLayoutViewData(const ViewLayout_t &layout, 
                            const Domain<Dim, DT> &dom)
  : LayoutBaseViewData<Dim,Dim2,UniformGridLayout<Dim2> >(
					      layout.pdata_m->layout_m,
					      layout,
					      layout.pdata_m->indexer_m, 
					      dom,
					      layout.internalGuards(),
					      layout.externalGuards())
  {
  }

  template <int OrigDim, class DT>
  UniformGridLayoutViewData(const UniformGridLayoutView<OrigDim,Dim2> &layout, 
                            const SliceDomain<DT> &dom)
  : LayoutBaseViewData<Dim,Dim2,UniformGridLayout<Dim2> >(
			     layout.pdata_m->layout_m,
			     layout,
			     Indexer_t(layout.pdata_m->indexer_m,dom),
			     dom)
  { 
  }

  // Destructor

  ~UniformGridLayoutViewData() 
  {
    typename List_t::iterator a;
    for (a = all_m.begin(); a != all_m.end(); ++a)
      delete (*a);
  }

};


//-----------------------------------------------------------------------------
//
// Full Description:
// UniformGridLayoutView<Dim, Dim2>
//
// UniformGridLayoutView is a Layout class that provides a view of an
// existing UniformGridLayout object. Dim is the logical dimension of
// the layout. Dim2 is the dimension of the UniformGridLayout
// contained within.
//
// To construct a UniformGridLayoutView, you need an existing
// UniformGridLayout or a UniformGridLayoutView and the subdomain that
// is being viewed. This class does not have a useful default
// constructor since it is based on an existing UniformGridLayout.
//
// Once created, UniformGridLayoutView has the same interface as
// Layout (see Layout.h). It also provides this extra interface:
//
// int globalID(const Loc<Dim> &pos) : return the globalID
//    of the node that contains the point.
//
//-----------------------------------------------------------------------------

template <int Dim, int Dim2>
class UniformGridLayoutView
  : public LayoutBaseView<Dim, Dim2, UniformGridLayoutViewData<Dim,Dim2> >
{
public:
  //============================================================
  // Typedefs and enumerations
  //============================================================

  // Enumerations.

  enum { dimensions = Dim };

  enum { dim = Dim };
  enum { dim2 = Dim2};
  // General public typedefs.
  
  typedef UniformGridLayoutViewData<Dim, Dim2>      LayoutData_t; 

  typedef typename LayoutData_t::Domain_t           Domain_t;
  typedef typename LayoutData_t::BaseDomain_t       BaseDomain_t;
  typedef typename LayoutData_t::Context_t          Context_t;
  typedef typename LayoutData_t::ID_t               ID_t;
  
  typedef typename LayoutData_t::Layout_t           Layout_t;
  typedef typename LayoutData_t::AllocatedDomain_t  AllocatedDomain_t;
  typedef typename LayoutData_t::Value_t            Value_t;
  
  typedef typename LayoutData_t::List_t             List_t;
  typedef typename LayoutData_t::Indexer_t          Indexer_t;
  typedef typename LayoutData_t::GuardLayers_t      GuardLayers_t;
    
  typedef UniformGridLayoutView<Dim, Dim2>          This_t;      // convenience
  typedef UniformGridLayoutView<Dim, Dim2>          ViewLayout_t;// convenience
  typedef LayoutBaseView<Dim,Dim2,LayoutData_t>     Base_t;

  // Iterator through nodes. Basically the same as the vector iterator
  // except it dereferences automatically.
  
  typedef DerefIterator<Value_t>                    iterator;
  typedef ConstDerefIterator<Value_t>               const_iterator;


  //============================================================
  // Constructors
  //============================================================

  // Default constructor. Final initialization should be done with
  // the assignment operator. 
  
  UniformGridLayoutView()
  : Base_t(new LayoutData_t())
  { }
  
  // Constructor building a UniformGridLayoutView from a
  // UniformGridLayout and a non-slice domain like an Interval<Dim> or
  // Range<Dim>.
  
  template <class DT>
  UniformGridLayoutView(const Layout_t &layout, const Domain<Dim2, DT> &dom)
  : LayoutBaseView<Dim,Dim2,UniformGridLayoutViewData<Dim,Dim2> >
    (new UniformGridLayoutViewData<Dim,Dim2>(layout,dom))
  { }

  // Constructor building a UniformGridLayoutView from a
  // UniformGridLayout and a slice domain like a
  // SliceInterval<Dim2,Dim> or SliceRange<Dim2,Dim>.
  
  template <class DT>
  UniformGridLayoutView(const Layout_t &layout, const SliceDomain<DT> &dom)
  : LayoutBaseView<Dim,Dim2,UniformGridLayoutViewData<Dim,Dim2> >
    (new UniformGridLayoutViewData<Dim,Dim2>(layout,dom))
  { }
  
  // Constructor building a UniformGridLayoutView from another
  // UniformGridLayoutView and a non-slice domain like an
  // Interval<Dim> or Range<Dim>.
  
  template <class DT>
  UniformGridLayoutView(const ViewLayout_t &layout, const Domain<Dim, DT> &dom)
  : LayoutBaseView<Dim,Dim2,UniformGridLayoutViewData<Dim,Dim2> >
    (new UniformGridLayoutViewData<Dim,Dim2>(layout,dom))
  { }

  // Constructor building a UniformGridLayoutView from another
  // UniformGridLayoutView and a slice domain like a
  // SliceInterval<Dim2,Dim> or SliceRange<Dim2,Dim>.
  
  template <int OldViewDim, class DT>
  UniformGridLayoutView(const UniformGridLayoutView<OldViewDim, Dim2> &layout, 
                        const SliceDomain<DT> &dom)
  : LayoutBaseView<Dim,Dim2,UniformGridLayoutViewData<Dim,Dim2> >
    (new UniformGridLayoutViewData<Dim,Dim2>(layout,dom))
  { }

  // Copy constructor & assignment operator
  // Shallow copies with reference counting.
  
  inline UniformGridLayoutView(const This_t &model) 
    : LayoutBaseView<Dim,
                     Dim2,
                     UniformGridLayoutViewData<Dim,Dim2> >(model.pdata_m)
  { }
  
  inline This_t &operator=(const This_t &model) 
  {
    if (this != &model)
      {
        pdata_m = model.pdata_m;
      }
    return *this;
  }

  // I think we need a version of this here ... just forward to
  // the RefCountedPtr.
  
  // makeOwnCopy???



  //============================================================
  // Destructor
  //============================================================

  // The actual data will be cleaned up by the UniformGridLayoutData
  // destructor if all references to the data go away, so there is
  // nothing to do here.
   
  inline ~UniformGridLayoutView() 
  { }

  //============================================================
  // Output
  //============================================================
    
  // Print a UniformGridLayoutView on an output stream

  template <class Ostream>
  void print(Ostream &ostr) const 
  {
    ostr << "UniformGridLayoutView " << ID() << " on global domain " 
      << domain() << ":" << '\n';
    ostr << "   Base ID:          " << baseID() << '\n';
    ostr << "   Base domain:      " << baseDomain() << '\n';
    ostr << "   Total subdomains: " << sizeGlobal() << '\n';
    ostr << "   Local subdomains: " << sizeLocal() << '\n';
    ostr << "  Remote subdomains: " << sizeRemote() << '\n';
    const_iterator a;
    for (a = beginGlobal(); a != endGlobal(); ++a)
      ostr << "  Global subdomain = " << *a << '\n';
    for (a = beginLocal(); a != endLocal(); ++a)
      ostr << "   Local subdomain = " << *a << '\n';
    for (a = beginRemote(); a != endRemote(); ++a)
      ostr << "  Remote subdomain = " << *a << '\n';
  }
  
#if !POOMA_NO_TEMPLATE_FRIENDS

  //protected:

  template <int OtherDim, int OtherDim2>
  friend class UniformGridLayoutView;
  
  template <int OtherDim, int OtherDim2>
  friend class UniformGridLayoutViewData;

#endif

  //============================================================
  // Data
  //============================================================

  // The data is stored in a LayoutBaseView::RefCounted class to simplify 
  // memory management. (LayoutViews have shallow copy semantics and they
  // own copies of the underlying Layout as well as their own sets of
  // Node lists, which must be cleaned up when all views are destroyed.)

};


//=============================================================================
// UniformGridLayout & UniformGridLayoutData inline method definitions
//=============================================================================

//-----------------------------------------------------------------------------
//
// Constructors and Initialize methods
//
//-----------------------------------------------------------------------------

// See comments in class definition above.

template <int Dim>
inline UniformGridLayout<Dim>::
UniformGridLayout()
: Observable<This_t>(*this), 
  LayoutBase<Dim,UniformGridLayoutData<Dim> >
    (new LayoutData_t())
{ 
  pdata_m->attach(*this);
}

template <int Dim>
inline UniformGridLayout<Dim>::
UniformGridLayout(const Domain_t &gdom,
		  const DistributedTag& t)
: Observable<This_t>(*this), 
  LayoutBase<Dim,UniformGridLayoutData<Dim> >
    (new LayoutData_t(gdom,
		      UniformGridPartition<Dim>(),
		      DistributedMapper<Dim>(UniformGridPartition<Dim>())))
{
  pdata_m->attach(*this);
}

template <int Dim>
inline UniformGridLayout<Dim>::
UniformGridLayout(const Domain_t &gdom,
		  const ReplicatedTag & t)
: Observable<This_t>(*this), 
  LayoutBase<Dim,UniformGridLayoutData<Dim> >
    (new LayoutData_t(gdom,
		      UniformGridPartition<Dim>(),
		      LocalMapper<Dim>()))
{
  pdata_m->attach(*this);
}

template <int Dim>
inline UniformGridLayout<Dim>::
UniformGridLayout(const Domain_t &gdom, 
		  const GuardLayers_t &gcs,
		  const DistributedTag &)
: Observable<This_t>(*this), 
  LayoutBase<Dim,UniformGridLayoutData<Dim> >
    (new LayoutData_t(gdom,
		      UniformGridPartition<Dim>(gcs),
		      DistributedMapper<Dim>(UniformGridPartition<Dim>(gcs))))
{
  pdata_m->attach(*this);
}

template <int Dim>
inline UniformGridLayout<Dim>::
UniformGridLayout(const Domain_t &gdom, 
		  const GuardLayers_t &gcs,
		  const ReplicatedTag & )
: Observable<This_t>(*this), 
  LayoutBase<Dim,UniformGridLayoutData<Dim> >
    (new LayoutData_t(gdom,
		      UniformGridPartition<Dim>(gcs),
		      LocalMapper<Dim>()))
{
  pdata_m->attach(*this);
}

template <int Dim>
inline UniformGridLayout<Dim>::
UniformGridLayout(const Domain_t &gdom, 
		  const Loc<Dim> &blocks,
		  const DistributedTag & )
: Observable<This_t>(*this), 
  LayoutBase<Dim,UniformGridLayoutData<Dim> >
    (new LayoutData_t(gdom,
		      UniformGridPartition<Dim>(blocks),
		      DistributedMapper<Dim>(
                         UniformGridPartition<Dim>(blocks))))
{
  pdata_m->attach(*this);
}

template <int Dim>
inline UniformGridLayout<Dim>::
UniformGridLayout(const Domain_t &gdom, 
		  const Loc<Dim> &blocks,
		  const ReplicatedTag & t)
: Observable<This_t>(*this), 
  LayoutBase<Dim,UniformGridLayoutData<Dim> >
    (new LayoutData_t(gdom,
		      UniformGridPartition<Dim>(blocks),
		      LocalMapper<Dim>()))
{
  pdata_m->attach(*this);
}

template <int Dim>
inline UniformGridLayout<Dim>::
UniformGridLayout(const Domain_t &gdom, 
                  const Loc<Dim> &blocks, 
                  const GuardLayers_t &igcs,
		  const DistributedTag &)
: Observable<This_t>(*this), 
 LayoutBase<Dim,UniformGridLayoutData<Dim> >
   (new LayoutData_t(gdom,
		     UniformGridPartition<Dim>(blocks,igcs),
		     DistributedMapper<Dim>(
                       UniformGridPartition<Dim>(blocks,igcs))))
{
  pdata_m->attach(*this);
}

template <int Dim>
inline UniformGridLayout<Dim>::
UniformGridLayout(const Domain_t &gdom, 
                  const Loc<Dim> &blocks, 
                  const GuardLayers_t &igcs,
		  const ReplicatedTag &)
  : Observable<This_t>(*this), 
    LayoutBase<Dim,UniformGridLayoutData<Dim> >
       (new LayoutData_t(gdom,
			 UniformGridPartition<Dim>(blocks,igcs),
			 LocalMapper<Dim>()))
{
  pdata_m->attach(*this);
}

template <int Dim>
inline UniformGridLayout<Dim>::
UniformGridLayout(const Domain_t &gdom, 
                  const Loc<Dim> &blocks, 
                  const GuardLayers_t &igcs, 
                  const GuardLayers_t &egcs,
		  const DistributedTag &)
     
: Observable<This_t>(*this), 
 LayoutBase<Dim,UniformGridLayoutData<Dim> >
(new LayoutData_t(gdom,
		  UniformGridPartition<Dim>(blocks,igcs,egcs),
		  DistributedMapper<Dim>(
		    UniformGridPartition<Dim>(blocks,igcs,egcs))))
{
  pdata_m->attach(*this);
}

template <int Dim>
inline UniformGridLayout<Dim>::
UniformGridLayout(const Domain_t &gdom, 
                  const Loc<Dim> &blocks, 
                  const GuardLayers_t &igcs, 
                  const GuardLayers_t &egcs,
		  const ReplicatedTag &t)
: Observable<This_t>(*this), 
 LayoutBase<Dim,UniformGridLayoutData<Dim> >
    (new LayoutData_t(gdom,
		      UniformGridPartition<Dim>(blocks,igcs,egcs),
		      LocalMapper<Dim>()))
{
  pdata_m->attach(*this);
}

template <int Dim>
template <class Partitioner>
inline UniformGridLayout<Dim>::
UniformGridLayout(const Domain_t &gdom, 
		  const Partitioner &gpar,
		  const DistributedTag & )
: Observable<This_t>(*this),
  LayoutBase<Dim,UniformGridLayoutData<Dim> >
   (new LayoutData_t(gdom,gpar,DistributedMapper<Dim>(gpar)))
{
  pdata_m->attach(*this);
}

template <int Dim>
template <class Partitioner>
inline UniformGridLayout<Dim>::
UniformGridLayout(const Domain_t &gdom, 
		  const Partitioner &gpar,
		  const ReplicatedTag &)
: Observable<This_t>(*this),
 LayoutBase<Dim,UniformGridLayoutData<Dim> >
   (new LayoutData_t(gdom,gpar,LocalMapper<Dim>()))
{
  pdata_m->attach(*this);
}

template <int Dim>
template <class Partitioner>
inline UniformGridLayout<Dim>::
UniformGridLayout(const Domain_t &gdom, 
		  const Partitioner &gpar,
		  const ContextMapper<Dim> & cmap)
: Observable<This_t>(*this), 
 LayoutBase<Dim,UniformGridLayoutData<Dim> >
   (new LayoutData_t(gdom,gpar,cmap))
{
  pdata_m->attach(*this);
}

template <int Dim>
inline UniformGridLayout<Dim>::
UniformGridLayout(const This_t &model) 
: Observable<This_t>(*this), 
 LayoutBase<Dim,UniformGridLayoutData<Dim> >(model.pdata_m)
{ 
   pdata_m->attach(*this);
}
  
template <int Dim>
inline UniformGridLayout<Dim> & UniformGridLayout<Dim>::
operator=(const This_t &model) 
{
  if (this != &model)
    {
      pdata_m->detach(*this);
      pdata_m = model.pdata_m;
      pdata_m->attach(*this);
    }
  return *this;
}

// Initialize methods...

template <int Dim>
inline void
UniformGridLayout<Dim>::
initialize(const Domain_t &gdom,
	   const DistributedTag &)
{
  PAssert(!initialized());

  // Initialize our global domain, and then do the partitioning.
  
  pdata_m->domain_m = gdom;
  pdata_m->innerdomain_m = gdom;
  pdata_m->partition(UniformGridPartition<Dim>(),
		     DistributedMapper<Dim>(UniformGridPartition<Dim>()));
}

template <int Dim>
inline void
UniformGridLayout<Dim>::
initialize(const Domain_t &gdom,
	   const ReplicatedTag &)
{
  PAssert(!initialized());

  // Initialize our global domain, and then do the partitioning.
  
  pdata_m->domain_m = gdom;
  pdata_m->innerdomain_m = gdom;
  pdata_m->partition(UniformGridPartition<Dim>(),
		     LocalMapper<Dim>());
}

template <int Dim>
inline void
UniformGridLayout<Dim>::
initialize(const Domain_t &gdom, 
	   const GuardLayers_t &gcs,
	   const DistributedTag &)
{
  PAssert(!initialized());

  // Initialize our global domain, and then do the partitioning.
  pdata_m->innerdomain_m = gdom;
  pdata_m->domain_m = gdom;
  pdata_m->partition(UniformGridPartition<Dim>(gcs),
		     DistributedMapper<Dim>(UniformGridPartition<Dim>(gcs) ));
}

template <int Dim>
inline void
UniformGridLayout<Dim>::
initialize(const Domain_t &gdom, 
	   const GuardLayers_t &gcs,
	   const ReplicatedTag &)
{
  PAssert(!initialized());

  // Initialize our global domain, and then do the partitioning.
  pdata_m->innerdomain_m = gdom;
  pdata_m->domain_m = gdom;
  pdata_m->partition(UniformGridPartition<Dim>(gcs),
		     LocalMapper<Dim>());
}

template <int Dim>
inline void
UniformGridLayout<Dim>::
initialize(const Domain_t &gdom,
	   const Loc<Dim> &blocks,
	   const DistributedTag &)
{
  PAssert(!initialized());
  
  // Initialize our global domain, and then do the partitioning.
  pdata_m->innerdomain_m = gdom;
  pdata_m->domain_m = gdom;
  pdata_m->partition(UniformGridPartition<Dim>(blocks),
		    DistributedMapper<Dim>(UniformGridPartition<Dim>(blocks)));
}

template <int Dim>
inline void
UniformGridLayout<Dim>::
initialize(const Domain_t &gdom,
	   const Loc<Dim> &blocks,
	   const ReplicatedTag &)
{
  PAssert(!initialized());
  pdata_m->innerdomain_m = gdom;
  pdata_m->domain_m = gdom;
  pdata_m->partition(UniformGridPartition<Dim>(blocks),
		     LocalMapper<Dim>());
}

template <int Dim>
inline void
UniformGridLayout<Dim>::
initialize(const Domain_t &gdom,
           const Loc<Dim> &blocks, 
           const GuardLayers_t &gcs,
	   const DistributedTag &)
{
  PAssert(!initialized()); 
  pdata_m->innerdomain_m = gdom;
  pdata_m->domain_m = gdom;
  pdata_m->partition(UniformGridPartition<Dim>(blocks, gcs),
		     DistributedMapper<Dim>(
		       UniformGridPartition<Dim>(blocks, gcs)));
}

template <int Dim>
inline void
UniformGridLayout<Dim>::
initialize(const Domain_t &gdom,
           const Loc<Dim> &blocks, 
           const GuardLayers_t &gcs,
	   const ReplicatedTag &)
{
  PAssert(!initialized()); 
  pdata_m->innerdomain_m = gdom;
  pdata_m->domain_m = gdom;
  pdata_m->partition(UniformGridPartition<Dim>(blocks, gcs),
		     LocalMapper<Dim>());
}

template <int Dim>
inline void
UniformGridLayout<Dim>::
initialize(const Domain_t &gdom,
           const Loc<Dim> &blocks, 
           const GuardLayers_t &igcs,
           const GuardLayers_t &egcs,
	   const DistributedTag &)
{
  PAssert(!initialized());
  
  // Initialize our global domain, and then do the partitioning.
  pdata_m->innerdomain_m = gdom;
  pdata_m->domain_m = gdom;
  pdata_m->partition(UniformGridPartition<Dim>(blocks, igcs, egcs),
		     DistributedMapper<Dim>(
		       UniformGridPartition<Dim>(blocks, igcs, egcs)));
}

template <int Dim>
inline void
UniformGridLayout<Dim>::
initialize(const Domain_t &gdom,
           const Loc<Dim> &blocks, 
           const GuardLayers_t &igcs,
           const GuardLayers_t &egcs,
	   const ReplicatedTag &)
{
  PAssert(!initialized());
  
  // Initialize our global domain, and then do the partitioning.
  pdata_m->innerdomain_m = gdom;
  pdata_m->domain_m = gdom;
  pdata_m->blocks_m = blocks; 
  pdata_m->partition(UniformGridPartition<Dim>(blocks, igcs, egcs),
		     LocalMapper<Dim>());
}


template <int Dim>
template <class Partitioner>
inline void
UniformGridLayout<Dim>::
initialize(const Domain_t &gdom,
           const Partitioner &p,
	   const DistributedTag &)
{
  PAssert(!initialized());

  // Initialize our global domain, and then do the partitioning.

  pdata_m->innerdomain_m = gdom;
  pdata_m->domain_m = gdom;
  pdata_m->blocks_m = p.blocks(); 
  pdata_m->partition(p,DistributedMapper<Dim>(p));
}

template <int Dim>
template <class Partitioner>
inline void
UniformGridLayout<Dim>::
initialize(const Domain_t &gdom,
           const Partitioner &p,
	   const ReplicatedTag &)
{
  PAssert(!initialized());

  // Initialize our global domain, and then do the partitioning.

  pdata_m->innerdomain_m = gdom;
  pdata_m->domain_m = gdom;
  pdata_m->blocks_m = p.blocks(); 
  pdata_m->partition(p,LocalMapper<Dim>());
}
template <int Dim>
template <class Partitioner>
inline void
UniformGridLayout<Dim>::
initialize(const Domain_t &gdom,
           const Partitioner &p,
	   const ContextMapper<Dim> &cmap)
{
  PAssert(!initialized());

  // Initialize our global domain, and then do the partitioning.

  pdata_m->innerdomain_m = gdom;
  pdata_m->domain_m = gdom;
  pdata_m->blocks_m = p.blocks();
  pdata_m->partition(p,cmap); 
}

// This initializer is intented to be used by the I/O system

template <int Dim>
void UniformGridLayout<Dim>::initialize(const Domain_t& idom,
				 const List_t& nodes,
				 const Loc<Dim>& blocks,
				 bool hasIG, bool hasEG,
				 const GuardLayers_t& ig,
				 const GuardLayers_t& eg)
{
  pdata_m->initialize(idom,nodes,blocks,hasIG,hasEG,ig,eg);
}

// Here are the implementations for globalID:

template <int Dim>
inline int
UniformGridLayoutData<Dim>::globalID(const Loc<Dim> &loc) const
{
  // Make sure the point is in our domain.
  PAssert(contains(domain_m, loc));
  int currloc;
  
  if (!hasExternalGuards_m) 
    {
      currloc = (loc[0].first() - firsti_m[0]) / blocksizes_m[0];
      for (int d = 1; d < Dim; ++d)
        currloc += blockstride_m[d] * 
          ((loc[d].first() - firsti_m[d]) / blocksizes_m[d]);
    }
  else
    {
      currloc = 0;
      for (int d = 0; d < Dim; ++d)
        {
          int l = loc[d].first();
      
          // If l < firsti_m[0], currloc is unchanged.
          
          if (l >= firsti_m[d]) 
            {
              if (l <= innerdomain_m[d].last())
                {
                  // The usual expression in this direction.
                
                  currloc += blockstride_m[d] * 
                    ((l - firsti_m[d]) / blocksizes_m[d]);
                }
              else
                {
                  // Must be in the last block in this direction.
                  
                  currloc += blockstride_m[d] * allDomain_m[d].last();
                }
            }
        }
    }
    
  // Return the globalID for the currloc's node

  PAssert(currloc >= 0 && currloc < all_m.size());
  return currloc;
}

template <int Dim>
inline int
UniformGridLayoutData<Dim>::globalID(int i0) const
{
  PAssert(Dim == 1);
  PAssert(i0 >= domain_m[0].first() && i0 <= domain_m[0].last());
  
  // Compute fortran-order index from position in block grid
  // See the Loc<Dim> version for comments. 
  
  int currloc;
  if (!hasExternalGuards_m)
    {
      currloc = (i0 - firsti_m[0]) / blocksizes_m[0];
    }
  else
    {
      currloc = 0;
      if (i0 >= firsti_m[0]) {
        if (i0 <= innerdomain_m[0].last())
          currloc = (i0 - firsti_m[0]) / blocksizes_m[0];
        else
          currloc = allDomain_m[0].last();
      }
    }
    
  // Return the globalID for the currloc's node.
  
  PAssert(currloc >= 0 && currloc < all_m.size());
  return currloc;
}

template <int Dim>
inline int
UniformGridLayoutData<Dim>::globalID(int i0, int i1) const
{
  PAssert(Dim == 2);
  PAssert(i0 >= domain_m[0].first() && i0 <= domain_m[0].last());
  PAssert(i1 >= domain_m[1].first() && i1 <= domain_m[1].last());

  // Compute fortran-order index from position in block grid

  int currloc;
  if (!hasExternalGuards_m)
    {
      currloc = (i0 - firsti_m[0]) / blocksizes_m[0]
              + blockstride_m[1] * ((i1 - firsti_m[1]) / blocksizes_m[1]);
    }
  else
    {
      currloc = 0;
      if (i0 >= firsti_m[0]) {
        if (i0 <= innerdomain_m[0].last())
          currloc = (i0 - firsti_m[0]) / blocksizes_m[0];
        else
          currloc = allDomain_m[0].last();
      }
      if (i1 >= firsti_m[1]) {
        if (i1 <= innerdomain_m[1].last())
          currloc += blockstride_m[1] * ((i1 - firsti_m[1]) / blocksizes_m[1]);
        else
          currloc += blockstride_m[1] * allDomain_m[1].last();
      }
    }
    
  // Return the globalID for the currloc's node
  
  PAssert(currloc >= 0 && currloc < all_m.size());
  return currloc;
}

template <int Dim>
inline int
UniformGridLayoutData<Dim>::globalID(int i0, int i1, int i2) const
{
  PAssert(Dim == 3);
  PAssert(i0 >= domain_m[0].first() && i0 <= domain_m[0].last());
  PAssert(i1 >= domain_m[1].first() && i1 <= domain_m[1].last());
  PAssert(i2 >= domain_m[2].first() && i2 <= domain_m[2].last());

  // Compute fortran-order index from position in block grid

  int currloc;
  if (!hasExternalGuards_m)
    {
      currloc = (i0 - firsti_m[0]) / blocksizes_m[0]
              + blockstride_m[1] * ((i1 - firsti_m[1]) / blocksizes_m[1])
              + blockstride_m[2] * ((i2 - firsti_m[2]) / blocksizes_m[2]);
    }
  else
    {
      currloc = 0; 
      if (i0 >= firsti_m[0]) {
        if (i0 <= innerdomain_m[0].last())
          currloc = (i0 - firsti_m[0]) / blocksizes_m[0];
        else
          currloc = allDomain_m[0].last();
      }
      if (i1 >= firsti_m[1]) {
        if (i1 <= innerdomain_m[1].last())
          currloc += blockstride_m[1] * ((i1 - firsti_m[1]) / blocksizes_m[1]);
        else
          currloc += blockstride_m[1] * allDomain_m[1].last();
      }
      if (i2 >= firsti_m[2]) {
        if (i2 <= innerdomain_m[2].last())
          currloc += blockstride_m[2] * ((i2 - firsti_m[2]) / blocksizes_m[2]);
        else
          currloc += blockstride_m[2] * allDomain_m[2].last();
      }
    }
    
  // Return the globalID for the currloc's node
  
  PAssert(currloc >= 0 && currloc < all_m.size());
  return currloc;
}

template <int Dim>
inline int
UniformGridLayoutData<Dim>::globalID(int i0, int i1, int i2, int i3) const
{
  PAssert(Dim == 4);
  PAssert(i0 >= domain_m[0].first() && i0 <= domain_m[0].last());
  PAssert(i1 >= domain_m[1].first() && i1 <= domain_m[1].last());
  PAssert(i2 >= domain_m[2].first() && i2 <= domain_m[2].last());
  PAssert(i3 >= domain_m[3].first() && i3 <= domain_m[3].last());

  // Compute fortran-order index from position in block grid

  int currloc;
  if (!hasExternalGuards_m)
    {
      currloc = (i0 - firsti_m[0]) / blocksizes_m[0]
              + blockstride_m[1] * ((i1 - firsti_m[1]) / blocksizes_m[1])
              + blockstride_m[2] * ((i2 - firsti_m[2]) / blocksizes_m[2])
              + blockstride_m[3] * ((i3 - firsti_m[3]) / blocksizes_m[3]);
    }
  else
    {
      currloc = 0;
      if (i0 >= firsti_m[0]) {
        if (i0 <= innerdomain_m[0].last())
           currloc = (i0 - firsti_m[0]) / blocksizes_m[0];
        else
           currloc = allDomain_m[0].last();
      }
      if (i1 >= firsti_m[1]) {
        if (i1 <= innerdomain_m[1].last())
          currloc += blockstride_m[1] * ((i1 - firsti_m[1]) / blocksizes_m[1]);
        else
          currloc += blockstride_m[1] * allDomain_m[1].last();
      }
      if (i2 >= firsti_m[2]) {
        if (i2 <= innerdomain_m[2].last())
          currloc += blockstride_m[2] * ((i2 - firsti_m[2]) / blocksizes_m[2]);
        else
          currloc += blockstride_m[2] * allDomain_m[2].last();
      }
      if (i3 >= firsti_m[3]) {
        if (i3 <= innerdomain_m[3].last())
          currloc += blockstride_m[3] * ((i3 - firsti_m[3]) / blocksizes_m[3]);
        else
          currloc += blockstride_m[3] * allDomain_m[3].last();
      }
    }
    
  // Return the globalID for the currloc's node
  
  PAssert(currloc >= 0 && currloc < all_m.size());
  return currloc;
}

template <int Dim>
inline int
UniformGridLayoutData<Dim>::globalID(int i0, int i1, int i2, int i3,
                                     int i4) const
{
  PAssert(Dim == 5);
  PAssert(i0 >= domain_m[0].first() && i0 <= domain_m[0].last());
  PAssert(i1 >= domain_m[1].first() && i1 <= domain_m[1].last());
  PAssert(i2 >= domain_m[2].first() && i2 <= domain_m[2].last());
  PAssert(i3 >= domain_m[3].first() && i3 <= domain_m[3].last());
  PAssert(i4 >= domain_m[4].first() && i4 <= domain_m[4].last());

  // Compute fortran-order index from position in block grid

  int currloc;
  if (!hasExternalGuards_m)
    {
      currloc = (i0 - firsti_m[0]) / blocksizes_m[0]
              + blockstride_m[1] * ((i1 - firsti_m[1]) / blocksizes_m[1])
              + blockstride_m[2] * ((i2 - firsti_m[2]) / blocksizes_m[2])
              + blockstride_m[3] * ((i3 - firsti_m[3]) / blocksizes_m[3])
              + blockstride_m[4] * ((i4 - firsti_m[4]) / blocksizes_m[4]);
    }
  else
    {
      currloc = 0;
      if (i0 >= firsti_m[0]) {
        if (i0 <= innerdomain_m[0].last())
           currloc = (i0 - firsti_m[0]) / blocksizes_m[0];
        else
           currloc = allDomain_m[0].last();
      }
      if (i1 >= firsti_m[1]) {
        if (i1 <= innerdomain_m[1].last())
          currloc += blockstride_m[1] * ((i1 - firsti_m[1]) / blocksizes_m[1]);
        else
          currloc += blockstride_m[1] * allDomain_m[1].last();
      }
      if (i2 >= firsti_m[2]) {
        if (i2 <= innerdomain_m[2].last())
          currloc += blockstride_m[2] * ((i2 - firsti_m[2]) / blocksizes_m[2]);
        else
          currloc += blockstride_m[2] * allDomain_m[2].last();
      }
      if (i3 >= firsti_m[3]) {
        if (i3 <= innerdomain_m[3].last())
          currloc += blockstride_m[3] * ((i3 - firsti_m[3]) / blocksizes_m[3]);
        else
          currloc += blockstride_m[3] * allDomain_m[3].last();
      }
      if (i4 >= firsti_m[4]) {
        if (i4 <= innerdomain_m[4].last())
          currloc += blockstride_m[4] * ((i4 - firsti_m[4]) / blocksizes_m[4]);
        else
          currloc += blockstride_m[4] * allDomain_m[4].last();
      }
    }
    
  // Return the globalID for the currloc's node
  
  PAssert(currloc >= 0 && currloc < all_m.size());
  return currloc;
}

template <int Dim>
inline int
UniformGridLayoutData<Dim>::globalID(int i0, int i1, int i2, int i3,
                                     int i4, int i5) const
{
  PAssert(Dim == 6);
  PAssert(i0 >= domain_m[0].first() && i0 <= domain_m[0].last());
  PAssert(i1 >= domain_m[1].first() && i1 <= domain_m[1].last());
  PAssert(i2 >= domain_m[2].first() && i2 <= domain_m[2].last());
  PAssert(i3 >= domain_m[3].first() && i3 <= domain_m[3].last());
  PAssert(i4 >= domain_m[4].first() && i4 <= domain_m[4].last());
  PAssert(i5 >= domain_m[5].first() && i5 <= domain_m[5].last());

  // Compute fortran-order index from position in block grid

  int currloc;
  if (!hasExternalGuards_m)
    {
      currloc = (i0 - firsti_m[0]) / blocksizes_m[0]
              + blockstride_m[1] * ((i1 - firsti_m[1]) / blocksizes_m[1])
              + blockstride_m[2] * ((i2 - firsti_m[2]) / blocksizes_m[2])
              + blockstride_m[3] * ((i3 - firsti_m[3]) / blocksizes_m[3])
              + blockstride_m[4] * ((i4 - firsti_m[4]) / blocksizes_m[4])
              + blockstride_m[5] * ((i5 - firsti_m[5]) / blocksizes_m[5]);
    }
  else
    {
      currloc = 0;
      if (i0 >= firsti_m[0]) {
        if (i0 <= innerdomain_m[0].last())
           currloc = (i0 - firsti_m[0]) / blocksizes_m[0];
        else
           currloc = allDomain_m[0].last();
      }
      if (i1 >= firsti_m[1]) {
        if (i1 <= innerdomain_m[1].last())
          currloc += blockstride_m[1] * ((i1 - firsti_m[1]) / blocksizes_m[1]);
        else
          currloc += blockstride_m[1] * allDomain_m[1].last();
      }
      if (i2 >= firsti_m[2]) {
        if (i2 <= innerdomain_m[2].last())
          currloc += blockstride_m[2] * ((i2 - firsti_m[2]) / blocksizes_m[2]);
        else
          currloc += blockstride_m[2] * allDomain_m[2].last();
      }
      if (i3 >= firsti_m[3]) {
        if (i3 <= innerdomain_m[3].last())
          currloc += blockstride_m[3] * ((i3 - firsti_m[3]) / blocksizes_m[3]);
        else
          currloc += blockstride_m[3] * allDomain_m[3].last();
      }
      if (i4 >= firsti_m[4]) {
        if (i4 <= innerdomain_m[4].last())
          currloc += blockstride_m[4] * ((i4 - firsti_m[4]) / blocksizes_m[4]);
        else
          currloc += blockstride_m[4] * allDomain_m[4].last();
      }
      if (i5 >= firsti_m[5]) {
        if (i5 <= innerdomain_m[5].last())
          currloc += blockstride_m[5] * ((i5 - firsti_m[5]) / blocksizes_m[5]);
        else
          currloc += blockstride_m[5] * allDomain_m[5].last();
      }
    }
    
  // Return the globalID for the currloc's node
  
  PAssert(currloc >= 0 && currloc < all_m.size());
  return currloc;
}

template <int Dim>
inline int
UniformGridLayoutData<Dim>::globalID(int i0, int i1, int i2, int i3,
                                     int i4, int i5, int i6) const
{
  PAssert(Dim == 7);
  PAssert(i0 >= domain_m[0].first() && i0 <= domain_m[0].last());
  PAssert(i1 >= domain_m[1].first() && i1 <= domain_m[1].last());
  PAssert(i2 >= domain_m[2].first() && i2 <= domain_m[2].last());
  PAssert(i3 >= domain_m[3].first() && i3 <= domain_m[3].last());
  PAssert(i4 >= domain_m[4].first() && i4 <= domain_m[4].last());
  PAssert(i5 >= domain_m[5].first() && i5 <= domain_m[5].last());
  PAssert(i6 >= domain_m[6].first() && i6 <= domain_m[6].last());

  // Compute fortran-order index from position in block grid

  int currloc;
  if (!hasExternalGuards_m)
    {
      currloc = (i0 - firsti_m[0]) / blocksizes_m[0]
              + blockstride_m[1] * ((i1 - firsti_m[1]) / blocksizes_m[1])
              + blockstride_m[2] * ((i2 - firsti_m[2]) / blocksizes_m[2])
              + blockstride_m[3] * ((i3 - firsti_m[3]) / blocksizes_m[3])
              + blockstride_m[4] * ((i4 - firsti_m[4]) / blocksizes_m[4])
              + blockstride_m[5] * ((i5 - firsti_m[5]) / blocksizes_m[5])
              + blockstride_m[6] * ((i6 - firsti_m[6]) / blocksizes_m[6]);
    }
  else
    {
      currloc = 0;
      if (i0 >= firsti_m[0]) {
        if (i0 <= innerdomain_m[0].last())
           currloc = (i0 - firsti_m[0]) / blocksizes_m[0];
        else
           currloc = allDomain_m[0].last();
      }
      if (i1 >= firsti_m[1]) {
        if (i1 <= innerdomain_m[1].last())
          currloc += blockstride_m[1] * ((i1 - firsti_m[1]) / blocksizes_m[1]);
        else
          currloc += blockstride_m[1] * allDomain_m[1].last();
      }
      if (i2 >= firsti_m[2]) {
        if (i2 <= innerdomain_m[2].last())
          currloc += blockstride_m[2] * ((i2 - firsti_m[2]) / blocksizes_m[2]);
        else
          currloc += blockstride_m[2] * allDomain_m[2].last();
      }
      if (i3 >= firsti_m[3]) {
        if (i3 <= innerdomain_m[3].last())
          currloc += blockstride_m[3] * ((i3 - firsti_m[3]) / blocksizes_m[3]);
        else
          currloc += blockstride_m[3] * allDomain_m[3].last();
      }
      if (i4 >= firsti_m[4]) {
        if (i4 <= innerdomain_m[4].last())
          currloc += blockstride_m[4] * ((i4 - firsti_m[4]) / blocksizes_m[4]);
        else
          currloc += blockstride_m[4] * allDomain_m[4].last();
      }
      if (i5 >= firsti_m[5]) {
        if (i5 <= innerdomain_m[5].last())
          currloc += blockstride_m[5] * ((i5 - firsti_m[5]) / blocksizes_m[5]);
        else
          currloc += blockstride_m[5] * allDomain_m[5].last();
      }
      if (i6 >= firsti_m[6]) {
        if (i6 <= innerdomain_m[6].last())
          currloc += blockstride_m[6] * ((i6 - firsti_m[6]) / blocksizes_m[6]);
        else
          currloc += blockstride_m[6] * allDomain_m[6].last();
      }
    }
    
  // Return the globalID for the currloc's node
  
  PAssert(currloc >= 0 && currloc < all_m.size());
  return currloc;
}

//============================================================
// NewDomain1 traits classes for UniformGridLayout and
// UniformGridLayoutView
//============================================================

//-----------------------------------------------------------------------------
//
// This is so an array can be initialized with a UniformGridLayout.
//
//-----------------------------------------------------------------------------

template <int Dim>
struct NewDomain1<UniformGridLayout<Dim> >
{
  typedef UniformGridLayout<Dim> &Type_t;

  inline static Type_t combine(const UniformGridLayout<Dim> &a)
    {
      return const_cast<Type_t>(a);
    }
};

//-----------------------------------------------------------------------------
//
// This is so an array can be initialized with a UniformGridLayoutView.
//
//-----------------------------------------------------------------------------

template <int Dim, int Dim2>
struct NewDomain1<UniformGridLayoutView<Dim, Dim2> >
{
  typedef UniformGridLayoutView<Dim, Dim2> &Type_t;

  inline static Type_t combine(const UniformGridLayoutView<Dim, Dim2> &a)
    {
      return const_cast<Type_t>(a);
    }
};

//-----------------------------------------------------------------------------
//
// ostream inserters for UniformGridLayout and UniformGridLayoutView:
//
//-----------------------------------------------------------------------------

template <int Dim>
std::ostream &operator<<(std::ostream &ostr, 
                         const UniformGridLayout<Dim> &layout)
{
  layout.print(ostr);
  return ostr;
}

template <int Dim, int Dim2>
std::ostream &operator<<(std::ostream &ostr, 
                         const UniformGridLayoutView<Dim, Dim2> &layout)
{
  layout.print(ostr);
  return ostr;
}


// } // namespace POOMA

#include "Layout/UniformGridLayout.cpp"

#endif // POOMA_LAYOUT_UNIFORMGRIDLAYOUT_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: UniformGridLayout.h,v $   $Author: luchini $
// $Revision: 1.80 $   $Date: 2000/06/27 19:11:34 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
