// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

//-----------------------------------------------------------------------------
// Field test 3: Simple test of differential operators.
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Includes
//-----------------------------------------------------------------------------

#include "Pooma/Fields.h"
#include "Utilities/Tester.h"

#include <iostream>


//-----------------------------------------------------------------------------
// Globals
//-----------------------------------------------------------------------------

const int NX = 6, NY = 6;
Interval<1> I(NX), J(NY);
Interval<2> pd(NX, NY), pdCell(NX - 1, NY - 1);
  

//-----------------------------------------------------------------------------
// Test function
//-----------------------------------------------------------------------------

template<class Geom, class T, class EngineTag>
void doTest(Pooma::Tester &tester, Field<Geom, T, EngineTag> &f)
{
  f.addBoundaryConditions(AllZeroFaceBC());
    
  for (int i = 0; i <= f.physicalDomain()[0].last(); i++)
    for (int j = 0; j <= f.physicalDomain()[1].last(); j++)
      f(i, j) = Vector<2>(2.0 * i, 3.0 * j);
      
  f.applyBoundaryConditions();
  
  tester.out() << f << std::endl;
  tester.out() << f.physicalDomain() << std::endl;
  tester.out() << f.totalDomain() << std::endl;
  tester.out() << f.x() << std::endl;
  
  tester.out() << div<Cell>(f).physicalDomain() << std::endl;
  tester.out() << div<Cell>(f).totalDomain() << std::endl;
  tester.out() << div<Cell>(f) << std::endl;
  
  tester.check("PD", div<Cell>(f).physicalDomain(), pdCell);
  tester.check("div<Cell>(f)(1,3)", div<Cell>(f)(1,3), 3.5);
  tester.check("div<Cell>(f)(-1,0)", div<Cell>(f)(-1,0), 0.75);
  tester.check("div<Cell>(f)(4,5)", div<Cell>(f)(4,5), -6.5);
  tester.check("sum(div<Cell>(f))", sum(div<Cell>(f)), 3.5*25);
}


//-----------------------------------------------------------------------------
// Main program
//-----------------------------------------------------------------------------

int main(int argc, char *argv[])
{
  Pooma::initialize(argc,argv);
  Pooma::Tester tester(argc, argv);

  // Create the mesh.
  
  Vector<2,double> origin;
  Vector<2,double> spacings;
  for (int d = 0; d < 2; d++) 
    {
      origin(d) = d;
      spacings(d) = d + 1;
    }
  UniformRectilinearMesh<2> mesh(I, J, origin, spacings);
  
  // Create the geometry.
  
  typedef DiscreteGeometry<Vert, UniformRectilinearMesh<2> > Geometry_t;
  
  Geometry_t geom(mesh, GuardLayers<2>(1));
  
  // Make a Brick-Engine-based field.
  
  Field<Geometry_t, Vector<2> > f(geom);

  doTest(tester, f);
  
  // Make a CompressibleBrick-Engine-based field.
  
  Field<Geometry_t, Vector<2>, CompressibleBrick> fc(geom);

  doTest(tester, fc);
  
  // Make a Multipatch-Engine-based field.

  Loc<2> blocks(2,2);
  UniformGridLayout<2> layout(pd, blocks, GuardLayers<2>(1), 
                              GuardLayers<2>(1), ReplicatedTag());
  typedef MultiPatch<UniformTag, Brick> MP_t;
  Field<Geometry_t, Vector<2>, MP_t> fm(geom, layout);

  doTest(tester, fm);
  
  // Make a Nonuniform Multipatch-Engine-based field.

  GridLayout<2> layout2(pd, blocks, GuardLayers<2>(1), GuardLayers<2>(1), 
                        ReplicatedTag());
  typedef MultiPatch<GridTag, Brick> MP2_t;
  Field<Geometry_t, Vector<2>, MP2_t> fg(geom, layout2);

  doTest(tester, fg);
    
  int ret = tester.results("field_test3");
  Pooma::finalize();
  return ret;
}

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: field_test3.cpp,v $   $Author: julianc $
// $Revision: 1.14 $   $Date: 2000/06/07 16:25:48 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
