// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license
//-----------------------------------------------------------------------------
// Dirty flag test.
//-----------------------------------------------------------------------------

#include "Pooma/Pooma.h"
#include "Pooma/UMPArrays.h"
#include "Layout/GuardLayers.h"
#include "Layout/Node.h"
#include "Utilities/Tester.h"

#include <iterator>
#include <vector>
using std::vector;

int main(int argc, char *argv[])
{
  Pooma::initialize(argc,argv);
  Pooma::Tester tester(argc, argv);
  
  // Useful typedefs, constants, declarations, &c.
  
  typedef Brick                      PTag_t;
  typedef BrickView                  PVTag_t;
  typedef UniformTag                 LTag_t;
  typedef MultiPatch<LTag_t,PTag_t>  MPTag_t;
  typedef Engine<2,int,MPTag_t>      UMPEngine_t;
  typedef Array<2,int,MPTag_t>       UMPArray_t;
  
  typedef Engine<2,int,PTag_t>       PatchEngine_t;
  typedef Engine<2,int,PVTag_t>      PatchViewEngine_t;
  typedef Array<2,int,PTag_t>        PatchArray_t;
  typedef Array<2,int,PVTag_t>       PatchViewArray_t;
  typedef Array<2,int,Brick>         BrickArray_t;
  
  typedef UniformGridLayout<2>       Layout_t; // Change if LTag_t is changed.
  typedef UniformGridLayoutView<2,2> ViewLayout_t;
  typedef Node<Interval<2> >         Node_t;
  
  typedef MultiPatchView<LTag_t,PTag_t,2> VTag_t;
  typedef Engine<2,int,VTag_t>            ViewEngine_t;
  typedef Array<2,int,VTag_t>             ViewArray_t;
  
  using std::endl;
  
  // Run parameters...
  
  const int size            = 9;       // Array will be size x size
  const int nblocks         = 3;       // Will have nblocks x nblocks patches
  const int internal_guards = 2;
  const int external_guards = 1;
  
  // Create the total domain.
  
  Interval<1> D(size);
  Interval<2> domain(D, D);
  
  // Create the block sizes.
  
  Loc<2> blocks(nblocks,nblocks);
  
  // OK, let's try some guard cells.
  
  GuardLayers<2> igcs(internal_guards), egcs(external_guards);

  // Create the partitioners.
  
  UniformGridPartition<2> partition(blocks,igcs,egcs);
  
  // Create the layout.
  
  UniformGridLayout<2> layout(domain, partition, ReplicatedTag());
  
  tester.out() << "\nCreating array a and assigning to it." << endl;

  UMPArray_t a(layout);

  a = 1;  // The ultimate test of whether POOMA is working 8-).

  tester.out() << "a's dirty flag is " << a.engine().isDirty() << endl;
  tester.check(a.engine().isDirty() == true);
  
  UMPArray_t b = a;

  tester.out() << "b's dirty flag is " << b.engine().isDirty() << endl;
  tester.check(b.engine().isDirty() == true);

  a.engine().fillGuards();

  tester.out() << "\nFilled a's guards." << endl;

  tester.out() << "a's dirty flag is " << a.engine().isDirty() << endl;
  tester.check(a.engine().isDirty() == false);
  
  tester.out() << "b's dirty flag is " << b.engine().isDirty() << endl;
  tester.check(b.engine().isDirty() == false);  
  
  // Create the view domain.
  
  CTAssert(size-2 > 2);
  Interval<1> DV(2,size-2);
  Interval<2> viewDomain(DV, DV);

  tester.out() << "\nCreating a view of a and assigning to it..." << endl;

  ViewArray_t av(a,viewDomain);
  
  av = 2;

  tester.out() << "a's dirty flag is " << a.engine().isDirty() << endl;
  tester.out() << "b's dirty flag is " << b.engine().isDirty() << endl;
  tester.check(a.engine().isDirty() == true);
  tester.check(b.engine().isDirty() == true);

  tester.out() << "av's dirty flag is " << av.engine().isDirty() << endl;
  tester.check(av.engine().isDirty() == true);

  tester.out() << "\nFilling av's guards..." << endl;

  av.engine().fillGuards();

  tester.out() << "a's dirty flag is " << a.engine().isDirty() << endl;
  tester.out() << "b's dirty flag is " << b.engine().isDirty() << endl;
  tester.check(a.engine().isDirty() == false);
  tester.check(b.engine().isDirty() == false);

  tester.out() << "av's dirty flag is " << av.engine().isDirty() << endl;
  tester.check(av.engine().isDirty() == false);

  int retval = tester.results("ump_test9: dirty flag test.");

  Pooma::finalize();    
  return retval;
}
    
// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: ump_test9.cpp,v $   $Author: sa_smith $
// $Revision: 1.7 $   $Date: 2000/07/11 23:06:54 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
