// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

#ifndef POOMA_DOMAIN_SPLIT_H
#define POOMA_DOMAIN_SPLIT_H

//-----------------------------------------------------------------------------
// Function:
//   bool split(domain, domain, domain);
// Class:
//   SplitDomain<T,Dim>
//   SplitDomainSingle<T,Dim,bool>
//-----------------------------------------------------------------------------

//////////////////////////////////////////////////////////////////////

//-----------------------------------------------------------------------------
// Overview: 
// void split(domain,domain,domain) is a global function which splits the
// first argument into two separate domains, roughly in the moddle.  If
// the first argument has zero length, this does nothing.  If the first
// argument has a length of one, the second argument is a copy of the
// first, and the second is set to be empty.
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Typedefs:
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Includes:
//-----------------------------------------------------------------------------

#include "Domain/DomainTraits.h"
#include "Utilities/PAssert.h"


//-----------------------------------------------------------------------------
// Forward Declarations:
//-----------------------------------------------------------------------------


//-----------------------------------------------------------------------------
//
// Full Description of SplitDomainSingle:
//
// SplitDomainSingle<T,Dim,bool strided>::split(a,b,c) splits just the
// Dim dimension of the first argument into the second and third argument.
// It is specialized on the third parameter indicating whether the
// domain has unit stride or not, and whether the type is int or not.
//
//-----------------------------------------------------------------------------

//
// The default (unit-stride) version of SplitDomainSingle, which assumes
// that the domains have unit stride.
//

template<class T, int Dim, bool strided>
struct SplitDomainSingle {
  static void split(const T &a, int axis, T &b, T &c) {
    // types for elements in these domains
    typedef typename DomainTraits<T>::Element_t E1_t;

    // a typedef for a single-dimension domain
    typedef typename DomainTraits<T>::OneDomain_t OneDomain_t;

    // split operation depends on length
    if (axis != (Dim - 1)) {
      b[Dim-1] = a[Dim-1];
      c[Dim-1] = a[Dim-1];
    } else if (a[Dim-1].length() < 2) {
      b[Dim-1] = a[Dim-1];
    } else {
      E1_t  a0 = a[Dim-1].first();
      E1_t  a1 = a[Dim-1].last();
      E1_t mid = a0 + a[Dim-1].length()/2;
      b[Dim-1] = OneDomain_t(a0, mid-1);
      c[Dim-1] = OneDomain_t(mid, a1);
    }
  }

  static void split(const T &a, int axis, int leftLength, T &b, T &c) {
    // types for elements in these domains
    typedef typename DomainTraits<T>::Element_t E1_t;

    // a typedef for a single-dimension domain
    typedef typename DomainTraits<T>::OneDomain_t OneDomain_t;

    // split operation depends on length
    if (axis != (Dim - 1)) {
      b[Dim-1] = a[Dim-1];
      c[Dim-1] = a[Dim-1];
    } else if (a[Dim-1].length() < 2) {
      b[Dim-1] = a[Dim-1];
    } else {
      E1_t  a0 = a[Dim-1].first();
      E1_t  a1 = a[Dim-1].last();
      E1_t mid = a0 + leftLength;
      b[Dim-1] = OneDomain_t(a0, mid-1);
      c[Dim-1] = OneDomain_t(mid, a1);
    }
  }

  static void split(const T &a, T &b, T &c) { split(a, Dim-1, b, c); }
};

//
// The non-unit-stride version of SplitDomainSingle.
//

template<class T, int Dim>
struct SplitDomainSingle<T,Dim,true> {
  static void split(const T &a, int axis, T &b, T &c) {
    // types for elements in these domains
    typedef typename DomainTraits<T>::Element_t E1_t;

    // a typedef for a single-dimension domain
    typedef typename DomainTraits<T>::OneDomain_t OneDomain_t;

    // split operation depends on length
    if (axis != (Dim - 1)) {
      b[Dim-1] = a[Dim-1];
      c[Dim-1] = a[Dim-1];
    } else if (a[Dim-1].length() < 2) {
      b[Dim-1] = a[Dim-1];
    } else {
      E1_t  a0 = a[Dim-1].first();
      E1_t  a1 = a[Dim-1].last();
      E1_t   s = a[Dim-1].stride();
      E1_t mid = a0 + (a[Dim-1].length()/2 * s);
      b[Dim-1] = OneDomain_t(a0, mid - s, s);
      c[Dim-1] = OneDomain_t(mid, a1, s);
    }
  }

  static void split(const T &a, int axis, int leftLength, T &b, T &c) {
    // types for elements in these domains
    typedef typename DomainTraits<T>::Element_t E1_t;

    // a typedef for a single-dimension domain
    typedef typename DomainTraits<T>::OneDomain_t OneDomain_t;

    // split operation depends on length
    if (axis != (Dim - 1)) {
      b[Dim-1] = a[Dim-1];
      c[Dim-1] = a[Dim-1];
    } else if (a[Dim-1].length() < 2) {
      b[Dim-1] = a[Dim-1];
    } else {
      E1_t  a0 = a[Dim-1].first();
      E1_t  a1 = a[Dim-1].last();
      E1_t   s = a[Dim-1].stride();
      E1_t mid = a0 + (leftLength * s);
      b[Dim-1] = OneDomain_t(a0, mid - s, s);
      c[Dim-1] = OneDomain_t(mid, a1, s);
    }
  }

  static void split(const T &a, T &b, T &c) { split(a, Dim-1, b, c); }
};

//
// Special version of SplitDomainSingle for int's, which must be
// handled uniquely.
//

template<int Dim, bool strided>
struct SplitDomainSingle<int,Dim,strided> {
  static void split(int a, int, int &b, int &c) {
    // splitting an integer just means to copy a to b, and set c to zero
    b = a;
    c = 0;
  }
  static void split(int a, int, int, int &b, int &c) {
    b = a;
    c = 0;
  }
  static void split(int a, int &b, int &c) {
    b = a;
    c = 0;
  }
};


//-----------------------------------------------------------------------------
//
// Full Description of SplitDomain:
//
// SplitDomain implements a basic template meta-program to split
// each dimension separately of the multidimensional domain.
// It uses SplitDomainSingle to do the single-domain splits,
// telling that struct whether the domain has unit stride or not.
// A general version of SplitDomain is defined, to split the
// domain in the 'Dim' dimension, and then a specialization is provided
// for Dim==1 that stops the metaprogram recursion.
//
//-----------------------------------------------------------------------------

template<class T, int Dim>
struct SplitDomain {
  static void split(const T &a, T &b, T &c) {
    // the number of relevant domains here which have unit stride
    const int USN = DomainTraits<T>::unitStride;

    // Split along the current dimension
    SplitDomainSingle<T,Dim,(USN == 0)>::split(a, b, c);

    // Split the remaining dimensions
    SplitDomain<T,Dim-1>::split(a, b, c);
  }

  static void split(const T &a, int axis, T &b, T &c) {
    // the number of relevant domains here which have unit stride
    const int USN = DomainTraits<T>::unitStride;

    // Split along the current dimension
    SplitDomainSingle<T,Dim,(USN == 0)>::split(a, axis, b, c);

    // Split the remaining dimensions
    SplitDomain<T,Dim-1>::split(a, axis, b, c);
  }

  static void split(const T &a, int axis, int leftLength, T &b, T &c) {
    // the number of relevant domains here which have unit stride
    const int USN = DomainTraits<T>::unitStride;

    // Split along the current dimension
    SplitDomainSingle<T,Dim,(USN == 0)>::split(a, axis, leftLength, b, c);

    // Split the remaining dimensions
    SplitDomain<T,Dim-1>::split(a, axis, leftLength, b, c);
  }
};

template<class T>
struct SplitDomain<T,1> {
  static void split(const T &a, T &b, T &c) {
    // the number of relevant domains here which have unit stride
    const int USN = DomainTraits<T>::unitStride;

    // Split along the current dimension
    SplitDomainSingle<T,1,(USN == 0)>::split(a, b, c);
  }

  static void split(const T &a, int axis, T &b, T &c) {
    // the number of relevant domains here which have unit stride
    const int USN = DomainTraits<T>::unitStride;

    // Split along the current dimension
    SplitDomainSingle<T,1,(USN == 0)>::split(a, axis, b, c);
  }

  static void split(const T &a, int axis, int leftLength, T &b, T &c)
  {
    // the number of relevant domains here which have unit stride
    const int USN = DomainTraits<T>::unitStride;

    // Split along the current dimension
    SplitDomainSingle<T,1,(USN == 0)>::split(a, axis, leftLength, b, c);
  }
};


//-----------------------------------------------------------------------------
//
// Full Description of split:
//
// void split(domain,domain,domain) is a global function which splits the
// first argument into two separate domains, roughly in the middle.  If
// the first argument has zero length, this does nothing.  If the first
// argument has a length of one, the second argument is a copy of the
// first, and the second is set to be empty.
//
// void split(domain,axis,domain,domain) is a global function which splits
// the first argument into two separate domains just along the Nth axis instead
// of along all axes.  Otherwise it is the same as the other global split.
//
// The implementation of split is deferred to the SplitDomain
// struct, which performs the split for each dimension.
//
//-----------------------------------------------------------------------------

template<class T>
inline void split(const T &a, T &b, T &c)
{
  SplitDomain<T,DomainTraits<T>::dimensions>::split(a, b, c);
}


template<class T>
inline void split(const T &a, int axis, T &b, T &c)
{
  SplitDomain<T,DomainTraits<T>::dimensions>::split(a, axis, b, c);
}


template<class T>
inline void split(const T &a, int axis, int leftLength, T &b, T &c)
{
  SplitDomain<T,DomainTraits<T>::dimensions>::split(a, axis, leftLength, b, c);
}


//////////////////////////////////////////////////////////////////////

#endif     // POOMA_DOMAIN_SPLIT_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: Split.h,v $   $Author: swhaney $
// $Revision: 1.8 $   $Date: 2000/03/07 13:16:39 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
