// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

//-----------------------------------------------------------------------------
// BCond<Field<Geom, T, EngineTag>, PeriodicFaceBC> template definitions.
//-----------------------------------------------------------------------------

// Include files.

#include "Geometry/CenteringTags.h"
#include "Meshes/MeshTraits.h"
#include "Utilities/PAssert.h"

#if defined(POOMA_DEBUG_BOUNDARY_CONDITIONS)
#include <iostream>
#endif

//-----------------------------------------------------------------------------
// BCond<Field<Geom, T, EngineTag>, PeriodicFaceBC>()
// 
// Constructor.
//-----------------------------------------------------------------------------

template<class Geom, class T, class EngineTag>
BCond<Field<Geom, T, EngineTag>, PeriodicFaceBC>::
BCond(const Field<Geom, T, EngineTag> &f, const PeriodicFaceBC &bc)
  : Base_t(f, f.totalDomain(), f.totalDomain()), bc_m(bc)
{
  // This BC only makes sense for geometries on logically rectilinear meshes
  CTAssert(MeshTraits<typename Geom::Mesh_t>::isLogicallyRectilinear);

  int adjust;
  typedef typename Geom::Centering_t Centering_t;
      
  // Get the direction.
  int d = bc.face() / 2;

  // Select the high or low face:

  if (bc.face() & 1) {

    // High face.

    // Get the number of guard layers in the upper direction.
    int nGuards = subject().geometry().guardLayers().upper(d);
	
    // The other directions span the subject's total domain. Therefore, we
    // just chop out the guard layers.

    if (subject().geometry().centering().centeringTag(d) == vert) {
      adjust = 1;
    } else {
      adjust = 0;
    }

    // For Vert centering, because boundary is at last physical vertex, which
    // is identified with the first physical vertex, destination domain is
    // extended to includes last physical vertex:
    destDomain()[d] = 
      Interval<1>(destDomain()[d].max() - (nGuards - 1 + adjust), 
    		  destDomain()[d].max());

    // N.B.: only do this here on the High face; don't do it again down below
    // on the low face. This follows the convention of r1.

    // Source domain is just the destination domain offset by the periodicity
    // length (number of *cells*):
    srcDomain()[d] = 
      Interval<1>(destDomain()[d].min() - 
		  (subject().physicalDomain()[d].length() - adjust),
		  destDomain()[d].max() - 
		  (subject().physicalDomain()[d].length() - adjust));

  } else {

    // Low face.
        
    // Get the number of guard layers in the lower direction.
    int nGuards = subject().geometry().guardLayers().lower(d);

    // The other directions span the subject's total domain. Therefore, we
    // just chop out the guard layers.

    if (subject().geometry().centering().centeringTag(d) == vert) {
      adjust = 1;
    } else {
      adjust = 0;
    }

    // For Vert centering, because boundary is at first physical vertex, which
    // is identified with the last physical vertex, destination domain is
    // extended to includes first physical vertex. However, *don't* include the
    // first physical vertex here on the Low face; see notes above regarding
    // the High face (following r1 convention):
    destDomain()[d] = 
      Interval<1>(destDomain()[d].min(), 
		  destDomain()[d].min() + (nGuards - 1));

    // Source domain is just the destination domain offset by the periodicity
    // length (number of *cells*):
    srcDomain()[d] = 
      Interval<1>(destDomain()[d].min() + 
		  (subject().physicalDomain()[d].length() - adjust),
		  destDomain()[d].max() + 
		  (subject().physicalDomain()[d].length() - adjust));

  }
}


//-----------------------------------------------------------------------------
// void applyBoundaryCondition()
// 
// The apply method for this boundary condition. (Kind of self-describing.)
//-----------------------------------------------------------------------------

template<class Geom, class T, class EngineTag>
void BCond<Field<Geom, T, EngineTag>, PeriodicFaceBC>::
applyBoundaryCondition()
{
  subject()(destDomain()) = subject()(srcDomain());
}


//-----------------------------------------------------------------------------
// Base_t *retarget()
// 
// This boundary condition's retarget function.
//-----------------------------------------------------------------------------

template<class Geom, class T, class EngineTag>
FieldBCondBase< Field<Geom, T, EngineTag> > *
BCond<Field<Geom, T, EngineTag>, PeriodicFaceBC>::
retarget(const Field<Geom, T, EngineTag> &f) const
{
  return new This_t(f, bc_m);
}

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: PeriodicFaceBC.cpp,v $   $Author: swhaney $
// $Revision: 1.10 $   $Date: 2000/03/07 13:16:12 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
