// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

//-----------------------------------------------------------------------------
// Class:
// 
// ConstantFaceBC
//-----------------------------------------------------------------------------

#ifndef POOMA_BCONDS_CONSTANTFACE_H
#define POOMA_BCONDS_CONSTANTFACE_H

//-----------------------------------------------------------------------------
// Overview: 
// ConstantFaceBC: BCondCategory class setting all guard layers beyond a
//                 specified (logically) rectilinear mesh face to a constant 
//                 value.
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Typedefs:
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Includes:
//-----------------------------------------------------------------------------

#include "BConds/BCond.h"
#include "BConds/FieldBCondBase.h"

//-----------------------------------------------------------------------------
// Forward Declarations:
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Full Description:
// 
// ConstantFaceBC is a specialization of BCondCategory.
// 
// It represents a Dirichlet boundary condition on a logically rectilinear
// domain where the value on that face is a constant. The setConstant()
// function provides a means to have a time-dependent BC.  A constructor switch
// allows the BC to enforce that the mesh-boundary value is zero; this affects
// only vertex-centered Field values/components because the boundary is defined
// to be the last vertex.
// 
// The T template parameter is the type of the constant value. For use with a
// scalar Field, say a Field<D,double>, the type would be the scalar type
// (double, in this example). For use with a Field of Vector<D,double>, T
// would be Vector<D,double>. For use with the ComponentBC wrapper class to
// set a BC for a single component of a Field<D,Vector<D,double> >, T would be
// double (the type of a single component of a Vector<D,double> object).
//-----------------------------------------------------------------------------

template <class T>
class ConstantFaceBC : public BCondCategory<ConstantFaceBC<T> >
{
public:

  //---------------------------------------------------------------------------
  // Constructors. 
  
  ConstantFaceBC(int face, T constant, bool enforceConstantBoundary = false) : 
    face_m(face), constant_m(constant), 
    enforceConstantBoundary_m(enforceConstantBoundary)  { }
  ConstantFaceBC(const ConstantFaceBC &model) : 
    face_m(model.face()), constant_m(model.constant()), 
    enforceConstantBoundary_m(model.enforceConstantBoundary())
  { }

  //---------------------------------------------------------------------------
  // Assignment operator. Does deep assignment.
  
  ConstantFaceBC &operator=(const ConstantFaceBC &rhs)
  {
    face_m = rhs.face();
    constant_m = rhs.constant();
    enforceConstantBoundary_m = rhs.enforceConstantBoundary();

    return *this;
  }

  //---------------------------------------------------------------------------
  // Accessors for data members.
  
  int face() const { return face_m; }
  T constant() const { return constant_m; }
  bool enforceConstantBoundary() const { return enforceConstantBoundary_m; }

  // User may want to change the constant's value, for time-dependence, e.g.:
  void setConstant(T newConstant) { constant_m = newConstant; }
  
private:

  int face_m;
  T constant_m;
  bool enforceConstantBoundary_m;
};


// ----------------------------------------------------------------------------
// An actual boundary condition class for ConstantFaceBC (partial
// specialization of the BCond class)
// ----------------------------------------------------------------------------

template<class Geom, class T, class EngineTag, class T1>
class BCond<Field<Geom, T, EngineTag>, ConstantFaceBC<T1> > :
  public FieldBCondBase<Field<Geom, T, EngineTag> >
{
public:

  typedef BCond<Field<Geom, T, EngineTag>, ConstantFaceBC<T1> > This_t;
  typedef FieldBCondBase<Field<Geom, T, EngineTag> > Base_t;
  
  //---------------------------------------------------------------------------
  // Constructors.

  BCond(const Field<Geom, T, EngineTag> &f, const ConstantFaceBC<T1> &bc);

  //---------------------------------------------------------------------------
  // Destructor.

  ~BCond() { }

  //---------------------------------------------------------------------------
  // Methods.

  // Apply the boundary conditions.
  
  void applyBoundaryCondition();
  
  // Clone and retarget this boundary condition.
  
  Base_t *retarget(const Field<Geom, T, EngineTag> &f) const;

private:

  // Store a copy of the BCondCategory object used to construct this.

  ConstantFaceBC<T1> bc_m;
};

//-----------------------------------------------------------------------------
// Full Description:
// 
// AllConstantFaceBC is a functor class that sets all of a Field's BCs to be 
// ConstantFaceBC.
//-----------------------------------------------------------------------------

template<class T>
class AllConstantFaceBC
{
public:

  AllConstantFaceBC(const T &constant, bool enforceConstantBoundary = false)
  : const_m(constant), ecb_m(enforceConstantBoundary) { }
  
  template<class Geom, class T1, class EngineTag>
  void operator()(Field<Geom, T1, EngineTag> &f) const
    {
      for (int i = 0; i < 2 * Geom::dimensions; i++)
        f.addBoundaryCondition(ConstantFaceBC<T>(i, const_m, ecb_m));
    }

private:

  T const_m;
  bool ecb_m;
};

#include "BConds/ConstantFaceBC.cpp"

#endif // POOMA_BCONDS_CONSTANTFACE_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: ConstantFaceBC.h,v $   $Author: swhaney $
// $Revision: 1.9 $   $Date: 2000/03/07 13:16:11 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
