// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license
//-----------------------------------------------------------------------------
// Array test 5: complex array elements.
//-----------------------------------------------------------------------------

// Include files

#include "Pooma/Arrays.h"
#include "Utilities/Tester.h"
#include "Engine/BrickEngine.h"
#include "Array/Array.h"
#include "Tiny/Vector.h"

#include <complex.h>
#include <math.h>

template<class T>
inline void check(const T &ans, const T &correct, Pooma::Tester &tester)
{
  tester.check(ans == correct);
}

int main(int argc, char *argv[])
{
  Pooma::initialize(argc, argv);
  Pooma::Tester tester(argc, argv);

  complex<double> x(1.0, 2.0);
  Array<2> a(2, 2, modelElement(7.0));
  Array<2, complex<double> > b(2, 2, modelElement(x)), c(2, 2);
  Array<2> d(2,2);
  int i, j;

  Pooma::blockAndEvaluate();
  for (j = 0; j < 2; j++)
    for (i = 0; i < 2; i++)
      {
        check(a(i,j), 7.0, tester);
        check(b(i,j), x, tester);
      }

  c = a + 2.0 * b;

  Pooma::blockAndEvaluate();
  for (j = 0; j < 2; j++)
    for (i = 0; i < 2; i++)
      {
        check(c(i,j), complex<double>(9.0,4.0), tester);
      }

  complex<double> y(-3, -4);
  c += a + y * conj(b);
  
  Pooma::blockAndEvaluate();
  for (j = 0; j < 2; j++)
    for (i = 0; i < 2; i++)
      {
        check(c(i,j), complex<double>(5.0,6.0), tester);
      }

  d = norm(a + y * conj(b));

  Pooma::blockAndEvaluate();
  for (j = 0; j < 2; j++)
    for (i = 0; i < 2; i++)
      {
        check(d(i,j), 20.0, tester);
      }

  d = real(y * pow(b, 2));

  bool OK = true;
  Pooma::blockAndEvaluate();  
  for (j = 0; j < 2; j++)
    for (i = 0; i < 2; i++)
      {
        OK = (OK && (fabs(d(i,j) - 25.0) < 1e-6));
      }
  check(OK, true, tester);
  
  Array<1, complex<double> > e(2);
  Array<1, Vector<2, complex<double> > > f(2), g(2);
  Vector<2, complex<double> > v(complex<double>(1.0, 2.0),
    complex<double>(3.0, 4.0));
  Vector<2, complex<double> > v1(complex<double>(-3.0, -1.0),
    complex<double>(-7.0, -1.0));
  e = complex<double>(-1.0, 1.0);
  f = v;
  g = f * e;

  tester.check(all(g == v1));
  
  int ret = tester.results( "array_test5" );
  Pooma::finalize();
  return ret; 
}

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: array_test5.cpp,v $   $Author: julianc $
// $Revision: 1.12 $   $Date: 2000/03/10 18:10:46 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
