// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

//-----------------------------------------------------------------------------
// Tiny example
//-----------------------------------------------------------------------------

// include files

#include "Pooma/Arrays.h"
#include <iostream>
#include <math.h>

int main(int argc, char* argv[])
{
  // Initialize Pooma.
  Pooma::initialize(argc, argv);

  // Make an array of 100 3D ray vectors.
  Loc<1> patchSize(25);
  UniformGridLayout<1> layout(Interval<1>(100), patchSize, ReplicatedTag());
  Array<1, Vector<3>, MultiPatch<UniformTag,Brick> > rays(layout);
  
  // Set the third component of all of the vectors to zero.
  rays.comp(2) = 0.0;
  
  // Starting some scalar code, must block.
  Pooma::blockAndEvaluate();
  
  // Fill the vectors with a random value for the first component.
  for (int i = 0; i < 100; i++)
    std::cin >> rays(i)(0);
  
  // Define a unit vector pointing in the y direction.
  Vector<3> n(0.0, 1.0, 0.0);
    
  // Set the second component so that the length is one.
  rays.comp(1) = sqrt(1.0 - rays.comp(0) * rays.comp(0));

  // Reflect the rays off of a plane perpendicular to the y axis.  
  rays += -2.0 * dot(rays, n) * n;
  
  // Define a diagonal tensor:
  Tensor<3,double,Diagonal> xyflip2(0.0);
  xyflip2(0,0) = -2.0; xyflip2(1,1) = -2.0;

  // Tensor-Vector dot product multiplies x and y components by -2.0:
  rays = dot(xyflip2, rays);

  // Output the rays.
  std::cout << rays << std::endl;
  
  // Clean up and leave.
  Pooma::finalize();
  return 0;
}

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: Tiny.cpp,v $   $Author: julianc $
// $Revision: 1.10 $   $Date: 2000/07/24 16:09:36 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
