/*
 * $Id: xim.c,v 1.6 1999/02/18 08:08:13 thhsieh Exp $
 */
/*
    Copyright (C) 1999 by  XCIN TEAM

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    For any question or suggestion, please mail to xcin mailing-list:
    xcin@linux.org.tw, or the maintainer Tung-Han Hsieh: thhsieh@linux.org.tw
*/      


#include <string.h>
#include <X11/Xlocale.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/keysym.h>
#include <Ximd/IMdkit.h>
#include <Ximd/Xi18n.h>
#include "xcintool.h"
#include "xcin_core.h"
#include "gui.h"
#include "xim.h"


xim_config_t  xim_conf;


/*----------------------------------------------------------------------------

	Tool Functions.

----------------------------------------------------------------------------*/

static void
xim_commit(IC *ic, char *str)
{
    char *cch_str_list[1];
    XTextProperty tp;
    int cch_len;
    IMCommitStruct call_data;

    if (! str)
	str = ic->cch;
    else {
        cch_len = strlen(str);
        if (ic->cch_size < cch_len+1) {
	    ic->cch_size = cch_len+1;
	    ic->cch = realloc(ic->cch, ic->cch_size);
        }
        strcpy(ic->cch, str);
    }
    DebugLog("commit str: %s\n", str);

    cch_str_list[0] = str;
    XmbTextListToTextProperty(gui_conf.display, cch_str_list, 1,
                                  XCompoundTextStyle, &tp);
    bzero(&call_data, sizeof(call_data));
    call_data.major_code = XIM_COMMIT;
    call_data.icid = ic->id;
    call_data.connect_id = ic->connect_id;
    call_data.flag = XimLookupChars;
    call_data.commit_string = tp.value;
    IMCommitString(xim_conf.ims, (XPointer)(&call_data));
    XFree(tp.value);
}

static int
process_IMKEY(IC *ic, keyinfo_t *keyinfo, unsigned int ret)
{
    char *cch;

    if (ret == 0) {
	/* This is IMKET_ABSORB */
	return True;
    }
    if ((ret & IMKEY_COMMIT) && ic->inpinfo.cch) {
	/* Send the result to IM lib. */
	xim_commit(ic, ic->inpinfo.cch);
    }
    if ((ret & IMKEY_BELL)) {
	/* The IM request a bell ring. */
	XBell(gui_conf.display, -50);
    }
    if ((ret & IMKEY_SHIFTESC) && keyinfo) {
	/* The IM request a ESC escape. */
	if ((ic->inp_state & IC_2BYTES))
	    cch = fullchar_ascii(&(ic->inpinfo), 1, keyinfo);
	else
	    cch = halfchar_ascii(&(ic->inpinfo), 1, keyinfo);
	if (cch)
	    xim_commit(ic, cch);
    }
    if (! (ret & IMKEY_IGNORE)) {
	/* The key will not be processed further more. */
	return True;
    }
    return False;
}

void
call_switch_in(IC *ic)
{
    xim_conf.ic = ic;
    gui_conf.winchange |= WIN_CHANGE_IM;

    if (ic->imodp && ic->imodp->switch_in && ! (ic->inp_state & IC_XIMFOCUS)) {
        ic->inpinfo.xcin_wlen = gui_conf.c_width;
        ic->imodp->switch_in(ic->imodp->conf, &(ic->inpinfo));
        ic->inp_state |= IC_XIMFOCUS;
    }
}

void
call_switch_out(IC *ic)
{
    if (ic->imodp && ic->imodp->switch_out && (ic->inp_state & IC_XIMFOCUS)) {
	ic->imodp->switch_out(ic->imodp->conf, &(ic->inpinfo));
	ic->inp_state &= ~(IC_XIMFOCUS);
    }
}

void
call_xim_init(IC *ic)
{
    if (ic->imodp && ! (ic->inp_state & IC_XIMINIT)) {
        ic->inpinfo.xcin_wlen = gui_conf.c_width;
        ic->imodp->xim_init(ic->imodp->conf, &(ic->inpinfo));
        ic->inp_state |= IC_XIMINIT;
    }
}

void
call_xim_end(IC *ic)
{
    unsigned int ret;

    if (ic->imodp && (ic->inp_state & IC_XIMINIT)) {
        if ((ret = ic->imodp->xim_end(ic->imodp->conf, &(ic->inpinfo))))
	    process_IMKEY(ic, NULL, ret);
	ic->inp_state &= ~(IC_XIMINIT);
    }
}

int
change_IM(IC *ic, int inp_num)
{
    cinput_t *cp;
    imodule_t *imodp=NULL;
    unsigned int ret;

/*
 *  Check if the module of the desired cinput has been loaded or not.
 */
    if (inp_num >= 0 && inp_num < MAX_INP_ENTRY) {
	cp = cinput + inp_num;
	if (! cp->modname)
	    return 0;

        if (! cp->inpmod && ! (cp->inpmod = 
	    load_module(cp->modname, cp->objname, MOD_CINPUT))) {
	    perr(XCINMSG_WARNING, "cannot load IM: %s, ignore.\n", cp->objname);
	    free_module(cp);
	    return 0;
	}
	imodp = cp->inpmod;
    }

/*
 *  The switch_out() xim_end() will only be called when really change IM.
 */
    gui_conf.winchange |= WIN_CHANGE_IM;
    if (ic->imodp && imodp != ic->imodp) {
	call_switch_out(ic);
	call_xim_end(ic);
    }
    if (inp_num < 0 || inp_num >= MAX_INP_ENTRY) {
    /* This will switch to English mode. */
        ic->inp_state &= ~(IC_CINPUT);
	ic->imodp = NULL;
        if ((xcin_conf.xcin_mode & XCIN_IM_FOCUS))
	    xcin_conf.xcin_mode &= ~(XCIN_RUN_IM_FOCUS);
	return 1;
    }

/*
 *  Initialize the new IM for this IC.
 */
    if (imodp) {
        ic->imodp = imodp;
        ic->inp_num = inp_num;
    }
    else {
	ic->imodp = cinput[xcin_conf.default_im].inpmod;
	ic->inp_num = xcin_conf.default_im;
    }
    if ((xcin_conf.xcin_mode & XCIN_IM_FOCUS)) {
	xcin_conf.xcin_mode |= XCIN_RUN_IM_FOCUS;
	xim_conf.im_focus = inp_num;
    }
    call_xim_init(ic);
    call_switch_in(ic);
    ic->inp_state |= IC_CINPUT;

    if ((xcin_conf.xcin_mode & XCIN_SHOW_CINPUT)) {
	ic->s_imodp = ic->imodp;
	ic->sinp_num = ic->inp_num;
    }
    else {
	ic->s_imodp = cinput[xcin_conf.default_im_sinmd].inpmod;
	ic->sinp_num = xcin_conf.default_im_sinmd;
    }

    return 1;
}

static void
circular_change_IM(IC *ic, int direction)
{
    int i, j;
    inp_state_t inpnum;

    inpnum = ((ic->inp_state & IC_CINPUT)) ? ic->inp_num : MAX_INP_ENTRY;
    for (i=0, j=inpnum+direction; i<MAX_INP_ENTRY+1; i++, j+=direction) {
	if (j > MAX_INP_ENTRY || j < 0)
	    j += ( (j < 0) ? (MAX_INP_ENTRY+1) : -(MAX_INP_ENTRY+1) );
	if (change_IM(ic, j))
	    return;
    }
}


static KeySym switch_keylist[MAX_INP_ENTRY] = {
    XK_0, XK_1, XK_2, XK_3, XK_4, XK_5, XK_6, XK_7, XK_8, XK_9,
    XK_minus, XK_equal
};

static int 
match_trigger_key(KeySym key, unsigned int state, IC *ic)
{
    KeySym keysym;
    unsigned int modifier;
    unsigned int modifier_mask;
    int inp_num;
    cinput_t *cp;

    keysym = XK_space;			/* ctrl+space: open/close IM */
    modifier = modifier_mask = ControlMask;
    if (keysym == key && (state & modifier_mask) == modifier) {
	if ((ic->inp_state & IC_CINPUT))
	    change_IM(ic, -1);
	else
	    change_IM(ic, ic->inp_num);
        return True;
    }

    keysym = XK_space;			/* shift+space: switch sb/2b */
    modifier = modifier_mask = ShiftMask;
    if (keysym == key && (state & modifier_mask) == modifier) {
	if ((ic->inp_state & IC_2BYTES))
	    ic->inp_state &= ~(IC_2BYTES);
	else
	    ic->inp_state |= IC_2BYTES;
	gui_conf.winchange |= WIN_CHANGE_IM;
	return True;
    }

    keysym = XK_Shift_L;		/* ctrl+shift: circular + */
    modifier = modifier_mask = ControlMask;
    if (keysym == key && (state & modifier_mask) == modifier) {
	circular_change_IM(ic, 1);
	return True;
    }

    keysym = XK_Control_L;		/* shift+ctrl: circular - */
    modifier = modifier_mask = ShiftMask;
    if (keysym == key && (state & modifier_mask) == modifier) {
	circular_change_IM(ic, -1);
	return True;
    }

    modifier = modifier_mask = ControlMask|Mod1Mask;	
					/* ctrl+alt+?: select IM */
    for (inp_num=0; inp_num<MAX_INP_ENTRY; inp_num++) {
	keysym = switch_keylist[inp_num];
        if (keysym == key && (state & modifier_mask) == modifier) {
	    change_IM(ic, inp_num);
	    return True;
	}
    }

    return False;
}

static void
call_simd(IC *ic)
{
    simdinfo_t simdinfo;
    int skey_len;

    simdinfo.icid = ic->id;
    simdinfo.xcin_wlen = ic->inpinfo.xcin_wlen;
    simdinfo.guimode = ic->inpinfo.guimode;
    simdinfo.cch_publish.wch = ic->inpinfo.cch_publish.wch;
    if (simdinfo.cch_publish.wch && 
	(ic->s_imodp->show_keystroke(ic->s_imodp->conf, &simdinfo))) {
        skey_len = wchs_len(simdinfo.s_keystroke);
        if (ic->skey_size < skey_len+1) {
	    ic->skey_size = skey_len+1;
	    ic->sinmd_keystroke = realloc(
		ic->sinmd_keystroke, ic->skey_size * sizeof(wch_t));
        }
        memcpy(ic->sinmd_keystroke, 
		simdinfo.s_keystroke, (skey_len+1)*sizeof(wch_t));
        gui_conf.winchange |= WIN_CHANGE_IM_CONTENT;
    }
    else
	ic->sinmd_keystroke[0].wch = (wchar_t)0;
}

static void
change_simd(IC *ic)
{
    int i;
    cinput_t *cp;

    for (i=ic->sinp_num+1; i!=ic->sinp_num; i++) {
	if (i >= MAX_INP_ENTRY)
	    i -= MAX_INP_ENTRY;
	cp = cinput + i;
	if (cp->modname) {
	    if (! cp->inpmod && ! (cp->inpmod =
		load_module(cp->modname, cp->objname, MOD_CINPUT))) {
		perr(XCINMSG_WARNING, "cannot load IM: %s, ignore.\n", cp->objname);
		free_module(cp);
	    }
	    else
	        break;
	}
    }

    ic->sinp_num = i;
    ic->s_imodp = cp->inpmod;
    xcin_conf.default_im_sinmd = i;
    xcin_conf.xcin_mode &= ~XCIN_SHOW_CINPUT;
    call_simd(ic);
}


/*----------------------------------------------------------------------------

	IM Protocol Handler.

----------------------------------------------------------------------------*/


static int 
im_open(XIMS ims, IMOpenStruct *call_data)
{
    return True;
}

static int 
im_close(XIMS ims, IMCloseStruct *call_data)
{
    ic_clean_all(call_data->connect_id);
    gui_conf.winchange |= WIN_CHANGE_IM;
    return True;
}

static int 
im_set_focus(XIMS ims, IMChangeFocusStruct *call_data) 
{
    IC *ic;

    if (! (ic = ic_find(call_data->icid)))
        return False;

    call_switch_in(ic);
    if ((xcin_conf.xcin_mode & XCIN_RUN_IM_FOCUS) && 
	(ic->inp_state & IC_NEWIC)) {
        IMPreeditStateStruct im_protocol;

        im_protocol.major_code = XIM_PREEDIT_START;
        im_protocol.minor_code = 0;
        im_protocol.connect_id = ic->connect_id;
        im_protocol.icid = ic->id;
        if (IMPreeditStart(xim_conf.ims, (XPointer)&im_protocol) == True)
            change_IM(ic, xim_conf.im_focus);
    }
    ic->inp_state &= ~(IC_NEWIC);

    return True;
}

static int 
im_unset_focus(XIMS ims, IMChangeFocusStruct *call_data)
{
    IC *ic;
    unsigned int ret;

    if (! (ic = ic_find(call_data->icid)))
	return False;
    
    call_switch_out(ic);
    return True;
}

static int 
im_trigger(XIMS ims, IMTriggerNotifyStruct *call_data)
{
    inp_state_t inp_num;
    cinput_t *cp;
    IC *ic;

    if (! (ic = ic_find(call_data->icid)))
	return False;
    if (call_data->flag == 0) { 		/* on key */
        /* 
	 *  Here, the start of preediting is notified from
         *  IMlibrary, which is the only way to start preediting
         *  in case of Dynamic Event Flow, because ON key is
         *  mandatary for Dynamic Event Flow. 
	 */
        /* xim_preedit_callback_start(ims, ic); */

	switch (call_data->key_index) {	
	case 0:					/* ctrl+space: default IM */
	    change_IM(ic, ic->inp_num);
	    break;
	case 3:					/* ctrl+shift: circuler + */
	    circular_change_IM(ic, 1);
	    break;
	case 6:					/* shift+ctrl: circuler - */
	    circular_change_IM(ic, -1);
	    break;
	case 9:
	    ic->inp_state |= IC_2BYTES;		/* shift+space: sb/2b */
	    gui_conf.winchange = WIN_CHANGE_IM;
	    break;
	default:				/* ctrl+alt+?: select IM */
	    inp_num = (call_data->key_index - 12) / 3;
	    cp = cinput + inp_num;
	    if (! change_IM(ic, inp_num))
	        return IMPreeditEnd(ims, (XPointer)call_data);
	    break;
	}
        return True;
    } 
    else 					/* never happens */
        return False;
}

static int
im_forward(XIMS ims, IMForwardEventStruct *call_data)
{
    XKeyEvent *kev;
    unsigned int modifier, ret;
    keyinfo_t keyinfo;
    IC *ic;

    if (! (ic = ic_find(call_data->icid)))
	return False;
    if (call_data->event.type != KeyPress)
        return True;    		/* doesn't matter */

    kev = &call_data->event.xkey;
    keyinfo.keystate = kev->state;
    keyinfo.keystr_len = XLookupString(
	kev, keyinfo.keystr, sizeof(keyinfo.keystr), &(keyinfo.keysym), NULL);
    keyinfo.keystr[(keyinfo.keystr_len >= 0) ? keyinfo.keystr_len : 0] = 0;

/*
 *  Process the special key binding.
 */
    modifier = ControlMask|Mod1Mask;
    if ((keyinfo.keystate & modifier) == modifier) {
        /* For simd switch: ctrl+alt+i */
        if (keyinfo.keysym == XK_i) {
            change_simd(ic);
	    return True;
	}
        /* Repeat the previous char: ctrl+alt+r */
	else if (keyinfo.keysym == XK_r) {
	    xim_commit(ic, NULL);
	    return True;
	}
	/* Change IM focus state */
	else if (keyinfo.keysym == XK_f) {
	    if ((xcin_conf.xcin_mode & XCIN_IM_FOCUS)) {
		xcin_conf.xcin_mode &= ~(XCIN_IM_FOCUS);
		xcin_conf.xcin_mode &= ~XCIN_RUN_IM_FOCUS;
	    }
	    else {
		xcin_conf.xcin_mode |= XCIN_IM_FOCUS;
		xcin_conf.xcin_mode |= XCIN_RUN_IM_FOCUS;
		xim_conf.im_focus = ic->inp_num;
	    }
	    gui_conf.winchange |= WIN_CHANGE_IM;
	    return True;
	}
    }

    /* For quick key phrase output: shift+alt+? */
    modifier = ShiftMask|Mod1Mask;
    if ((keyinfo.keystate & modifier) == modifier) {
	char *str;
	if ((str = qphrase_str(keyinfo.keystr[0])))
	    xim_commit(ic, str);
	return True;
    }

    /* For trigger keys */
    if (match_trigger_key(keyinfo.keysym, keyinfo.keystate, ic)) {
	if (! (ic->inp_state & IC_CINPUT) && ! (ic->inp_state & IC_2BYTES))
	    return IMPreeditEnd(ims, (XPointer)call_data);
	else
	    return True;
    }

/*
 *  Forward key to IM.
 */
    gui_conf.winchange |= WIN_CHANGE_IM_CONTENT;
    if ((ic->inp_state & IC_CINPUT)) {
        ic->inpinfo.xcin_wlen = gui_conf.c_width;
        ret = ic->imodp->keystroke(ic->imodp->conf, &(ic->inpinfo), &keyinfo);
	call_simd(ic);
	if ((process_IMKEY(ic, &keyinfo, ret)))
	    return True;
    }
    if ((ic->inp_state & IC_2BYTES)) {
	char *cch;
        if ((cch = fullchar_keystroke(&(ic->inpinfo), keyinfo.keysym))) {
	    xim_commit(ic, cch);
	    return True;
	}
    }
    IMForwardEvent(xim_conf.ims, (XPointer)call_data);
    return True;
}


static int 
im_protocol_handler(XIMS ims, IMProtocol *call_data)
{
    int ret, icid=-1;

    switch (call_data->major_code) {
    case XIM_OPEN:      
        DebugLog("XIM_OPEN\n");
//	icid = call_data->imopen.id;
        ret = im_open(ims, &call_data->imopen);
	break;
    case XIM_CLOSE:     
        DebugLog("XIM_CLOSE\n");
//	icid = call_data->imclose.id;
        ret = im_close(ims, &call_data->imclose);
	break;
    case XIM_CREATE_IC: 
        DebugLog("XIM_CREATE_IC\n");
	icid = call_data->changeic.icid;
        ret = ic_create(ims, &call_data->changeic);
	break;
    case XIM_DESTROY_IC:
        DebugLog("XIM_DESTORY_IC\n");
	icid = call_data->destroyic.icid;
        ret = ic_destroy(ims, &call_data->destroyic);
	break;
    case XIM_SET_IC_FOCUS: 
        DebugLog("XIM_SET_IC_FOCUS\n");
	icid = call_data->changefocus.icid;
        ret = im_set_focus(ims, &call_data->changefocus);
	break;
    case XIM_UNSET_IC_FOCUS: 
        DebugLog("XIM_UNSET_IC_FOCUS\n");
	icid = call_data->changefocus.icid;
        ret = im_unset_focus(ims, &call_data->changefocus);
	break;
    case XIM_TRIGGER_NOTIFY: /* start/end of pre-editing signal */
        DebugLog("XIM_TRIGGER_NOTIFY\n");
	icid = call_data->triggernotify.icid;
        ret = im_trigger(ims, &call_data->triggernotify);
	break;
    case XIM_FORWARD_EVENT: 
        DebugLog("XIM_FORWARD_EVENT\n");
	icid = call_data->forwardevent.icid;
        ret = im_forward(ims, &call_data->forwardevent);
	break;
    case XIM_SET_IC_VALUES: 
        DebugLog("XIM_SET_IC_VALUES\n");
	icid = call_data->changeic.icid;
        ret = ic_set(ims, &call_data->changeic);
	break;
    case XIM_GET_IC_VALUES: 
        DebugLog("XIM_GET_IC_VALUES\n");
	icid = call_data->changeic.icid;
        ret = ic_get(ims, &call_data->changeic);
	break;



    case XIM_RESET_IC:
        DebugLog("XIM_RESET_IC_FOCUS:\n");
	ret = True;
	break;
    case XIM_PREEDIT_START_REPLY:
        DebugLog("XIM_PREEDIT_START_REPLY\n");
	ret = False;
	break;
    case XIM_PREEDIT_CARET_REPLY:
        DebugLog("XIM_PREEDIT_CARET_REPLY\n");
	ret = False;
	break;
    default:
	DebugLog("XIM Unknown\n");
	ret = False;
	break;
    }
    check_ic_exist(icid);

    if (gui_conf.winchange)
        win_draw_map();
    return  ret;
}


/*----------------------------------------------------------------------------

	XIM Initialization.

----------------------------------------------------------------------------*/

/* Supported input styles */
static XIMStyle defaultStyles[] = {
#if 0
    XIMPreeditCallbacks|XIMStatusCallbacks,
    XIMPreeditPosition|XIMStatusArea,
    XIMPreeditPosition|XIMStatusNothing,
    XIMPreeditArea|XIMStatusArea,
#endif
    XIMPreeditNothing|XIMStatusNothing,
    0
};

/* Supported Encodings */
static XIMEncoding zhEncodings[] = {
    "COMPOUND_TEXT",
    NULL
};

/* Supported on trigger keys. */
static XIMTriggerKey onTriggerKeys[] = {
    {XK_space, ControlMask, ControlMask},	/* ctrl+space: to IM */
    {XK_Shift_L, ControlMask, ControlMask},	/* ctrl+shift: IM recycle */
    {XK_Control_L, ShiftMask, ShiftMask},	/* shift+ctrl: IM recycle */
    {XK_space, ShiftMask, ShiftMask},		/* shift+space: 2B & SB */
    {XK_0, ControlMask|Mod1Mask, ControlMask|Mod1Mask},
    {XK_1, ControlMask|Mod1Mask, ControlMask|Mod1Mask},
    {XK_2, ControlMask|Mod1Mask, ControlMask|Mod1Mask},
    {XK_3, ControlMask|Mod1Mask, ControlMask|Mod1Mask},
    {XK_4, ControlMask|Mod1Mask, ControlMask|Mod1Mask},
    {XK_5, ControlMask|Mod1Mask, ControlMask|Mod1Mask},
    {XK_6, ControlMask|Mod1Mask, ControlMask|Mod1Mask},
    {XK_7, ControlMask|Mod1Mask, ControlMask|Mod1Mask},
    {XK_8, ControlMask|Mod1Mask, ControlMask|Mod1Mask},
    {XK_9, ControlMask|Mod1Mask, ControlMask|Mod1Mask},
    {XK_minus, ControlMask|Mod1Mask, ControlMask|Mod1Mask},
    {XK_equal, ControlMask|Mod1Mask, ControlMask|Mod1Mask},
    {0L, 0L, 0L}
};


void xim_init(void)
{
    char transport[128], xim_name[128];
    unsigned int transport_type;
    XIMStyles input_styles;
    XIMTriggerKeys on_keys;
    XIMEncodings encodings;
    int i, j;

    transport_type = TRANSPORT_X;

    if (transport_type == TRANSPORT_LOCAL) {
        char hostname[64];
        char *address = "/tmp/.ximsock";
        gethostname(hostname, 64);
        sprintf(transport, "local/%s:%s", hostname, address);
    } 
    else if (transport_type == TRANSPORT_TCP) {
        char hostname[64];
        int port_number = XIM_TCP_PORT;
        gethostname(hostname, 64);
        sprintf(transport, "tcp/%s:%d", hostname, port_number);
    } 
    else 
        strcpy(transport, "X/");

    input_styles.count_styles = sizeof(defaultStyles)/sizeof(XIMStyle) - 1;
    input_styles.supported_styles = defaultStyles;

    encodings.count_encodings = sizeof(zhEncodings)/sizeof(XIMEncoding) - 1;
    encodings.supported_encodings = zhEncodings;

    on_keys.count_keys = sizeof(onTriggerKeys)/sizeof(XIMTriggerKey) - 1;
    on_keys.keylist = onTriggerKeys;

    if (xcin_conf.xim_name)
	xim_conf.xim_name = xcin_conf.xim_name;
    else if (strcasecmp("zh_TW.Big5", xcin_conf.locale.lc_ctype)) {
	sprintf(xim_name, "%s-%s", DEFAULT_XIMNAME, xcin_conf.locale.lc_ctype);
	xim_conf.xim_name = strdup(xim_name);
    }
    else
	xim_conf.xim_name = DEFAULT_XIMNAME;

    xim_conf.ims = IMOpenIM(gui_conf.display,
                   	    IMServerWindow,    gui_conf.window,
                   	    IMModifiers,       "Xi18n",
                   	    IMServerName,      xim_conf.xim_name,
                   	    IMLocale,          xcin_conf.locale.lc_ctype,
                   	    IMServerTransport, transport,
                   	    IMInputStyles,     &input_styles,
                            IMEncodingList,    &encodings,
                            IMProtocolHandler, im_protocol_handler,
                            IMFilterEventMask, KeyPressMask,
                  	    IMOnKeysList,      &on_keys,
                   	    NULL);
    if (xim_conf.ims == 0)
	perr(XCINMSG_ERROR, 
	_("IMOpenIM() with name \"%s\" locale \"%s\" transport \"%s\" failed.\n"),
		xim_conf.xim_name, xcin_conf.locale.lc_ctype, transport);
    else
	perr(XCINMSG_NORMAL,
	_("XIM server \"%s\" locale \"%s\" transport \"%s\" inp_style \"Root\".\n"),
		xim_conf.xim_name, xcin_conf.locale.lc_ctype, transport);
}

void
xim_close(void)
{
    IMCloseIM(xim_conf.ims);
}
