/*
 * $Id: xcin_main.c,v 1.7 1999/02/18 08:08:09 thhsieh Exp $
 */
/*
    Copyright (C) 1999 by  XCIN TEAM

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    For any question or suggestion, please mail to xcin mailing-list:
    xcin@linux.org.tw, or the maintainer Tung-Han Hsieh: thhsieh@linux.org.tw
*/      


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <X11/Xlocale.h>
#include "xcintool.h"
#include "xcin_core.h"
#include "syscin.h"
#include "imodule.h"
#include "gui.h"
#include "xim.h"

core_config_t xcin_conf;		/* XCIN global configuration. */
cinput_t cinput[MAX_INP_ENTRY];		/* CInput reference tables. */


/*----------------------------------------------------------------------------

        Initial Input.

----------------------------------------------------------------------------*/

static char *
check_default_dir(char *path)
{
    if (path && path[0] && check_file_exist(path, FTYPE_DIR))
	return  strdup(path);
    else if (check_file_exist(XCIN_DEFAULT_DIR, FTYPE_DIR))
	return  XCIN_DEFAULT_DIR;
    else {
	perr(XCINMSG_ERROR, _("the default xcin dir does not exist.\n"));
	return  NULL;
    }
}

static char *
check_user_dir(char *path, char *home)
{
    char str[256];

    if (! path) {
	snprintf(str, 256, "%s/%s", home, XCIN_USER_DIR);
        if (check_file_exist(str, FTYPE_DIR))
	    return  strdup(str);
    }
    else if (path[0] == '/' && check_file_exist(path, FTYPE_DIR))
	return  strdup(path);
    else {
	snprintf(str, 256, "%s/%s", home, path);
        if (check_file_exist(str, FTYPE_DIR))
	    return  strdup(str);
    }
    return  NULL;
}


static void
print_usage(void)
{
    int  i;

    perr(XCINMSG_EMPTY,
	_("Usage:  xcin [-h] [-d DISPLAY] [-x XIM_name] [-r rcfile]\n"
	  "Options:  -h: print this message.\n"
	  "          -d: set X Display.\n"
	  "          -x: register XIM server name to Xlib.\n"
	  "          -r: set user specified rcfile.\n\n"
	  "Useful environment variables:  $XCIN_RCFILE.\n\n"));
}

static void
command_switch(int argc, char **argv)
/* Command line arguements. */
{
    int  rev;
    char value[256] = {'\0'}, *home, *cmd[1];

    perr(XCINMSG_EMPTY,
	_("XCIN (Chinese XIM server) version %s.\n" 
	  "(module ver: %s, syscin ver: %s).\n"
          "(use \"-h\" option for help)\n\n"), 
	XCIN_VERSION, MODULE_VERSION, SYSCIN_VERSION);

    opterr = 0;
    while ((rev = getopt(argc, argv, "hd:x:r:")) != EOF) {
        switch (rev) {
        case 'h':
            print_usage();
            exit(0);
	case 'd':
	    xcin_conf.display_name = strdup(optarg);
	    break;
	case 'x':
	    xcin_conf.xim_name = strdup(optarg);
	    break;
        case 'r':
            strncpy(value, optarg, 256);
            break;
        case ':':
            perr(XCINMSG_ERROR, 
		_("arguement expected for option  -%c.\n"), optopt);
            break;
        case '?':
            perr(XCINMSG_ERROR, _("unknown option  -%c.\n"), optopt);
            break;
        }
    }
    if (! (home = getenv("HOME")))
	home = getenv("home");

/*
 *  Read rcfile & check default_dir and user_dir.
 */
    xcin_conf.rcfile = read_xcinrc(value, home);

    cmd[0] = RCKEY_XCIN_DEFAULT_DIR;
    if (get_resource(cmd, value, 256, 1))
	xcin_conf.default_dir = check_default_dir(value);
    else
	xcin_conf.default_dir = check_default_dir(NULL);

    cmd[0] = RCKEY_XCIN_USER_DIR;
    if (get_resource(cmd, value, 256, 1))
        xcin_conf.user_dir = check_user_dir(value, home);
    else
	xcin_conf.user_dir = check_user_dir(NULL, home);
}


/*----------------------------------------------------------------------------

        Resource Reading & Checking.

----------------------------------------------------------------------------*/


static void
read_core_config(void)
{
    char *cmd[1], value[256];
/*
 *  GUI config reading.
 */
    cmd[0] = RCKEY_INDEX_FONT;
    if (get_resource(cmd, value, 256, 1))
        set_data(&(xcin_conf.indexfont), RC_STRING, value, 0, 0);
    cmd[0] = RCKEY_FG_COLOR;
    if (get_resource(cmd, value, 256, 1))
        set_data(&(xcin_conf.fg_color), RC_STRING, value, 0, 0);
    cmd[0] = RCKEY_BG_COLOR;
    if (get_resource(cmd, value, 256, 1))
        set_data(&(xcin_conf.bg_color), RC_STRING, value, 0, 0);
    cmd[0] = RCKEY_M_FG_COLOR;
    if (get_resource(cmd, value, 256, 1))
        set_data(&(xcin_conf.mfg_color), RC_STRING, value, 0, 0);
    cmd[0] = RCKEY_M_BG_COLOR;
    if (get_resource(cmd, value, 256, 1))
        set_data(&(xcin_conf.mbg_color), RC_STRING, value, 0, 0);
    cmd[0] = RCKEY_XCIN_HIDE;
    if (get_resource(cmd, value, 256, 1))
        set_data(&(xcin_conf.xcin_mode), RC_IFLAG, value, XCIN_MODE_HIDE, 0);
    cmd[0] = RCKEY_X_GEOMETRY;
    if (get_resource(cmd, value, 256, 1))
        set_data(&(xcin_conf.geometry), RC_STRING, value, 0, 0);
    cmd[0] = RCKEY_XKILL_DISABLE;
    if (get_resource(cmd, value, 256, 1))
        set_data(&(xcin_conf.xcin_mode), RC_IFLAG, value, XCIN_XKILL_OFF, 0);
    cmd[0] = RCKEY_IM_FOCUS_ON;
    if (get_resource(cmd, value, 256, 1))
        set_data(&(xcin_conf.xcin_mode), RC_IFLAG, value, XCIN_IM_FOCUS, 0);
}

static void
read_core_config_locale(void)
{
    char *cmd[2], value[256], *s;
/*
 *  Read locale specific global setting.
 */
    cmd[0] = xcin_conf.locale.lc_ctype;
    cmd[1] = RCKEY_FONTSET;
    if (get_resource(cmd, value, 256, 2))
	set_data(&(xcin_conf.fontset), RC_STRING, value, 0, 0);

    cmd[1] = RCKEY_PHRASE;
    if (get_resource(cmd, value, 256, 2))
	set_data(&(xcin_conf.phrase_fn), RC_STRING, value, 0, 0);

    cmd[1] = RCKEY_DEFAULT_IM;
    if (! get_resource(cmd, value, 256, 2))
	perr(XCINMSG_ERROR, 
	_("rcfile: %s: value not specified.\n"), RCKEY_DEFAULT_IM);
    else
        set_data(&(xcin_conf.default_im_name), RC_STRING, value, 0, 0);

    cmd[1] = RCKEY_DEFAULT_IM_MODULE;
    if (! get_resource(cmd, value, 256, 2))
	perr(XCINMSG_ERROR, 
	_("rcfile: %s: value not specified.\n"), RCKEY_DEFAULT_IM_MODULE);
    else 
	set_data(&(xcin_conf.default_im_mod_name), RC_STRING, value, 0, 0);
    if ((s = strrchr(xcin_conf.default_im_mod_name, '.')) && !strcmp(s, ".so"))
	*s = '\0';

    cmd[1] = RCKEY_DEFAULT_IM_SINMD;
    if (! get_resource(cmd, value, 256, 2))
	perr(XCINMSG_ERROR, 
	_("rcfile: %s: value not specified.\n"), RCKEY_DEFAULT_IM_SINMD);
    else
	set_data(&(xcin_conf.default_im_sinmd_name), RC_STRING, value, 0, 0);

    cmd[1] = RCKEY_CINPUT;
    if (! get_resource(cmd, value, 256, 2))
	perr(XCINMSG_ERROR, 
	_("rcfile: %s: value not specified.\n"), RCKEY_CINPUT);
    else
	set_data(&(xcin_conf.im_objname), RC_STRING, value, 0, 0);
}

static void
read_core_config_cinput(void)
{
    char *cmd[2], value[256], *s, *s1;
    char objname[100];
    cinput_t *cinp;
    int setkey;

/*
 *  Go to each CINPUT sub-node and read important keywords.
 */
    s1 = xcin_conf.im_objname;
    while (get_word(&s1, objname, 100, NULL)) {
	cmd[0] = objname;
	cmd[1] = RCKEY_SETKEY;		/* setkey should be uniquely defined */
        if (! get_resource(cmd, value, 256, 2))
	    perr(XCINMSG_ERROR, _("rcfile: %s: %s: value not specified.\n"), 
		objname, RCKEY_SETKEY);
	else
	    set_data(&setkey, RC_INT, value, 0, 0);
	if (setkey < 0 || setkey > MAX_INP_ENTRY || cinput[setkey].objname)
	    perr(XCINMSG_ERROR, _("rcfile: %s: %s: not valid value.\n"), 
		objname, RCKEY_SETKEY);
	else
	    cinp = cinput + setkey;

	cmd[1] = RCKEY_MODULE;
        if (get_resource(cmd, value, 256, 2)) {
            if ((s = strrchr(value, '.')) && !strcmp(s, ".so"))
		*s = '\0';
	    cinp->modname = strdup(value);
	}
	else
	    cinp->modname = strdup(xcin_conf.default_im_mod_name);
	cinp->objname = strdup(objname);
    }
}


/*----------------------------------------------------------------------------

        Initialization

----------------------------------------------------------------------------*/

void load_syscin()
{
    FILE *fp;
    int len;
    char buf[40], truefn[256];
    char inpn_english[CIN_CNAME_LENGTH];
    char inpn_sbyte[CIN_CNAME_LENGTH];
    char inpn_2bytes[CIN_CNAME_LENGTH];
    char inpn_zhhex[CIN_CNAME_LENGTH];
    wch_t ascii[N_ASCII_KEY];
    charcode_t ccp[WCH_SIZE];

    fp = open_data("sys.tab", "rb", xcin_conf.default_dir, xcin_conf.user_dir,
		"tab", xcin_conf.locale.lc_ctype, truefn, 256, XCINMSG_ERROR);
    if (fread(buf, sizeof(char), MODULE_ID_SIZE, fp) != MODULE_ID_SIZE ||
	strcmp(buf, "syscin"))
	perr(XCINMSG_ERROR, _("invalid tab file: %s.\n"), truefn);
    len = sizeof(SYSCIN_VERSION);
    if (fread(buf, len, 1, fp) != 1 || ! check_version(SYSCIN_VERSION, buf, 5))
	perr(XCINMSG_ERROR, _("invalid sys.tab version: %s.\n"), buf);

    if (fread(inpn_english, sizeof(char), CIN_CNAME_LENGTH, fp) != CIN_CNAME_LENGTH ||
        fread(inpn_sbyte, sizeof(char), CIN_CNAME_LENGTH, fp)   != CIN_CNAME_LENGTH ||
        fread(inpn_2bytes, sizeof(char), CIN_CNAME_LENGTH, fp)  != CIN_CNAME_LENGTH ||
        fread(inpn_zhhex, sizeof(char), CIN_CNAME_LENGTH, fp)   != CIN_CNAME_LENGTH ||
        fread(ascii, sizeof(wch_t), N_ASCII_KEY, fp) != N_ASCII_KEY ||
	fread(ccp, sizeof(charcode_t), WCH_SIZE, fp) != WCH_SIZE)
	perr(XCINMSG_ERROR, _("sys.tab reading error.\n"));
    fclose(fp);

    xcin_conf.inpn_english = strdup(inpn_english);
    xcin_conf.inpn_sbyte   = strdup(inpn_sbyte);
    xcin_conf.inpn_2bytes  = strdup(inpn_2bytes);
    xcin_conf.inpn_zhhex   = strdup(inpn_zhhex);
    fullascii_init(ascii);
    ccode_init(ccp, WCH_SIZE);
}

static void
load_default_cinput(void)
{
    int i;
    cinput_t *cinp;

/*
 *  Try to load the default input method.
 */
    for (i=0, cinp=cinput; i<MAX_INP_ENTRY; i++, cinp++) {
	if (cinp->objname && 
	    ! strcmp(cinp->objname, xcin_conf.default_im_name))
	    break;
    }
    if (i >= MAX_INP_ENTRY)
	perr(XCINMSG_WARNING, _("not valid %s: %s, ignore\n"),
		RCKEY_DEFAULT_IM, xcin_conf.default_im_name);
    else {
        if (! (cinp->inpmod = 
		load_module(cinp->modname, cinp->objname, MOD_CINPUT))) {
	    perr(XCINMSG_WARNING, 
		_("error loading IM: %s, ignore.\n"), cinp->objname);
	    free(cinp->modname);
	    free(cinp->objname);
	    cinp->modname = NULL;
	    cinp->objname = NULL;
	}
	else {
	    xcin_conf.default_im = i;
	    return;
	}
    }

/*
 *  If false, try to load any specified input method.
 */
    i = 0;
    cinp = cinput;
    while (1) {
        for (; i<MAX_INP_ENTRY; i++, cinp++) {
	    if (cinp->objname)
	        break;
        }
        if (i >= MAX_INP_ENTRY)
	    perr(XCINMSG_ERROR, _("no valid input method loaded.\n"));

        if (! (cinp->inpmod = 
		load_module(cinp->modname, cinp->objname, MOD_CINPUT))) {
	    perr(XCINMSG_WARNING, 
		_("error loading IM: %s, ignore.\n"), cinp->objname);
	    free_module(cinp);
	}
	else {
	    xcin_conf.default_im_name = strdup(cinp->objname);
	    xcin_conf.default_im = i;
	    break;
	}
    }
}
    
static void
load_default_sinmd(void)
{
    int i;
    cinput_t *cinp;

/*
 *  Check the default show_cinput.
 */
    if (! strcmp(xcin_conf.default_im_sinmd_name, "DEFAULT"))
	xcin_conf.xcin_mode |= XCIN_SHOW_CINPUT;
    else {
        for (i=0, cinp=cinput; i<MAX_INP_ENTRY; i++, cinp++) {
	    if (cinp->objname && 
		! strcmp(cinp->objname, xcin_conf.default_im_sinmd_name))
		break;
	}
        if (i < MAX_INP_ENTRY) {
	    if (cinp->inpmod || (cinp->inpmod = 
		load_module(cinp->modname, cinp->objname, MOD_CINPUT))) {
		xcin_conf.default_im_sinmd = i;
	        return;
	    }
	    else
	        free_module(cinp);
	}
	perr(XCINMSG_WARNING, _("error loading IM: %s, ignore.\n"), 
		xcin_conf.default_im_sinmd_name);
	xcin_conf.xcin_mode |= XCIN_SHOW_CINPUT;
    }
}


/*----------------------------------------------------------------------------

        Main Program.

----------------------------------------------------------------------------*/

int
main(int argc, char **argv)
{
    char *s = NULL;

    set_perr("xcin");
    locale_setting(&(xcin_conf.locale.lc_ctype),
		&(xcin_conf.locale.lc_messages), XCINMSG_ERROR);
    if (XSupportsLocale() != True)
        perr(XCINMSG_ERROR, 
		_("X locale \"%s\" is not supported by your system.\n"),
		xcin_conf.locale.lc_ctype);

/*
 *  Option piroity:  command line > environment variable > user rcfile.
 */
    command_switch(argc, argv);
    read_core_config();
    read_core_config_locale();
    read_core_config_cinput();

    load_syscin();
    load_default_cinput();
    load_default_sinmd();
    qphrase_init();
    gui_init(argc, argv);
    xim_init();

#ifdef DEBUG
    {
	core_config_t *core_config = &xcin_conf;
        imodule_t *modp = imodules;
	int i;

	DebugLog("xcin_core:\n");
	DebugLog("    rcfile = %s\n", core_config->rcfile);
	DebugLog("    default_dir = %s\n", core_config->default_dir);
	DebugLog("    user_dir = %s\n", core_config->user_dir);
	DebugLog("    default_im_mod_name = %s\n", core_config->default_im_mod_name);
	DebugLog("    default_im_name = %s\n", core_config->default_im_name);
	DebugLog("    default_im = %d\n", core_config->default_im);
	if ((core_config->xcin_mode & XCIN_MODE_HIDE))
	    DebugLog("    xcin hide: yes\n");
	for (i=0; i<MAX_INP_ENTRY; i++) {
	    if (cinput[i].objname) {
	        DebugLog("    cinput: %s\n", cinput[i].objname);
	        DebugLog("\tmodname = %s\n", cinput[i].modname);
	        DebugLog("\tsetkey = %d\n", i);
	    }
	}

        do {
	    DebugLog("module: %s %s\n", modp->name, modp->version);
	    if (modp->objname)
	        DebugLog("\tobjname=%s\n", modp->objname);
	    modp = modp->next;
        } while(modp);
    }
#endif

    gui_loop();
}
