/*
    Copyright (C) 1999 by  XCIN TEAM

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    For any question or suggestion, please mail to xcin mailing-list:
    xcin@linux.org.tw, or the maintainer Tung-Han Hsieh: thhsieh@linux.org.tw
*/      

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <locale.h>
#include <unistd.h>
#include "xcintool.h"
#include "gencin.h"
#include "module.h"
#include "cin2tab.h"

static ccode_info_t ccinfo;

static cintab_head_t th;
static icode_idx_t *icode_idx;
static ichar_t *ichar;
static icode_t *icode1;
static icode_t *icode2;


/*--------------------------------------------------------------------------

	Normal cin2tab functions.

--------------------------------------------------------------------------*/

static void
cin_ename(char *arg, cintab_t *cintab)
{
    if (! arg[0])
	perr(XCINMSG_ERROR, _("%s(%d): arguement expected.\n"), 
		cintab->fname_cin, cintab->lineno);
    strncpy(th.ename, arg, CIN_ENAME_LENGTH);
    th.ename[CIN_ENAME_LENGTH - 1] = '\0';
}

static void
cin_cname(char *arg, cintab_t *cintab)
{
    if (! arg[0])
	perr(XCINMSG_ERROR, _("%s(%d): arguement expected.\n"), 
		cintab->fname_cin, cintab->lineno);
    strncpy(th.cname, arg, CIN_CNAME_LENGTH);
    th.cname[CIN_CNAME_LENGTH - 1] = '\0';
}

static void
cin_selkey(char *arg, cintab_t *cintab)
{
    if (! arg[0])
	perr(XCINMSG_ERROR, _("%s(%d): arguement expected.\n"), 
		cintab->fname_cin, cintab->lineno);
    th.n_selkey = strlen(arg);
    if (th.n_selkey > SELECT_KEY_LENGTH)
	perr(XCINMSG_ERROR, _("%s(%d): too many selection keys defined.\n"),
	     cintab->fname_cin, cintab->lineno);
    strcpy(th.selkey, arg);
}

static void
cin_endkey(char *arg, cintab_t *cintab)
{
    if (! arg[0])
	perr(XCINMSG_ERROR, _("%s(%d): arguement expected.\n"), 
		cintab->fname_cin, cintab->lineno);
    th.n_endkey = strlen(arg);
    if (th.n_endkey > END_KEY_LENGTH)
	perr(XCINMSG_ERROR, _("%s(%d): too many end keys defined.\n"),
	     cintab->fname_cin, cintab->lineno);
    strcpy(th.endkey, arg);
}

static void
cin_keyname(char *arg, cintab_t *cintab)
{
    char cmd1[64], arg1[64];
    int k;

    if (! arg[0] || strcmp(arg, "begin") != 0)
	perr(XCINMSG_ERROR, _("%s(%d): arguement \"begin\" expected.\n"),
	     cintab->fname_cin, cintab->lineno);

    while (cmd_arg(cmd1, 64, arg1, 64, NULL)) {
        if (! arg1[0])
	    perr(XCINMSG_ERROR, _("%s(%d): arguement expected.\n"), 
		cintab->fname_cin, cintab->lineno);
	if (! strcmp("%keyname", cmd1) && ! strcmp("end", arg1))
	    break;

	if (! (k = key2code(cmd1[0])))
	    perr(XCINMSG_ERROR, _("%s(%d): illegal key \"%c\" specified.\n"),
		 cintab->fname_cin, cintab->lineno, cmd1[0]);
	if (th.keyname[k].wch)
	    perr(XCINMSG_ERROR, _("%s(%d): key \"%c\" is already in used.\n"),
		 cintab->fname_cin, cintab->lineno, cmd1[0]);
	strncpy(th.keyname[k].s, arg1, WCH_SIZE);
	th.n_keyname++;
    }
}

/*------------------------------------------------------------------------*/

typedef struct {
    int char_idx;
    unsigned int key[2];
    char mark;
} cin_char_t;

static int
cchar_cmp(const void *a, const void *b)
{
    cin_char_t *aa=(cin_char_t *)a, *bb=(cin_char_t *)b;

    if (aa->char_idx == bb->char_idx) {
	if (aa->mark)
	    return -1;
	else if (bb->mark)
	    return 1;
	else
	    return 0;
    }
    else if (aa->char_idx > bb->char_idx)
	return 1;
    else
	return -1;
}

static int
icode_cmp(const void *a, const void *b)
{
    cin_char_t *aa=(cin_char_t *)a, *bb=(cin_char_t *)b;

    if (aa->key[0] == bb->key[0]) {
	if (aa->key[1] == bb->key[1])
	    return 0;
	else if (aa->key[1] > bb->key[1])
	    return 1;
	else
	    return -1;
    }
    else if (aa->key[0] > bb->key[0])
	return 1;
    else
	return -1;
}

static void
cin_chardef(char *arg, cintab_t *cintab)
{
    char cmd1[64], arg1[64], arg2[2], *idx;
    wch_t ch;
    cin_char_t *cchar, *cch;
    int len, ret, cch_size;
    unsigned int i, j;

    if (! arg[0] || strcmp(arg, "begin") != 0)
	perr(XCINMSG_ERROR, _("%s(%d): arguement \"begin\" expected.\n"),
	     cintab->fname_cin, cintab->lineno);

    cch_size = ccinfo.total_char;
    cch = cchar = malloc(cch_size * sizeof(cin_char_t));
    idx = calloc(cch_size, sizeof(char));

    while ((ret=cmd_arg(cmd1, 64, arg1, 64, arg2, 2, NULL))) {
        if (! arg1[0])
	    perr(XCINMSG_ERROR, _("%s(%d): arguement expected.\n"), 
		cintab->fname_cin, cintab->lineno);
	if (th.n_icode >= cch_size) {
	    cch_size += ccinfo.total_char;
	    cchar = realloc(cchar, cch_size * sizeof(cin_char_t));
	    cch = cchar + th.n_icode;
	}
	if (! strcmp("%chardef", cmd1) && ! strcmp("end", arg1))
	    break;

	/*
	 *  Fill in the cin_char_t *cch unit.
	 */
	bzero(&ch, sizeof(wch_t));
	strncpy(ch.s, arg1, WCH_SIZE);
	if ((i = ccode_to_idx(&ch)) == -1)
	    perr(XCINMSG_ERROR, _("%s(%d): not recognized char: %s\n"), 
		cintab->fname_cin, cintab->lineno, ch.s);
	cch->char_idx = i;
	keys2codes(cch->key, 2, cmd1);
	cch->mark = (ret==3 && arg2[0]=='*') ? 1 : 0;

	if (! idx[i])
	    th.n_ichar ++;
	else
	    idx[i] = 1;
	th.n_icode ++;
	cch ++;

	len = strlen(cmd1);
	if (th.n_max_keystroke < len)
	    th.n_max_keystroke = len;
    }

    /*
     *  Determine the memory model.
     */
    ret = (th.n_max_keystroke <= 5) ? ICODE_MODE1 : ICODE_MODE2;
    th.icode_mode = ret;

    /*
     *  Fill in the ichar, icode_idx and icode1, icode2 arrays.
     */
    stable_sort(cchar, th.n_icode, sizeof(cin_char_t), icode_cmp);

    ichar = calloc(cch_size, sizeof(ichar_t));
    icode_idx = malloc(sizeof(icode_idx_t) * th.n_icode);
    icode1 = malloc(th.n_icode * sizeof(icode_t));
    if (ret == ICODE_MODE2)
        icode2 = malloc(th.n_icode * sizeof(icode_t));
    bzero(idx, ccinfo.total_char);

    for (i=0, cch=cchar; i<th.n_icode; i++, cch++) {
	icode_idx[i] = j = (icode_idx_t)(cchar[i].char_idx);
	icode1[i] = (icode_t)(cch->key[0]);
        if (ret == ICODE_MODE2)
	    icode2[i] = (icode_t)(cch->key[1]);

	if (! idx[j] || cch->mark) {
	    ichar[j] = (ichar_t)i;
	    idx[j] = 1;
	}
    }
    free(cchar);
    free(idx);
}


/*----------------------------------------------------------------------------

	Main Functions.

----------------------------------------------------------------------------*/

struct genfunc_s {
    char *name;
    void (*func) (char *, cintab_t *);
    char gotit;
};

static struct genfunc_s genfunc[] = {
    {"%ename", cin_ename, 0},
    {"%cname", cin_cname, 0},
    {"%keyname", cin_keyname, 0},
    {"%selkey", cin_selkey, 0},
    {"%endkey", cin_endkey, 1},
    {"%chardef", cin_chardef, 0},
    {NULL, NULL, 0}
};

void
gencin(cintab_t *cintab)
{
    int i;
    char cmd[64], arg[64];

    strncpy(th.version, GENCIN_VERSION, 20);
    load_systab(cintab->sysfn);
    ccode_info(&ccinfo);

    while (cmd_arg(cmd, 64, arg, 64, NULL)) {
	for (i=0; genfunc[i].name != NULL; i++) {
	    if (! strcmp(genfunc[i].name, cmd)) {
		genfunc[i].func(arg, cintab);
		genfunc[i].gotit = 1;
		break;
	    }
	}
	if (genfunc[i].name == NULL)
	    perr(XCINMSG_ERROR, _("%s(%d): unknown command: %s.\n"),
		 cintab->fname_cin, cintab->lineno, cmd);
    }

    for (i=0; genfunc[i].name != NULL; i++) {
	if (genfunc[i].gotit == 0)
	    perr(XCINMSG_ERROR, _("%s: command \"%s\" not specified.\n"),
		 cintab->fname_cin, genfunc[i].name);
    }
    perr(XCINMSG_NORMAL,
	_("number of keyname: %d\n"), th.n_keyname);
    perr(XCINMSG_NORMAL, 
	_("max length of keystroke: %d\n"), th.n_max_keystroke);
    perr(XCINMSG_NORMAL, 
	_("total char number of this encoding: %d\n"), ccinfo.total_char);
    perr(XCINMSG_NORMAL, 
	_("number of char defined: %d\n"), th.n_ichar);
    perr(XCINMSG_NORMAL, 
	_("number of keystroke code defined: %d\n"), th.n_icode);
    perr(XCINMSG_NORMAL, 
	_("memory model: %d\n"), th.icode_mode);

    fwrite(&th, sizeof(cintab_head_t), 1, cintab->fw);
    fwrite(icode_idx, sizeof(icode_idx_t), th.n_icode, cintab->fw);
    fwrite(ichar, sizeof(ichar_t), ccinfo.total_char, cintab->fw);
    fwrite(icode1, sizeof(icode_t), th.n_icode, cintab->fw);
    if (th.icode_mode == ICODE_MODE2)
	fwrite(icode2, sizeof(icode_t), th.n_icode, cintab->fw);

#ifdef DEBUG
{
    char cch[5], keycode[11];
    unsigned int codelist[2], ic_idx, cch_idx;

    for (i=0; i<th.n_icode; i++) {
	ic_idx = icode_idx[i];
	ccode_to_char(ic_idx, cch, 5);

	codelist[0] = icode1[i];
	if (! icode2)
	    codes2keys(codelist, 1, keycode, 11);
	else {
	    codelist[1] = icode2[i];
	    codes2keys(codelist, 2, keycode, 11);
	}

	DebugLog("keycode: %5s  ==>  icode_idx: %5d  ==>  cch: %s\n", 
		keycode, ic_idx, cch);
    }
    DebugLog("===================================================\n");
    for (i=0; i<ccinfo.total_char; i++) {
	if (! ichar[i])
	    continue;

	ccode_to_char(i, cch, 5);
	cch_idx = ichar[i];
	codelist[0] = icode1[cch_idx];
	if (! icode2)
	    codes2keys(codelist, 1, keycode, 11);
	else {
	    codelist[1] = icode2[cch_idx];
	    codes2keys(codelist, 2, keycode, 11);
	}
	DebugLog("cch: %s  ==>  icode_idx: %5d  ==>  keycode: %5s\n",
		cch, cch_idx, keycode);
    }
}
#endif
}

