/*
 * Copyright 1999, TaBE Project, All Rights Reserved.
 * Copyright 1999, Pai-Hsiang Hsiao, All Rights Reserved.
 *
 * $Id: tabe_tsidbint.c,v 1.15 1999/05/31 10:20:21 shawn Exp $
 *
 */
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <netinet/in.h>
#include <db.h>

#include "tabe.h"

static void tabeTsiDBClose(struct TsiDB *tsidb);
static int  tabeTsiDBRecordNumber(struct TsiDB *tsidb);
static int  tabeTsiDBStoreTsi(struct TsiDB *tsidb, struct TsiInfo *tsi);
static int  tabeTsiDBLookupTsi(struct TsiDB *tsidb, struct TsiInfo *tsi);
static int  tabeTsiDBCursorSet(struct TsiDB *tsidb, struct TsiInfo *tsi);
static int  tabeTsiDBCursorNext(struct TsiDB *tsidb, struct TsiInfo *tsi);
static int  tabeTsiDBCursorPrev(struct TsiDB *tsidb, struct TsiInfo *tsi);

struct TsiDBDataDB {
  unsigned long int  refcount;
  unsigned long int  yinnum;
/*
  unsigned char      reserved[32];
*/
};

static int  TsiDBStoreTsiDB(struct TsiDB *tsidb, struct TsiInfo *tsi);
static int  TsiDBLookupTsiDB(struct TsiDB *tsidb, struct TsiInfo *tsi);
static void TsiDBPackDataDB(struct TsiInfo *tsi, DBT *dat);
static void TsiDBUnpackDataDB(struct TsiInfo *tsi, DBT *dat);

/*
 * open a TsiDB with the given type and name
 *
 * return pointer to TsiDB if success, NULL if failed
 *
 */
struct TsiDB *
tabeTsiDBOpen(type, db_name, flags)
int type;
const char *db_name;
int flags;
{
  struct TsiDB *tsidb;
  DB *dbp;

  switch(type) {
  case DB_TYPE_DB:
    tsidb = (struct TsiDB *)malloc(sizeof(struct TsiDB));
    if (!tsidb) {
      perror("tabeTsiDBOpen()");
      return(NULL);
    }
    memset(tsidb, 0, sizeof(struct TsiDB));
    tsidb->type = type;
    tsidb->flags = flags;

    tsidb->Close = tabeTsiDBClose;
    tsidb->RecordNumber = tabeTsiDBRecordNumber;
    tsidb->Put = tabeTsiDBStoreTsi;
    tsidb->Get = tabeTsiDBLookupTsi;
    tsidb->CursorSet = tabeTsiDBCursorSet;
    tsidb->CursorNext = tabeTsiDBCursorNext;
    tsidb->CursorPrev = tabeTsiDBCursorPrev;

    if (tsidb->flags & DB_FLAG_CREATEDB) {
      if (tsidb->flags & DB_FLAG_READONLY) {
        return(NULL);
      }
      else {
	errno = db_open(db_name, DB_BTREE, DB_CREATE,
			0644, NULL, NULL, &dbp);
      }
    }
    else {
      if (tsidb->flags & DB_FLAG_READONLY) {
	errno = db_open(db_name, DB_BTREE, DB_RDONLY, 0444, NULL, NULL, &dbp);
      }
      else {
	errno = db_open(db_name, DB_BTREE, 0        , 0644, NULL, NULL, &dbp);
      }
    }
    if (errno > 0) {
      fprintf(stderr, "tabeTsiDBOpen(): Can not open DB file %s(%s).\n",
	     db_name, strerror(errno));
      free(tsidb);
      return(NULL);
    }
    if (errno < 0) {
      /* DB specific errno */
      fprintf(stderr, "tabeTsiDBOpen(): DB error opening DB File %s.\n",
	      db_name);
      free(tsidb);
      return(NULL);
    }
    tsidb->db_name = strdup(db_name);
    tsidb->dbp = (void *)dbp;
    return(tsidb);
  default:
    fprintf(stderr, "tabeTsiDBOpen(): Unknown DB type.\n");
    break;
  }

  return(NULL);
}

/*
 * close and flush DB file
 */
static void
tabeTsiDBClose(tsidb)
struct TsiDB *tsidb;
{
  DB  *dbp;
  DBC *dbcp;

  switch(tsidb->type) {
  case DB_TYPE_DB:
    dbp = (DB *)tsidb->dbp;
    dbcp = (DBC *)tsidb->dbcp;
    /* close it and reset the DB pointer */
    if (dbcp) {
      dbcp->c_close(dbcp);
      dbcp = (void *)NULL;
    }
    if (dbp) {
      dbp->close(dbp, 0);
      dbp = (void *)NULL;
    }
    return;
  default:
    fprintf(stderr, "tabeTsiDBClose(): Unknown DB type.\n");
    break;
  }
  return;
}

/*
 * returns the number of record in Tsi DB
 */
static int
tabeTsiDBRecordNumber(tsidb)
struct TsiDB *tsidb;
{
  DB *dbp;
  DB_BTREE_STAT *sp;

  switch(tsidb->type) {
  case DB_TYPE_DB:
    dbp = (DB *)tsidb->dbp;
    errno = dbp->stat(dbp, &sp, NULL, 0);
    if (!errno) {
      return(sp->bt_nrecs);
    }
    break;
  default:
    fprintf(stderr, "tabeTsiDBRecordNumber(): Unknown DB type.\n");
    break;
  }
  return(0);
}

/*
 * store Tsi in designated DB
 *
 * return 0 if success, -1 if failed
 *
 */
static int
tabeTsiDBStoreTsi(tsidb, tsi)
struct TsiDB *tsidb;
struct TsiInfo *tsi;
{
  int rval;

  if (tsidb->flags & DB_FLAG_READONLY) {
    fprintf(stderr, "tabeTsiDBStoreTsi(): writing a read-only DB.\n");
    return(-1);
  }

  switch(tsidb->type) {
  case DB_TYPE_DB:
    rval = TsiDBStoreTsiDB(tsidb, tsi);
    return(rval);
  default:
    fprintf(stderr, "tabeTsiDBStoreTsi(): Unknown DB type.\n");
    break;
  }

  return(-1);
}

/*
 * lookup Tsi in designated DB
 *
 * return 0 if success, -1 if failed
 *
 */
static int
tabeTsiDBLookupTsi(tsidb, tsi)
struct TsiDB *tsidb;
struct TsiInfo *tsi;
{
  int rval;

  switch(tsidb->type) {
  case DB_TYPE_DB:
    rval = TsiDBLookupTsiDB(tsidb, tsi);
    return(rval);
  default:
    fprintf(stderr, "tabeTsiDBLookupTsi(): Unknown DB type.\n");
    break;
  }

  return(-1);
}

static void
TsiDBPackDataDB(tsi, dat)
struct TsiInfo *tsi;
DBT *dat;
{
  struct TsiDBDataDB *d;
  int datalen, i, yinlen;
  unsigned char *data;

  yinlen = tsi->yinnum * strlen(tsi->tsi)/2;
  datalen = sizeof(struct TsiDBDataDB) + sizeof(Yin)*yinlen;
  data = (unsigned char *)malloc(sizeof(unsigned char)*datalen);
  memset(data, 0, sizeof(unsigned char)*datalen);
  d = (struct TsiDBDataDB *)data;

  /* convert to network byte order */
  d->refcount = htonl(tsi->refcount);
  d->yinnum   = htonl(tsi->yinnum);
  for (i = 0; i < yinlen; i++) {
    ((Yin *)(data+sizeof(struct TsiDBDataDB)))[i] = htons(tsi->yindata[i]);
  }

  dat->data = data;
  dat->size = datalen;
}

static void
TsiDBUnpackDataDB(tsi, dat)
struct TsiInfo *tsi;
DBT *dat;
{
  int i, yinlen;
  struct TsiDBDataDB d;

  memset(&d, 0, sizeof(struct TsiDBDataDB));
  memcpy(&d, dat->data, sizeof(struct TsiDBDataDB));
  /* convert to system byte order */
  tsi->refcount = ntohl(d.refcount);
  tsi->yinnum   = ntohl(d.yinnum);
  yinlen        = tsi->yinnum*(strlen(tsi->tsi)/2);

  if (tsi->yindata) {
    free(tsi->yindata);
    tsi->yindata = (Yin *)NULL;
  }

  if (yinlen) {
    tsi->yindata = (Yin *)malloc(sizeof(Yin)*yinlen);
    memcpy(tsi->yindata, dat->data+sizeof(struct TsiDBDataDB),
	   sizeof(Yin)*yinlen);
  }

  /* convert to system byte order */
  for (i = 0; i < yinlen; i++) {
    tsi->yindata[i] = ntohs(tsi->yindata[i]);
  }
}

static int
TsiDBStoreTsiDB(tsidb, tsi)
struct TsiDB *tsidb;
struct TsiInfo *tsi;
{
  DBT key, dat;
  DB *dbp;

  memset(&key, 0, sizeof(key));
  memset(&dat, 0, sizeof(dat));

  key.data = tsi->tsi;
  key.size = strlen(tsi->tsi);

  TsiDBPackDataDB(tsi, &dat);

  dbp = tsidb->dbp;
  if (tsidb->flags & DB_FLAG_OVERWRITE) {
    errno = dbp->put(dbp, NULL, &key, &dat, 0);
  }
  else {
    errno = dbp->put(dbp, NULL, &key, &dat, DB_NOOVERWRITE);
  }
  if (errno > 0) {
    fprintf(stderr, "TsiDBStoreTsiDB(): can not store DB. (%s)\n",
	    strerror(errno));
    return(-1);
  }
  if (errno < 0) {
    switch(errno) {
    case DB_KEYEXIST:
#ifdef MYDEBUG
      fprintf(stderr, "TsiDBStoreTsiDB(): tsi exist.\n");
#endif
      return(-1);
    default:
      fprintf(stderr, "TsiDBStoreTsiDB(): unknown DB error.\n");
      return(-1);
    }
  }

  free(dat.data);
  return(0);
}

static int
TsiDBLookupTsiDB(tsidb, tsi)
struct TsiDB *tsidb;
struct TsiInfo *tsi;
{
  DBT key, dat;
  DB *dbp;

  memset(&key, 0, sizeof(key));
  memset(&dat, 0, sizeof(dat));

  key.data = tsi->tsi;
  key.size = strlen(tsi->tsi);

  dbp = tsidb->dbp;
  errno = dbp->get(dbp, NULL, &key, &dat, 0);
  if (errno > 0) {
    fprintf(stderr, "TsiDBLookupTsiDB(): can not lookup DB. (%s)\n",
	    strerror(errno));
    return(-1);
  }
  if (errno < 0) {
    switch(errno) {
    case DB_NOTFOUND:
#ifdef MYDEBUG
      fprintf(stderr, "TsiDBLookupTsiDB(): tsi does not exist.\n");
#endif
      return(-1);
    default:
      fprintf(stderr, "TsiDBLookupTsiDB(): unknown DB error.\n");
      return(-1);
    }
  }

  TsiDBUnpackDataDB(tsi, &dat);

  return(0);
}

static int
tabeTsiDBCursorSet(tsidb, tsi)
struct TsiDB *tsidb;
struct TsiInfo *tsi;
{
  DB  *dbp;
  DBC *dbcp;
  DBT  key, dat;

  dbp  = tsidb->dbp;
  dbcp = tsidb->dbcp;
  if (dbcp) {
    dbcp->c_close(dbcp);
  }

#if DB_VERSION_MINOR > 6 || (DB_VERSION_MINOR == 6 && DB_VERSION_PATCH > 4)
  dbp->cursor(dbp, NULL, &dbcp, 0);
#else
  dbp->cursor(dbp, NULL, &dbcp);
#endif
  tsidb->dbcp = dbcp;

  memset(&key, 0, sizeof(key));
  memset(&dat, 0, sizeof(dat));

  if (strlen(tsi->tsi)) {
    key.data = tsi->tsi;
    key.size = strlen(tsi->tsi);
    errno = dbcp->c_get(dbcp, &key, &dat, DB_SET);
  }
  else {
    errno = dbcp->c_get(dbcp, &key, &dat, DB_FIRST);
  }
  if (errno > 0) {
    fprintf(stderr, "tabeTsiDBCursorSet(): error setting cursor. (%s)\n",
	    strerror(errno));
    return(-1);
  }
  if (errno < 0) {
    switch(errno) {
    default:
      fprintf(stderr, "tabeTsiDBCursorSet(): Unknown error.\n");
      return(-1);
    }
  }

  /* we depends on the caller to allocate enough large buffer */
  strcpy(tsi->tsi, key.data);
  tsi->tsi[key.size] = (unsigned char)NULL;
  TsiDBUnpackDataDB(tsi, &dat);  

  return(0);
}

static int
tabeTsiDBCursorNext(tsidb, tsi)
struct TsiDB *tsidb;
struct TsiInfo *tsi;
{
  DB  *dbp;
  DBC *dbcp;
  DBT  key, dat;

  dbp  = tsidb->dbp;
  dbcp = tsidb->dbcp;
  if (!dbcp) {
    return(-1);
  }

  memset(&key, 0, sizeof(key));
  memset(&dat, 0, sizeof(dat));

  errno = dbcp->c_get(dbcp, &key, &dat, DB_NEXT);
  if (errno < 0) {
    switch(errno) {
    case DB_NOTFOUND:
      return(-1);
    default:
      return(-1);
    }
  }

  /* we depends on the caller to allocate buffer large enough */
  strcpy(tsi->tsi, key.data);
  tsi->tsi[key.size] = (unsigned char)NULL;

  TsiDBUnpackDataDB(tsi, &dat);

  return(0);
}

static int
tabeTsiDBCursorPrev(tsidb, tsi)
struct TsiDB *tsidb;
struct TsiInfo *tsi;
{
  DB  *dbp;
  DBC *dbcp;
  DBT  key, dat;

  dbp  = tsidb->dbp;
  dbcp = tsidb->dbcp;
  if (!dbcp) {
    return(-1);
  }

  memset(&key, 0, sizeof(key));
  memset(&dat, 0, sizeof(dat));

  errno = dbcp->c_get(dbcp, &key, &dat, DB_PREV);
  if (errno < 0) {
    switch(errno) {
    case DB_NOTFOUND:
      return(-1);
    default:
      return(-1);
    }
  }

  /* we depends on the caller to allocate enough large buffer */
  strcpy(tsi->tsi, key.data);
  tsi->tsi[key.size] = (unsigned char)NULL;

  TsiDBUnpackDataDB(tsi, &dat);

  return(0);
}
