/*
 *  $Id: gui.c,v 1.6 1999/02/18 08:08:06 thhsieh Exp $
 */
/*
    Copyright (C) 1999 by  XCIN TEAM

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    For any question or suggestion, please mail to xcin mailing-list:
    xcin@linux.org.tw, or the maintainer Tung-Han Hsieh: thhsieh@linux.org.tw
*/

#include <stdlib.h>
#include <string.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xos.h>
#include "xcintool.h"
#include "xcin_core.h"
#include "gui.h"
#include "xim.h"

#define FIELD_STEP  10

gui_config_t gui_conf;


/*----------------------------------------------------------------------------

	GUI tool functions.

----------------------------------------------------------------------------*/

static char *ims_english;
static char *ims_sbyte;
static char *ims_2bytes;
static char s_inpname[CIN_CNAME_LENGTH*2];
static char e_inpname[CIN_ENAME_LENGTH];

static void
inpstate_content(inp_state_t inp_state)
{
    char *inpn, *inpb, *str;

    inpn = (! (inp_state & IC_CINPUT)) ?
        	ims_english : xim_conf.ic->inpinfo.inp_cname;
    inpb = (! (inp_state & IC_2BYTES)) ? ims_sbyte : ims_2bytes;
    snprintf(s_inpname, CIN_CNAME_LENGTH*2, "[%s][%s]", inpn, inpb);
    gui_conf.inpname.x = FIELD_STEP;
    gui_conf.inpname.width = XmbTextEscapement(
		gui_conf.fontset, s_inpname, strlen(s_inpname));

    if (! xim_conf.ic)
	return;

    gui_conf.coding.width = xim_conf.ic->inpinfo.area3_len * gui_conf.ef_width;
    gui_conf.coding.x = 2*FIELD_STEP + 
		gui_conf.inpname.x + gui_conf.inpname.width;
    gui_conf.show_coding.x = 2*FIELD_STEP + 
		gui_conf.coding.x + gui_conf.coding.width;

    str = xim_conf.ic->inpinfo.inp_ename;
    if ((xim_conf.ic->inp_state & IC_CINPUT)) {
	if (! (xcin_conf.xcin_mode & XCIN_IM_FOCUS))
	    strncpy(e_inpname, str, CIN_ENAME_LENGTH);
	else
	    snprintf(e_inpname, CIN_ENAME_LENGTH, 
			"*%s", xim_conf.ic->inpinfo.inp_ename);
        gui_conf.e_inpname.width = 
		XTextWidth(gui_conf.indexfont, str, strlen(str));
    }
}

static void
change_win_attrib(XConfigureEvent *event)
{
    if (event->x >= 0 && event->x <= gui_conf.display_width &&
	event->y >= 0 && event->y <= gui_conf.display_height) {
        gui_conf.pos_x = event->x;
        gui_conf.pos_y = event->y;
    }
    else {
	XMoveWindow(gui_conf.display, gui_conf.window, 
			gui_conf.pos_x, gui_conf.pos_y);
    }
    gui_conf.width = event->width;
    gui_conf.c_width = gui_conf.width / gui_conf.ef_width;
}

void 
win_map_change(int state)
{
    if (state) {
        XMapWindow(gui_conf.display, gui_conf.window);
	xcin_conf.xcin_mode |= XCIN_MODE_WORKING;
    }
    else {
        XUnmapWindow(gui_conf.display, gui_conf.window);
	xcin_conf.xcin_mode &= ~(XCIN_MODE_WORKING);
    }
}

void
win_draw_multich(inpinfo_t *inpinfo)
{
    int i, x, y, len;
    wch_t *selkey, *cch;
    char *pgstate;
    GC spot_gc;

    x = FIELD_STEP;
    y = gui_conf.ef_ascent;
    selkey = inpinfo->s_selkey;
    cch = inpinfo->mcch;
    spot_gc = ((inpinfo->guimode & GUIMOD_SELKEYSPOT)) ? 
		gui_conf.gcm : gui_conf.gc;

    if (inpinfo->n_mcch == 1) {
	len = strlen(cch->s);
        XmbDrawImageString(gui_conf.display, gui_conf.window,
                gui_conf.fontset, gui_conf.gc, x, y, cch->s, len);
    }
    else {
        for (i=0; i<inpinfo->n_mcch; i++, selkey++, cch++) {
	    if ((len = strlen(selkey->s))) {
                XmbDrawImageString(gui_conf.display, gui_conf.window,
                    gui_conf.fontset, spot_gc, x, y, selkey->s, len);
	        x += (XmbTextEscapement(gui_conf.fontset, selkey->s, len) + 5);
	    }
	    len = strlen(cch->s);
            XmbDrawImageString(gui_conf.display, gui_conf.window,
                gui_conf.fontset, gui_conf.gc, x, y, cch->s, len);
	    x += (XmbTextEscapement(gui_conf.fontset, cch->s, len) + FIELD_STEP);
        }
        switch (inpinfo->mcch_pgstate) {
        case MCCH_BEGIN:
	    pgstate = ">";
	    len = 1;
	    break;
        case MCCH_MIDDLE:
	    pgstate = "</>";
	    len = 3;
	    break;
        case MCCH_END:
	    pgstate = "<";
	    len = 1;
	    break;
        default:
	    pgstate = NULL;
	    break;
        }
        if (pgstate)
            XmbDrawImageString(gui_conf.display, gui_conf.window,
                gui_conf.fontset, gui_conf.gc, x, y, pgstate, len);
    }
}

void
win_draw_listcch(inpinfo_t *inpinfo)
{
    int x, y, edit_pos, len;
    wch_t tmp;
    static int str_size;
    static char *str=NULL;

    x = FIELD_STEP;
    y = gui_conf.ef_ascent;
    len = WCH_SIZE * inpinfo->n_lcch;

    if (len >= str_size) {
	str_size = len+1;
	if (str)
	    free(str);
	str = calloc(str_size, sizeof(char));
    }

    edit_pos = inpinfo->edit_pos;
    if (edit_pos < inpinfo->n_lcch) {
	tmp.wch = inpinfo->lcch[edit_pos].wch;
	inpinfo->lcch[edit_pos].wch = (wchar_t)0;

	if (edit_pos > 0) {
            wchs_to_mbs(str, inpinfo->lcch, str_size);
	    len = strlen(str);
            XmbDrawImageString(gui_conf.display, gui_conf.window,
                gui_conf.fontset, gui_conf.gc, x, y, str, len);
	    x += XmbTextEscapement(gui_conf.fontset, str, len);
	}

	len = strlen(tmp.s);
        XmbDrawImageString(gui_conf.display, gui_conf.window,
            gui_conf.fontset, gui_conf.gcm, x, y, tmp.s, len);
	x += XmbTextEscapement(gui_conf.fontset, str, len);

	if (edit_pos < inpinfo->n_lcch - 1) {
            wchs_to_mbs(str, inpinfo->lcch+edit_pos+1, str_size);
            XmbDrawImageString(gui_conf.display, gui_conf.window,
                gui_conf.fontset, gui_conf.gc, x, y, str, strlen(str));
	}
	inpinfo->lcch[edit_pos].wch = tmp.wch;
    }
    else {
        wchs_to_mbs(str, inpinfo->lcch, str_size);
	len = strlen(str);
	if (len) {
            XmbDrawImageString(gui_conf.display, gui_conf.window,
                gui_conf.fontset, gui_conf.gc, x, y, str, len);
	    x += XmbTextEscapement(gui_conf.fontset, str, len);
	}
	else
	    x = FIELD_STEP;
        XFillRectangle(gui_conf.display, gui_conf.window, gui_conf.gcrm, 
	    x, 0, gui_conf.ef_width, gui_conf.ef_height);
    }
}

void 
win_draw(inp_state_t inp_state)
{
    int x, y;
    char *str, buf[256];
    IC *ic = xim_conf.ic;

    if (gui_conf.winchange) {
	XClearWindow(gui_conf.display, gui_conf.window);
        if ((gui_conf.winchange & WIN_CHANGE_IM) == WIN_CHANGE_IM)
	    inpstate_content(inp_state);
    }

    /* Draw area 1. */
    if (ic && (ic->inp_state & IC_CINPUT)) {
	if ((ic->inpinfo.guimode & GUIMOD_LISTCHAR) && ic->inpinfo.n_mcch <= 1)
	    win_draw_listcch(&(ic->inpinfo));
	else if (ic->inpinfo.n_mcch > 0)
	    win_draw_multich(&(ic->inpinfo));
    }

    /* Draw area 2. */
    x = gui_conf.inpname.x;
    y = gui_conf.ef_height + gui_conf.ef_ascent;
    str = s_inpname;
    XmbDrawImageString(gui_conf.display, gui_conf.window,
            gui_conf.fontset, gui_conf.gc, x, y, str, strlen(str));

    if (ic && (ic->inp_state & IC_CINPUT)) {
	/* Draw area 3. */
        x = gui_conf.coding.x;
        XFillRectangle(gui_conf.display, gui_conf.window, gui_conf.gcrm, x, 
	    y-gui_conf.ef_ascent, gui_conf.coding.width, gui_conf.ef_height);
        if (wchs_to_mbs(buf, ic->inpinfo.s_keystroke, 256)) 
            XmbDrawImageString(gui_conf.display, gui_conf.window, 
		gui_conf.fontset, gui_conf.gcm, x, y, buf, strlen(buf));

	/* Draw area 4. */
	if (ic->inpinfo.cch_publish.wch) {
	    int slen;

            slen = snprintf(buf, 256, "%s:", ic->inpinfo.cch_publish.s);
	    str = buf + slen;
            if (ic->sinmd_keystroke[0].wch &&
	        wchs_to_mbs(str, ic->sinmd_keystroke, 256-slen)) {
	        if ((ic->inpinfo.guimode & GUIMOD_SINMDLINE1)) {
		    x = FIELD_STEP;
		    y = gui_conf.ef_ascent;
	        }
	        else
                    x = gui_conf.show_coding.x;
                XmbDrawImageString(gui_conf.display, gui_conf.window, 
		    gui_conf.fontset, gui_conf.gc, x, y, buf, strlen(buf));
            }
	}

	/* Draw area 5. */
        str = e_inpname;
        if (str[0]) {
            x = gui_conf.width - FIELD_STEP - gui_conf.e_inpname.width;
            y = gui_conf.height - gui_conf.if_descent;
            XDrawString(gui_conf.display, gui_conf.window, 
	            gui_conf.gce, x, y, str, strlen(str));
        }
    }
}

void
win_draw_map(void)
{
    inp_state_t inp_state;

    inp_state = (xim_conf.ic) ? xim_conf.ic->inp_state : 0;
    win_draw(inp_state);
    if ((xcin_conf.xcin_mode & XCIN_MODE_HIDE)) {
	if (! (inp_state & IC_CINPUT) && ! (inp_state & IC_2BYTES) &&
	      (xcin_conf.xcin_mode & XCIN_MODE_WORKING))
	    win_map_change(0);
	else if (((inp_state & IC_CINPUT) || (inp_state & IC_2BYTES))
	      && ! (xcin_conf.xcin_mode & XCIN_MODE_WORKING))
	    win_map_change(1);
    }
    if ((gui_conf.winchange & WIN_RAISE))
	XRaiseWindow(gui_conf.display, gui_conf.window);
    gui_conf.winchange = 0;
}


/*----------------------------------------------------------------------------

	GUI setting & initialization.

----------------------------------------------------------------------------*/

static void 
x_set_display(char *display_name)
{
    if (! (gui_conf.display = XOpenDisplay(display_name)))
	perr(XCINMSG_ERROR, _("cannot open display: %s\n"), display_name);
    gui_conf.screen = DefaultScreen(gui_conf.display);
    gui_conf.colormap = DefaultColormap(gui_conf.display, gui_conf.screen);
    gui_conf.display_width = DisplayWidth(gui_conf.display, gui_conf.screen);
    gui_conf.display_height = DisplayHeight(gui_conf.display, gui_conf.screen);
}

static void
x_set_font(char *fontset_name)
{
    int  fsize, efsize;
    char *s;
    int  charset_count, fontset_count=0;
    char **charset_list;
    char *def_string;
    XFontStruct **font_structs;
    int i;

/*
 *  compute the number of input font names.
 */
    if (*fontset_name != '\0')
	fontset_count ++;
    while ((s=strchr(fontset_name, ' ')) != NULL) {
	if (*(s+1) != '\0') {
	    *s = ',';
	    fontset_count ++;
	}
    }

/*
 *  see gdkfont.c for more details.
 */
    gui_conf.fontset = XCreateFontSet(gui_conf.display, fontset_name,
            	&charset_list, &charset_count, &def_string);
    if (charset_count || ! gui_conf.fontset) {
        for (i=0; i<charset_count; i++)
            perr(XCINMSG_NORMAL, _("invalid font %s.\n"), charset_list[i]);
	perr(XCINMSG_ERROR, _("fontset setting error.\n"));
    }
    charset_count = XFontsOfFontSet(gui_conf.fontset, 
				&font_structs, &charset_list);
    if (charset_count != fontset_count) {
        for (i=0; i<charset_count; i++)
	    perr(XCINMSG_NORMAL, _("only valid font %s.\n"), charset_list[i]);
	perr(XCINMSG_ERROR, _("fontset setting error.\n"));
    }

    for (i=0; i<charset_count; i++) {
	fsize = font_structs[i]->max_bounds.width / 2;
	if (fsize > gui_conf.ef_width)
            gui_conf.ef_width = fsize;
        fsize = font_structs[i]->ascent + font_structs[i]->descent;
	if (fsize > gui_conf.ef_height) {
	    gui_conf.ef_height = fsize;
	    gui_conf.ef_ascent = font_structs[i]->ascent;
	}
    }

/*
 *  Create the index font.
 */
    for (i=0; i<charset_count; i++) {
	if (! strcmp(xcin_conf.indexfont, charset_list[i]))
	    break;
    }
    if (i < charset_count)
	gui_conf.indexfont = font_structs[i];
    else if(! (gui_conf.indexfont = 
	XLoadQueryFont(gui_conf.display, xcin_conf.indexfont))) {
	perr(XCINMSG_ERROR, _("invalid %s: %s\n"), 
		RCKEY_INDEX_FONT, xcin_conf.indexfont);
	gui_conf.if_descent = gui_conf.indexfont->descent;
    }
}

static unsigned long 
x_set_color(char *color_name)
{
    XColor color;

    if (! XParseColor(gui_conf.display, gui_conf.colormap, color_name, &color))
	return  0;
    if (! XAllocColor(gui_conf.display, gui_conf.colormap, &color))
	return  0;

    return  color.pixel;
}

static void
x_set_geometry(char *value)
{
    int  r=0;
    int  pos_x=0, pos_y=0;
    unsigned int  width=0, height=0;
    
    r = XParseGeometry(value, &pos_x, &pos_y, &width, &height);

    gui_conf.pos_x = ((r & XValue)) ? pos_x : 100;
    gui_conf.pos_y = ((r & YValue)) ? pos_y : 100;
/*
 *  The width and height are measured in English characters.
 */
    gui_conf.c_width = MIN_WIN_WIDTH;
    if ((r & WidthValue))
	gui_conf.c_width = (width < MIN_WIN_WIDTH) ? MIN_WIN_WIDTH : width;
    gui_conf.c_height = 2;
}

static void 
set_wm_property(int argc, char **argv, Bool negative_x, Bool negative_y)
{
    char *win_name = XCIN_VERSION, *icon_name = "xcin";
    XTextProperty windowName, iconName;
    XSizeHints *size_hints;
    XWMHints *wm_hints;
    XClassHint *class_hints;

    if (! XStringListToTextProperty(&win_name, 1, &windowName) ||
    	! XStringListToTextProperty(&icon_name, 1, &iconName))
	perr(XCINMSG_IERROR, _("string text property error.\n"));

    if (! (size_hints = XAllocSizeHints()) ||
    	! (wm_hints = XAllocWMHints()) ||
	! (class_hints = XAllocClassHint()))
	perr(XCINMSG_IERROR, _("insufficient memory.\n"));

    size_hints->flags = 
	USPosition | USSize | PMinSize | PResizeInc | PMaxSize | PWinGravity;
    size_hints->x = gui_conf.pos_x;
    size_hints->y = gui_conf.pos_y;
    size_hints->width = gui_conf.width;
    size_hints->height = gui_conf.height;
    size_hints->min_width = MIN_WIN_WIDTH * gui_conf.ef_width;
    size_hints->max_width = MAX_WIN_WIDTH * gui_conf.ef_width;
    size_hints->min_height = gui_conf.height;
    size_hints->max_height = size_hints->min_height;
    size_hints->width_inc = gui_conf.ef_width * 2;
    size_hints->height_inc = 0;
    if (negative_x)
        size_hints->win_gravity = 
	    (negative_y) ? SouthEastGravity: NorthEastGravity;
    else
        size_hints->win_gravity = 
	    (negative_y) ? SouthWestGravity: NorthWestGravity;

    wm_hints->flags = InputHint | StateHint;
    wm_hints->input = False;
    wm_hints->initial_state = NormalState;

    class_hints->res_name = "xcin";
    class_hints->res_class = "xcin";

    XSetWMProperties(gui_conf.display, gui_conf.window, &windowName,
		&iconName, argv, argc, size_hints, wm_hints, class_hints);

    XFree(size_hints);
    XFree(wm_hints);
    XFree(class_hints);
    XFree(windowName.value);
    XFree(iconName.value);
}

static void 
set_GC(void)
{
    gui_conf.gc = XCreateGC(gui_conf.display, gui_conf.window, 0, NULL);
    XSetForeground(gui_conf.display, gui_conf.gc, gui_conf.fg_color);
    XSetBackground(gui_conf.display, gui_conf.gc, gui_conf.bg_color);

    gui_conf.gce = XCreateGC(gui_conf.display, gui_conf.window, 0, NULL);
    XSetForeground(gui_conf.display, gui_conf.gce, gui_conf.fg_color);
    XSetBackground(gui_conf.display, gui_conf.gce, gui_conf.bg_color);
    XSetFont(gui_conf.display, gui_conf.gce, gui_conf.indexfont->fid);

    gui_conf.gcm = XCreateGC(gui_conf.display, gui_conf.window, 0, NULL);
    XSetForeground(gui_conf.display, gui_conf.gcm, gui_conf.mfg_color);
    XSetBackground(gui_conf.display, gui_conf.gcm, gui_conf.mbg_color);

    gui_conf.gcrm = XCreateGC(gui_conf.display, gui_conf.window, 0, NULL);
    XSetForeground(gui_conf.display, gui_conf.gcrm, gui_conf.mbg_color);
    XSetBackground(gui_conf.display, gui_conf.gcrm, gui_conf.mbg_color);
}


void 
gui_init(int argc, char **argv)
{
    Bool negative_x=0, negative_y=0;

    x_set_display(xcin_conf.display_name);
    x_set_font(xcin_conf.fontset);

    gui_conf.fg_color  = x_set_color(xcin_conf.fg_color);
    gui_conf.bg_color  = x_set_color(xcin_conf.bg_color);
    gui_conf.mfg_color = x_set_color(xcin_conf.mfg_color);
    gui_conf.mbg_color = x_set_color(xcin_conf.mbg_color);
    
    x_set_geometry(xcin_conf.geometry);
    gui_conf.width = gui_conf.c_width * gui_conf.ef_width;
    gui_conf.height = gui_conf.c_height * gui_conf.ef_height;
    if (gui_conf.pos_x < 0) {
	gui_conf.pos_x += (DisplayWidth(gui_conf.display, gui_conf.screen) - 
		gui_conf.width);
	negative_x = 1;
    }
    if (gui_conf.pos_y < 0) {
	gui_conf.pos_y += (DisplayHeight(gui_conf.display, gui_conf.screen) -
		gui_conf.height);
	negative_y = 1;
    }
    gui_conf.border = 1;
    gui_conf.root = RootWindow(gui_conf.display, gui_conf.screen);
    gui_conf.window = XCreateSimpleWindow(gui_conf.display, gui_conf.root,
		gui_conf.pos_x, gui_conf.pos_y,
		gui_conf.width, gui_conf.height, gui_conf.border,
		gui_conf.fg_color, gui_conf.bg_color);
    set_wm_property(argc, argv, negative_x, negative_y);

    set_GC();
    XSelectInput(gui_conf.display, gui_conf.window, 
	(ExposureMask | StructureNotifyMask));

    if (! (xcin_conf.xcin_mode & XCIN_XKILL_OFF)) {
        gui_conf.wm_del_win = 
		XInternAtom(gui_conf.display, "WM_DELETE_WINDOW", False);
        XSetWMProtocols(gui_conf.display, gui_conf.window, 
		&(gui_conf.wm_del_win), 1);
    }

    if (! (xcin_conf.xcin_mode & XCIN_MODE_HIDE))
        XMapWindow(gui_conf.display, gui_conf.window);

    ims_english = xcin_conf.inpn_english;
    ims_sbyte   = xcin_conf.inpn_sbyte;
    ims_2bytes  = xcin_conf.inpn_2bytes;

    inpstate_content(0);
}


/*----------------------------------------------------------------------------

	GUI main loop.

----------------------------------------------------------------------------*/

void 
gui_loop(void)
{
    XEvent event;

    while(1) {
        XNextEvent(gui_conf.display, &event);   
        if (XFilterEvent(&event, None) == True)
            continue;

	switch (event.type) {
	case Expose:
	    win_draw_map();
	    break;
	case GraphicsExpose:
	    win_draw_map();
	    break;
	case ConfigureNotify:
	    change_win_attrib(&(event.xconfigure));
	    break;
	case ClientMessage:
	    if (! (xcin_conf.xcin_mode & XCIN_XKILL_OFF) &&
		event.xclient.format == 32 && 
		event.xclient.data.l[0] == gui_conf.wm_del_win) {
		xim_close();
		exit(0);
	    }
	    break;
        }
    }
}
