/*
 *  $Id: phone.c,v 1.1 1999/02/18 08:22:50 thhsieh Exp $
 */
/*
    Copyright (C) 1999 by  XCIN TEAM

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    For any question or suggestion, please mail to xcin mailing-list:
    xcin@linux.org.tw, or the maintainer Tung-Han Hsieh: thhsieh@linux.org.tw
*/      


#include <string.h>
#include <X11/Xlib.h>
#include <X11/keysym.h>
#include "xcintool.h"
#include "module.h"
#include "gen_inp.h"


/*----------------------------------------------------------------------------

        gen_inp_init()

----------------------------------------------------------------------------*/

static int
loadtab(gen_inp_conf_t *cf, FILE *fp)
{
    int n, nn, ret=1;
    char modID[MODULE_ID_SIZE];

    if (fread(modID, sizeof(char), MODULE_ID_SIZE, fp) != MODULE_ID_SIZE ||
	strcmp(modID, "gencin")) {
	perr(XCINMSG_WARNING, _("gen_inp: %s: invalid tab file.\n"), cf->tabfn);
	return False;
    }
    if (fread(&(cf->header), sizeof(cintab_head_t), 1, fp) != 1) {
	perr(XCINMSG_WARNING, _("gen_inp: %s: reading error.\n"), cf->tabfn);
	return False;
    }
    if (! check_version(GENCIN_VERSION, cf->header.version, 5)) {
	perr(XCINMSG_WARNING, _("gen_inp: %s: invalid version.\n"), cf->tabfn);
	return False;
    }
    if (! cf->inp_cname)
	cf->inp_cname = cf->header.cname;

    n  = cf->header.n_icode;
    nn = cf->ccinfo.total_char;
    cf->icidx = malloc(sizeof(icode_idx_t) * n);
    cf->ichar = malloc(sizeof(ichar_t) * nn);
    cf->ic1 = malloc(sizeof(icode_t) * n);
    if (!n || !nn ||
	fread(cf->icidx, sizeof(icode_idx_t), n, fp) != n ||
	fread(cf->ichar, sizeof(ichar_t), nn, fp) != nn ||
	fread(cf->ic1, sizeof(icode_t), n, fp) != n)
	ret = 0;
    if (ret && cf->header.icode_mode == ICODE_MODE2) {
        cf->ic2 = malloc(sizeof(icode_t) * n);
	if (fread(cf->ic2, sizeof(icode_t), nn, fp) != n)
	    ret = 0;
    }

    if (! ret) {
	perr(XCINMSG_WARNING, _("gen_inp: %s: reading error.\n"), cf->tabfn);
	return False;
    }
    else
	return True;
}

static int
gen_inp_init(void *conf, char *objname, core_config_t *xc)
{
    char *s, *cmd[2], value[50], truefn[256];
    gen_inp_conf_t *cf = (gen_inp_conf_t *)conf;
    FILE *fp;
    int ret;

    cmd[0] = objname;
    cmd[1] = "INP_CNAME";				/* inp_names */
    if (get_resource(cmd, value, 50, 2))
        cf->inp_cname = strdup(value);
    cf->inp_ename = objname;

    cmd[1] = "SETKEY";					/* setkey */
    if (get_resource(cmd, value, 50, 2))
        cf->setkey = (char)atoi(value);
    else {
        perr(XCINMSG_WARNING, _("%s: %s: value not found.\n"), objname, cmd);
	return False;
    }

    cmd[1] = "AUTO_COMPOSE";				/* auto compose */
    if (get_resource(cmd, value, 50, 2))
        set_data(&(cf->mode), RC_IFLAG, value, INP_MODE_AUTOCOMPOSE, 0);

    cmd[1] = "AUTO_UPCHAR";				/* auto_up char */
    if (get_resource(cmd, value, 50, 2))
        set_data(&(cf->mode), RC_IFLAG, value, INP_MODE_AUTOUPCHAR, 0);
    if ((cf->mode & INP_MODE_AUTOUPCHAR)) {
        cmd[1] = "SPACE_AUTOUP";			/* space key auto_up */
        if (get_resource(cmd, value, 50, 2))
            set_data(&(cf->mode), RC_IFLAG, value, INP_MODE_SPACEAUTOUP, 0);
        cmd[1] = "SELKEY_SHIFT";			/* selkey shift */
        if (get_resource(cmd, value, 50, 2))	
	    set_data(&(cf->mode), RC_IFLAG, value, INP_MODE_SELKEYSHIFT, 0);
    }

    cmd[1] = "AUTO_FULLUP";				/* auto full_up */
    if (get_resource(cmd, value, 50, 2))
        set_data(&(cf->mode), RC_IFLAG, value, INP_MODE_AUTOFULLUP, 0);
    if ((cf->mode & INP_MODE_AUTOFULLUP)) {
        cmd[1] = "SPACE_IGNORE";			/* space ignore */
        if (get_resource(cmd, value, 50, 2))	
	    set_data(&(cf->mode), RC_IFLAG, value, INP_MODE_SPACEIGNOR, 0);
    }

    cmd[1] = "AUTO_RESET";				/* auto reset error */
    if (get_resource(cmd, value, 50, 2))
	set_data(&(cf->mode), RC_IFLAG, value, INP_MODE_AUTORESET, 0);
    if (! (cf->mode & INP_MODE_AUTORESET)) {
        cmd[1] = "SPACE_RESET";				/* space reset error */
        if (get_resource(cmd, value, 50, 2))
	    set_data(&(cf->mode), RC_IFLAG, value, INP_MODE_SPACERESET, 0);
    }

    cmd[1] = "SINMD_IN_LINE1";				/* sinmd in line1 mode*/
    if (get_resource(cmd, value, 50, 2))
	set_data(&(cf->mode), RC_IFLAG, value, INP_MODE_SINMDLINE1, 0);
    cmd[1] = "WILD_ENABLE";				/* wild key enable */
    if (get_resource(cmd, value, 50, 2))
	set_data(&(cf->mode), RC_IFLAG, value, INP_MODE_WILDON, 0);
    cmd[1] = "BEEP_WRONG";				/* beep mode: wrong */
    if (get_resource(cmd, value, 50, 2))
        set_data(&(cf->mode), RC_IFLAG, value, INP_MODE_BEEPWRONG, 0);
    cmd[1] = "BEEP_DUPCHAR";				/* beep mode: dupchar */
    if (get_resource(cmd, value, 50, 2))
        set_data(&(cf->mode), RC_IFLAG, value, INP_MODE_BEEPDUP, 0);

    ccode_info(&(cf->ccinfo));

/*
 *  Read the IM tab file.
 */
    if (! (s = strrchr(cf->inp_ename, '.')) || strcmp(s+1, "tab"))
        snprintf(value, 50, "%s.tab", cf->inp_ename);
    if ((fp = open_data(value, "rb", xc->default_dir, xc->user_dir, "tab", 
		xc->locale.lc_ctype, truefn, 256, XCINMSG_WARNING))) {
	cf->tabfn = strdup(truefn);
        ret = loadtab(cf, fp);
	fclose(fp);
    }
    else
	return False;

    if (cf->header.n_endkey) {
        cmd[1] = "END_KEY";				/* end key enable */
        if (get_resource(cmd, value, 50, 2))
	    set_data(&(cf->mode), RC_IFLAG, value, INP_MODE_ENDKEY, 0);
    }
    return  ret;
}


/*----------------------------------------------------------------------------

        gen_inp_xim_init()

----------------------------------------------------------------------------*/

static int
gen_inp_xim_init(void *conf, inpinfo_t *inpinfo)
{
    gen_inp_conf_t *cf = (gen_inp_conf_t *)conf;
    int i;

    inpinfo->inp_cname = cf->inp_cname;
    inpinfo->inp_ename = cf->inp_ename;
    inpinfo->area3_len = cf->header.n_max_keystroke * 2 + 2;
    inpinfo->keystroke_len = 0;
    inpinfo->n_mcch = 0;
    inpinfo->n_lcch = 0;
    inpinfo->guimode = 0;
    inpinfo->cch_publish.wch = (wchar_t)0;
    inpinfo->s_keystroke = calloc(INP_CODE_LENGTH+1, sizeof(wch_t));
    inpinfo->iccf = calloc(1, sizeof(gen_inp_iccf_t));

    if ((cf->mode & INP_MODE_SINMDLINE1))
	inpinfo->guimode |= GUIMOD_SINMDLINE1;
    for (i=0; i<SELECT_KEY_LENGTH+1; i++)
        inpinfo->s_selkey[i].wch = (wchar_t)0;
    if (! (cf->mode & INP_MODE_SELKEYSHIFT)) {
        inpinfo->n_selkey = cf->header.n_selkey;
        for (i=0; i<SELECT_KEY_LENGTH && i<cf->header.n_selkey; i++)
	    inpinfo->s_selkey[i].s[0] = cf->header.selkey[i];
    }
    else {
        inpinfo->n_selkey = cf->header.n_selkey+1;
        for (i=0; i<SELECT_KEY_LENGTH && i<cf->header.n_selkey; i++)
	    inpinfo->s_selkey[i+1].s[0] = cf->header.selkey[i];
    }
    return  True;
}

static unsigned int
gen_inp_xim_end(void *conf, inpinfo_t *inpinfo)
{
    gen_inp_iccf_t *iccf = (gen_inp_iccf_t *)inpinfo->iccf;

    if (iccf->n_mcch_list)
	free(iccf->mcch_list);
    free(inpinfo->iccf);
    free(inpinfo->s_keystroke);
    return IMKEY_ABSORB;
}


/*----------------------------------------------------------------------------

        gen_inp_keystroke(), gen_inp_show_keystroke

----------------------------------------------------------------------------*/

static unsigned int
return_wrong(gen_inp_conf_t *cf)
{
    return ((cf->mode & INP_MODE_BEEPWRONG)) ? IMKEY_BELL : IMKEY_ABSORB;
}

static unsigned int
return_correct(gen_inp_conf_t *cf)
{
    return ((cf->mode & INP_MODE_BEEPDUP)) ? IMKEY_BELL : IMKEY_ABSORB;
}

static void
reset_keystroke(inpinfo_t *inpinfo, gen_inp_iccf_t *iccf)
{
    inpinfo->s_keystroke[0].wch = (wchar_t)0;
    inpinfo->keystroke_len = 0;
    inpinfo->n_mcch = 0;
    iccf->keystroke[0] = '\0';
    iccf->mode = 0;
    if (iccf->n_mcch_list) {
	free(iccf->mcch_list);
	iccf->n_mcch_list = 0;
    }
}


/*------------------------------------------------------------------------*/

static int
cmp_icvalue(icode_t *ic1, icode_t *ic2, unsigned int idx,
	    icode_t icode1, icode_t icode2, int mode)
{
    if (ic1[idx] > icode1)
	return 1;
    else if (ic1[idx] < icode1)
	return -1;
    else {
	if (! mode)
	    return 0;
	else if (ic2[idx] > icode2)
	    return 1;
	else if (ic2[idx] < icode2)
	    return -1;
	else
	    return 0;
    }
}

static int 
bsearch_char(icode_t *ic1, icode_t *ic2, 
	     icode_t icode1, icode_t icode2, int size, int mode, int wild)
{
    int head, middle, end, ret;

    head   = 0;
    middle = size / 2;
    end    = size;
    while ((ret=cmp_icvalue(ic1, ic2, middle, icode1, icode2, mode))) {
        if (ret > 0)
            end = middle;
        else
            head = middle + 1;
        middle = (end + head) / 2;
        if (middle == head && middle == end)
            break;
    }
    if (ret == 0) {
	while(middle > 0 &&
	      ! cmp_icvalue(ic1, ic2, middle-1, icode1, icode2, mode)) 
	    middle --;
	return middle;
    }
    else
	return (wild) ? middle : -1;
}


static int
match_keystroke_wild(gen_inp_conf_t *cf, 
		       inpinfo_t *inpinfo, gen_inp_iccf_t *iccf)
{
    icode_t icode[2];
    int size, md, idx, n_ich=0;
    char *s;

    size = cf->header.n_icode;
    md = (cf->header.icode_mode == ICODE_MODE2) ? 1 : 0;
    icode[0] = icode[1] = 0;
    if ((s=strchr(iccf->keystroke, '*')))
	*s = '\0';

    /*
     *  Search for the first char.
     */
    keys2codes(icode, 2, iccf->keystroke);
    idx = bsearch_char(cf->ic1, cf->ic2, icode[0], icode[1], size, md, 1);
    iccf->mcch_hidx = idx;		/* refer to index of cf->icidx */

    /*
     *  Pick up the remaining chars;
     */
    for (n_ich=0; idx<size && n_ich<inpinfo->n_selkey; n_ich++, idx++) {
	if (! ccode_to_char(cf->icidx[idx], inpinfo->mcch[n_ich].s, WCH_SIZE))
	    return 0;
    }
    inpinfo->n_mcch = n_ich;
    inpinfo->mcch_pgstate = (idx+n_ich >= size) ? MCCH_ONEPG : MCCH_BEGIN;

    return 1;
}

static int
match_keystroke_normal(gen_inp_conf_t *cf, 
		       inpinfo_t *inpinfo, gen_inp_iccf_t *iccf)
{
    icode_t icode[2];
    int size, md, idx, n_ich=0, mcch_size;
    wch_t *mcch;

    size = cf->header.n_icode;
    md = (cf->header.icode_mode == ICODE_MODE2) ? 1 : 0;
    icode[0] = icode[1] = 0;

    /*
     *  Search for the first char.
     */
    keys2codes(icode, 2, iccf->keystroke);
    if ((idx = bsearch_char(cf->ic1, cf->ic2, 
		icode[0], icode[1], size, md, 0)) == -1)
	return 0;

    /*
     *  Search for all the chars with the same keystroke.
     */
    mcch_size = inpinfo->n_selkey;
    mcch = malloc(mcch_size * sizeof(wch_t));
    do {
	if (mcch_size <= n_ich) {
	    mcch_size *= 2;
	    mcch = realloc(mcch, mcch_size * sizeof(wch_t));
	}
	if (! ccode_to_char(cf->icidx[idx], mcch[n_ich].s, WCH_SIZE))
	    return 0;
	n_ich ++;
        idx ++;
    } while (idx < size &&
             ! cmp_icvalue(cf->ic1, cf->ic2, idx, icode[0], icode[1], md));

    /*
     *  Prepare mcch for display.
     */
    for (idx=0; idx<inpinfo->n_selkey && idx<n_ich; idx++)
	inpinfo->mcch[idx].wch = mcch[idx].wch;
    inpinfo->n_mcch = idx;

    if (idx >= n_ich) {
        inpinfo->mcch_pgstate = MCCH_ONEPG;
	free(mcch);
    }
    else {
	inpinfo->mcch_pgstate = MCCH_BEGIN;
        iccf->mcch_list = mcch;
        iccf->n_mcch_list = n_ich;
	iccf->mcch_hidx = 0;		/* refer to index of iccf->mcch_list */
    }
    return 1;
}

static int 
match_keystroke(gen_inp_conf_t *cf, inpinfo_t *inpinfo, gen_inp_iccf_t *iccf)
/* return: 1: success, 0: false */
{
    int ret;

    inpinfo->n_mcch = 0;
    if (! (iccf->mode & INPINFO_MODE_INWILD))
	ret = match_keystroke_normal(cf, inpinfo, iccf);
    else
	ret = match_keystroke_wild(cf, inpinfo, iccf);

    if (inpinfo->n_mcch > 1 && (iccf->mode & INPINFO_MODE_SPACE))
	iccf->mode &= ~(INPINFO_MODE_SPACE);
    return ret;
}


/*------------------------------------------------------------------------*/

static void
commit_char(inpinfo_t *inpinfo, gen_inp_iccf_t *iccf, wch_t *cch)
{
    static char cch_s[WCH_SIZE+1];

    inpinfo->cch = cch_s;
    strncpy(cch_s, cch->s, WCH_SIZE);
    cch_s[WCH_SIZE] = '\0';

    inpinfo->keystroke_len = 0;
    inpinfo->s_keystroke[0].wch = (wchar_t)0;
    inpinfo->n_mcch = 0;
    inpinfo->cch_publish.wch = cch->wch;

    iccf->mode &= ~(INPINFO_MODE_MCCH);
    iccf->mode &= ~(INPINFO_MODE_INWILD);
    inpinfo->guimode &= ~(GUIMOD_SELKEYSPOT);

    DebugLog("char = %s\n", cch->s);
}

static unsigned int
commit_keystroke(gen_inp_conf_t *cf, inpinfo_t *inpinfo, gen_inp_iccf_t *iccf)
/* return: the IMKEY state */
{
    if (match_keystroke(cf, inpinfo, iccf)) {
        if (inpinfo->n_mcch == 1) {
            commit_char(inpinfo, iccf, inpinfo->mcch);
            return IMKEY_COMMIT;
        }
        else {
            iccf->mode |= INPINFO_MODE_MCCH;
	    inpinfo->guimode |= GUIMOD_SELKEYSPOT;
            return return_correct(cf);
        }
    }
    else {
	if ((cf->mode & INP_MODE_AUTORESET))
	    reset_keystroke(inpinfo, iccf);
	else
	    iccf->mode |= INPINFO_MODE_WRONG;
        return return_wrong(cf);
    }
}


/*------------------------------------------------------------------------*/

static int
mcch_choosech(gen_inp_conf_t *cf, 
		inpinfo_t *inpinfo, gen_inp_iccf_t *iccf, int idx)
{
    int min;
    wch_t wch;

    if (inpinfo->n_mcch == 0 && ! match_keystroke(cf, inpinfo, iccf))
        return 0;
    min = (inpinfo->n_selkey > inpinfo->n_mcch) ? 
                inpinfo->n_mcch : inpinfo->n_selkey;
    if (idx >= min)
        return 0;

    if (idx < 0)
	wch.wch = inpinfo->mcch[0].wch;
    else if ((cf->mode & INP_MODE_SELKEYSHIFT))
	wch.wch = inpinfo->mcch[idx+1].wch;
    else
	wch.wch = inpinfo->mcch[idx].wch;
    commit_char(inpinfo, iccf, &wch);
    reset_keystroke(inpinfo, iccf);

    return 1;
}

static int
fillpage(gen_inp_conf_t *cf, inpinfo_t *inpinfo, gen_inp_iccf_t *iccf, char dir)
{
    int i, j, n_pg=inpinfo->n_selkey, total;

    total = ((iccf->mode & INPINFO_MODE_INWILD)) ?
		cf->header.n_icode : iccf->n_mcch_list;
    switch (dir) {
    case 0:
	iccf->mcch_hidx = 0;
	break;
    case 1:
	if (iccf->mcch_hidx + n_pg < total)
	    iccf->mcch_hidx += n_pg;
	else
	    return 0;
	break;
    case -1:
	if (iccf->mcch_hidx - n_pg >= 0)
	    iccf->mcch_hidx -= n_pg;
	else
	    return 0;
	break;
    }
	    
    if (! (iccf->mode & INPINFO_MODE_INWILD)) {
        for (i=0, j=iccf->mcch_hidx; i<n_pg && j<total; i++, j++)
	    inpinfo->mcch[i].wch = iccf->mcch_list[j].wch;
    }
    else {
        for (i=0, j=iccf->mcch_hidx; i<n_pg && j<total; i++, j++) {
	    if (! ccode_to_char(cf->icidx[j], inpinfo->mcch[i].s, WCH_SIZE))
                return return_wrong(cf);
	}
    }

    if (iccf->mcch_hidx == 0)
        inpinfo->mcch_pgstate = (i < total) ? MCCH_BEGIN : MCCH_ONEPG;
    else if (total - iccf->mcch_hidx >= n_pg)
	inpinfo->mcch_pgstate = MCCH_MIDDLE;
    else
	inpinfo->mcch_pgstate = MCCH_END;
    inpinfo->n_mcch = i;

    return 1;
}

static int
mcch_nextpage(gen_inp_conf_t *cf, 
	      inpinfo_t *inpinfo, gen_inp_iccf_t *iccf, char key)
{
    int ret;

    switch (inpinfo->mcch_pgstate) {
    case MCCH_ONEPG:
	switch (key) {
	case ' ':
	    if ((cf->mode & INP_MODE_AUTOUPCHAR))
                return (mcch_choosech(cf, inpinfo, iccf, -1)) ?
                	IMKEY_COMMIT : return_wrong(cf);
            else
                return return_correct(cf);
	case '<':
	case '>':
	    return return_correct(cf);
	defalut:
	    return return_wrong(cf);
	}
	break;

    case MCCH_END:
	switch (key) {
	case ' ':
	case '>':
	    return (fillpage(cf, inpinfo, iccf, 0)) ? 
			IMKEY_ABSORB : return_wrong(cf);
	case '<':
	    return (fillpage(cf, inpinfo, iccf, -1)) ?
			IMKEY_ABSORB : return_wrong(cf);
	default:
	    return return_wrong(cf);
	}
	break;

    case MCCH_BEGIN:
	switch (key) {
	case ' ':
	case '>':
	    return (fillpage(cf, inpinfo, iccf, 1)) ? 
			IMKEY_ABSORB : return_wrong(cf);
	case '<':
	    return return_correct(cf);
	default:
	    return return_wrong(cf);
	}
	break;

    default:
	switch (key) {
	case ' ':
	case '>':
	    return (fillpage(cf, inpinfo, iccf, 1)) ? 
			IMKEY_ABSORB : return_wrong(cf);
	case '<':
	    return (fillpage(cf, inpinfo, iccf, -1)) ?
			IMKEY_ABSORB : return_wrong(cf);
	default:
	    return return_wrong(cf);
	}
	break;
    }
}

/*------------------------------------------------------------------------*/


static unsigned int
gen_inp_keystroke(void *conf, inpinfo_t *inpinfo, keyinfo_t *keyinfo)
{
    gen_inp_conf_t *cf = (gen_inp_conf_t *)conf;
    gen_inp_iccf_t *iccf = (gen_inp_iccf_t *)inpinfo->iccf;
    KeySym keysym = keyinfo->keysym;
    char *keystr = keyinfo->keystr;
    int len, max_len;
    char sp_ignore=0;

    len = inpinfo->keystroke_len;
    max_len = cf->header.n_max_keystroke;
    if ((iccf->mode & INPINFO_MODE_SPACE)) {
        sp_ignore = 1;
        iccf->mode &= ~(INPINFO_MODE_SPACE);
    }

    if ((keysym == XK_BackSpace || keysym == XK_Delete) && len) {
        iccf->keystroke[len-1] = '\0';
        inpinfo->s_keystroke[len-1].wch = (wchar_t)0;
        inpinfo->keystroke_len --;
	inpinfo->n_mcch = 0;
	inpinfo->cch_publish.wch = (wchar_t)0;
	iccf->mode = 0;
	if (len-1 > 0 && (cf->mode & INP_MODE_AUTOCOMPOSE))
	    match_keystroke(cf, inpinfo, iccf);
        return IMKEY_ABSORB;
    }

    else if (keysym == XK_Escape && len) {
	reset_keystroke(inpinfo, iccf);
        inpinfo->cch_publish.wch = (wchar_t)0;
        return IMKEY_ABSORB;
    }

    else if (keysym == XK_space) {
        inpinfo->cch_publish.wch = (wchar_t)0;

	if ((cf->mode & INP_MODE_SPACEAUTOUP) && inpinfo->n_mcch>1) {
            if ((mcch_choosech(cf, inpinfo, iccf, -1)))
                return IMKEY_COMMIT;
            else {
                if ((cf->mode & INP_MODE_AUTORESET))
                    reset_keystroke(inpinfo, iccf);
                else
                    iccf->mode |= INPINFO_MODE_WRONG;
                return return_wrong(cf);
            }
	}
	else if ((iccf->mode & INPINFO_MODE_MCCH))
	    return mcch_nextpage(cf, inpinfo, iccf, ' ');
        else if ((cf->mode & INP_MODE_SPACERESET) &&
                 (iccf->mode & INPINFO_MODE_WRONG)) {
            reset_keystroke(inpinfo, iccf);
            return IMKEY_ABSORB;
        }
        else if (sp_ignore)
            return IMKEY_ABSORB;
	else if (inpinfo->keystroke_len)
	    return commit_keystroke(cf, inpinfo, iccf);
    }

    else if (keyinfo->keystr_len == 1) {
        unsigned int ret=IMKEY_ABSORB;
	int selkey_idx;
        char keycode, *s;
        wch_t wch;

        inpinfo->cch_publish.wch = (wchar_t)0;
	keycode = key2code(keystr[0]);
	wch.wch = cf->header.keyname[keycode].wch;
	selkey_idx = ((s = strchr(cf->header.selkey, keystr[0]))) ? 
		(int)(s - cf->header.selkey) : -1;

	if (inpinfo->keystroke_len && selkey_idx != -1 && ! wch.wch) {
	    /* Don't enter the multi-cch selection, but selkey pressed. */
	    return (mcch_choosech(cf, inpinfo, iccf, selkey_idx)) ?
			IMKEY_COMMIT : return_wrong(cf);
	}
	else if ((keystr[0]=='<' || keystr[0]=='>') && inpinfo->n_mcch > 1)
	    return mcch_nextpage(cf, inpinfo, iccf, keystr[0]);
	else if ((iccf->mode & INPINFO_MODE_MCCH)) {
	    /* Enter the multi-cch selection. */
	    if (selkey_idx != -1)
		return (mcch_choosech(cf, inpinfo, iccf, selkey_idx)) ?
			IMKEY_COMMIT : return_wrong(cf);
	    else if ((cf->mode & INP_MODE_AUTOUPCHAR)) {
		if (! mcch_choosech(cf, inpinfo, iccf, -1))
		    return return_wrong(cf);
		ret |= IMKEY_COMMIT;
	    }
	    else
		return return_wrong(cf);
	}

	/* The previous cch might be committed, so len might be 0 */
	len = inpinfo->keystroke_len;

	if (! wch.wch)
	    return (ret | IMKEY_IGNORE);
        else if ((keyinfo->keystate & ShiftMask)) {
	    if (! (cf->mode & INP_MODE_WILDON) || keystr[0] != '*')
                return (ret | IMKEY_SHIFTESC);
	    else
	        iccf->mode |= INPINFO_MODE_INWILD;
	}
        else if (len >= max_len)
            return return_wrong(cf);

        iccf->keystroke[len] = keystr[0];
        iccf->keystroke[len+1] = '\0';
	if ((iccf->mode & INPINFO_MODE_INWILD))
	    inpinfo->s_keystroke[len].s[0] = '*';
	else
            inpinfo->s_keystroke[len].wch = wch.wch;
        inpinfo->s_keystroke[len+1].wch = (wchar_t)0;
        inpinfo->keystroke_len ++;
        len ++;

	if ((cf->mode & INP_MODE_AUTOCOMPOSE))
	    match_keystroke(cf, inpinfo, iccf);

	if ((cf->mode & INP_MODE_ENDKEY) && 
		 (s=strchr(cf->header.endkey, keystr[0])))
	    return commit_keystroke(cf, inpinfo, iccf);
	else if ((cf->mode & INP_MODE_AUTOFULLUP)) {
	    if (len == max_len) {
	        if ((cf->mode & INP_MODE_SPACEIGNOR))
		    iccf->mode |= INPINFO_MODE_SPACE;
	        return commit_keystroke(cf, inpinfo, iccf);
	    }
	}
	return ret;
    }

    return IMKEY_IGNORE;
}

static int 
gen_inp_show_keystroke(void *conf, simdinfo_t *simdinfo)
{
    gen_inp_conf_t *cf = (gen_inp_conf_t *)conf;
    int i, idx;
    wchar_t tmp;
    char *k, keystroke[INP_CODE_LENGTH+1];
    static wch_t keystroke_list[INP_CODE_LENGTH+1];

    if ((i = ccode_to_idx(&(simdinfo->cch_publish))) == -1)
        return False;

    idx = cf->ichar[i];
    if (cf->header.icode_mode == ICODE_MODE1)
	codes2keys(&(cf->ic1[idx]), 1, keystroke, SELECT_KEY_LENGTH+1);
    else if (cf->header.icode_mode == ICODE_MODE2) {
        unsigned int klist[2];

	klist[0] = cf->ic1[idx];
	klist[1] = cf->ic2[idx];
	codes2keys(klist, 2, keystroke, SELECT_KEY_LENGTH+1);
    }
    for (i=0, k=keystroke; i<INP_CODE_LENGTH && *k; i++, k++) {
	idx = key2code(*k);
	if ((tmp = cf->header.keyname[idx].wch))
	    keystroke_list[i].wch = tmp;
	else {
	    keystroke_list[i].wch = (wchar_t)0;
	    keystroke_list[i].s[0] = '?';
	}
    }
    keystroke_list[i].wch = (wchar_t)0;
    simdinfo->s_keystroke = keystroke_list;

    return (i) ? True : False;
}

/*----------------------------------------------------------------------------

        Definition of general input method module (templet).

----------------------------------------------------------------------------*/

static char *gen_inp_valid_objname[] = { "*", NULL };


module_t module_ptr = {
    "gen_inp",					/* name */
    MODULE_VERSION,				/* version */
    "general input method module.",		/* comments */
    gen_inp_valid_objname,			/* valid_objname */
    MOD_CINPUT,					/* module_type */
    sizeof(gen_inp_conf_t),			/* conf_size */
    gen_inp_init,				/* init */
    gen_inp_xim_init,				/* xim_init */
    gen_inp_xim_end,				/* xim_end */
    NULL,					/* switch_in */
    NULL,					/* switch_out */
    gen_inp_keystroke,				/* keystroke */
    gen_inp_show_keystroke,			/* show_keystroke */
};
