/*
 * Author:      William Chia-Wei Cheng (bill.cheng@acm.org)
 *
 * Copyright (C) 2001, William Chia-Wei Cheng.
 *
 * This file may be distributed under the terms of the Q Public License
 * as defined by Trolltech AS of Norway and appearing in the file
 * LICENSE.QPL included in the packaging of this file.
 *
 * THIS FILE IS PROVIDED AS IS WITH NO WARRANTY OF ANY KIND, INCLUDING
 * THE WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY SPECIAL,
 * INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
 * FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 * NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION
 * WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 * @(#)$Header: /mm/src/tgif/v4/RCS/wb.c,v 4.71 2001/10/16 20:22:01 william Exp $
 */

#define _INCLUDE_FROM_WB_C_

#define FOREVER while(1)

#include <pthread.h>
#include "tgifdefs.h"
#include "cmdids.h"

#ifdef _TGIF_WB2
#ifdef _HAS_STREAMS_SUPPORT
#include <stropts.h>
#include <sys/types.h>
#endif /* _HAS_STREAMS_SUPPORT */
#include "rmcast/rmcast.h"
#endif /* _TGIF_WB2 */

#include "auxtext.e"
/* #include "chat.e" */
#include "drawing.e"
#include "choice.e"
#include "cmd.e"
#include "dialog.e"
#include "file.e"
#include "grid.e"
#include "http.e"
#include "mark.e"
#include "menu.e"
#include "msg.e"
#include "obj.e"
#include "page.e"
#include "remote.e"
#include "select.e"
#include "setup.e"
#include "stk.e"
#include "strtbl.e"
#include "tcp.e"
#include "util.e"
#include "wb.e"
#include "wb_buff.e"
#include "wb_seg.e"

struct WhiteBoardRec	gstWBInfo;


int	cmdLineNoWhiteBoard=FALSE;
int	cmdLineWhiteBoardListenOnly=FALSE;

static int	recordCmdLogicalClock=0;
static char	recordCmdSenderProcID[MAXSTRING];

#ifdef _TGIF_WB2
pthread_mutex_t  message_rcv_mutex    = PTHREAD_MUTEX_INITIALIZER;
#ifdef _TGIF_DBG_WB2
static int		wb2DebugLevel=0;
#endif /* _TGIF_DBG_WB2 */
#endif /* _TGIF_WB2 */



/* =================== Remote Related Functions =================== */

static
void CleanUpWBSockets()
{
   if (gstWBInfo.listening) {
      if (gstWBInfo.listen_socket != (-1)) {
         close(gstWBInfo.listen_socket);
         gstWBInfo.listen_socket = (-1);
      }
      if (gstWBInfo.listen_fp != NULL) {
         fclose(gstWBInfo.listen_fp);
         gstWBInfo.listen_fp = NULL;
      }
      if (*gstWBInfo.listen_fname != '\0') {
         unlink(gstWBInfo.listen_fname);
         *gstWBInfo.listen_fname = '\0';
      }
      gstWBInfo.listening = FALSE;
   }
   if (gstWBInfo.send_socket != (-1)) {
      close(gstWBInfo.send_socket);
      gstWBInfo.send_socket = (-1);
   }
   if (gstWBInfo.send_fp != NULL) {
      fclose(gstWBInfo.send_fp);
      gstWBInfo.send_fp = NULL;
   }
   if (*gstWBInfo.send_fname != '\0') {
      unlink(gstWBInfo.send_fname);
      *gstWBInfo.send_fname = '\0';
   }
}

static
void CleanUpWBCmds()
{
   struct CmdRec *cmd_ptr=NULL;
   int num_records=0;

   for (cmd_ptr=gstWBInfo.last_cmd; cmd_ptr != NULL; cmd_ptr=cmd_ptr->prev) {
      num_records++;
   }
   if (num_records > 0) {
      struct CmdRec *prev_cmd=NULL;
      double inc=(100.0/((double)num_records)), percent_start=0.0;

      ShowInterrupt(1);
      SaveStatusStrings();
      for (cmd_ptr=gstWBInfo.last_cmd; cmd_ptr != NULL; cmd_ptr=prev_cmd,
            percent_start+=inc) {
         prev_cmd = cmd_ptr->prev;
         DeleteARedoRecord(cmd_ptr, percent_start,
               min(((double)percent_start+inc),((double)100.0)));
      }
      RestoreStatusStrings();
      HideInterrupt();
   }
   gstWBInfo.first_cmd = gstWBInfo.last_cmd = gstWBInfo.cur_cmd = NULL;
}

#ifdef _TGIF_WB2
static
int CreateWBListenSocket()
{
   gstWBInfo.listening = TRUE;

   sprintf(gstWBInfo.listen_fname, "%sTgifLXXXXXX", TMP_DIR);
   mktemp(gstWBInfo.listen_fname);
   unlink(gstWBInfo.listen_fname);
   if ((gstWBInfo.listen_fp=fopen(gstWBInfo.listen_fname, "w+")) == NULL) {
      fprintf(stderr, "Fail to create '%s' for deserialization.\n",
            gstWBInfo.listen_fname);
      return FALSE;
   }

   return TRUE;
}

void PrintFullIDsOfObjects(psz_prefix)
   char *psz_prefix;
{
#ifdef _TGIF_DBG_WB2
   if (wb2DebugLevel > 0) {
      struct ObjRec *obj_ptr=NULL;
      int stacking_order=0;

      if (psz_prefix != NULL) {
         if (strcmp(psz_prefix, "\t") == 0) {
            fprintf(stderr, "\tin PrintFullIDsOfObjects():\n");
         } else {
            fprintf(stderr, "%s, in PrintFullIDsOfObjects():\n", psz_prefix);
         }
      } else {
         fprintf(stderr, "In PrintFullIDsOfObjects():\n");
      }
      for (obj_ptr=topObj; obj_ptr != NULL; obj_ptr=obj_ptr->next,
            stacking_order++) {
         char buf[MAXSTRING], obj_type[MAXSTRING];

         if (obj_ptr->creator_full_id == NULL) {
            sprintf(buf, "(NULL)%1d/%s", obj_ptr->id, gszLocalPID);
         } else {
            strcpy(buf, obj_ptr->creator_full_id);
         }
         switch (obj_ptr->type) {
         case OBJ_POLY: strcpy(obj_type, "p"); break;
         case OBJ_BOX: strcpy(obj_type, "b"); break;
         case OBJ_OVAL: strcpy(obj_type, "o"); break;
         case OBJ_TEXT:
            strcpy(obj_type, "t");
            {
               MiniLinesInfo *minilines=(&obj_ptr->detail.t->minilines);
               char *psz=minilines->first->first_block->seg->dyn_str.s;
               char buf1[MAXSTRING];
               int len=strlen(psz);

               if (len > 5) {
                  char saved_ch=psz[5];

                  sprintf(buf1, " - \"%s...\"", psz);
                  psz[5] = saved_ch;
               } else {
                  sprintf(buf1, " - \"%s\"", psz);
               }
               strcat(buf, buf1);
            }
            break;
         case OBJ_POLYGON: strcpy(obj_type, "g"); break;
         case OBJ_ARC: strcpy(obj_type, "a"); break;
         case OBJ_RCBOX: strcpy(obj_type, "rcb"); break;
         case OBJ_XBM: strcpy(obj_type, "xbm"); break;
         case OBJ_XPM: strcpy(obj_type, "xpm"); break;
         case OBJ_GROUP:
         case OBJ_ICON:
         case OBJ_SYM:
         case OBJ_PIN: strcpy(obj_type, "r"); break;
         }
         fprintf(stderr, "\t%d/%s:\t%s\n", stacking_order, obj_type, buf);
      }
   }
#endif /* _TGIF_DBG_WB2 */
}


int RecvWBData(flag, ppsz_buf, pn_buf_sz)
   int flag, *pn_buf_sz;
   char **ppsz_buf;
{
   
   int retval;
   
   retval = buff_rem(gstWBInfo.bd_commands, (void**)ppsz_buf);
   
   return retval;
}

static
void ProcessWBInputData(buf)
   char *buf;
   /*
    * Here the WB input data is in the HTTP data format.
    * So we use HttpExtractText() to convert the data into
    *       something that can be handled by ProcessRemoteCmd().
    * In the future, if the WB input data is binary, this is
    *       where the conversion takes place.
    */
{
   char *psz_content=NULL, *psz_content_type=NULL;
   int content_sz=0;

   psz_content = HttpExtractText(buf, &content_sz, NULL, &psz_content_type);
   if (psz_content != NULL) {
      ProcessRemoteCmd(psz_content, content_sz);
      FreeRemoteBuf(psz_content);
   }
   if (psz_content_type != NULL) UtilFree(psz_content_type);
}

int WBHasReadData(pn_flag, pn_retry)
   int *pn_flag, *pn_retry;
{
   /* leave *pn_retry alone */
   
   if ((!(gstWBInfo.BlockRemoteCmdDepth <= 1)) &&  (buff_items(gstWBInfo.bd_commands) > 0) ) 
   {
    // fprintf(stderr,"gstWBInfo.BlockRemoteCmdDepth > 1 e existem dados no buffer. (WBHasReadData)\n");   
    // exit(1);
   }
   
   return (gstWBInfo.BlockRemoteCmdDepth <= 1 &&
         buff_items(gstWBInfo.bd_commands) > 0);
}



void TryHandleWBInputData()
{
   int flag=0, retry=TRUE;
   char *buf;
   int buf_sz;
       

     while (retry && WBHasReadData(&flag, &retry)) { 
      
      buf=NULL;
      buf_sz=0;

      deserializingFile = TRUE;

      if (RecvWBData(flag, &buf, &buf_sz)) {
         ProcessWBInputData(buf);
         FreeRemoteBuf(buf);
      }
      deserializingFile = FALSE;
     } 


}

int SendWBData(pszWBData, logical_clock)
   char *pszWBData;
   int   logical_clock;
{
   int    nPackets1, i;
   struct SegmentationPack *pack1;
   
#ifdef DEBUG   
   fprintf(stderr,"Mensagem enviada:\n%s\n", pszWBData);
#endif   

   pack1 = Segment( pszWBData, strlen(pszWBData), gszLocalPID, logical_clock, &nPackets1 );

   for( i = 0; i < nPackets1; i++ )
   {
      RM_sendto(gstWBInfo.MCastSock, (char*)&pack1[i], sizeof(struct SegmentationPack));
        
      /* FIXME: Contorna a limitacao de buffer do UDP (aguarda 0.1 segs. entre envio de pcts) */
        
      /*
	  if(i%10 == 9)
	    usleep( 100000 );
      */
   }
   free( pack1 );

#ifdef _TGIF_DBG_WB2
   fprintf(stderr, "SendWBData(), pid = %ld\n", gstWBInfo.pid);
#endif /* _TGIF_DBG_WB2 */

   return TRUE;
}

#endif /* _TGIF_WB2 */

/* =================== Cmd Related Functions =================== */

static
int SaveCmd(FP, CmdType, SubCmdPtr, TopSel, BotSel, NumObjs)
   FILE *FP;
   int CmdType, NumObjs;
   struct SubCmdRec *SubCmdPtr;
   struct SelRec *TopSel, *BotSel;
{
   struct SelRec *sel_ptr=NULL;
   int count=0;

   if (recordCmdIncludeTgifObj) {
      sprintf(gszMsgBox, "%s.\n%s %s.\n\n%s %s.",
            "Warning:  recordCmdIncludeTgifObj is TRUE in SaveCmd()",
            TOOL_NAME, "WhiteBoard may crash",
            "Please try to reproduce this error and",
            "send bug report to william@cs.ucla.edu");
      MsgBox(gszMsgBox, TOOL_NAME, STOP_MB);
      return FALSE;
   } else if (CmdType == CMD_GOTO_PAGE) {
      /*
       * GotoPage() is disabled for _TGIF_WB2.
       */
      return FALSE;
   } else if (CmdType == CMD_MOVE) {
      if (fprintf(FP, "cmd(%1d,%1d,%1d,", CmdType,
            SubCmdPtr->detail.mv->dx, SubCmdPtr->detail.mv->dy) == EOF) {
         writeFileFailed = TRUE;
      }
   } else {
      if (fprintf(FP, "cmd(%1d,", CmdType) == EOF) {
         writeFileFailed = TRUE;
      }
   }
   if (fprintf(FP, "%1d,%1d,%1d,\"%s\").\n",
         recordCmdIncludeTgifObj, recordCmdUsesNewColormap,
         gstWBInfo.logical_clock, gszLocalPID) == EOF) {
      writeFileFailed = TRUE;
   }
   /* Lamport's Algorithm increments logical_clock */
   gstWBInfo.logical_clock++;

   if (botSelBeforeInCmd != NULL) {
      if (fprintf(FP, "before_image(%1d,[\n", stackingCount) == EOF) {
         writeFileFailed = TRUE;
      }
      for (count=0, sel_ptr=topSelBeforeInCmd; sel_ptr != NULL;
            sel_ptr=sel_ptr->next, count++) {
         struct ObjRec *obj_ptr=sel_ptr->obj;
         char buf[MAXSTRING];

         if (stackingPositionHasIds) {
            sprintf(buf, "%s", ((char**)stackingPosition)[count]);
         } else {
            if (obj_ptr->creator_full_id == NULL) {
               sprintf(buf, "%1d/%s", obj_ptr->id, gszLocalPID);
            } else {
               strcpy(buf, obj_ptr->creator_full_id);
            }
         }
         if (fprintf(FP, "\t\"") == EOF) writeFileFailed = TRUE;
         SaveString(FP, buf);
         if (sel_ptr->next == NULL) {
            if (fprintf(FP, "\"\n") == EOF) writeFileFailed = TRUE;
         } else {
            if (fprintf(FP, "\",\n") == EOF) writeFileFailed = TRUE;
         }
      }
      if (fprintf(FP, "]).\n") == EOF) writeFileFailed = TRUE;
   }
   if (BotSel != NULL) {
      struct SelRec *sel_ptr;
      int *stacking_pos=NULL, stacking_count=0;

      sel_ptr = PrepareStackingInfo(TopSel, BotSel, NumObjs, FALSE,
            &stacking_pos, &stacking_count, NULL);
      if (sel_ptr == NULL && stacking_count == NumObjs) {
         if (fprintf(FP, "after_positions(%1d,[\n", NumObjs) == EOF) {
            writeFileFailed = TRUE;
         }
         for (count=0, sel_ptr=TopSel;
               count < stacking_count && sel_ptr != NULL;
               count++, sel_ptr=sel_ptr->next) {
            char buf[MAXSTRING];

            sprintf(buf, "%1d", stacking_pos[count]);
            if (fprintf(FP, "\t") == EOF) writeFileFailed = TRUE;
            SaveString(FP, buf);
            if (sel_ptr->next == NULL) {
               if (fprintf(FP, "\n") == EOF) writeFileFailed = TRUE;
            } else {
               if (fprintf(FP, ",\n") == EOF) writeFileFailed = TRUE;
            }
         }
         if (fprintf(FP, "]).\n") == EOF) writeFileFailed = TRUE;

         if (fprintf(FP, "after_image(%1d,[\n", NumObjs) == EOF) {
            writeFileFailed = TRUE;
         }
         for (count=0, sel_ptr=TopSel; sel_ptr != NULL;
               sel_ptr=sel_ptr->next, count++) {
            SaveObj(FP, sel_ptr->obj, 1);
            if (sel_ptr->next == NULL) {
               if (fprintf(FP, "\n") == EOF) writeFileFailed = TRUE;
            } else {
               if (fprintf(FP, ",\n") == EOF) writeFileFailed = TRUE;
            }
         }
         if (fprintf(FP, "]).\n") == EOF) writeFileFailed = TRUE;
      }
      if (stacking_pos != NULL) free(stacking_pos);
   }
   return TRUE;
}

#ifdef _TGIF_WB2
#ifdef _TGIF_DBG_WB2
static
void DebugDumpCmd(psz_prefix, cmd_type, logical_clock, psz_process_id)
   char *psz_prefix, *psz_process_id;
   int cmd_type, logical_clock;
{
   if (psz_prefix == NULL) {
      fprintf(stderr, "cmd(%1d,%1d,\"%s\")\n", cmd_type, logical_clock,
            psz_process_id);
   } else {
      fprintf(stderr, "%s, cmd(%1d,%1d,\"%s\")\n", psz_prefix, cmd_type,
            logical_clock, psz_process_id);
   }
}
#endif /* _TGIF_DBG_WB2 */
#endif /* _TGIF_WB2 */

char *SerializeCmd(CmdType, SubCmdPtr, TopSel, BotSel, NumObjs,
      pnAssignedLogicalClock)
   int CmdType, NumObjs, *pnAssignedLogicalClock;
   struct SubCmdRec *SubCmdPtr;
   struct SelRec *TopSel, *BotSel;
{
   char *buf=NULL, header[MAXSTRING<<1];
   int content_sz=0, ok=TRUE, header_sz=0;
   time_t tloc;

   if (gstWBInfo.send_fp == NULL) {
      sprintf(gszMsgBox, "Cannot open '%s'.\n\nSerialization aborted.",
            gstWBInfo.send_fname);
      MsgBox(gszMsgBox, TOOL_NAME, INFO_MB);
      return NULL;
   }
   *pnAssignedLogicalClock = gstWBInfo.logical_clock;

   rewind(gstWBInfo.send_fp);
   writeFileFailed = FALSE;

   time(&tloc);
   if (fprintf(gstWBInfo.send_fp, "%%TGWB begin - %ld\n", (long)tloc) == EOF) {
      writeFileFailed = TRUE;
   }
   Save(gstWBInfo.send_fp, NULL, 0, 1);
   ok = SaveCmd(gstWBInfo.send_fp, CmdType, SubCmdPtr, TopSel, BotSel, NumObjs);

   if (fprintf(gstWBInfo.send_fp, "%%TGWB end - %ld\n", (long)tloc) == EOF) {
      writeFileFailed = TRUE;
   }
   if (writeFileFailed) {
      sprintf(gszMsgBox, "Fail to write to '%s'.\n\nFile system may be full.",
            gstWBInfo.send_fname);
      MsgBox(gszMsgBox, TOOL_NAME, INFO_MB);
      return NULL;
   }
   if (!ok) {
      return NULL;
   }
   content_sz = (int)ftell(gstWBInfo.send_fp);
   /*
    * From: <PID>:<IP>\r\n
    * Content-Type: application/x-tgif-cmd\r\n
    * Content-Length: <LENGTH>\r\n
    * \r\n
    * <DATA of size LENGTH>
    */
   sprintf(header, "%s%s\r\n%s%s\r\n%s%1d\r\n\r\n",
         "From: ", gszLocalPID,
         "Content-Type: ", "application/x-tgif-cmd",
         "Content-Length: ", content_sz);
   header_sz = strlen(header);
   buf = (char*)malloc((header_sz+content_sz+1)*sizeof(char));
   if (buf == NULL) FailAllocMessage();
   strcpy(buf, header);
   rewind(gstWBInfo.send_fp);
   if (fread(&buf[header_sz], sizeof(char), content_sz, gstWBInfo.send_fp) !=
         content_sz) {
      sprintf(gszMsgBox, "Error in reading '%s'.\n\nSerialization aborted.",
            gstWBInfo.send_fname);
      MsgBox(gszMsgBox, TOOL_NAME, INFO_MB);

      UtilFree(buf);
      return NULL;
   }
   buf[header_sz+content_sz] = '\0';
#ifdef _TGIF_WB2
#ifdef _TGIF_DBG_WB2
   if (wb2DebugLevel == 0) {
      fprintf(stderr, "Sending %1d bytes, pid = %ld!\n",
            header_sz+content_sz+2, gstWBInfo.pid);
   } else {
      fprintf(stderr, "%s\nSending %1d bytes, pid = %ld!\n", buf,
            header_sz+content_sz+2, gstWBInfo.pid);
   }
   DebugDumpCmd(">>> In SerializeCmd()",
         CmdType, *pnAssignedLogicalClock, gszLocalPID);
   PrintFullIDsOfObjects("In SerializeCmd()");
#endif /* _TGIF_DBG_WB2 */
#endif /* _TGIF_WB2 */

   return buf;
}

void SerializeFreeBuf(pszWBData)
   char *pszWBData;
{
   UtilFree(pszWBData);
}

static struct CmdRec gstDeserializeCmd;

static char **gppszBeforeImage=NULL;
static int gnBeforeImageCount=0;

static int *gpnAfterPositions=NULL;
static int gnAfterPositionsCount=0;

static
void CleanUpDeserializeCmd()
{
   memset(&gstDeserializeCmd, 0, sizeof(gstDeserializeCmd));
   gstDeserializeCmd.type = INVALID;
}

static
void CleanUpBeforeImage()
{
   if (gppszBeforeImage != NULL) {
      int i;

      for (i=0; i < gnBeforeImageCount; i++) {
         if (gppszBeforeImage[i] != NULL) {
            UtilFree(gppszBeforeImage[i]);
         } else {
            break;
         }
      }
      free(gppszBeforeImage);
   }
   gppszBeforeImage = NULL;
   gnBeforeImageCount = 0;
}

static
void CleanUpAfterPositions()
{
   if (gpnAfterPositions != NULL) free(gpnAfterPositions);
   gpnAfterPositions = NULL;
   gnAfterPositionsCount = 0;
}

static
void DeserializationCleanUp()
{
   CleanUpDeserializeCmd();
   CleanUpBeforeImage();
   CleanUpAfterPositions();
}

int ReadCmd(FP, Inbuf)
   FILE *FP;
   char *Inbuf;
{
   char *c_ptr=NULL;

   c_ptr = FindChar((int)'(', Inbuf);
   InitScan(c_ptr, "\t\n, ()");
   if (GETINT("cmd", gstDeserializeCmd.type, "cmd_type") == INVALID) {
      return FALSE;
   }
   if (gstDeserializeCmd.type == INVALID) {
      return FALSE;
   } else if (gstDeserializeCmd.type == CMD_GOTO_PAGE) {
      return FALSE;
   } else if (gstDeserializeCmd.type == CMD_MOVE) {
      if (GETINT("cmd", gstDeserializeCmd.dx, "dx") == INVALID ||
          GETINT("cmd", gstDeserializeCmd.dy, "dy") == INVALID) {
         return FALSE;
      }
   }
   *recordCmdSenderProcID = '\0';
   if (GETINT("cmd", recordCmdIncludeTgifObj, "include_tgif_obj") == INVALID ||
       GETINT("cmd", recordCmdUsesNewColormap, "use_new_colormap") == INVALID ||
       GETINT("cmd", recordCmdLogicalClock, "logical_clock") == INVALID ||
       GETSTR("cmd", recordCmdSenderProcID, "sender_process_id") == INVALID) {
      return FALSE;
   }
   UtilRemoveQuotes(recordCmdSenderProcID);
   return TRUE;
}

int ReadBeforeImage(FP, Inbuf)
   FILE *FP;
   char *Inbuf;
{
   char *c_ptr, *line;
   int index=0, ok=TRUE;

   c_ptr = FindChar((int)'(', Inbuf);
   InitScan(c_ptr, "\t\n, ");
   if (GETINT("before_image", gnBeforeImageCount, "before_image_count") ==
         INVALID) {
      return FALSE;
   }
   if (gnBeforeImageCount > 0) {
      gppszBeforeImage = (char**)malloc(gnBeforeImageCount*sizeof(char*));
      if (gppszBeforeImage == NULL) FailAllocMessage();
      memset(gppszBeforeImage, 0, gnBeforeImageCount*sizeof(char*));
   }
   index = 0;
   while ((line=UtilGetALine(FP)) != NULL) {
      char full_id[MAXSTRING];

      scanLineNum++;
      if (*line == ']') {
         free(line);
         break;
      }
      if (index >= gnBeforeImageCount) {
         fprintf(stderr, "Warning: too many lines in before_image.\n");
         free(line);
         break;
      }
      c_ptr = FindChar((int)'"', line);
      c_ptr = ParseStr(c_ptr, (int)'"', full_id, sizeof(full_id));
      if (c_ptr == NULL) {
         ok = FALSE;
         free(line);
         break;
      }
      gppszBeforeImage[index] = UtilStrDup(full_id);
      if (gppszBeforeImage[index] == NULL) FailAllocMessage();
      index++;
      free(line);
   }
   if (ok && index < gnBeforeImageCount) {
      fprintf(stderr, "Warning: too few lines in before_image.\n");
      ok = FALSE;
   }
   if (!ok) {
      CleanUpBeforeImage();
   }
   return ok;
}

int ReadAfterPositions(FP, Inbuf)
   FILE *FP;
   char *Inbuf;
{
   char *c_ptr, *line;
   int index=0, ok=TRUE;

   c_ptr = FindChar((int)'(', Inbuf);
   InitScan(c_ptr, "\t\n, ");
   if (GETINT("after_positions", gnAfterPositionsCount,
         "after_positions_count") == INVALID) {
      return FALSE;
   }
   if (gnAfterPositionsCount > 0) {
      gpnAfterPositions = (int*)malloc(gnAfterPositionsCount*sizeof(int));
      if (gpnAfterPositions == NULL) FailAllocMessage();
      memset(gpnAfterPositions, 0, gnAfterPositionsCount*sizeof(int));
   }
   index = 0;
   while ((line=UtilGetALine(FP)) != NULL) {
      int pos=(-1);

      scanLineNum++;
      if (*line == ']') {
         free(line);
         break;
      }
      if (index >= gnAfterPositionsCount) {
         fprintf(stderr, "Warning: too many lines in before_image.\n");
         free(line);
         break;
      }
      InitScan(line, "\t\n, ");
      if (GETINT("after_positions", pos, "after_positions_pos") == INVALID) {
         ok = FALSE;
         free(line);
         break;
      }
      gpnAfterPositions[index++] = pos;
      free(line);
   }
   if (ok && index < gnAfterPositionsCount) {
      fprintf(stderr, "Warning: too few lines in after_positions.\n");
      ok = FALSE;
   }
   if (!ok) {
      CleanUpBeforeImage();
   }
   return ok;
}

int ReadAfterImage(FP, Inbuf)
   FILE *FP;
   char *Inbuf;
{
   struct ObjRec *obj_ptr=NULL;
   char *c_ptr;
   int ok=TRUE, count_expected=(-1), count=0;

   c_ptr = FindChar((int)'(', Inbuf);
   InitScan(c_ptr, "\t\n, ");
   if (GETINT("after_image", count_expected, "after_image_count") == INVALID) {
      return FALSE;
   }
   count = 0;
   while (ReadObj(FP, &obj_ptr)) {
      if (obj_ptr == NULL) {
         ok = FALSE;
         break;
      }
      obj_ptr->next = NULL;
      obj_ptr->prev = botObj;
      if (botObj == NULL) {
         topObj = obj_ptr;
      } else {
         botObj->next = obj_ptr;
      }
      botObj = obj_ptr;
      obj_ptr = NULL;
      count++;
   }
   if (ok && count != count_expected) {
      fprintf(stderr, "Warning: object count mismatched in after_image.\n");
      ok = FALSE;
   }
   return ok;
}

static
void SelectObjects(top_obj, bot_obj)
   struct ObjRec *top_obj, *bot_obj;
{
   struct ObjRec *obj_ptr;

   for (obj_ptr=bot_obj; obj_ptr != NULL; obj_ptr=obj_ptr->prev) {
      obj_ptr->tmp_parent = NULL;
      AddObjIntoSel(obj_ptr, NULL, topSel, &topSel, &botSel);
   }
   UpdSelBBox();
}

static
struct CmdRec *DeserializeCmd(pszBuffer)
   char *pszBuffer;
{
   struct StkRec *stk_ptr=NULL;
   struct ObjRec *obj_ptr=NULL, *saved_top_obj=NULL, *saved_bot_obj=NULL;
   int read_status=TRUE, ok=TRUE;
   struct CmdRec *cmd_ptr=NULL;

   if (gstWBInfo.listen_fp == NULL) {
      sprintf(gszMsgBox, "Cannot open '%s'.\n\n%s pid=%ld.",
            gstWBInfo.listen_fname, "Deserialization aborted for",
            gstWBInfo.pid);
      MsgBox(gszMsgBox, TOOL_NAME, INFO_MB);
      return NULL;
   }
#ifdef _TGIF_WB2
#ifdef _TGIF_DBG_WB2
   if (wb2DebugLevel == 0) {
      fprintf(stderr, "Received %1d bytes, pid = %ld, timestamp = %1d/%s\n",
            strlen(pszBuffer), gstWBInfo.pid, recordCmdLogicalClock,
            recordCmdSenderProcID);
   } else {
      fprintf(stderr, "%s\nReceived %1d bytes, pid = %ld, timestamp = %1d/%s\n",
            pszBuffer, strlen(pszBuffer), gstWBInfo.pid, recordCmdLogicalClock,
            recordCmdSenderProcID);
   }
   PrintFullIDsOfObjects("In the beginning of DeserializeCmd()");
#endif /* _TGIF_DBG_WB2 */
#endif /* _TGIF_WB2 */

   rewind(gstWBInfo.listen_fp);
   writeFileFailed = FALSE;
   if (fprintf(gstWBInfo.listen_fp, "%s", pszBuffer) == EOF) {
      writeFileFailed = TRUE;
   }
   if (writeFileFailed) {
      sprintf(gszMsgBox, "Fail to write to '%s'.\n\nFile system may be full.",
            gstWBInfo.listen_fname);
      MsgBox(gszMsgBox, TOOL_NAME, INFO_MB);
      return NULL;
   }
   stk_ptr = SaveFileInfo();
   ResetFileInfo();

   recordCmdIncludeTgifObj = FALSE;
   recordCmdUsesNewColormap = FALSE;

   memset(&gstDeserializeCmd, 0, sizeof(gstDeserializeCmd));
   gstDeserializeCmd.type = INVALID;
   gppszBeforeImage = NULL;
   gpnAfterPositions = NULL;
   gnBeforeImageCount = gnAfterPositionsCount = 0;

   rewind(gstWBInfo.listen_fp);
   importingFile = TRUE;

   strcpy(scanFileName, gstWBInfo.listen_fname);
   scanLineNum = 0;
   readingPageNum = loadedCurPageNum = 0;
   foundGoodStateObject = FALSE;
   while ((read_status=ReadObj(gstWBInfo.listen_fp, &obj_ptr)) == TRUE) {
      if (obj_ptr != NULL) {
         obj_ptr->tmp_parent = NULL;
         AdjForOldVersion(obj_ptr);
         AddObj(NULL, topObj, obj_ptr);
      }
   }
   importingFile = FALSE;

   if (read_status == INVALID) {
      sprintf(gszMsgBox, "File version too large (=%1d).  %s!",
            fileVersion, "Deserialization aborted");
      MsgBox(gszMsgBox, TOOL_NAME, INFO_MB);
      ok = FALSE;
   } else {
#ifdef _TGIF_WB2
#ifdef _TGIF_DBG_WB2
      fprintf(stderr, "%s, pid = %ld, timestamp = %1d/%s\n",
            "Command successfully parsed", gstWBInfo.pid, recordCmdLogicalClock,
            recordCmdSenderProcID);
#endif /* _TGIF_DBG_WB2 */
#endif /* _TGIF_WB2 */
      if (recordCmdLogicalClock+1 > gstWBInfo.logical_clock) {
         /* Lamport's Algorithm set logical_clock */
         gstWBInfo.logical_clock = recordCmdLogicalClock+1;
#ifdef _TGIF_WB2
#ifdef _TGIF_DBG_WB2
         fprintf(stderr, "Logical clock for pid = %ld set to %1d.\n",
               gstWBInfo.pid, gstWBInfo.logical_clock);
#endif /* _TGIF_DBG_WB2 */
#endif /* _TGIF_WB2 */
      }
   }
   if (!ok) {
      DelAllObj();
   } else {
      saved_top_obj = topObj;
      saved_bot_obj = botObj;
      topObj = botObj = NULL;
   }
   RestoreFileInfo(stk_ptr);
   ResetOnePageSize();
   free(stk_ptr);

   if (ok) {
      cmd_ptr = (struct CmdRec *)malloc(sizeof(struct CmdRec));
      if (cmd_ptr == NULL) FailAllocMessage();
      memset(cmd_ptr, 0, sizeof(struct CmdRec));
      cmd_ptr->serialized = TRUE; /* created from deserialization */
      cmd_ptr->top_before = cmd_ptr->bot_before = NULL;
      cmd_ptr->pos_before = (int*)gppszBeforeImage;
      cmd_ptr->count_before = gnBeforeImageCount;
      cmd_ptr->type = gstDeserializeCmd.type;
      cmd_ptr->undone = TRUE;
      cmd_ptr->include_tgif_obj = recordCmdIncludeTgifObj;
      cmd_ptr->new_colormap = recordCmdUsesNewColormap;
#ifdef _TGIF_WB2
      cmd_ptr->logical_clock = recordCmdLogicalClock;
      if (*recordCmdSenderProcID != '\0') {
         cmd_ptr->sender_process_id = UtilStrDup(recordCmdSenderProcID);
         if (cmd_ptr->sender_process_id == NULL) FailAllocMessage();
      }
      cmd_ptr->first_redo_after_deserialize = TRUE;
      cmd_ptr->skipped = FALSE;
#ifdef _TGIF_DBG_WB2
      DebugDumpCmd("<<< In DeserializeCmd()", cmd_ptr->type,
            cmd_ptr->logical_clock, cmd_ptr->sender_process_id);
#endif /* _TGIF_DBG_WB2 */
#endif /* _TGIF_WB2 */
      cmd_ptr->dx = gstDeserializeCmd.dx;
      cmd_ptr->dy = gstDeserializeCmd.dy;

      if (saved_top_obj != NULL) {
         SelectObjects(saved_top_obj, saved_bot_obj);
         CopySel(topSel, numObjSelected,
               &(cmd_ptr->top_after), &(cmd_ptr->bot_after));
         cmd_ptr->pos_after = gpnAfterPositions;
         cmd_ptr->count_after = gnAfterPositionsCount;
         RemoveAllSel();
      } else {
         cmd_ptr->top_after = cmd_ptr->bot_after = NULL;
         cmd_ptr->pos_after = NULL;
         cmd_ptr->count_after = 0;
      }
      gppszBeforeImage = NULL;
      gnBeforeImageCount = 0;
      gpnAfterPositions = NULL;
      gnAfterPositionsCount = 0;
   }
   DeserializationCleanUp();

#ifdef _TGIF_WB2
#ifdef _TGIF_DBG_WB2
   PrintFullIDsOfObjects("At the end of DeserializeCmd()");
#endif /* _TGIF_DBG_WB2 */
#endif /* _TGIF_WB2 */

   return cmd_ptr;
}

static
int TotalOrderForTwo(older_cmd, newer_cmd)
   struct CmdRec *older_cmd, *newer_cmd;
   /*
    * Note: If logical clocks and host names are the same, and if pid in
    *       older_cmd is less than pid in newer_cmd, will return TRUE!
    */
{
   int comparison=0;

   if (older_cmd == NULL || newer_cmd == NULL) return TRUE;

   if (older_cmd->logical_clock > newer_cmd->logical_clock) {
      return FALSE;
   } else if (older_cmd->logical_clock < newer_cmd->logical_clock) {
      return TRUE;
   }
   comparison = strcmp(older_cmd->sender_process_id,
         newer_cmd->sender_process_id);
   TgAssert(comparison != 0,
         "Identical logical clock detected in TotalOrderForTwo()", NULL);
   return (comparison < 0);
}

#ifdef _TGIF_WB2
#ifdef _TGIF_DBG_WB2
static
void DumpCmdStack(prefix_string)
   char *prefix_string;
   /* Dump the command stack, last one first */
{
   struct CmdRec *cmd_ptr=NULL;

   if (prefix_string != NULL) fprintf(stderr, "%s\n", prefix_string);
   for (cmd_ptr=gstWBInfo.last_cmd; cmd_ptr != NULL; cmd_ptr=cmd_ptr->prev) {
      char buf[256];

      if (cmd_ptr->skipped) {
         sprintf(buf, "\tcmd_ptr->next = 0x%08lx, (SKIPPED)",
               (long)cmd_ptr->next);
      } else {
         sprintf(buf, "\tcmd_ptr->next = 0x%08lx", (long)cmd_ptr->next);
      }
      DebugDumpCmd(buf, cmd_ptr->type, cmd_ptr->logical_clock,
            cmd_ptr->sender_process_id);
   }
}
#endif /* _TGIF_DBG_WB2 */
#endif /* _TGIF_WB2 */

void FindCmdInsertionPoint(cmd_to_insert, pp_immed_right_cmd)
   struct CmdRec *cmd_to_insert, **pp_immed_right_cmd;
   /*
    * On return, cmd_to_insert is to be inserted immediately to the left
    *       of *pp_immed_right_cmd.
    */
{
   struct CmdRec *cmd_ptr=NULL;

   for (cmd_ptr=gstWBInfo.last_cmd; cmd_ptr != NULL; cmd_ptr=cmd_ptr->prev) {
      if (TotalOrderForTwo(cmd_ptr, cmd_to_insert)) {
         *pp_immed_right_cmd = cmd_ptr->next;
         return;
      }
   }
   *pp_immed_right_cmd = gstWBInfo.first_cmd;
}

static
void HighLightExistingObjects()
   /*
    * If select object no longer exists, don't select and highlight it!
    */
{
   int something_deleted=FALSE;
   struct SelRec *sel_ptr=NULL, *next_sel=NULL;

   for (sel_ptr=topSel; sel_ptr != NULL; sel_ptr=next_sel) {
      next_sel = sel_ptr->next;
      if (!IsTopLevelObject(sel_ptr->obj)) {
         UnlinkSel(sel_ptr, &topSel, &botSel);
         free(sel_ptr);
         something_deleted = TRUE;
      }
   }
   if (something_deleted) UpdSelBBox();
   HighLightForward();
}

void CleanUpObsoletedWBCmds(stopped_cmd_ptr)
   struct CmdRec *stopped_cmd_ptr;
{
   struct CmdRec *cmd_ptr=NULL;
   int num_records=0;

   for (cmd_ptr=stopped_cmd_ptr->prev; cmd_ptr != NULL; cmd_ptr=cmd_ptr->prev) {
      num_records++;
   }
   if (num_records > 0) {
      struct CmdRec *prev_cmd=NULL;
      double inc=(100.0/((double)num_records)), percent_start=0.0;

      ShowInterrupt(1);
      SaveStatusStrings();
      for (cmd_ptr=stopped_cmd_ptr->prev; cmd_ptr != NULL; cmd_ptr=prev_cmd,
            percent_start+=inc) {
         prev_cmd = cmd_ptr->prev;
         DeleteARedoRecord(cmd_ptr, percent_start,
               min(((double)percent_start+inc),((double)100.0)));
      }
      RestoreStatusStrings();
      HideInterrupt();
   }
   gstWBInfo.first_cmd = stopped_cmd_ptr;
   gstWBInfo.first_cmd->prev = NULL;
}

void HandleChatLine(cmd_ptr)
   struct CmdRec *cmd_ptr;
{
   /*
    * Handle the chat line...
    */
   DeleteARedoRecord(cmd_ptr, 0, 0);
   gstWBInfo.cur_cmd = NULL;
}

void ProcessRemoteCmd(psz_buf, buf_sz)
   char *psz_buf;
   int buf_sz;
{
   struct SelRec *saved_top_sel=topSel, *saved_bot_sel=botSel;
   struct CmdRec *cmd_ptr=NULL;
   int highlighted_before=FALSE;

   if (topSel != NULL) {
      highlighted_before = TRUE;
      HighLightReverse();
      topSel = botSel = NULL;
   }
   gstWBInfo.cur_cmd = DeserializeCmd(psz_buf);

   if (gstWBInfo.cur_cmd != NULL) {
      struct CmdRec *immed_right_cmd=NULL;
      int cur_cmd_is_wb_clearall=(gstWBInfo.cur_cmd->type == CMD_WB_CLEARALL);

      if (gstWBInfo.cur_cmd->type == CMD_CHAT_A_LINE) {
         HandleChatLine(gstWBInfo.cur_cmd);
      } else {
#ifdef _TGIF_WB2
#ifdef _TGIF_DBG_WB2
         DumpCmdStack("Before FindCmdInsertionPoint()");
#endif /* _TGIF_DBG_WB2 */
#endif /* _TGIF_WB2 */
         /*
          * FindCmdInsertionPoint() examine the logical clock of the new command
          *       and decides where to insert it.
          */
         FindCmdInsertionPoint(gstWBInfo.cur_cmd, &immed_right_cmd);

         if (immed_right_cmd == NULL) {
            /* append */
            InsertCmd(gstWBInfo.last_cmd, NULL, gstWBInfo.cur_cmd,
                  &gstWBInfo.first_cmd, &gstWBInfo.last_cmd);
         } else {
            /* insert */
            InsertCmd(immed_right_cmd->prev, immed_right_cmd, gstWBInfo.cur_cmd,
                  &gstWBInfo.first_cmd, &gstWBInfo.last_cmd);
         }
#ifdef _TGIF_WB2
#ifdef _TGIF_DBG_WB2
         DumpCmdStack("After FindCmdInsertionPoint()");
#endif /* _TGIF_DBG_WB2 */
#endif /* _TGIF_WB2 */
         if (immed_right_cmd != NULL) {
            /*
             * Undo all the commands to the right of gstWBInfo.cur_cmd.
             * Do not perform the undo if the command is skipped.
             */
            for (cmd_ptr=gstWBInfo.last_cmd; cmd_ptr != gstWBInfo.cur_cmd &&
                  cmd_ptr->type != CMD_WB_CLEARALL; cmd_ptr=cmd_ptr->prev) {
               if (cmd_ptr->skipped) {
#ifdef _TGIF_WB2
#ifdef _TGIF_DBG_WB2
                  strcpy(gszMsgBox, "*** No need to UndoACmd()");
                  DebugDumpCmd(gszMsgBox, cmd_ptr->type, cmd_ptr->logical_clock,
                        cmd_ptr->sender_process_id);
#endif /* _TGIF_DBG_WB2 */
#endif /* _TGIF_WB2 */
                  cmd_ptr->skipped = FALSE;
               } else {
#ifdef _TGIF_WB2
#ifdef _TGIF_DBG_WB2
                  strcpy(gszMsgBox, "*** Just before UndoACmd()");
                  DebugDumpCmd(gszMsgBox, cmd_ptr->type, cmd_ptr->logical_clock,
                        cmd_ptr->sender_process_id);
                  PrintFullIDsOfObjects("\t");
#endif /* _TGIF_DBG_WB2 */
#endif /* _TGIF_WB2 */
                  UndoACmd(cmd_ptr, FALSE);
#ifdef _TGIF_WB2
#ifdef _TGIF_DBG_WB2
                  DebugDumpCmd("*** After UndoACmd()",
                        cmd_ptr->type, cmd_ptr->logical_clock,
                        cmd_ptr->sender_process_id);
                  PrintFullIDsOfObjects("\t");
#endif /* _TGIF_DBG_WB2 */
#endif /* _TGIF_WB2 */
                  if (topSel != NULL) RemoveAllSel();
               }
            }
            if (cmd_ptr != gstWBInfo.cur_cmd) {
               gstWBInfo.cur_cmd = cmd_ptr;
            }
         }
         /*
          * Redo all the commands starting from gstWBInfo.cur_cmd.
          */
         if (gstWBInfo.cur_cmd->type == CMD_WB_CLEARALL) {
            CleanUpObsoletedWBCmds(gstWBInfo.cur_cmd);
            if (cur_cmd_is_wb_clearall) {
               CleanUpDrawingWindow();
               ClearFileInfo(TRUE);
               ClearAndRedrawDrawWindow();
               CheckFileAttrsInLoad();
               Msg("WhiteBoard cleared by peer.");
               RedrawTitleWindow();
               DelAllPages();
               lastPageNum = 1;
               InitPage();
               ShowPage();
            }
         }
         for (cmd_ptr=gstWBInfo.cur_cmd; cmd_ptr != NULL;
               cmd_ptr=cmd_ptr->next) {
#ifdef _TGIF_WB2
#ifdef _TGIF_DBG_WB2
            DebugDumpCmd("--- Just before RedoACmd()",
                  cmd_ptr->type, cmd_ptr->logical_clock,
                  cmd_ptr->sender_process_id);
            PrintFullIDsOfObjects("\t");
#endif /* _TGIF_DBG_WB2 */
#endif /* _TGIF_WB2 */
            if (!RedoACmd(cmd_ptr, FALSE)) {
#ifdef _TGIF_WB2
#ifdef _TGIF_DBG_WB2
               cmd_ptr->skipped = TRUE;
               DebugDumpCmd("=== Skip RedoACmd()",
                     cmd_ptr->type, cmd_ptr->logical_clock,
                     cmd_ptr->sender_process_id);
               PrintFullIDsOfObjects("\t");
#endif /* _TGIF_DBG_WB2 */
#endif /* _TGIF_WB2 */
            } else {
#ifdef _TGIF_WB2
#ifdef _TGIF_DBG_WB2
               DebugDumpCmd("--- Just after RedoACmd()",
                     cmd_ptr->type, cmd_ptr->logical_clock,
                     cmd_ptr->sender_process_id);
               PrintFullIDsOfObjects("\t");
#endif /* _TGIF_DBG_WB2 */
#endif /* _TGIF_WB2 */
               if (topSel != NULL) RemoveAllSel();
            }
         }
#ifdef _TGIF_WB2
#ifdef _TGIF_DBG_WB2
         PrintFullIDsOfObjects("At end of ProcessRemoteCmd()");
         fprintf(stderr, "%s%s\n",
               "========================================",
               "========================================");
#endif /* _TGIF_DBG_WB2 */
#endif /* _TGIF_WB2 */
      }
   }
   topSel = saved_top_sel;
   botSel = saved_bot_sel;
   UpdSelBBox();
   if (highlighted_before) {
      HighLightExistingObjects();
   }
}

/* =================== SIGPOLL Handler function =================== */

#ifdef _TGIF_WB2
static
void *ReceivePacket( void *arg )
{
    char *buf=NULL, *aux=NULL;
    char id[300];
    struct SegmentationPack pack1;
    char *buffer1;
    int byread;



    while( 1 )
    {
    
        if( (byread = RM_recv(gstWBInfo.MCastSock, (char*) &pack1, sizeof(struct SegmentationPack))) > 0 )
        {
        
#ifdef DEBUG0
            fprintf(stdout, "__________# of received packet: %u of %u\n", ntohl(pack1.NumSeq), ntohl(pack1.NumPackets));
#endif

            if( (buffer1 = DeSegment( pack1, &byread )) != NULL ){
                    buf = (char *)malloc( byread + 1 );
                    memcpy( buf, buffer1, byread );
	                buf[byread]= '\0';
                    free(buffer1);
#ifdef DEBUG0                    
                    fprintf(stdout, "******Desegmentation OK\n");
#endif                    
            }
            else
                continue;
        }else
             continue;

	    /*
         * verifica se a mensagem nao eh dele proprio
         */

        
		if (strncmp(buf, "From: ", 6) != 0 || sscanf(&buf[6], "%s\r\n", id) != 1) {
           fprintf(stderr, "%s.  %s, pid = %ld=tid=%s.\nbuf=%s.\n",
                 "Fatal error", "Cannot find the sender id in IO_SignalHandler()",
                 gstWBInfo.pid, id, buf);
           continue;
        }

        if(strcmp(id, gszLocalPID) == 0) {
           continue;
        }

        /* Verifica se pode processar o comando agora, ou se deve armazenar */
#ifdef DEBUG0        
        fprintf(stderr,"Process the comand now? ");
#endif        

        if ((aux = (char *)malloc( byread + 1 )) == NULL)
        {
         fprintf(stderr,"Allocation error. (ReceivePacket)\n");
         exit(1);
        }
        strncpy(aux, buf, byread);
        aux[byread] = (char)0;
        buff_ins(gstWBInfo.bd_commands, aux);

        if(gstWBInfo.BlockRemoteCmdDepth > 0) {
#ifdef DEBUG0        
           fprintf(stderr,"no!\n");
#endif           

#ifdef DEBUG           
           buff_show(gstWBInfo.bd_commands);
#endif           
           continue;
        }
        else
        {
            static char c='a';

            if (write(talkToSelfFiledes[1], &c, 1) != 1) {
               sprintf(gszMsgBox, TgLoadString(STID_WRITE_TO_SELF_PIPE_FAIL),
                     TOOL_NAME);
               fprintf(stderr, "%s\n", gszMsgBox);
            }
            if (c == 'g') {
               c = 'a';
            } else {
               c++;
            }
#ifdef DEBUG0                
            fprintf(stderr,"yes!\n");
#endif        
        }
    }
    return( NULL );
}
#endif /* _TGIF_WB2 */

/* =================== Init and CleanUp Functions =================== */

void CleanUpWhiteBoard()
{
#ifdef _TGIF_WB2
   buff_destroy(gstWBInfo.bd_commands);
#endif /* _TGIF_WB2 */
   CleanUpWBSockets();
   CleanUpWBCmds();
   memset(&gstWBInfo, 0, sizeof(struct WhiteBoardRec));
/* CleanUpChat(); */
}

int InitWhiteBoard()
{
#ifdef _TGIF_WB2
   char localhost_name[31];
   char pid[6];
   char logfilename[MAXPATHLENGTH];
   pthread_t tid;
#endif /* _TGIF_WB2 */

   InitLocalPID(); /* initializes gszLocalPID */

   memset(&gstWBInfo, 0, sizeof(struct WhiteBoardRec));

   gstWBInfo.BlockRemoteCmdDepth = 1; /* FIXME! deveria ser zero? */
   gstWBInfo.pid = (long)getpid();

#ifdef _TGIF_WB2
   gstWBInfo.do_whiteboard = TRUE;
#endif /* _TGIF_WB2 */
   if (cmdLineNoWhiteBoard) gstWBInfo.do_whiteboard = FALSE;

   *gstWBInfo.send_fname = '\0';
   gstWBInfo.send_fp = NULL;
   gstWBInfo.send_socket = (-1);

   gstWBInfo.listening = FALSE;

   *gstWBInfo.listen_fname = '\0';
   gstWBInfo.listen_fp = NULL;
   gstWBInfo.listen_socket = (-1);

#ifdef _TGIF_WB2
   if (gstWBInfo.do_whiteboard) {
      if (!CreateWBListenSocket()) {
         CleanUpWBSockets();
         exit(-1);
      }
      /* fim do trecho */

      sprintf(gstWBInfo.send_fname, "%sTgifSXXXXXX", TMP_DIR);
      mktemp(gstWBInfo.send_fname);
      unlink(gstWBInfo.send_fname);
      if ((gstWBInfo.send_fp=fopen(gstWBInfo.send_fname, "w+")) == NULL) {
         fprintf(stderr, "Fail to create '%s' for serialization.\n",
               gstWBInfo.send_fname);
         exit(-1);
      }
      /* Creation of multicast socket */
      
      /* sprintf(logfile, "%snome.txt", TMP_DIR); */
      
      gethostname(localhost_name, 30);

      strcpy(logfilename,"rmcastlog.");
      strcat(logfilename,localhost_name);
      strcat(logfilename,".");
      sprintf(pid,"%d",(int)getpid());
      strcat(logfilename,pid);

      RM_initialize(0, NULL/* logfilename */);
      
      gstWBInfo.MCastSock = RM_joinGroup("225.0.0.5", 5000);

      gstWBInfo.bd_commands = buff_init (10, WB_PACK, UNSORTED, NULL );
      
/* debugging RMCAST
      fprintf(stderr,"buffer descriptor: %d\n", gstWBInfo.bd_commands);
*/      
      if( pthread_create( &tid, NULL, ReceivePacket, NULL ) != 0 )
      {
          fprintf( stderr, "Could create receive thread\n" );
          exit( -1 );
      }
   }
#ifdef _TGIF_DBG_WB2
   {
      char *c_ptr=NULL;

      wb2DebugLevel = 0;
      if ((c_ptr=XGetDefault(mainDisplay, TOOL_NAME, "DebugWB2")) != NULL) {
         wb2DebugLevel = atoi(c_ptr);
      }
   }
#endif /* _TGIF_DBG_WB2 */
/* noChatWindow = FALSE;
   InitChat();
 */
#endif /* _TGIF_WB2 */

   return TRUE;
}

