/***************************************************************************
                             rminternals.c
                             -------------------
    begin                : May 2001
    copyright            : (C) 2001 by Jorge Allyson Azevedo
                                       Milena Scanferla
                                       Magnos Martinello
                                       Daniel Sadoc
    email                : {allyson,milena,magnos,sadoc}@land.ufrj.br
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
 
/************************************************************************************************
 *																								*
 * Reliable Multicast Protocol (rminternals.c)													*
 *																								*
 * Internal functions to send and receive message packets										*
 *																								*
 ************************************************************************************************/

#ifndef RMINTERNALS_C
#define RMINTERNALS_C

#define  LOSS_PROB 0.3

#include "rminternals.h"

#include <pthread.h>
#include <string.h>
#include <signal.h>
#include <errno.h>
#include <error.h>

#include <sys/time.h>
#include <unistd.h>
#include <stdio.h>
#include <time.h>

extern pthread_t rcv_thread, signal_handler_thread;
extern char *EVENT_ACTION_NAME[6];
extern char *MESSAGE_TYPE_NAME[9];

FILE *logfile;
EVENT_LIST *event_list;
CACHE      *cache;
USER_INFO  local_user_info;
sigset_t alrmset;

extern pthread_mutex_t  change_local_user_sn;
extern pthread_mutex_t  event_list_mutex;
extern pthread_mutex_t  cache_mutex;


/*********************** Main routines ************************************************************/

void lock_eventlist_and_cache()
{
    pthread_mutex_lock(&event_list_mutex);
    pthread_mutex_lock(&cache_mutex);    
}

void unlock_eventlist_and_cache()
{
    pthread_mutex_unlock(&cache_mutex);    
    pthread_mutex_unlock(&event_list_mutex);
}


/*****************************************************************************************************
 *
 * int	rmcastCreateSocket  (int port)
 *
 * Creates a multi-cast socket, using the specified port.  
 * 
 * Returns the socket handler.
 *
 ******************************************************************************************************/

int	rmcastCreateSocket  (int port)
{
	
	int aux = 1  ;
	struct sockaddr_in saddr;
	
	local_user_info.socket = socket( AF_INET, SOCK_DGRAM, IPPROTO_UDP);
	
	bzero( (char*) &saddr, sizeof(saddr) );
	
	saddr.sin_family = AF_INET;
	saddr.sin_port = htons(port);
	saddr.sin_addr.s_addr = htonl(INADDR_ANY);
	
	if ( setsockopt ( local_user_info.socket, SOL_SOCKET, SO_REUSEADDR, (char*)&aux, sizeof(aux)) == -1)
		perror( "reuse addr error" );
	
	if ( bind(local_user_info.socket, (struct sockaddr *)&saddr, sizeof(saddr)) == -1)
	{
		perror( "binding error" );
		close ( local_user_info.socket );
		local_user_info.socket = -1;
	}
	
	rmcastSetTTLValue ( local_user_info.socket, 1 );
	rmcastSetLoopBack ( local_user_info.socket, 1 );
	
	return ( local_user_info.socket );
}


void rmcastSetTTLValue (int soc, int ttl)
{
	if (setsockopt (soc, IPPROTO_IP, IP_MULTICAST_TTL, (char*)&ttl, sizeof (ttl)) ==-1)
		perror ("ttl");
}	

void rmcastSetLoopBack (int soc, int loop)
{
	if (setsockopt (soc, IPPROTO_IP, IP_MULTICAST_LOOP, (char*)&loop, sizeof (loop)) ==-1)
		perror ("loop");
}

/*****************************************************************************************************
 *
 * void rmcastJoinGroup  (int soc, char *group_addr)
 *
 * Joins the user using the socket soc to the multicast group.
 *
 * Arguments:   the socket, in soc;
 *              the multicast group ip, in group_addr
 *
 ******************************************************************************************************/


void rmcastJoinGroup  (int soc, char *group_addr)
{
	struct sockaddr_in group;
	struct ip_mreq mreq;
	
	printf( "Joining group %s\n", group_addr );
	if( (group.sin_addr.s_addr = inet_addr(group_addr)) == (unsigned)-1 )
		perror( "inet_addr " );
	
	mreq.imr_multiaddr	  = group.sin_addr;
	mreq.imr_interface.s_addr = INADDR_ANY;
	
	if( setsockopt( soc, IPPROTO_IP, IP_ADD_MEMBERSHIP, (char *)&mreq, sizeof(mreq)) == -1 )
		perror( "ADD_MEMBERSHIP" );

#ifdef REFRESH	
    eventListInsert(&event_list, NULL, REF_SND_WAIT, -1 );		
#endif
    
}

/*****************************************************************************************************
 *
 * void rmcastLeaveGroup  (int soc, char *group_addr)
 *
 * Used when the user using the socket soc wants to leave the multicast group.
 *
 * Arguments:   the socket, in soc;
 *              the multicast group ip, in group_addr
 *
 ******************************************************************************************************/

void rmcastLeaveGroup  (int soc, char *group_addr)
{
	struct sockaddr_in group;
	struct ip_mreq mreq;
	
	fprintf( stderr, "Disconnecting... %s ", group_addr );
	if( (group.sin_addr.s_addr = inet_addr(group_addr)) == (unsigned)-1 )
		perror( "leave inet_addr" );
	
	mreq.imr_multiaddr	  = group.sin_addr;
	mreq.imr_interface.s_addr = INADDR_ANY;
	
	if( setsockopt( soc, IPPROTO_IP, IP_DROP_MEMBERSHIP, (char *)&mreq, sizeof(mreq)) == -1 )
		perror( "DROP_MEMBERSHIP" );
#ifdef DEBUG2        
	fprintf(stderr, "ok (rmcastLeaveGroup)\n");
#endif    
	
}


void showLogMsg(char type, PACKET_INFO *pckt_info)
{
    struct timeval time_of_day;
    int sn=0;
    char buffer[200];
    char last_sn[16];
    
    struct tm time;    
    
    if (logfile!= NULL)
    
    {

        gettimeofday(&time_of_day, 0);    


        time = *gmtime((const time_t*)&time_of_day);

        time_of_day. tv_sec =     ( (time.tm_hour * 60 + time.tm_min) * 60 ) + time.tm_sec;


        if (pckt_info !=NULL)
        {
            switch (pckt_info->type)
            {
                case DATA_PACKET_TYPE:

                     sn = pckt_info->packet_data.data_packet.sn; 

                     sprintf(buffer,"%15s   %6d",pckt_info->sender_id.ip,pckt_info->sender_id.pid);

                     /*
                     strcpy(member_id,pckt_info->sender_id.ip);
                     strcat(member_id,"\t");
                     sprintf(pid,"%d",pckt_info->sender_id.pid);
                     strcat(member_id,pid);
                     */

                     break;

                case NAK_PACKET_TYPE:

                     sn = pckt_info->packet_data.nak_packet.sn; 

        /* FIXME - relativo ao agrupamento de naks           
                     sprintf(last_sn," -> %d",pckt_info->packet_data.nak_packet.last_sn);
        */

                     sprintf(buffer,"%15s   %6d   %15s   %6d",
                             pckt_info->sender_id.ip,
                             pckt_info->sender_id.pid,
                             pckt_info->packet_data.nak_packet.requested_member_id.ip,
                             pckt_info->packet_data.nak_packet.requested_member_id.pid);

                     /*
                     strcpy(member_id,pckt_info->packet_data.nak_packet.requested_member_id.ip);
                     strcat(member_id,"\t");
                     sprintf(pid,"%d",pckt_info->packet_data.nak_packet.requested_member_id.pid);
                     strcat(member_id,pid);
                     */

                     break;

                case RETRANSM_PACKET_TYPE:

                     sn = pckt_info->packet_data.retransm_packet.data_packet.sn; 

                     sprintf(buffer,"%15s   %6d   %15s   %6d",
                             pckt_info->sender_id.ip,
                             pckt_info->sender_id.pid,
                             pckt_info->packet_data.retransm_packet.original_sender_id.ip,
                             pckt_info->packet_data.retransm_packet.original_sender_id.pid);

                     /*
                     strcpy(member_id,pckt_info->packet_data.retransm_packet.original_sender_id.ip);
                     strcat(member_id,"\t");
                     sprintf(pid,"%d",pckt_info->packet_data.retransm_packet.original_sender_id.pid);
                     strcat(member_id,pid);
                     */

                     break;                

        #ifdef REFRESH        

                case REFRESH_PACKET_TYPE:

                     sn = pckt_info->packet_data.refresh_packet.sn_of_last_msg_sent; 

                     sprintf(buffer,"%15s   %6d",pckt_info->sender_id.ip,pckt_info->sender_id.pid);

                     /*
                     strcpy(member_id,pckt_info->sender_id.ip);
                     strcat(member_id,"\t");
                     sprintf(pid,"%d",pckt_info->sender_id.pid);
                     strcat(member_id,pid);
                     */

                     break;
        #endif

                default:
                     fprintf(stderr,"Unknown packet type: %c\n",pckt_info->type);                     
                     exit(1);
                     break;
            }

            fprintf(logfile,"%.0Lf   %c   %s   %20s   %3d%s\n", 
                ( (((long double)time_of_day.tv_sec)*1000000+(long double)time_of_day.tv_usec) ) ,
                type,
                MESSAGE_TYPE_NAME[(int)pckt_info->type],
                buffer, sn, 
                ((type == NAK_PACKET_TYPE) ? last_sn:""));
        }
        else
        {
           fprintf(logfile,"%.0Lf     %c\n", 
                    ( (((long double)time_of_day.tv_sec)*1000000+(long double)time_of_day.tv_usec) ) ,
                    type);
        }

    }            
}

/*****************************************************************************************************
 *
 * void* rmcastReceivePackets(void *user_soc)
 *
 * Receives packets from socket soc.
 *
 * Arguments:   the socket number, in soc;
 *
 * Side effects:  may change the state of the event list or cache.
 *
 ******************************************************************************************************/

void *rmcastReceivePackets(void *user_soc)
{
	int soc = (int)user_soc;
	struct sockaddr_in sender;
#ifdef LOSS_SIM    
    static int first = 1, first_pckts_lost = 50;
#endif    
	int	 len, sn;
	int	 nbytes; /* number of bytes received */
	char	 message[MAX_BUFFER];
	PACKET_INFO pckt_info;
//	CACHE *cache_sender_entry;
	MEMBER_ID aux_member_id;
	
   
	do {
		
		bzero(message, MAX_BUFFER);
#ifdef DEBUG0        
		fprintf(stderr,"Waiting to receive a message...\n");
#endif        
		len  = sizeof(struct sockaddr_in);
		nbytes = (int)recvfrom( soc, message, MAX_BUFFER, 0, (struct sockaddr *) &sender, (socklen_t*)&len);
#ifdef DEBUG0
		fprintf(stderr,"A message was received from the network! (message size: %d)! ",nbytes);
#endif
        if (nbytes <= 0)
        {
            fprintf(stderr,"error while receiving a message from the net. ");
            fprintf(stderr,"(%d) \n",nbytes);
        }

		
		if (nbytes>0)
		{
			msgPcktUnmountMessage(&pckt_info, (BYTE*)message);

            if (memberIsEqual(&(pckt_info.sender_id),&(local_user_info.member_id)))
            {    
      	        showLogMsg(RECEIVE,&pckt_info);
#ifdef DONT_SEND_NAK_AGAIN_IMMEDIATLY
                if ( pckt_info.type  == NAK_PACKET_TYPE )
                {
                    cacheUpdateNakListNakWasReceived(&cache, &(pckt_info.packet_data.nak_packet.requested_member_id), (pckt_info.packet_data.nak_packet.sn));
                }
#endif                
                continue;
            }
#ifdef DEBUG0            
            fprintf(stderr,"   message type: %s ",MESSAGE_TYPE_NAME[(int)(pckt_info.type)]);
#endif            

            switch (pckt_info.type)
            {
                case DATA_PACKET_TYPE:
                     sn = pckt_info.packet_data.data_packet.sn; break;
                case NAK_PACKET_TYPE:
                     sn = pckt_info.packet_data.nak_packet.sn; break;
                case RETRANSM_PACKET_TYPE:
                     sn = pckt_info.packet_data.retransm_packet.data_packet.sn; break;                
                case REFRESH_PACKET_TYPE:
                     sn = pckt_info.packet_data.refresh_packet.sn_of_last_msg_sent; break;
                default:
                     fprintf(stderr,"   sn          : %d \n",sn);                     
                     exit(1);
                     break;
                     
            }
#ifdef DEBUG0            
            fprintf(stderr,"   sn          : %d \n",sn);
#endif
            
#ifdef DEBUG
			msgPcktShowMessage((BYTE *)message);
#endif
            
           
            /* ------------------------ Loss Simulation ----------------------------------------- */
            
#ifdef LOSS_SIM            

//          if (pckt_info.type != NAK_PACKET_TYPE && pckt_info.type != RETRANSM_PACKET_TYPE)
            {
                if (generate_uniform() < LOSS_PROB || ((pckt_info.type == DATA_PACKET_TYPE) && (sn == 0) && (first == 1)) 
#ifdef LOSE_FIRST_50                
                     || (first_pckts_lost > 0)
#endif                     
                     ) 
                     
                {
#ifdef DEBUG0                
                    fprintf(stderr,"*\n*\nThis packet was lost. (rmcastReceivePackets)\n*\n*\n");
#endif                    
                    if (logfile != NULL)
                    {
      	                showLogMsg(RECEIVE_AND_LOSE,&pckt_info);
                    }

                    if (pckt_info.type == DATA_PACKET_TYPE)
                        
                        first = 0;  
                                          
                    if (first_pckts_lost>0)
                    
                        first_pckts_lost--;
                    
                    /* msgPcktShowMessage((BYTE*)message); */
                    continue;
                }
            }
#endif            

            /* --------------------------------------------------------------------------------- */
            

            lock_eventlist_and_cache();
            
            showLogMsg('K', NULL);

            if (logfile != NULL)
            {
      	        showLogMsg(RECEIVE,&pckt_info);
            }
            
            switch (pckt_info.type)
			{
				
			case NAK_PACKET_TYPE:
#ifdef DEBUG0            
                fprintf(stderr, "a NAK was received!\n");
#endif				
				if (rmcastRemoveEvent(&pckt_info,NAK_SND_WAIT))
                {
#ifdef DEBUG0
                    fprintf(stderr, "a NAK was received. inserting a RET_RCV_WAIT.\n");
#endif                    
                    rmcastInsertEvent(&pckt_info,RET_RCV_WAIT);
			    }
            	else if (!rmcastFindEvent(&pckt_info,RET_SND_WAIT))
				{
					if (rmcastCacheContains(&pckt_info))
					{
#ifdef DEBUG0                                
                        fprintf(stderr, "a NAK was received. inserting a RET_SND_WAIT.\n");
#endif                        
						rmcastInsertEvent(&pckt_info,RET_SND_WAIT);
					}


/*
 *  					else
 *  					{
 *  						if (!rmcastFindEvent(&pckt_info,RET_RCV_WAIT))
 *  						{
 *  							cache_sender_entry = cacheLookForMember(&cache,
 *  								&(pckt_info.packet_data.nak_packet.requested_member_id));
 *  
 *  							if (cache_sender_entry->last->data_pckt.sn < pckt_info.packet_data.nak_packet.sn)
 * 
 *  								rmcastInsertEvent(&pckt_info, RET_RCV_WAIT);
 * 						    }
 *                      }
 *  					
 * Esse codigo foi retirado em 25/09/2001 por Milena e Daniel para otimizacao do programa
 */
                    


				}	  
				
				break;
				
				
			case SM_PACKET_TYPE:
				
				/* FIXME - comparar cada campo de cada linha da "sm table" com a cache 
				- se a cache possuir valores menores entao pacotes foram perdidos,
				- se nao existir t1 ou t2 na lista entao incluir t1.
				*/
				
				break;
				
			case LEAVE_GROUP_PACKET_TYPE:
				
				cacheDesactivateMember(cache,&(pckt_info.sender_id));
				/* FIXME- o usuario que enviou a leave msg deve indicar tb o sn do ultimo 
				pacote de dados enviado por ele, para que o receptor possa verificar
				o recebimento deste ultimo pacote */
				
				break;
				
				
			case JOIN_GROUP_PACKET_TYPE: 
				
				/* FIXME O usuario q deseja se juntar ao grupo envia esta msg. Deve ser 
				disparado um timer para enviar uma msg JOIN_ACK q vai permitir que o 
				usuario q deseja entrar abra uma conexao unicast com o membro q enviou
				o JOIN_ACK. */
				
				break;
				
			case JOIN_ACK_PACKET_TYPE:
				
				/* FIXME Se existir um evento JOIN na lista de eventos, abrir uma conexao 
				unicast com o usuario q enviou o JOIN_ACK para copiar o estado atual 
				daquele usuario. */
				
				break;
				
			case REFRESH_PACKET_TYPE:
#ifdef REFRESH				
				rmcastProcessRefreshPacket(pckt_info);			            	
#endif				
				break;

			case RETRANSM_PACKET_TYPE:
            
                rmcastRemoveEvent(&pckt_info,RET_SND_WAIT);
				
				if (!rmcastRemoveEvent(&pckt_info,NAK_SND_WAIT) &&
                    !rmcastRemoveEvent(&pckt_info,RET_RCV_WAIT))
                {   
                    if (rmcastCacheContains(&pckt_info))
                        break;
                }
                
				/* Now we will get the content of the retransmition and put it in the cache. */

				strcpy(aux_member_id.ip, pckt_info.packet_data.retransm_packet.original_sender_id.ip);
				aux_member_id.pid=pckt_info.packet_data.retransm_packet.original_sender_id.pid;
				pckt_info.packet_data.data_packet = pckt_info.packet_data.retransm_packet.data_packet;
                pckt_info.sender_id = aux_member_id;
                
                /* continue... */
				
			case DATA_PACKET_TYPE: 
				
                rmcastProcessDataPacket(pckt_info); /* inserts data in cache... */
                
        		break;

			default:
            
				fprintf(stderr,"Invalid message type. (%d)\n",pckt_info.type);
				exit(1);
                
				break;
			}
            showLogMsg('U', NULL);                                                
            unlock_eventlist_and_cache();
		}
#ifdef DEBUG		
		fprintf(stderr,"received, ok!\n");		
#endif

    } FOREVER;
    
    return NULL;
}



/*****************************************************************************************************
 *
 * void  rmcastSendPacket(int soc, BYTE *message, int message_size)
 *
 * Sends a message packet of size message_size to socket soc.
 *
 ******************************************************************************************************/

void	rmcastSendPacket(int soc, BYTE *message, int message_size)
{
    PACKET_INFO pckt_info;
	struct sockaddr_in dest;
    int sn;
#ifdef DEBUG0    
    struct in_addr     inaddr;
#endif        
#ifdef SHOW_MSG
    int i;
#endif
//  int j=0;
	int retval;	
    char type = *((char*)message);

    char local_message[message_size+1];

#ifdef SHOW_MSG
    fprintf(stderr,"will copy a message of size %d to local var. ",message_size);

    for (i=0; i< message_size; i++)
    {
        fprintf(stderr,"(%d)%d",i,(unsigned int)message[i]);
    }
    fprintf(stderr,"\n");
 
#endif        
    memcpy(local_message, message, message_size);
    
	dest.sin_family = AF_INET;

	
	dest.sin_port        = htons    ( local_user_info.group_id.port );
    
    
	dest.sin_addr.s_addr = inet_addr( local_user_info.group_id.ip );

	
	if(type == DATA_PACKET_TYPE ) 
	{
      	pthread_mutex_lock(&change_local_user_sn);
            
		local_user_info.sn++;
   
      	pthread_mutex_unlock(&change_local_user_sn);
        
	}
    
    msgPcktUnmountMessage(&pckt_info, local_message);
    
    switch (pckt_info.type)
       {
           case DATA_PACKET_TYPE:
                sn = pckt_info.packet_data.data_packet.sn; break;
           case NAK_PACKET_TYPE:
                sn = pckt_info.packet_data.nak_packet.sn; break;
           case RETRANSM_PACKET_TYPE:
                sn = pckt_info.packet_data.retransm_packet.data_packet.sn; break;
           case REFRESH_PACKET_TYPE:
                sn = pckt_info.packet_data.refresh_packet.sn_of_last_msg_sent; break;               

       }
#ifdef DEBUG0
    fprintf(stdout,"will send a %s - sn: %d\n", MESSAGE_TYPE_NAME[(int)type],sn);
#endif    
    if (logfile != NULL)
    {
      	showLogMsg(SEND,&pckt_info);
    }

	retval = sendto( soc, local_message, message_size, 0, (struct sockaddr *)&dest, sizeof(dest) );

#ifdef USLEEP_SENDTO
    
    usleep(10L);
    
#endif    

#ifdef DEBUG0     
	if( retval < 0 )
	{
   	    fprintf(stderr,"Error! retval < 0\n");
  		perror( "send_message" );

 	    fprintf(stderr,"port: %d\n",ntohs(dest.sin_port));
        inaddr.s_addr = (unsigned long int)dest.sin_addr.s_addr;
        
        fprintf(stderr,"ip  : %s\n",inet_ntoa(inaddr));
        msgPcktShowMessage(message);
        fprintf(stderr,"exiting.\n");
        exit(1);
	}
#endif            
}



/*****************************************************************************************************
 *
 *  void *rmcastHandleSignals(void *arg)
 *
 *  This thread will be the only one that threats the alarm events. 
 *
 *  Communication with it should be done using SIGUSR1. 
 *
 *****************************************************************************************************/
 
extern const char * const sys_siglist[];  

void *rmcastHandleSignals(void *arg)
{
    int sig;
    int err;

	struct itimerval value;   
#ifdef DEBUG0      
    fprintf (stderr,"(handle_alrm)\n");    
#endif    
    
    while (1)
    {
        err = sigwait(&alrmset, &sig);
#ifdef DEBUG0      
        fprintf (stderr,"signal %s (%d) received (rmcastHandleSignals)\n",sys_siglist[sig], sig);
#endif        
        if (err || (sig!=SIGALRM && sig!=SIGUSR1))
        {
            fprintf(stderr,"err: %d\n",err);
            if (sig==SIGINT)
                exit(1);
            else
                continue; 
        }
        lock_eventlist_and_cache();   
        
        showLogMsg('k', NULL);                                 
        
        if (sig==SIGALRM || (event_list!=NULL && event_list->timer_value == 0))
        {
            rmcastCatchEvents(0);    
        }

#ifdef DEBUG0       
        fprintf (stderr,"(rmcastHandleSignals)\n");
#endif        
        eventListShow(event_list);

        if (event_list!=NULL)
        {
#ifdef DEBUG0               
           fprintf (stderr,"setting timer to %ld (rmcastHandleSignals)\n",event_list->timer_value);                   
#endif           
           setTimerValue(&value,event_list->timer_value);
           setitimer(ITIMER_REAL,&value,NULL);
        }    
        else
        {
#ifdef DEBUG0                       
           fprintf (stderr,"resetting timer (rmcastHandleSignals)\n");                               
#endif           
           setTimerValue(&value,0);
           setitimer(ITIMER_REAL,&value,NULL);
        }
        showLogMsg('u', NULL);                    
        unlock_eventlist_and_cache();        
    }
    return (NULL);
}





/*****************************************************************************************************
 *
 * void  rmcastCatchEvents   (int i)
 *
 * Catch and execute event list events.
 *
 ******************************************************************************************************/


void	rmcastCatchEvents   (int i)
{
	
	CACHE *cache_sender_entry;
	

    int retval, update_nak_list_retval;
	EVENT_LIST first;
	int msgsize;
	BYTE *message;
	PACKET_INFO pack;
    
	DATA_PACKET *data;
    
#ifdef DEBUG0               
    fprintf(stderr,"Event catched!\n");
#endif
    if (event_list == NULL)
    {
#ifdef DEBUG0                   
            fprintf(stderr,"Internal warning! first == NULL. (rmcastCatchEvents)\n");
#endif            
            return;
    }
    
	first = *event_list;
    
	eventListShow(event_list);
    
#ifdef DEBUG0                   
    fprintf(stderr,"event list: %p\n",&(*event_list));
#endif    
	
	do
	{
#ifdef DEBUG0                   
		fprintf(stderr,"*** Removing event: action %s\n\ttimer value: %ld\n",
            EVENT_ACTION_NAME[(int)first.action],
            first.timer_value);
#endif        
        if (eventListRemoveFirst(&event_list) == -1)
        {
#ifdef DEBUG0        
            fprintf(stderr,"Internal warning. (rmcastCatchEvents)\n");
#endif
        }
        
        
        eventListShow(event_list);
        
		switch (first.action)
		{	  
        
		case NAK_SND_WAIT:
			
			pack.type=NAK_PACKET_TYPE;
			pack.sender_id = local_user_info.member_id;
			pack.packet_size=sizeof(NAK_PACKET); 
			strcpy(pack.packet_data.nak_packet.requested_member_id.ip,first.member_id->ip);
			pack.packet_data.nak_packet.requested_member_id.pid=first.member_id->pid;
			pack.packet_data.nak_packet.sn=first.sn;
			msgPcktMountMessage(&pack, &message, &msgsize);
#ifdef DEBUG0            
            fprintf(stderr,"will send nak ");
#endif            
			rmcastSendPacket(local_user_info.socket, message, msgsize);
#ifdef DEBUG0                           
            fprintf(stderr,"ok (rmcastCatchEvents)\n");
#endif            
			
			cache_sender_entry = cacheLookForMember(&cache,first.member_id);
			retval = eventListInsert(&event_list,&(cache_sender_entry->sm_info.member_id), RET_RCV_WAIT, first.sn);
#ifdef DEBUG0                           
			fprintf(stderr,"RET_RCV_WAIT was inserted in the event list.\n");
#endif            
			break;
			
		case RET_RCV_WAIT:
			
			/* Next we will check if it is possible to send one more NAK for the lost message,
			i.e., the NAK should be sent only MAX_NAK times */
#ifdef DEBUG0                           			
			fprintf(stderr," \nWaiting for retransmission timed out\n ip: %s - pid: %d - sn: %d\n",
				first.member_id->ip, first.member_id->pid, first.sn);
#endif                
			
			if ((update_nak_list_retval = cacheUpdateNakList(&cache, first.member_id, first.sn)) > 0)
			{
				cache_sender_entry = cacheLookForMember(&cache,first.member_id);
				retval = eventListInsert(&event_list,&(cache_sender_entry->sm_info.member_id), NAK_SND_WAIT, first.sn);
#ifdef DEBUG0                
				fprintf(stderr,"NAK_SND_WAIT was inserted in the event list.\n");
#endif                
			} else if (update_nak_list_retval == 0)
			{
				fprintf(stderr,"Error inserting NAK node or max number of NAKs reached!\n"); 
				
				/* FIXME Provavelmente o usuario devera sair do grupo pois nao
				conseguiu recuperar uma msg. */
                
                exit(1);
                
			} else if (update_nak_list_retval == -1)
            {
                cache_sender_entry = cacheLookForMember(&cache,first.member_id);
				retval = eventListInsert(&event_list,&(cache_sender_entry->sm_info.member_id), RET_RCV_WAIT, first.sn);
            }
			break;
			
		case RET_SND_WAIT:
			
			if((data=cacheLookForMessage(&cache, first.member_id, first.sn))!=NULL)
			{
				pack.type=RETRANSM_PACKET_TYPE;
				pack.sender_id = local_user_info.member_id;
				pack.packet_size=sizeof(RETRANSM_PACKET);
				pack.packet_data.retransm_packet.original_sender_id=*first.member_id;
				pack.packet_data.retransm_packet.data_packet=*data;
				msgPcktMountMessage(&pack, &message, &msgsize);
#ifdef DEBUG0                
                fprintf(stderr,"will send retr ");
#endif                
				rmcastSendPacket(local_user_info.socket, message, msgsize);
#ifdef DEBUG0                                           
                fprintf(stderr,"ok (rmcastCatchEvents)\n");                
#endif                
			}
#ifdef DEBUG0                           			
			fprintf(stderr," \nMessage retransmited\n ip: %s - pid: %d - sn: %d\n",
				first.member_id->ip, first.member_id->pid, first.sn);
#endif                
			
			break;
            
#ifdef REFRESH                

        case REF_SND_WAIT:
#ifdef DEBUG0        
            fprintf(stderr,"Processing the refresh event.\n");
#endif            

            pack.type = REFRESH_PACKET_TYPE;
            pack.sender_id = local_user_info.member_id;
            pack.packet_size = sizeof(REFRESH_PACKET);
      		
            pthread_mutex_lock(&change_local_user_sn);
            pack.packet_data.refresh_packet.sn_of_last_msg_sent = local_user_info.sn - 1;
       		pthread_mutex_unlock(&change_local_user_sn);
#ifdef DEBUG0       
            fprintf(stderr,"Mounting refresh packet.\n");
#endif                 
            msgPcktMountMessage(&pack, &message, &msgsize);
			rmcastSendPacket(local_user_info.socket, message, msgsize);            
#ifdef DEBUG0
            fprintf(stderr,"refresh packet mounted.\n");            
#endif
		   	eventListInsert(&event_list, NULL, REF_SND_WAIT, -1 );    
#ifdef DEBUG0            
            fprintf(stderr,"Refresh event processed.\n");
#endif           
            break;
        
#endif
			
		default:
			fprintf(stderr,"Error: Action Undefined %d\n",first.action);
		}

        
        
  		if (event_list == NULL) 
        {
            break;
        }
		
        first = *event_list;
        
  	
	} while (first.timer_value == 0);
	
	eventListShow(event_list);


	return;
}

/*********************** End of the Main routines *************************************************/




/**************** Routines to facilitate the interface with the event list and the cache **********/



/*****************************************************************************************************
 *
 * void rmcastProcessDataPacket(PACKET_INFO pckt_info)
 *
 * Processes the data packet.
 *
 * Arguments: the packet to be processed, in pckt_info;
 *
 * Side effects: may affect the event list, the cache and the message queue.
 *
 ******************************************************************************************************/

void rmcastProcessDataPacket(PACKET_INFO pckt_info)
{
    int i, retval;
    
    
  	MEMBER_ID *aux_member_id = &(pckt_info.sender_id);
    
  	CACHE_NODE *aux_pckt_info;
    
  	CACHE *cache_sender_entry;
    
    char *auxbuff;
    
    
	if( pckt_info.type == DATA_PACKET_TYPE )
	{
		/* Delete all events about this packet */

		if (!rmcastRemoveEvent(&pckt_info,NAK_SND_WAIT))
		{
			if (!rmcastRemoveEvent(&pckt_info,RET_RCV_WAIT))
			{
				rmcastRemoveEvent(&pckt_info,RET_SND_WAIT);
			}
		} 
	}
    
    pckt_info.type = DATA_PACKET_TYPE;
#ifdef DEBUG
	fprintf(stderr,"Inserting data in cache: ip: %s pid: %d\n",aux_member_id->ip,aux_member_id->pid);
#endif    
    
    
	retval = cacheInsertMessage(&cache, &(pckt_info.sender_id), &(pckt_info.packet_data.data_packet));
    
    cacheShow(cache);
    
    
	cache_sender_entry = cacheLookForMember(&cache, aux_member_id);
   
    
	aux_pckt_info = cache_sender_entry->last_inserted;
    

	if (retval==1) /* the inserted packet is in sequence */
	{
#ifdef DEBUG0
		fprintf(stderr,"The inserted packet is in sequence...\n");	  
#endif        


		for (i=cache_sender_entry->sm_info.member_status.last_rcv; 
			i<=cache_sender_entry->sm_info.member_status.last_seq_rcv; 
			i++)
		{
			// pthread_mutex_lock(&remove_queues_lock);
#ifdef DEBUG0
			fprintf(stderr,"A \"reliable\" message will be enqueued: (size: %d)\n",
                pckt_info.packet_data.data_packet.data_size);
#endif                
            auxbuff = (char*)aux_pckt_info->data_pckt.data;
             
    		messageQueueEnter((char*)(aux_pckt_info->data_pckt.data),pckt_info.packet_data.data_packet.data_size);
			aux_pckt_info = aux_pckt_info->previous;

		}
        
        if (cache_sender_entry->sm_info.member_status.last_received_or_nak < cache_sender_entry->sm_info.member_status.last_rcv)
        {
            cache_sender_entry->sm_info.member_status.last_received_or_nak = cache_sender_entry->sm_info.member_status.last_rcv;
        }
	}
	else if (retval==2) /* the inserted packet is not in sequence */
	{
#ifdef DEBUG0    
		fprintf(stderr,"The inserted packet is NOT in sequence... ");	
#endif
		for (i=cache_sender_entry-> sm_info.member_status.last_received_or_nak+1;
            i<cache_sender_entry->sm_info.member_status.last_rcv;
		   	i++) 			
		{
            pckt_info.packet_data.data_packet.sn = i;
#ifdef DEBUG0            
            fprintf(stderr," (sn: %d) ", i);
#endif            

			if (!rmcastFindEvent(&pckt_info,NAK_SND_WAIT) && !rmcastFindEvent(&pckt_info, RET_RCV_WAIT))
			{
				if (cacheLookForMessage(&cache, aux_member_id, i)==NULL)
				{
					if (cacheUpdateNakList(&cache, aux_member_id, i)!=0)
					{
						rmcastInsertEvent(&pckt_info,NAK_SND_WAIT); 
#ifdef DEBUG0
						fprintf(stderr,"NAK_SND_WAIT was inserted in the event list. ");
#endif                        
					}
					else
					{
#ifdef DEBUG0                    
						fprintf(stderr,"Error inserting NAK node or max number of NAKs reached! "); 
#endif                        

						/* FIXME Provavelmente o usuario devera sair do grupo pois nao
						conseguiu recuperar uma msg. */
					}
				}
			}
		}
        
        if (cache_sender_entry->sm_info.member_status.last_received_or_nak < cache_sender_entry->sm_info.member_status.last_rcv)
        {
            cache_sender_entry->sm_info.member_status.last_received_or_nak = cache_sender_entry->sm_info.member_status.last_rcv;
        }
#ifdef DEBUG0        
        fprintf(stderr,"ok\n");
#endif        
	}
	else 
	{
#ifdef DEBUG0            
		fprintf(stderr,"Couldnt insert cache node.\n");
#endif        
	}
    
			
}

/*****************************************************************************************************
 *
 * void rmcastProcessRefreshPacket(PACKET_INFO pckt_info)
 *
 * Processes the refresh packet.
 *
 * Arguments: the packet to be processed, in pckt_info;
 *
 * Side effects: may affect the event list, the cache and the message queue.
 *
 ******************************************************************************************************/

#ifdef REFRESH

void rmcastProcessRefreshPacket(PACKET_INFO pckt_info)
{
	int i, new_value_of_last_received_or_nak;
	CACHE *cache_sender_entry;
	cache_sender_entry = cacheLookForMember(&cache, &(pckt_info.sender_id));
    
    if (pckt_info.packet_data.refresh_packet.sn_of_last_msg_sent < 0)
        return;
    
    if (cache_sender_entry == NULL)
    {
        cache_sender_entry  = cacheInsertMember(&cache, &(pckt_info.sender_id));
    }
    
    new_value_of_last_received_or_nak = pckt_info.packet_data.refresh_packet.sn_of_last_msg_sent;

	for (i = pckt_info.packet_data.refresh_packet.sn_of_last_msg_sent; 
    	 i > cache_sender_entry->sm_info.member_status.last_received_or_nak;
         i --)
    {
#ifdef DEBUG0    
      	fprintf(stderr,"Will wait to send a nak, i: %d last received or nak: %d.\n", i, cache_sender_entry->sm_info.member_status.last_received_or_nak);
#endif        
    	pckt_info.packet_data.refresh_packet.sn_of_last_msg_sent = i;
		rmcastInsertEvent(&pckt_info,NAK_SND_WAIT);     	

   }
   cache_sender_entry->sm_info.member_status.last_received_or_nak = new_value_of_last_received_or_nak;  
}


#endif

/*****************************************************************************************************
 *
 * int rmcastRemoveEvent(PACKET_INFO *packt, int event_type)
 *
 * Removes an event from the event list.
 *
 * Arguments: the network packet, in packt;
 *            the event type, in event_type.
 *
 * Returns: 1, on sucess;
 *          0, otherwise.
 *
 * Side effects: it affects the event list.
 *
 ******************************************************************************************************/

int rmcastRemoveEvent(PACKET_INFO *packt, int event_type)
{
    int retval;
    
    switch (packt->type)
    {
        case    RETRANSM_PACKET_TYPE:

            retval = eventListRemove(&event_list, &(packt->packet_data.retransm_packet.original_sender_id),
                                     event_type, packt->packet_data.retransm_packet.data_packet.sn); 
            break;
        
        case    DATA_PACKET_TYPE:    
        
            retval = eventListRemove(&event_list, &(packt->sender_id), 
                                     event_type, packt->packet_data.data_packet.sn);
            break;
        
        case    NAK_PACKET_TYPE:
        
            retval = eventListRemove(&event_list, &(packt->packet_data.nak_packet.requested_member_id), 
                                     event_type, packt->packet_data.nak_packet.sn);   
            break;
    }
    
    return  retval;
}

/*****************************************************************************************************
 *
 * int rmcastFindEvent(PACKET_INFO *packt, int event_type)
 *
 * Searches for an event in the event list.
 *
 * Arguments: the network packet, in packt;
 *            the event type, in event_type.
 *
 * Returns: 1, on sucess;
 *          0, otherwise.
 *
 ******************************************************************************************************/

int rmcastFindEvent(PACKET_INFO *packt, int event_type)
{
    int retval;
    
	EVENT_LIST *aux;
    
    switch (packt->type)
    {
        case    NAK_PACKET_TYPE:
        
            retval = ((eventListFind(&event_list, &(packt->packet_data.nak_packet.requested_member_id),
                                     event_type, packt->packet_data.nak_packet.sn,&aux))!=NULL);
            break;
            
        case    DATA_PACKET_TYPE:   
        
            retval = ((eventListFind(&event_list, &(packt->sender_id),
                                     event_type, packt->packet_data.data_packet.sn,&aux))!=NULL);
            break;
            
        default:
        
            fprintf(stderr,"Not yet implemented. (rmcastFindEvent)\n");
            exit(1);
            break;
    }
    
    return retval;
}


/*****************************************************************************************************
 *
 * int rmcastCacheContains(PACKET_INFO *packt)
 *
 * Searches for a packet in the cache.
 *
 * Arguments: the network packet, in packt;
 *
 * Returns: 1, on sucess;
 *          0, otherwise.
 *
 ******************************************************************************************************/

int rmcastCacheContains(PACKET_INFO *packt)
{

    int retval;
    
    switch (packt->type)
    {
        case NAK_PACKET_TYPE:
        
            retval = ((cacheLookForMessage(&cache, &(packt->packet_data.nak_packet.requested_member_id), 
                                           packt->packet_data.nak_packet.sn))!=NULL);			
            break;

        case RETRANSM_PACKET_TYPE:
            retval = ((cacheLookForMessage(&cache, &(packt->packet_data.retransm_packet.original_sender_id),
                                           packt->packet_data.retransm_packet.data_packet.sn))!=NULL);
            break;
            
        default:
        
            fprintf(stderr,"Not yet implemented. (rmcastCacheContains)\n");
            exit(1);
            break;
    }
    
    return retval;        
}


/*****************************************************************************************************
 *
 * int rmcastInsertEvent(PACKET_INFO *packt, int event_type)
 *
 * Inserts an event in the event list.
 *
 * Arguments: the network packet, in packt;
 *            the event type, in event_type.
 *
 * Returns: 1, on sucess;
 *          0, otherwise.
 *
 * Side effects: it affects the event list.
 *
 ******************************************************************************************************/


int rmcastInsertEvent(PACKET_INFO *packt, int event_type)
{
    int retval;
    
  	CACHE *cache_sender_entry;

    switch (packt->type)
    {
        case NAK_PACKET_TYPE:
        
            cache_sender_entry = cacheLookForMember(&cache,
                                        &(packt->packet_data.nak_packet.requested_member_id));	
            retval = eventListInsert(&event_list, &(cache_sender_entry->sm_info.member_id), 
                                     event_type, (packt->packet_data.nak_packet.sn) );		
            break;

        case DATA_PACKET_TYPE:
    
    		cache_sender_entry = cacheLookForMember(&cache, &(packt->sender_id));
            retval = eventListInsert(&event_list,&(cache_sender_entry->sm_info.member_id), 
                                     event_type, packt->packet_data.data_packet.sn );
            break;
            
        case REFRESH_PACKET_TYPE:
        
    		cache_sender_entry = cacheLookForMember(&cache, &(packt->sender_id));
            
            if (cache_sender_entry!=NULL)
            {
                retval = eventListInsert(&event_list,&(cache_sender_entry->sm_info.member_id), 
                                     event_type, packt->packet_data.refresh_packet.sn_of_last_msg_sent );
            }
        
            break;
        
        
        default:
        
            fprintf(stderr,"Not yet implemented. (rmcastInsertEvent)\n");
            exit(1);
            break;
    }
    
    return retval;                
}

/**********End of the Routines to facilitate the interface with the event list and the cache ******/

double generate_uniform()
{
	double retval;
	retval = ((double)rand())/(RAND_MAX+1.0);
#ifdef DEBUG0    
	fprintf(stderr,"uniform var: %f\n",retval);
#endif    
	return (retval);
}

#endif

