/***************************************************************************
                             rmevent.c
                             -------------------
    begin                : May 2001
    copyright            : (C) 2001 by Jorge Allyson Azevedo
                                       Milena Scanferla
                                       Magnos Martinello
                                       Daniel Sadoc
    email                : {allyson,milena,magnos,sadoc}@land.ufrj.br
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
/******************************************************************************************************
 *																									  *
 * Reliable Multicast Protocol (rmevent.c)															  *
 *																									  *
 * Reliable Multicast event list functions. 														  *
 *																									  *
 * use tab spacing = 4																				  *
 *																									  *
 ******************************************************************************************************/
 
#ifndef RMEVENT_C
#define RMEVENT_C

#undef DEBUG
#undef DEBUG2

#include <signal.h>
#include <stdlib.h>
#include <sys/time.h>
#include <unistd.h>
#include <error.h>
#include <errno.h>
#include <pthread.h>

#include "rmstruct.h"
#include "rmevent.h"

/*---------------- Global variables ------------------------------------------------------------------*/

extern pthread_t rcv_thread, signal_handler_thread;

static long DEFAULT_TIMER_VALUE [6] = {     0,
                                            1000000,
                                            2000000,
                                            1000000,
                                            10000000,
                                            0 };
                                            
char *EVENT_ACTION_NAME[6] =          { "", 
                                        "NAK_SND_WAIT", 
                                        "RET_RECV_WAIT", 
                                        "RET_SND_WAIT",
                                        "REF_SND_WAIT",
                                        "UNKNOWN"};

pthread_mutex_t   event_list_mutex = PTHREAD_MUTEX_INITIALIZER;

inline void getRemainingTime(EVENT_LIST **el, long double*remaining_time, char *msg);

EVENT_LIST * eventListFind2(EVENT_LIST **el1,MEMBER_ID *member_id, char action, int sn, EVENT_LIST **antNode);

/*---------------- Auxiliary functions to manipulate the timer ---------------------------------------*/


/***************************************************************************************
 *
 * float generateSample(char distribution)
 *
 * Generates a random sample.
 *
 * Arguments:	distribution, the distribution of the random var (not yet implemented).
 *
 * Returns: 	a random number between 0 and 1.
 *
 ***************************************************************************************/

float generateSample(char distribution)
{
    return ((rand()/(RAND_MAX+1.0)));
}

/***************************************************************************************
 *
 * int generateTimerValue(char action, char distribution)
 *
 * Generates a timer value corresponding to the defined action.  This value indicates
 * in how many microseconds the action will be executed.
 *
 * Arguments:	action, the action to be executed.
 *				distribution, the distribution of the random var (not yet implemented).
 *
 * Returns: 	the timer value, in microseconds.
 *
 ***************************************************************************************/


int generateTimerValue(char action, char distribution)
{
    int retval;
    retval = DEFAULT_TIMER_VALUE[(int)action] /* + (int)generateSample(distribution)*/;    
    return (retval);
}

/***************************************************************************************
 *
 * inline void getRemainingTime(EVENT_LIST **el, long double*remaining_time, char *msg)
 *
 * Gets the remaining time to execute a specific action in the event list.
 *
 * Arguments:	el, the event list;
 *				remaining_time, the remaining time to execute the action;
 *				msg, an error message to be shown if remaining_time <= 0.
 *
 ***************************************************************************************/

inline void getRemainingTime(EVENT_LIST **el, long double*remaining_time, char *msg)
{
    struct timeval time_of_day;
    gettimeofday(&time_of_day, 0);    

	(*remaining_time) = DEFAULT_TIMER_VALUE[(int)((*el)->action)] - ( (((long double)time_of_day.tv_sec)*1000000+(long double)time_of_day.tv_usec) - getTimerValue(&((*el)->time_of_day)) );	

    if (*remaining_time <= 0)
    {
#ifdef DEBUG0    
       fprintf(stderr,"!!! Error: remaining time < 0\n");
       fprintf(stderr,"\tremaining time: %.0Lf \n",*remaining_time);
       fprintf(stderr,"\ttime to live  : %ld \n",DEFAULT_TIMER_VALUE[(int)((*el)->action)] );
       fprintf(stderr,"\ttime of day   : %.0Lf \n",( (((long double)time_of_day.tv_sec)*1000000+(long double)time_of_day.tv_usec) ) );
       fprintf(stderr,"\tenter time    : %.0Lf \n",getTimerValue(&((*el)->time_of_day)) );
       if (msg!=NULL)
           fprintf(stderr,"\tmsg           : %s \n",msg );
#endif           
        (*remaining_time) = 0;
    }
}


/***************************************************************************************
 *
 * void setTimerValue(struct itimerval *value, long int time)
 *
 * Converts the time from microseconds to struct itimerval.
 *
 * Arguments:	value, where the converted value will be stored (a struct itimerval);
 *				time, the time in microseconds.
 *
 ***************************************************************************************/

void setTimerValue(struct itimerval *value, long int time)
{
	 (*value).it_interval.tv_sec = 0;
	 (*value).it_interval.tv_usec = 0;

 	 (*value).it_value.tv_sec = time/1000000;
	 (*value).it_value.tv_usec = time%1000000;
}

/***************************************************************************************
 *
 * long double getTimerValue(struct timeval *value )
 *
 * Converts the time from struct timeval to microseconds.
 *
 * Arguments:	value, the struct timeval source.
 *
 * Returns: 	the time, in microseconds.
 *
 ***************************************************************************************/


long double getTimerValue(/* struct itimerval *value */ struct timeval *value )
{
	return (((long double)(*value).tv_sec*1000000)+((long double)(*value).tv_usec));
}

/*---------------- Main functions to manipulate the event list ---------------------------------------*/

/***************************************************************************************
 *
 * void eventListInit(EVENT_LIST **el)
 *
 * Initialize the event list.
 *
 * Arguments:	el, the event list;
 *
 ***************************************************************************************/


void eventListInit(EVENT_LIST **el)
{
    *el = NULL;
}

/***************************************************************************************
 *
 * int eventListAllocNode(EVENT_LIST **el_node, MEMBER_ID *member, char action, 
 *							int sn, long timer_value)
 *
 * Allocates a event list node, and fills it with specified info.
 *
 * Arguments:	el_node, the node which will be created;
 *				member, action, sn and timer_value hold info. to be stored in new node.
 *
 * Returns: 	0 on error, 1 on success.
 *
 ***************************************************************************************/

int eventListAllocNode(EVENT_LIST **el_node, MEMBER_ID *member, 
                        char action, int sn, long timer_value)
{
    if ((*el_node = (EVENT_LIST*)malloc(sizeof(EVENT_LIST)))==NULL)
    {
        return (0);
    }
    (*el_node)->member_id = member;
    (*el_node)->action = action;
    (*el_node)->timer_value = timer_value;
    (*el_node)->sn = sn;
    gettimeofday( &((*el_node)->time_of_day), 0) ;    
    (*el_node)->next = NULL;    
    return (1);
}


/***************************************************************************************
 *
 * int	eventListInsert(EVENT_LIST **el, MEMBER_ID *member_id,	char action, int sn)
 *
 * Inserts an event node in the event list.
 *
 * Arguments:	el, the event list;
 *				member_id, the member identification;
 *				action, event action to be executed;
 *				sn, sequence number of the message to retransmited/requested.
 *
 ***************************************************************************************/

int  eventListInsert(EVENT_LIST **el, MEMBER_ID *member_id, 
                     char action, int sn)
{
    EVENT_LIST *newNode, *auxNode, *antNode;
    long int timer_value=0, accumulator=0;
    char distribution=0;
	long double remaining_time; 
    struct timeval time_of_day;                             
     
    gettimeofday(&time_of_day, 0);    

    auxNode = *el;
    
 
    if (*el!=NULL)
    {
        
        getRemainingTime(el,&remaining_time,"(eventListInsert)\n");        
        
    }
    else
    {
        remaining_time = 0;
    }        
#ifdef DEBUG0    
    fprintf(stderr,"### current event list:\n");
    
    eventListShow(*el);

    fprintf(stderr,"### node to be inserted:\n");
    fprintf(stderr,"### action: %d sn: %d\n", action, sn);
    fprintf(stderr,"### remaining time: %.0Lf\n", remaining_time);
    fprintf(stderr,"### curr time %ld:%ld (%.0Lf)\n", (long int)time_of_day.tv_sec,(long int)time_of_day.tv_usec,
        (((long double)time_of_day.tv_sec)*1000000+time_of_day.tv_usec) );    
    fprintf(stderr,"###\n");
#endif    

    if (*el!=NULL)
    {
        
		  /* .. and set the timer value of the header of the list to the 
		  	     remaining time to the first event occur */		  
                 
	     (*el)->timer_value = remaining_time;
         

    }

    timer_value=generateTimerValue(action,distribution);
    
    if ((eventListAllocNode(&newNode,member_id,action,sn,timer_value))==(int)NULL)
            
    {    
         return (0);    
    }    
    if (auxNode == NULL) /* if the list is empty... */
    {
         (*el) = newNode; 
    }
    else
    {
        if (newNode->timer_value < auxNode->timer_value)
        {
   
		     /* if the node to be inserted is the first of the list...*/
             
            (newNode)->next = auxNode;
            (*el) = newNode;
            (newNode)->next->timer_value -= newNode->timer_value;
        }
        else
        {
            antNode = auxNode;
            accumulator = auxNode->timer_value;
            while(accumulator < newNode->timer_value)
            {
                antNode = auxNode;
	    		auxNode = auxNode->next;
    			if (auxNode==NULL)
                    break;
                accumulator += auxNode->timer_value;	  
            }
            if (auxNode == NULL)
            {
		    	/* ... if the node to be inserted is the last of the list... */
            
                antNode->next = newNode;              
                newNode->timer_value -= accumulator;
            }
            else
            {
                if (auxNode!=antNode)
					 {
                
					    /* ... if the node to be inserted is in the middle of the list */
					    newNode->next = auxNode;   
					 	antNode->next = newNode;
#ifdef DEBUG2												
 					 	fprintf(stderr,"accumulator: %d newNode->timer_value: %d antNode->timer_value: %d auxNode->timer_value: %d", 
					 	  accumulator , newNode->timer_value ,  antNode->timer_value, auxNode->timer_value);
#endif						
					 	newNode->timer_value -= (accumulator - auxNode->timer_value);
                	    auxNode->timer_value -= newNode->timer_value;
					 }
					 else
    				 {
                  
                      /* ... finally, if the node to be inserted has a time value 
				      equal to the first of the list... */
                  
	                    newNode->next = antNode->next;   
						antNode->next = newNode;
					 	newNode->timer_value = 0;
					 }
            }
        }
    } 
	   
#ifdef DEBUG	 
	 eventListShow(*el);	 
#endif
	 
    pthread_kill(signal_handler_thread, SIGUSR1);       
    

     
    return (1);
}

/***************************************************************************************
 *
 * void eventListShow(EVENT_LIST *el)
 *
 * Shows the event list.
 *
 * Arguments:	el, the event list.
 *
 ***************************************************************************************/

void eventListShow(EVENT_LIST *el)
{
#ifdef EL_SHOW
     EVENT_LIST *aux;
 	 int i=0;
     long double remaining_time;
     
    struct timeval time_of_day;
    
    gettimeofday(&time_of_day, 0);    
     
    aux = el;
     
	 fprintf(stderr,"\nEvent list:\n");
	 for (;aux!=NULL;aux=aux->next)
	 {
        fprintf(stderr," (ip: %s pid: %d action: %2d sn: %2d time of day: %ld:%ld) %8ld [%p] -> ", 
            ((aux->member_id==NULL)?aux->member_id->ip:"NULL"),  
            aux->member_id->pid,aux->action, 
            aux->sn, 
            (long int)aux->time_of_day.tv_sec, 
            (long int)aux->time_of_day.tv_usec, 
            aux->timer_value, 
            aux);
		if (i%2) fprintf(stderr,"\n");
 	    i++;
	 }
	 fprintf(stderr," NULL\n");
     
    if (el!=NULL)
    {
	    remaining_time = DEFAULT_TIMER_VALUE[(int)(el->action)] - 
            ( (((long double)time_of_day.tv_sec)*1000000+(long double)time_of_day.tv_usec) - 
            getTimerValue(&(el->time_of_day)) );	

        fprintf(stderr,"###\n");
        fprintf(stderr,"\tremaining time: %.0Lf \n",remaining_time);
        fprintf(stderr,"\ttime to live  : %ld \n",DEFAULT_TIMER_VALUE[(int)((el)->action)] );
        fprintf(stderr,"\ttime of day   : %.0Lf \n",( (((long double)time_of_day.tv_sec)*1000000+(long double)time_of_day.tv_usec) ) );
        fprintf(stderr,"\tenter time    : %.0Lf \n",getTimerValue(&((el)->time_of_day)) );
        fprintf(stderr,"###\n");
    }     
#endif
}


/***************************************************************************************
 *
 * int eventListRemoveFirst(EVENT_LIST **el)
 *
 * Removes the first element of the event list.  
 * 
 * Arguments: el, the event list.
 *
 * Return:	1, on success;
 *			0, on fail.
 *
 * Pay Attention! 
 *
 * This function neither sends any signal, nor calls any time related function.
 *
 ***************************************************************************************/

int eventListRemoveFirst(EVENT_LIST **el)
{
	EVENT_LIST *ant;
   
	if (*el==NULL)
	{
        return (0);
	}
	else
	{
		ant = *el;
		(*el) = (*el)->next;
      
		free(ant);
        return (1);
	}
}

/***************************************************************************************
 *
 * int eventListRemove(EVENT_LIST **el, MEMBER_ID *member_id, char action, int sn)
 *
 * Arguments:	el, the event list;
 *				member_id, action and sn, identify the node to be removed;
 *
 * Return value: 1 on success;
 *				 0 otherwise.
 *
 ***************************************************************************************/

int eventListRemove(EVENT_LIST **el, MEMBER_ID *member_id, char action, int sn)
{
    EVENT_LIST *ant_node;  
    EVENT_LIST *node_to_be_removed;
    long double remaining_time=0;
   
    node_to_be_removed= eventListFind(el,member_id,action,sn,&ant_node);    
    
    if (node_to_be_removed==NULL)
    {   
        return (0);
    }
    else
    {
        if (ant_node == NULL)
        {       
             /* the node to be removed is the first node of the list */
            
             if (*el!=NULL)
             {
                 getRemainingTime(el,&remaining_time,"(eventListRemove)\n");
             }
             else
             {
                 remaining_time = 0;
             }        
			 *el = (*el)->next;
				
			 if (*el!=NULL)
             {  
                  (*el)->timer_value += remaining_time;
            
                  
                  if ((*el)->timer_value == 0)
                  {
#ifdef DEBUG0                  
                    fprintf(stderr,"Internal warning: the alarm was restarted. (eventListRemove)\n");
#endif                 
                    eventListShow(*el);
                    /* exit(1); */
                  }
			 }
			
             pthread_kill(signal_handler_thread, SIGUSR1);       

        }
        else
        {
            if (node_to_be_removed->next!=NULL)
            {
                 node_to_be_removed->next->timer_value += node_to_be_removed->timer_value;
                 ant_node->next = node_to_be_removed -> next;
            }
            else
            {
                 /* the node to be removed  is the last node of the list */
                 ant_node->next = NULL;
            }
        }
    }
#ifdef DEBUG0    
    fprintf(stderr,"here. (eventListRemove)\n ");        
#endif    
    free(node_to_be_removed);
    node_to_be_removed = NULL;
    
    return (1);
}

int eventListIsEqual(EVENT_LIST *node1, EVENT_LIST *node2)
{
    if ((node1->member_id == NULL) || (node2->member_id == NULL))
    {
        if ( (node1->action == node2->action)  &&
             (node1->sn     == node2->sn))
             
            return 1;
            
        else
        
            return 0;
    }
    
    if ((!strcmp(node1->member_id->ip, node2->member_id->ip))
        && (node1->member_id->pid == node2->member_id->pid)
        && (node1->action == node2->action) 
        && (node1->sn     == node2->sn))

        return (1);
    else            
        return (0);
}

/***************************************************************************************
 *
 * EVENT_LIST * eventListFind(EVENT_LIST **el1,MEMBER_ID *member_id, char action, 
 *							  int sn, EVENT_LIST **antNode)
 *
 * Search for a node in the event list.
 *
 * Arguments:	el1, the event list;
 *				member_id, action and sn, identify the node to be found;
 *				antNode, return a pointer to the node that points to the one,
 *						 or NULL if the node removed is the last one.
 *
 * Return value: a pointer to the node found, or NULL if the node wasn't found.
 *
 ***************************************************************************************/


EVENT_LIST * eventListFind(EVENT_LIST **el1,MEMBER_ID *member_id, char action, int sn, EVENT_LIST **antNode)
{
    EVENT_LIST *el;
    
    EVENT_LIST *event_node, *auxPointer;
    
    *antNode = NULL;
    
    el = *el1;
    
    auxPointer = el;
    
#ifdef DEBUG2
  
  
      fprintf(stderr,"######Inside listfind\n ip: %s\n pid: %d\n action:%d\n sn: %d\n######\n", 
                               member_id->ip,
                               member_id->pid,
                               action,
                               sn);
  
                              
  
#endif
   
    eventListShow(el);
    
    if ((eventListAllocNode(&event_node, member_id, action, sn, 0))==(int)NULL)
    {
        perror("Alloc list node at eventListFind");
        exit(0);
        return (0);
    }
    
    
    while (auxPointer!=NULL)
    {
    
        
        if (eventListIsEqual(event_node, auxPointer))
        {
            /*  The event to be removed was found! */
            
            break;
        }
        *antNode = auxPointer;
        auxPointer = auxPointer->next;
    }

    return (auxPointer);  
}

#endif
