/***************************************************************************
                             rmcast.c
                             -------------------
    begin                : May 2001
    copyright            : (C) 2001 by Jorge Allyson Azevedo
                                       Milena Scanferla
                                       Magnos Martinello
                                       Daniel Sadoc
    email                : {allyson,milena,magnos,sadoc}@land.ufrj.br
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
/******************************************************************************************************
 *																									  *
 * Reliable Multicast Protocol (rmcast.c)															  *
 *																									  *
 * Top-level routines used by the reliable multicast library.										  *
 *																									  *
 * use tab spacing = 4																				  *
 *																									  *
 ******************************************************************************************************/

#ifndef RMCAST_C
#define RMCAST_C

#include <math.h>
#include <unistd.h> 
#include <signal.h>
#include <pthread.h>  
#include <sys/types.h>
#include <sys/ipc.h>
#include <sys/msg.h>
#include <netdb.h>

#include "rmcast.h"

#include "rminternals.h"


pthread_t rcv_thread, signal_handler_thread;


extern int		    h_errno;
extern EVENT_LIST   *event_list;
extern CACHE	    *cache;
extern USER_INFO    local_user_info;
extern sigset_t     alrmset;
extern FILE         *logfile;

static int r_qid = -1; /* message queue identifier */

pthread_mutex_t  change_local_user_sn = PTHREAD_MUTEX_INITIALIZER;


/**************************************************************************************
 *
 * void RM_initialize(int options, char *logfile)
 *
 * Initializes internal structures (cache, event list and message queue) and
 * local user information (ip, pid and sn).
 *
 * Parameters: until now, unused. 
 *
 **************************************************************************************/

void RM_initialize(int options, char *logfilename)
{
	struct hostent *localhost_ent;
	int i;
	char localhost_ip[MAX_IP_STRING_SIZE], localhost_name[31];
    
    if (logfilename!=NULL)
    {
        if ((logfile = fopen(logfilename,"w"))==NULL)
        {
    	    fprintf(stderr,"Could not open %s.",logfilename);
            exit(1);
        }
    }
    else
    {
    	logfile = NULL;
    }
	
	sigemptyset(&alrmset);			 /* makes alrmset empty */
	sigaddset(&alrmset, SIGALRM);	 /* puts SIGALRM in the set */
	sigaddset(&alrmset, SIGUSR1);	 /* puts SIGUSR1 in the set */	 
	
	/* blocks SIGALRM and SIGUSR1,because they wont be handled by this thread */
	pthread_sigmask(SIG_BLOCK, &alrmset, NULL); 
	
	
	for (i=0; i< MAX_IP_STRING_SIZE; i++ ) localhost_ip[i]=0;
	
	/* localhost_ip <- local ip */
	
	gethostname(localhost_name, 30);

    if (logfile!=NULL)
    {
  		fprintf (logfile,"host: %s\n", localhost_name );

    }

#ifdef DEBUG0	
    fprintf (stderr,"local host name: %s\n", localhost_name );
#endif                
	

	localhost_ent  = gethostbyname(localhost_name);
	strcpy(localhost_ip, 
		inet_ntoa(*((struct in_addr*)(localhost_ent->h_addr_list[0]))));
	
	
	/* initialization of the local cache, event list and message queue */
	
	cacheInit(&cache);
	
	eventListInit(&event_list);  
    
	if (r_qid == -1 && (r_qid = init_queue() ) == -1)
	{	
		fprintf(stderr,"Couldn't create message queue.\n");
		exit(1);
	} 
	
	/* initialization of local user info */
	
	pthread_mutex_lock(&change_local_user_sn);
	
	local_user_info.sn = 0;    
	
	pthread_mutex_unlock(&change_local_user_sn);
	
	local_user_info.member_id.pid = (int)getpid();		   
	strncpy(local_user_info.member_id.ip, localhost_ip, MAX_IP_STRING_SIZE);

	if (logfile!=NULL)	
    {
     	fprintf(logfile, "ip: %s\n",localhost_ip);
        fprintf(logfile, "pid: %d\n",local_user_info.member_id.pid);
        fprintf (logfile, "-----------------------------------------------------------\n");
    	fprintf (logfile, "Time  Send/Rec/Loss  Packet_type  sender_ip  sender_pid  requested_ip  requested_pid  sn\n");
        fprintf (logfile, "-----------------------------------------------------------\n");
    
    }
    
#ifdef DEBUG0	
	fprintf(stderr, "local host ip: %s\n",localhost_ip);
#endif
	
}


/**************************************************************************************
 *
 * int	RM_joinGroup (char *group, int port)
 *
 * When this routine is used, RM_initialize must have been called at least once.
 *  
 * Sets up multicast group information, creates the thread responsible for receiving 
 * packets and activates the alarm responsible for catching events from the event list.
 *
 * Parameters: group, the multicast group ip;
 *			   port, the multicast group port.
 *
 * Returns: the socket created.
 *
 **************************************************************************************/

int  RM_joinGroup (char *group, int port)
{
	/* Now, we are seting the multicast info (multicast port and multicast ip), */
	
	local_user_info.group_id.port = port ;	  
	
	strncpy(local_user_info.group_id. ip, group, MAX_IP_STRING_SIZE) ;	  
	
	local_user_info.socket = rmcastCreateSocket  (port); 
	
	
	/* creating the thread responsible for receiving packets */
	
	pthread_create( &rcv_thread, NULL, rmcastReceivePackets, (void*)local_user_info.socket);
	
	pthread_create( &signal_handler_thread, NULL, rmcastHandleSignals, NULL);
	
	rmcastJoinGroup 	(local_user_info.socket,group);


	if (logfile!=NULL)	
    	fprintf(logfile, "joined to group\n");

#ifdef DEBUG0	
	fprintf(stderr,"joined to group\n");
#endif
	
	/* and activating the alarm responsible for catching events from the event list. */

	signal(SIGINT,	remove_queues);    
	signal(SIGTERM, remove_queues);    
	signal(SIGHUP,	remove_queues);    

/*
   	eventListInsert(&event_list, &(local_user_info.member_id), REF_SND_WAIT, 0 );     
*/	
	return local_user_info.socket;
}

/**************************************************************************************
 *
 * int	RM_leaveGroup(int  sock,   char *group)
 *
 * Used to leave from the multicast group.
 *
 * Parameters: sock, the multicast socket;
 *			   group, the multicast ip.
 *
 * Returns: always TRUE.
 *
 **************************************************************************************/

int  RM_leaveGroup(int	sock,	char *group)
{
	rmcastLeaveGroup  (sock, group);   
	
	remove_queues(0);
	
	return TRUE;
}

/**************************************************************************************
 *
 * void RM_terminate (void)
 *
 * Exits from the program. 
 *
 **************************************************************************************/

void RM_terminate (void)
{

#ifdef DEBUG0
	fprintf(stderr,"Exiting...\n");
#endif
	
	exit(1);	
}

/**************************************************************************************
 *
 * int	RM_sendto	 (int socket, char *buffer, int buffsize)
 *
 * Sends a message, contained in buffer, to the multicast group.
 *
 * Parameters: socket, the socket number;
 *			   buffer, message to be sent;
 *			   buffsize, maximum number of chars to be sent.
 *
 * Returns: 1, on success;
 *			0, on error.
 *
 **************************************************************************************/

int  RM_sendto	  (int socket, char *buffer, int buffsize)
{
	PACKET_INFO pckt_info;
	
	BYTE* local_buffer = NULL;
	
	int final_message_size, retval;
	
	pckt_info.type = DATA_PACKET_TYPE;
	strcpy(pckt_info.sender_id.ip, local_user_info.member_id.ip);
	pckt_info.sender_id.pid = local_user_info.member_id.pid;
	
    lock_eventlist_and_cache();
    
	pthread_mutex_lock(&change_local_user_sn);	  

	pckt_info.packet_data.data_packet.sn = local_user_info.sn;
	
	pthread_mutex_unlock(&change_local_user_sn);
    	
	pckt_info.packet_data.data_packet.data_size = buffsize;
	
	pckt_info.packet_data.data_packet.data = (BYTE*)malloc((sizeof(char)*buffsize)+1);
	memcpy((char*)(pckt_info.packet_data.data_packet.data), buffer, buffsize);
	   
	retval = cacheInsertMessage(&cache, &(pckt_info.sender_id), &(pckt_info.packet_data.data_packet));
    
/*
    rmcastRemoveEvent(NULL,REF_SND_WAIT);
    retval = eventListInsert(&event_list, &(local_user_info.member_id), REF_SND_WAIT, 0 );		
*/
    
    unlock_eventlist_and_cache();
        
	if (retval == 0)
	{
		fprintf(stderr,"Error inserting message in local cache. (RM_sendto)\n");
		exit(0);
	}
    
	   
	msgPcktMountMessage(&pckt_info, &local_buffer, &final_message_size);

#ifdef DEBUG0	
	fprintf(stderr,"Will send a message... ");
#endif
	
	rmcastSendPacket (socket, local_buffer, final_message_size);
    
#ifdef DEBUG0	
	fprintf(stderr,"ok!\n");
#endif
	
#ifdef DEBUG		
	msgPcktShowMessage(local_buffer);
#endif	  
	
	free(local_buffer);
	
	return TRUE;
}

/**************************************************************************************
 *
 * int	RM_recv 	 (int socket, char *buffer, int buffsize)
 *
 * Waits to receive a message.
 *
 * Parameters: socket, the socket number;
 *			   buffer, message received;
 *			   buffsize, maximum number of chars to be received.
 *
 * Returns: number of bytes received, on success;
 *			-1, on error.
 *
 **************************************************************************************/

int  RM_recv	  (int socket, char *buffer, int buffsize)
{
	struct q_entry r_entry;
	int mlen, size, power, ind;

#ifdef DEBUG0
	fprintf(stderr,"Waiting to receive a \"reliable\" message...\n");
#endif
	
	/* waiting to receive a reliable message trhough the message queue */
	
	if ((mlen = msgrcv(r_qid, &r_entry, MAXOBN, 
		0, MSG_NOERROR)) == -1) {
		
		perror("msgrcv failed");
		exit(1);
	}
	
	else {
		
		/* a reliable message was received */
		
		size =0;
		
		/* convert mtext from base 256 to base 10 - mtext[0] is the least signif. digit of mtext */
		
		for (power = 1, ind = 0; ind < MSG_QUEUE_HEADER_SIZE;  power = (int)pow(256,ind))
		{
			size += (((unsigned char)(r_entry.mtext[ind])) * power);
#ifdef DEBUG2			 
			fprintf(stderr,"power: %d r_entry.mtext[%d]: %d size: %d\n",power,ind,r_entry.mtext[ind],size);
#endif			  
			ind++;
		}
		
		if (size > buffsize)
		{
			fprintf(stderr,"Internal warning. Message size: %d, Buffer size: %d. (RM_recv)\n", size, buffsize);
			exit(1);
		}
		
		else
		{
			memcpy(buffer,&(r_entry.mtext[MSG_QUEUE_HEADER_SIZE]),size);
		}
		
#ifdef DEBUG1		
		fprintf(stderr,"A \"reliable\" message was received: %s!\n",buffer);
#endif

	}
#ifdef DEBUG2	 
	fprintf(stderr,"message size (send to app): %d\n",size);
#endif	  
	
	
	return size;
}



/**************************************************************************************/

void RMDEBUG_setpidip(int pid, char *ip)
{
	local_user_info.member_id.pid = pid;		 
	strncpy(local_user_info.member_id.ip, ip, MAX_IP_STRING_SIZE);
}		 

/**************************************************************************************/


void RMDEBUG_setsn(int sn)
{
	pthread_mutex_lock(&change_local_user_sn);
	
	local_user_info.sn = sn;   
	
	pthread_mutex_unlock(&change_local_user_sn);
}
/**************************************************************************************/

#endif
