/*
 * Author:      William Chia-Wei Cheng (william@cs.ucla.edu)
 *
 * Copyright (C) 1990-1994, William Cheng.
 *
 * Permission limited to the use, copy, modify, and distribute this software
 * and its documentation for any purpose is hereby granted by the Author without
 * fee, provided that the above copyright notice appear in all copies and
 * that both the copyright notice and this permission notice appear in
 * supporting documentation, and that the name of the Author not be used
 * in advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission.  The Author makes no
 * representations about the suitability of this software for any purpose.
 * It is provided "as is" without express or implied warranty.  All other
 * rights (including the right to sell "tgif" and the right to sell derivative
 * works of tgif) are reserved by the Author.
 *
 * THE AUTHOR DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO
 * EVENT SHALL THE AUTHOR BE LIABLE FOR ANY SPECIAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF
 * USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR
 * OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
 * PERFORMANCE OF THIS SOFTWARE.
 */
#ifndef lint
static char RCSid[] =
      "@(#)$Header: /amnt/maui/tangram/u/william/X11/TGIF2/RCS/special.c,v 2.74 1994/04/04 00:49:06 william Exp $";
#endif

#include <stdio.h>
#include <string.h>
#include <X11/Xlib.h>
#include "const.h"
#include "types.h"

#include "animate.e"
#include "attr.e"
#include "choice.e"
#include "cmd.e"
#include "color.e"
#include "copypaste.e"
#include "cursor.e"
#include "dialog.e"
#include "drawing.e"
#include "dup.e"
#include "edit.e"
#include "file.e"
#include "grid.e"
#include "group.e"
#include "mainloop.e"
#include "mark.e"
#include "menu.e"
#include "msg.e"
#include "move.e"
#include "names.e"
#include "obj.e"
#include "raster.e"
#include "ruler.e"
#include "select.e"
#include "setup.e"
#ifndef _NO_EXTERN
#include "special.e"
#endif
#include "stk.e"

char * specialMenuStr[] =
   {  "MakeSymbolic   ^#m",
      "UnMakeSymbolic ^#n",
      "Instantiate     ^i",
      "MakeIconic     ^#i",
      "UnMakeIconic   ^#j",
      "Push            ^v",
      "Pop             ^k",
      "AttachAttrs     #a",
      "DetachAttrs     #t",
      "ShowAttr        #-",
      "ShowAttrName    #n",
      "HideAttr       ^#h",
      "HideAttrName    #j",
      "Move/JustfyAttr #m",
      "EditAttrs         ",
      "AnimateSend     #e",
      "AnimateFlash    #f",
      "AttachFileAttrs   ",
      "DetachFileAttrs   ",
      "EditFileAttrs     ",
      "UpdateSymbols  ^#u"
   };
static char * specialMenuDescription[] =
   {  "Turn a selected grouped/icon object into a symbol object",
      "Turn a selected symbol object into a grouped object",
      "Instantiate a building-block object from the current domain",
      "Turn a selected grouped/symbol object into an icon object",
      "Turn an icon object into a grouped object",
      "Push into (edit) the symbol file which defines the selected icon object",
      "Pop back to a high lever (reverse of Push)",
      "Attach selected text objects as attributes for the non-text object",
      "Detach all attributes of selected objects",
      "Make all attributes of selected objects visible",
      "Make all attribute names of selected objects visible",
      "Hide all attributes for selected objects",
      "Hide all attribute names for selected objects",
      "Move/justify an attribute for a selected object",
      "Edit attributes of a selected object",
      "Animate a little token on a selected poly (not very useful)",
      "Flash a selected poly (not very useful)",
      "Attach selected text objects as file attributes",
      "Detach all file attributes",
      "Edit file attributes",
      "Refresh selected icon objects from their definition files",
      NULL
   };

struct ObjRec * ReadSymbol (FP)
   FILE	* FP;
{
   register struct AttrRec	* attr_ptr;
   struct ObjRec		* obj_ptr;
   char				msg[MAXSTRING];
   int				read_status;

   SetWatchCursor (drawWindow);
   SetWatchCursor (mainWindow);

   importingFile = TRUE; /* ignore 'state' info but set fileVersion */
   readingPageNum = 0;
   while ((read_status = ReadObj (FP, &obj_ptr)) == TRUE)
      if (obj_ptr != NULL)
      {
         if (obj_ptr->type == OBJ_SYM)
         {
            UnlockAnObj (obj_ptr);
            obj_ptr->type = OBJ_ICON;
            attr_ptr = obj_ptr->lattr;
            for ( ; attr_ptr != NULL; attr_ptr = attr_ptr->prev)
               attr_ptr->inherited = TRUE;
            importingFile = FALSE;
            SetDefaultCursor (mainWindow);
            SetDefaultCursor (drawWindow);
            return (obj_ptr);
         }
         else
            FreeObj (obj_ptr);
      }
   importingFile = FALSE;
   if (read_status == INVALID)
   {
      sprintf (msg, "File version too large (=%1d).  Read symbol aborted!",
            fileVersion);
      Msg (msg);
   }
   SetDefaultCursor (mainWindow);
   SetDefaultCursor (drawWindow);
   return (NULL);
}

struct ObjRec * GetObjRepresentation (PathName, SymName)
   char	* PathName, * SymName;
{
   char			file_name[MAXPATHLENGTH], msg[MAXSTRING], * rest;
   int			short_name;
   struct ObjRec	* obj_ptr;
   FILE			* fp;
   int			tmp_linenum;
   char			tmp_filename[MAXPATHLENGTH];

   sprintf (file_name, "%s/%s.%s", PathName, SymName, SYM_FILE_EXT);
   if ((short_name = IsPrefix (bootDir, file_name, &rest))) ++rest;

   if ((fp = fopen (file_name, "r")) == NULL)
   {
      if (short_name)
         sprintf (msg, "Can not open '%s'\n", rest);
      else
         sprintf (msg, "Can not open '%s'\n", file_name);
      Msg (msg);
      return (NULL);
   }

   strcpy (tmp_filename, scanFileName);
   tmp_linenum = scanLineNum;
   strcpy (scanFileName, (short_name ? rest : file_name));
   scanLineNum = 0;

   if ((obj_ptr = ReadSymbol (fp)) != NULL)
   {
      obj_ptr->id = objId++;
      obj_ptr->dirty = FALSE;
      strcpy (obj_ptr->detail.r->s, SymName);
      obj_ptr->detail.r->rotate = ROTATE0;
      obj_ptr->detail.r->flip = NO_FLIP;
      AdjObjBBox (obj_ptr);
   }

   strcpy (scanFileName, tmp_filename);
   scanLineNum = tmp_linenum;

   fclose (fp);
   return (obj_ptr);
}

void PlaceTopObj (ObjPtr)
   struct ObjRec	* ObjPtr;
{
   int		x, y, grid_x, grid_y, dx, dy, placing = TRUE;
   int		cursor_x, cursor_y, orig_x, orig_y;
   int		obj_ltx, obj_lty, obj_rbx, obj_rby;
   int		grid_obj_ltx, grid_obj_lty, grid_dx=0, grid_dy=0;
   Window	root_win, child_win;
   int		root_x, root_y, first_time=TRUE;
   unsigned int	status;
   XEvent	input, ev;

   RedrawMsg ();

   XFlush (mainDisplay);
   XSync (mainDisplay, False);

   if (XCheckMaskEvent (mainDisplay, ExposureMask, &ev) ||
         XCheckMaskEvent (mainDisplay, VisibilityChangeMask, &ev))
      ExposeEventHandler (&ev, TRUE);

   obj_ltx = OFFSET_X(ObjPtr->obbox.ltx); obj_lty = OFFSET_Y(ObjPtr->obbox.lty);
   obj_rbx = OFFSET_X(ObjPtr->obbox.rbx); obj_rby = OFFSET_Y(ObjPtr->obbox.rby);
   GridXY (obj_ltx, obj_lty, &grid_obj_ltx, &grid_obj_lty);

   dx = dy = 0;
   XSetInputFocus (mainDisplay, drawWindow, RevertToNone, CurrentTime);
   XGrabPointer (mainDisplay, drawWindow, False,
         PointerMotionMask | ButtonPressMask,
         GrabModeAsync, GrabModeAsync, None, handCursor, CurrentTime);

   if (XCheckMaskEvent (mainDisplay, PointerMotionMask, &input))
   {
      first_time = FALSE;

      cursor_x = input.xmotion.x;
      cursor_y = input.xmotion.y;
      while (XCheckMaskEvent (mainDisplay, PointerMotionMask, &input))
      {
         cursor_x = input.xmotion.x;
         cursor_y = input.xmotion.y;
      }
      GridXY (cursor_x, cursor_y, &orig_x, &orig_y);
      grid_dx = orig_x-grid_obj_ltx; grid_dy = orig_y-grid_obj_lty;
      SelBox (drawWindow, revDefaultGC, obj_ltx+grid_dx, obj_lty+grid_dy,
            obj_rbx+grid_dx, obj_rby+grid_dy);
   }

   while (placing)
   {
      XNextEvent (mainDisplay, &input);

      if (first_time)
      {
         first_time = FALSE;

         XQueryPointer (mainDisplay, drawWindow, &root_win, &child_win,
               &root_x, &root_y, &cursor_x, &cursor_y, &status);

         GridXY (cursor_x, cursor_y, &orig_x, &orig_y);
         grid_dx = orig_x-grid_obj_ltx; grid_dy = orig_y-grid_obj_lty;
         SelBox (drawWindow, revDefaultGC, obj_ltx+grid_dx, obj_lty+grid_dy,
               obj_rbx+grid_dx, obj_rby+grid_dy);
      }

      if (input.type == Expose || input.type == VisibilityNotify)
      {
         SelBox (drawWindow, revDefaultGC, obj_ltx+grid_dx, obj_lty+grid_dy,
               obj_rbx+grid_dx, obj_rby+grid_dy);
         ExposeEventHandler (&input, TRUE);
         SelBox (drawWindow, revDefaultGC, obj_ltx+grid_dx, obj_lty+grid_dy,
               obj_rbx+grid_dx, obj_rby+grid_dy);
      }
      else if (input.type == ButtonPress)
      {
         XUngrabPointer (mainDisplay, CurrentTime);
         placing = FALSE;
         SelBox (drawWindow, revDefaultGC, obj_ltx+grid_dx+dx,
               obj_lty+grid_dy+dy, obj_rbx+grid_dx+dx, obj_rby+grid_dy+dy);
         grid_dx = ABS_SIZE(grid_dx+dx);
         grid_dy = ABS_SIZE(grid_dy+dy);
         MoveObj (ObjPtr, grid_dx, grid_dy);
         numRedrawBBox = 0;
         ShowInterrupt ();
         DrawObj (drawWindow, ObjPtr);
         HideInterrupt ();
      }
      else if (input.type == MotionNotify)
      {
         x = input.xmotion.x;
         y = input.xmotion.y;
         GridXY (x, y, &grid_x, &grid_y);
         SelBox (drawWindow, revDefaultGC, obj_ltx+grid_dx+dx,
               obj_lty+grid_dy+dy, obj_rbx+grid_dx+dx, obj_rby+grid_dy+dy);
         dx = grid_x - orig_x;
         dy = grid_y - orig_y;
         SelBox (drawWindow, revDefaultGC, obj_ltx+grid_dx+dx,
               obj_lty+grid_dy+dy, obj_rbx+grid_dx+dx, obj_rby+grid_dy+dy);
         MarkRulers (grid_x, grid_y);
         while (XCheckMaskEvent (mainDisplay, PointerMotionMask, &ev)) ;
      }
   }
   XSync (mainDisplay, True);
}

static
int UnMakeIconicOnInstantiate (obj_ptr)
   struct ObjRec	* obj_ptr;
{
   register struct AttrRec	* attr_ptr;

   for (attr_ptr=obj_ptr->lattr; attr_ptr!=NULL; attr_ptr=attr_ptr->prev)
      if (*attr_ptr->name=='\0' &&
            strcmp (attr_ptr->s, "unmakeiconic_on_instantiate")==0)
         return (TRUE);
   return (FALSE);
}

static
void DeleteUnMakeIconicOnInstantiateText ()
{
   struct SelRec	* sel_ptr;
   struct ObjRec	* text_obj_ptr=NULL;

   for (sel_ptr=botSel; sel_ptr != NULL; sel_ptr=sel_ptr->prev)
      if (sel_ptr->obj->type == OBJ_TEXT &&
            strcmp (sel_ptr->obj->detail.t->first->s,
            "unmakeiconic_on_instantiate") == 0)
      {
         text_obj_ptr = sel_ptr->obj;
         break;
      }
   if (text_obj_ptr == NULL)
   {
      char	msg[MAXSTRING+1];

      fprintf (stderr, "Warning:  Can not find the '%s' text object.\n",
            "unmakeiconic_on_instantiate");
      fprintf (stderr, "Please send bug report to william@cs.ucla.edu.\n");
      strcpy (msg,
            "Possible fatal error.  Please read the terminal error output.");
      Dialog (msg, NULL, NULL);
      XFlush (mainDisplay);
      XSync (mainDisplay, False);
      return;
   }
   MakeQuiescent ();
   topSel = (struct SelRec *) calloc (1, sizeof(struct SelRec));
   topSel->next = NULL;
   topSel->obj = text_obj_ptr;
   topSel->prev = NULL;
   botSel = topSel;
   UpdSelBBox ();
   HighLightForward ();
   DelAllSelObj ();
}

void UnMakeIconic ();

void Instantiate ()
{
   char			file_name[MAXPATHLENGTH], * rest, msg[MAXSTRING];
   char			sym_name[MAXPATHLENGTH], path_name[MAXPATHLENGTH];
   char			ext_str[MAXPATHLENGTH];
   int			short_name;
   struct ObjRec	* obj_ptr;
   FILE			* fp;
   XEvent		ev;
   int			tmp_linenum, len, ext_len;
   char			tmp_filename[MAXPATHLENGTH];

   TieLooseEnds ();
   SetCurChoice (NOTHING);
   if (topSel!=NULL) { HighLightReverse (); RemoveAllSel (); }

   sprintf (msg, "Please select a symbol to INSTANTIATE in the '%s' domain...",
         curDomainName);
   if (SelectFromLibrary (msg, SYM_FILE_EXT, sym_name, path_name) == INVALID)
      return;

   if (XCheckMaskEvent (mainDisplay, ExposureMask, &ev) ||
         XCheckMaskEvent (mainDisplay, VisibilityChangeMask, &ev))
      ExposeEventHandler (&ev, TRUE);

   sprintf (file_name, "%s/%s", path_name, sym_name);

   len = strlen (sym_name);
   sprintf (ext_str, ".%s", SYM_FILE_EXT);
   ext_len = strlen (ext_str);
   if (len > ext_len && strcmp (ext_str, &sym_name[len-ext_len]) == 0)
      sym_name[len-ext_len] = '\0';

   if ((short_name = IsPrefix (bootDir, file_name, &rest))) ++rest;

   if ((fp = fopen (file_name, "r")) == NULL)
   {
      if (short_name)
         sprintf (msg, "Can not instantiate '%s'.", rest);
      else
         sprintf (msg, "Can not instantiate '%s'.", file_name);
      Msg (msg);
      return;
   }

   strcpy (tmp_filename, scanFileName);
   tmp_linenum = scanLineNum;
   strcpy (scanFileName, (short_name ? rest : file_name));
   scanLineNum = 0;

   SetWatchCursor (drawWindow);
   SetWatchCursor (mainWindow);
   if ((obj_ptr = ReadSymbol (fp)) != NULL)
   {
      if (short_name)
         sprintf (msg, "Instantiating '%s'...", rest);
      else
         sprintf (msg, "Instantiating '%s'...", file_name);
      Msg (msg);
      obj_ptr->id = objId++;
      obj_ptr->dirty = FALSE;
      strcpy (obj_ptr->detail.r->s, sym_name);
      obj_ptr->detail.r->rotate = ROTATE0;
      obj_ptr->detail.r->flip = NO_FLIP;
      PlaceTopObj (obj_ptr);
      AssignNewObjIds (obj_ptr);
      AddObj (NULL, topObj, obj_ptr);
      AdjObjBBox (obj_ptr);

      SelectTopObj ();
      if (UnMakeIconicOnInstantiate (obj_ptr))
      {
         StartCompositeCmd ();
         RecordNewObjCmd ();
         UnMakeIconic ();
         UngroupSelObj ();
         DeleteUnMakeIconicOnInstantiateText ();
         EndCompositeCmd ();
      }
      else
         RecordNewObjCmd ();
      SetFileModified (TRUE);
      justDupped = FALSE;
   }
   fclose (fp);

   strcpy (scanFileName, tmp_filename);
   scanLineNum = tmp_linenum;

   SetDefaultCursor (mainWindow);
   SetDefaultCursor (drawWindow);
   Msg ("");
}

void MakeSymbolic ()
{
   if (topSel!=NULL && topSel==botSel && topSel->obj->type!=OBJ_SYM)
   {
      HighLightReverse ();
      PrepareToRecord (CMD_REPLACE, topSel, botSel, numObjSelected);

      if (topSel->obj->type != OBJ_GROUP && topSel->obj->type != OBJ_ICON)
      {
         GroupSingleObj ();
         if (topSel->obj->fattr != NULL)
            Msg ("Object's attributes are promoted to the new symbol object.");
      }

      topSel->obj->type = OBJ_SYM;
      Msg ("Selected object is now SYMBOLIC.");
      AdjObjBBox (topSel->obj);
      UpdSelBBox ();
      RecordCmd (CMD_REPLACE, NULL, topSel, botSel, numObjSelected);
      RedrawAnArea (botObj,
            botSel->obj->obbox.ltx-QUARTER_INCH-GRID_ABS_SIZE(1),
            botSel->obj->obbox.lty-QUARTER_INCH-GRID_ABS_SIZE(1),
            botSel->obj->obbox.rbx+QUARTER_INCH+GRID_ABS_SIZE(1),
            botSel->obj->obbox.rby+QUARTER_INCH+GRID_ABS_SIZE(1));
      HighLightForward ();
      SetFileModified (TRUE);
      justDupped = FALSE;
   }
   else
      Msg ("Please select one non-symbol object to make it Symbolic.");
}

void UnMakeSymbolic ()
{
   register struct ObjRec	* obj_ptr;
   register int			ltx = 0, lty = 0, rbx = 0, rby = 0;
   struct SelRec		* sel_ptr = topSel;
   int				modified = FALSE;

   StartCompositeCmd ();
   for ( ; sel_ptr != NULL; sel_ptr = sel_ptr->next)
   {
      obj_ptr = sel_ptr->obj;
      if (obj_ptr->type == OBJ_SYM)
      {
         PrepareToReplaceAnObj (obj_ptr);
         obj_ptr->type = OBJ_GROUP;
         AdjObjBBox (obj_ptr);
         RecordReplaceAnObj (obj_ptr);
         if (modified)
         {
            if (obj_ptr->bbox.ltx < ltx) ltx = obj_ptr->bbox.ltx;
            if (obj_ptr->bbox.lty < lty) lty = obj_ptr->bbox.lty;
            if (obj_ptr->bbox.rbx > rbx) rbx = obj_ptr->bbox.rbx;
            if (obj_ptr->bbox.rby > rby) rby = obj_ptr->bbox.rby;
         }
         else
         {
            ltx = obj_ptr->bbox.ltx; lty = obj_ptr->bbox.lty;
            rbx = obj_ptr->bbox.rbx; rby = obj_ptr->bbox.rby;
            modified = TRUE;
         }
      }
   }
   EndCompositeCmd ();
   if (modified)
   {
      HighLightReverse ();
      UpdSelBBox ();
      RedrawAnArea (botObj, ltx-QUARTER_INCH-GRID_ABS_SIZE(1),
            lty-QUARTER_INCH-GRID_ABS_SIZE(1),rbx+QUARTER_INCH+GRID_ABS_SIZE(1),
            rby+QUARTER_INCH+GRID_ABS_SIZE(1));
      HighLightForward ();
      SetFileModified (TRUE);
      justDupped = FALSE;
   }
}

void MakeIconic ()
{
   char 		icon_name[MAXPATHLENGTH], file_name[MAXPATHLENGTH];
   char 		s[MAXPATHLENGTH], icon_full_name[MAXPATHLENGTH], * rest;
   char 		obj_ext_str[MAXSTRING], sym_ext_str[MAXSTRING];
   FILE			* fp;
   int			len, short_name, obj_ext_len, sym_ext_len;
   int			ltx, lty, rbx, rby;

   if (topSel!=NULL && topSel==botSel)
   {
      Dialog ("Please Enter Name of the Icon:",
            "( <CR>: accept, <ESC>: cancel )", icon_name);
      len = strlen (icon_name);
      if (*icon_name == '\0')
      {
         Msg ("Name not specified, icon not created.");
         return;
      }

      sprintf (obj_ext_str, ".%s", OBJ_FILE_EXT);
      sprintf (sym_ext_str, ".%s", SYM_FILE_EXT);
      obj_ext_len = strlen (obj_ext_str);
      sym_ext_len = strlen (sym_ext_str);

      if (len >= obj_ext_len || len >= sym_ext_len)
      {
         if (strcmp (&icon_name[len-obj_ext_len], obj_ext_str) == 0)
         {
            Msg ("Can not save as an object file, icon not created.");
            return;
         }
         else if (strcmp (&icon_name[len-sym_ext_len], sym_ext_str) != 0)
         {
            strcpy (icon_full_name, icon_name);
            strcat (icon_full_name, sym_ext_str);
         }

         if (strlen (icon_full_name) == sym_ext_len)
         {
            Msg ("No file name specified.  File not saved.");
            return;
         }
      }
      else
      {
         strcpy (icon_full_name, icon_name);
         strcat (icon_full_name, sym_ext_str);
      }

      sprintf (file_name, "%s/%s", curDir, icon_full_name);

      if (!OkayToCreateFile (file_name)) return;
      if ((short_name = IsPrefix (bootDir, file_name, &rest))) ++rest;

      if ((fp = fopen (file_name, "w")) == NULL)
      {
         if (short_name)
            sprintf (s, "Can not create '%s', icon not created.", rest);
         else
            sprintf (s, "Can not create '%s', icon not created.", file_name);
         Msg (s);
         return;
      }

      if (!DirInSymPath (curDir)) UpdateSymInfo ();

      if (short_name)
         sprintf (s, "Creating '%s' ...", rest);
      else
         sprintf (s, "Creating '%s' ...", file_name);
      Msg (s);

      ltx = selLtX; lty = selLtY; rbx = selRbX; rby = selRbY;

      PrepareToRecord (CMD_REPLACE, topSel, botSel, numObjSelected);
      if (topSel->obj->type == OBJ_GROUP || topSel->obj->type == OBJ_SYM ||
            topSel->obj->type == OBJ_ICON)
         JustMoveSelToTop ();
      else
      {
         GroupSingleObj ();
         if (topSel->obj->fattr != NULL)
            Msg ("Object's attributes are promoted to the new icon object.");
      }

      topSel->obj->type = OBJ_SYM;
      strcpy (topSel->obj->detail.r->s, icon_name);
      topSel->obj->detail.r->rotate = ROTATE0;
      topSel->obj->detail.r->flip = NO_FLIP;

      writeFileFailed = FALSE;
      Save (fp, topSel->obj, 0, 1);

      if (writeFileFailed)
      {
         writeFileFailed = FALSE;
         sprintf (s, "%s '%s'.  %s.  '%s' deleted.",
               "Fail to write to", file_name, "File system may be full",
               file_name);
         Dialog (s, NULL, NULL);
         unlink (file_name);
      }
      else
      {
         if (short_name)
            sprintf (s, "File '%s' created.", rest);
         else
            sprintf (s, "File '%s' created.", file_name);
         Msg (s);
      }
      fclose (fp);

      HighLightReverse ();
      topSel->obj->type = OBJ_ICON;
      topSel->obj->id = objId++;
      Msg ("Selected object is now ICONIC.");
      AdjObjBBox (topSel->obj);
      UpdSelBBox ();
      RecordCmd (CMD_REPLACE, NULL, topSel, botSel, numObjSelected);
      ltx = min(ltx,selLtX); lty = min(lty,selLtY);
      rbx = max(rbx,selRbX); rby = max(rby,selRbY);
      RedrawAnArea (botObj, ltx-GRID_ABS_SIZE(1), lty-GRID_ABS_SIZE(1),
            rbx+GRID_ABS_SIZE(1), rby+GRID_ABS_SIZE(1));
      HighLightForward ();
      SetFileModified (TRUE);
      justDupped = FALSE;
   }
   else
      Msg ("Please select one object to make it ICONIC.");
}

void UnMakeIconic ()
{
   register struct ObjRec	* obj_ptr;
   struct SelRec		* sel_ptr;
   struct AttrRec		* attr_ptr;
   int				modified = FALSE;

   HighLightReverse ();
   StartCompositeCmd ();
   for (sel_ptr = topSel; sel_ptr != NULL; sel_ptr = sel_ptr->next)
   {
      obj_ptr = sel_ptr->obj;
      if (obj_ptr->type == OBJ_ICON)
      {
         modified = TRUE;
         PrepareToReplaceAnObj (obj_ptr);
         obj_ptr->type = OBJ_GROUP;
         attr_ptr = obj_ptr->fattr;
         for ( ; attr_ptr != NULL; attr_ptr = attr_ptr->next)
            attr_ptr->inherited = FALSE;
         AdjObjBBox (obj_ptr);
         RecordReplaceAnObj (obj_ptr);
      }
   }
   EndCompositeCmd ();
   if (modified)
   {
      Msg ("Selected ICONIC objects are GROUP objects now.");
      UpdSelBBox ();
      RedrawAnArea (botObj, selLtX-GRID_ABS_SIZE(1), selLtY-GRID_ABS_SIZE(1),
            selRbX+GRID_ABS_SIZE(1), selRbY+GRID_ABS_SIZE(1));
      SetFileModified (TRUE);
      justDupped = FALSE;
   }
   HighLightForward ();
}

void SpecialSubMenu (index)
   int	index;
{
   switch (index)
   {
      case SPECIAL_SYM: MakeSymbolic (); break;
      case SPECIAL_UNSYM: UnMakeSymbolic (); break;
      case SPECIAL_INST: Instantiate (); break;
      case SPECIAL_ICON: MakeIconic (); break;
      case SPECIAL_UNICON: UnMakeIconic (); break;
      case SPECIAL_PUSH: PushIcon (); break;
      case SPECIAL_POP: PopIcon (); break;
      case SPECIAL_ADDATTR: AddAttrs (); break;
      case SPECIAL_DETACHATTR: DetachAttrs (); break;
      case SPECIAL_SHOWATTR: ShowAllAttrs (); break;
      case SPECIAL_SHOWATTRNAME: ShowAllAttrNames (); break;
      case SPECIAL_HIDEATTR: HideAllAttrs (); break;
      case SPECIAL_HIDEATTRNAME: HideAllAttrNames (); break;
      case SPECIAL_MOVEATTR: MoveAttr (); break;
      case SPECIAL_EDITATTR: EditAttrs (); break;
      case SPECIAL_ANIMATESEND: AnimateSel (); break;
      case SPECIAL_ANIMATEFLASH: FlashSelColor (); break;
      case SPECIAL_ADDFILEATTR: AddFileAttrs (); break;
      case SPECIAL_DETACHFILEATTR: DetachFileAttrs (); break;
      case SPECIAL_EDITFILEATTR: EditFileAttrs (); break;
      case SPECIAL_UPDATESYMS: UpdateSymbols (); break;
   }
}

int SpecialMenu (X, Y, TrackMenubar)
   int	X, Y, TrackMenubar;
{
   register int	index;
   int		* fore_colors, * valid, * init_rv;

   DefaultColorArrays (MAXSPECIALMENUS, &fore_colors, &valid, &init_rv, NULL);
   activeMenu = MENU_SPECIAL;
   index = TextMenuLoop (X, Y, specialMenuStr, MAXSPECIALMENUS, fore_colors,
         valid, init_rv, specialMenuDescription, SINGLECOLOR, TrackMenubar);

   if (index >= 0) SpecialSubMenu (index);
   return (index);
}
