/* The GIMP -- an image manipulation program
 * Copyright (C) 1999 Manish Singh
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "config.h"

#include <gtk/gtk.h>

#include "libgimpbase/gimpbase.h"
#include "libgimpwidgets/gimpwidgets.h"

#include "display-types.h"

#include "core/gimpimage.h"

#include "widgets/gimpeditor.h"

#include "gimpdisplay.h"
#include "gimpdisplayshell.h"
#include "gimpdisplayshell-filter.h"
#include "gimpdisplayshell-filter-dialog.h"

#include "libgimp/gimpintl.h"


#define LIST_WIDTH  150
#define LIST_HEIGHT 100

#define UPDATE_DISPLAY(shell) G_STMT_START      \
{	                                        \
  gimp_display_shell_expose_full (shell);       \
  gimp_display_shell_flush (shell);             \
} G_STMT_END  


typedef struct _ColorDisplayDialog ColorDisplayDialog;

struct _ColorDisplayDialog
{
  GimpDisplayShell *shell;

  GtkWidget        *dialog;

  GtkTreeStore     *src;
  GtkTreeStore     *dest;

  GtkTreeSelection *src_sel;
  GtkTreeSelection *dest_sel;

  gboolean          modified;

  GList            *old_nodes;
  GList            *conf_nodes;

  GtkWidget        *add_button;
  GtkWidget        *remove_button;
  GtkWidget        *up_button;
  GtkWidget        *down_button;
  GtkWidget        *configure_button;
};


static void   make_dialog                      (ColorDisplayDialog *cdd);

static void   color_display_ok_callback        (GtkWidget          *widget,
                                                ColorDisplayDialog *cdd);
static void   color_display_cancel_callback    (GtkWidget          *widget,
                                                ColorDisplayDialog *cdd);
static void   color_display_add_callback       (GtkWidget          *widget,
                                                ColorDisplayDialog *cdd);
static void   color_display_remove_callback    (GtkWidget          *widget,
                                                ColorDisplayDialog *cdd);
static void   color_display_up_callback        (GtkWidget          *widget,
                                                ColorDisplayDialog *cdd);
static void   color_display_down_callback      (GtkWidget          *widget,
                                                ColorDisplayDialog *cdd);
static void   color_display_configure_callback (GtkWidget          *widget,
                                                ColorDisplayDialog *cdd);

static void   src_list_populate                (const char         *name,
                                                GtkTreeStore       *src);
static void   dest_list_populate               (GList              *node_list,
                                                GtkTreeStore       *dest);

static void   src_selection_changed            (GtkTreeSelection   *sel,
                                                ColorDisplayDialog *cdd);
static void   dest_selection_changed           (GtkTreeSelection   *sel,
                                                ColorDisplayDialog *cdd);

static void   color_display_update_up_and_down (ColorDisplayDialog *cdd);


static void
make_dialog (ColorDisplayDialog *cdd)
{
  GtkWidget *hbox;
  GtkWidget *editor;
  GtkWidget *scrolled_win;
  GtkWidget *tv;
  GtkWidget *vbox;
  GtkWidget *image;

  cdd->dialog = gimp_dialog_new (_("Color Display Filters"), "display_filters",
                                 gimp_standard_help_func,
                                 "dialogs/display_filters/display_filters.html",
                                 GTK_WIN_POS_NONE,
                                 FALSE, TRUE, FALSE,

                                 GTK_STOCK_CANCEL, color_display_cancel_callback,
                                 cdd, NULL, NULL, FALSE, TRUE,

                                 GTK_STOCK_OK, color_display_ok_callback,
                                 cdd, NULL, NULL, TRUE, FALSE,

                                 NULL);

  hbox = gtk_hbox_new (FALSE, 6);
  gtk_container_set_border_width (GTK_CONTAINER (hbox), 6);
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (cdd->dialog)->vbox), hbox,
		      TRUE, TRUE, 0);

  scrolled_win = gtk_scrolled_window_new (NULL, NULL);
  gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW (scrolled_win),
                                       GTK_SHADOW_IN);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled_win),
				  GTK_POLICY_AUTOMATIC, 
				  GTK_POLICY_AUTOMATIC);
  gtk_box_pack_start (GTK_BOX (hbox), scrolled_win, TRUE, TRUE, 0);
  gtk_widget_show (scrolled_win);

  cdd->src = gtk_tree_store_new (1, G_TYPE_STRING);
  tv = gtk_tree_view_new_with_model (GTK_TREE_MODEL (cdd->src));
  g_object_unref (G_OBJECT (cdd->src));

  gtk_widget_set_size_request (tv, LIST_WIDTH, LIST_HEIGHT);
  gtk_tree_view_set_headers_clickable (GTK_TREE_VIEW (tv), FALSE);

  gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (tv),
                                               0, _("Available Filters"),
                                               gtk_cell_renderer_text_new (),
                                               "text", 0, NULL);
  gtk_container_add (GTK_CONTAINER (scrolled_win), tv);
  gtk_widget_show (tv);

  cdd->src_sel = gtk_tree_view_get_selection (GTK_TREE_VIEW (tv));

  g_signal_connect (G_OBJECT (cdd->src_sel), "changed",
                    G_CALLBACK (src_selection_changed),
                    cdd);

  vbox = gtk_vbox_new (FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox), vbox, FALSE, FALSE, 0);
  gtk_widget_show (vbox);

  cdd->add_button = gtk_button_new ();
  gtk_box_pack_start (GTK_BOX (vbox), cdd->add_button, FALSE, FALSE, 16);
  gtk_widget_set_sensitive (cdd->add_button, FALSE);
  gtk_widget_show (cdd->add_button);

  image = gtk_image_new_from_stock (GTK_STOCK_GO_FORWARD, GTK_ICON_SIZE_BUTTON);
  gtk_container_add (GTK_CONTAINER (cdd->add_button), image);
  gtk_widget_show (image);

  gimp_help_set_help_data (cdd->add_button,
                           _("Add the selected filter to the list of "
                             "active filters."), NULL);

  g_signal_connect (G_OBJECT (cdd->add_button), "clicked",
                    G_CALLBACK (color_display_add_callback),
                    cdd);

  cdd->remove_button = gtk_button_new ();
  gtk_box_pack_start (GTK_BOX (vbox), cdd->remove_button, FALSE, FALSE, 0);
  gtk_widget_set_sensitive (cdd->remove_button, FALSE);
  gtk_widget_show (cdd->remove_button);

  image = gtk_image_new_from_stock (GTK_STOCK_GO_BACK, GTK_ICON_SIZE_BUTTON);
  gtk_container_add (GTK_CONTAINER (cdd->remove_button), image);
  gtk_widget_show (image);

  gimp_help_set_help_data (cdd->remove_button,
                           _("Remove the selected filter from the list of "
                             "active filters."), NULL);

  g_signal_connect (G_OBJECT (cdd->remove_button), "clicked",
                    G_CALLBACK (color_display_remove_callback),
                    cdd);

  editor = gimp_editor_new ();
  gtk_box_pack_start (GTK_BOX (hbox), editor, TRUE, TRUE, 0);
  gtk_widget_show (editor);

  cdd->up_button =
    gimp_editor_add_button (GIMP_EDITOR (editor),
                            GTK_STOCK_GO_UP,
                            _("Move the selected filter up"),
                            NULL,
                            G_CALLBACK (color_display_up_callback),
                            NULL,
                            cdd);

  cdd->down_button =
    gimp_editor_add_button (GIMP_EDITOR (editor),
                            GTK_STOCK_GO_DOWN,
                            _("Move the selected filter down"),
                            NULL,
                            G_CALLBACK (color_display_down_callback),
                            NULL,
                            cdd);

  cdd->configure_button =
    gimp_editor_add_button (GIMP_EDITOR (editor),
                            GIMP_STOCK_EDIT,
                            _("Configure the selected filter"),
                            NULL,
                            G_CALLBACK (color_display_configure_callback),
                            NULL,
                            cdd);

  gtk_widget_set_sensitive (cdd->up_button,        FALSE);
  gtk_widget_set_sensitive (cdd->down_button,      FALSE);
  gtk_widget_set_sensitive (cdd->configure_button, FALSE);

  scrolled_win = gtk_scrolled_window_new (NULL, NULL);
  gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW (scrolled_win),
                                       GTK_SHADOW_IN);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled_win),
				  GTK_POLICY_AUTOMATIC, 
				  GTK_POLICY_AUTOMATIC);
  gtk_container_add (GTK_CONTAINER (editor), scrolled_win);
  gtk_widget_show (scrolled_win);

  cdd->dest = gtk_tree_store_new (2, G_TYPE_STRING, G_TYPE_POINTER);
  tv = gtk_tree_view_new_with_model (GTK_TREE_MODEL (cdd->dest));
  g_object_unref (G_OBJECT (cdd->dest));

  gtk_widget_set_size_request (tv, LIST_WIDTH, LIST_HEIGHT);
  gtk_tree_view_set_headers_clickable (GTK_TREE_VIEW (tv), FALSE);

  gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (tv),
                                               0, _("Active Filters"),
                                               gtk_cell_renderer_text_new (),
                                               "text", 0, NULL);
  gtk_container_add (GTK_CONTAINER (scrolled_win), tv);
  gtk_widget_show (tv);

  cdd->dest_sel = gtk_tree_view_get_selection (GTK_TREE_VIEW (tv));

  g_signal_connect (G_OBJECT (cdd->dest_sel), "changed",
                    G_CALLBACK (dest_selection_changed),
                    cdd);

  gtk_widget_show (hbox);
}

static void
color_display_ok_callback (GtkWidget          *widget,
			   ColorDisplayDialog *cdd)
{
  GList *list;

  gtk_widget_destroy (GTK_WIDGET (cdd->dialog));
  cdd->shell->filters_dialog = NULL;

  if (cdd->modified)
    {
      for (list = cdd->old_nodes; list; list = g_list_next (list))
	{
	  if (! g_list_find (cdd->shell->filters, list->data))
	    gimp_display_shell_filter_detach_destroy (cdd->shell, list->data);
	}

      g_list_free (cdd->old_nodes);

      UPDATE_DISPLAY (cdd->shell);
    }
}

static void
color_display_cancel_callback (GtkWidget          *widget,
			       ColorDisplayDialog *cdd)
{
  GList *list;
  GList *next;

  gtk_widget_destroy (GTK_WIDGET (cdd->dialog));
  cdd->shell->filters_dialog = NULL;
  
  if (cdd->modified)
    {
      list = cdd->shell->filters;
      cdd->shell->filters = cdd->old_nodes;

      while (list)
	{
	  next = list->next;

	  if (! g_list_find (cdd->old_nodes, list->data))
	    gimp_display_shell_filter_detach_destroy (cdd->shell, list->data);

	  list = next;
	}

      UPDATE_DISPLAY (cdd->shell);
    }
}

static void 
color_display_update_up_and_down (ColorDisplayDialog *cdd)
{
  GtkTreeModel *model;
  GtkTreeIter   iter;
  gboolean      up_sensitive   = FALSE;
  gboolean      down_sensitive = FALSE;

  if (gtk_tree_selection_get_selected (cdd->dest_sel, &model, &iter))
    {
      GtkTreePath *path;
      gint        *indices;

      path    = gtk_tree_model_get_path (model, &iter);
      indices = gtk_tree_path_get_indices (path);

      up_sensitive   = indices[0] > 0;
      down_sensitive = indices[0] < (g_list_length (cdd->shell->filters) - 1);

      gtk_tree_path_free (path);
    }

  gtk_widget_set_sensitive (cdd->up_button,   up_sensitive);
  gtk_widget_set_sensitive (cdd->down_button, down_sensitive);
}

static void
color_display_add_callback (GtkWidget          *widget,
			    ColorDisplayDialog *cdd)
{
  GtkTreeModel *model;
  GtkTreeIter   iter;

  if (gtk_tree_selection_get_selected (cdd->src_sel, &model, &iter))
    {
      ColorDisplayNode *node;
      GValue            val = { 0, };

      gtk_tree_model_get_value (model, &iter, 0, &val);

      node = gimp_display_shell_filter_attach (cdd->shell,
                                               g_value_get_string (&val));

      g_value_unset (&val);

      gtk_tree_store_append (cdd->dest, &iter, NULL);

      gtk_tree_store_set (cdd->dest, &iter,
                          0, node->cd_name,
                          1, node,
                          -1);

      cdd->modified = TRUE;

      color_display_update_up_and_down (cdd);

      UPDATE_DISPLAY (cdd->shell);
    }
}

static void
color_display_remove_callback (GtkWidget          *widget,
			       ColorDisplayDialog *cdd)
{
  GtkTreeModel *model;
  GtkTreeIter   iter;

  if (gtk_tree_selection_get_selected (cdd->dest_sel, &model, &iter))
    {
      ColorDisplayNode *node;
      GValue            val = { 0, };

      gtk_tree_model_get_value (model, &iter, 1, &val);

      node = g_value_get_pointer (&val);

      g_value_unset (&val);

      gtk_tree_store_remove (cdd->dest, &iter);

      cdd->modified = TRUE;

      if (g_list_find (cdd->old_nodes, node))
        gimp_display_shell_filter_detach (cdd->shell, node);
      else
        gimp_display_shell_filter_detach_destroy (cdd->shell, node);

      color_display_update_up_and_down (cdd);

      UPDATE_DISPLAY (cdd->shell);
    }
}

static void
color_display_up_callback (GtkWidget          *widget,
			   ColorDisplayDialog *cdd)
{
  GtkTreeModel *model;
  GtkTreeIter   iter1;
  GtkTreeIter   iter2;

  if (gtk_tree_selection_get_selected (cdd->dest_sel, &model, &iter1))
    {
      GtkTreePath      *path;
      ColorDisplayNode *node1;
      ColorDisplayNode *node2;

      path = gtk_tree_model_get_path (model, &iter1);
      gtk_tree_path_prev (path);
      gtk_tree_model_get_iter (model, &iter2, path);
      gtk_tree_path_free (path);

      gtk_tree_model_get (model, &iter1, 1, &node1, -1);
      gtk_tree_model_get (model, &iter2, 1, &node2, -1);

      gtk_tree_store_set (GTK_TREE_STORE (model), &iter1,
                          0, node2->cd_name,
                          1, node2,
                          -1);
      gtk_tree_store_set (GTK_TREE_STORE (model), &iter2,
                          0, node1->cd_name,
                          1, node1,
                          -1);

      gimp_display_shell_filter_reorder_up (cdd->shell, node1);

      cdd->modified = TRUE;

      gtk_tree_selection_select_iter (cdd->dest_sel, &iter2);

      UPDATE_DISPLAY (cdd->shell);
    }
}

static void
color_display_down_callback (GtkWidget          *widget,
			     ColorDisplayDialog *cdd)
{
  GtkTreeModel *model;
  GtkTreeIter   iter1;
  GtkTreeIter   iter2;

  if (gtk_tree_selection_get_selected (cdd->dest_sel, &model, &iter1))
    {
      GtkTreePath      *path;
      ColorDisplayNode *node1;
      ColorDisplayNode *node2;

      path = gtk_tree_model_get_path (model, &iter1);
      gtk_tree_path_next (path);
      gtk_tree_model_get_iter (model, &iter2, path);
      gtk_tree_path_free (path);

      gtk_tree_model_get (model, &iter1, 1, &node1, -1);
      gtk_tree_model_get (model, &iter2, 1, &node2, -1);

      gtk_tree_store_set (GTK_TREE_STORE (model), &iter1,
                          0, node2->cd_name,
                          1, node2,
                          -1);
      gtk_tree_store_set (GTK_TREE_STORE (model), &iter2,
                          0, node1->cd_name,
                          1, node1,
                          -1);

      gimp_display_shell_filter_reorder_down (cdd->shell, node1);

      cdd->modified = TRUE;

      gtk_tree_selection_select_iter (cdd->dest_sel, &iter2);

      UPDATE_DISPLAY (cdd->shell);
    }
}

static void
color_display_configure_callback (GtkWidget          *widget,
				  ColorDisplayDialog *cdd)
{
  GtkTreeModel *model;
  GtkTreeIter   iter;

  if (gtk_tree_selection_get_selected (cdd->dest_sel, &model, &iter))
    {
      ColorDisplayNode *node;
      GValue            val = { 0, };

      gtk_tree_model_get_value (model, &iter, 1, &val);

      node = g_value_get_pointer (&val);

      g_value_unset (&val);

      if (! g_list_find (cdd->conf_nodes, node))
        cdd->conf_nodes = g_list_append (cdd->conf_nodes, node);

      gimp_display_shell_filter_configure (node, NULL, NULL, NULL, NULL);

      cdd->modified = TRUE;

      UPDATE_DISPLAY (cdd->shell);
    }
}

void
gimp_display_shell_filter_dialog_new (GimpDisplayShell *shell)
{
  ColorDisplayDialog *cdd;

  cdd = g_new0 (ColorDisplayDialog, 1);

  make_dialog (cdd);

  color_display_foreach ((GimpCDFunc) src_list_populate, cdd->src);

  cdd->old_nodes = shell->filters;
  dest_list_populate (shell->filters, cdd->dest);
  shell->filters = g_list_copy (cdd->old_nodes);

  cdd->shell = shell;

  shell->filters_dialog = cdd->dialog;
}

static void
src_list_populate (const char   *name,
		   GtkTreeStore *src)
{
  GtkTreeIter iter;

  gtk_tree_store_append (src, &iter, NULL);

  gtk_tree_store_set (src, &iter,
                      0, name,
                      -1);
}

static void
dest_list_populate (GList        *node_list,
    		    GtkTreeStore *dest)
{
  ColorDisplayNode *node;
  GList            *list;
  GtkTreeIter       iter;

  for (list = node_list; list; list = g_list_next (list))
    {
      node = (ColorDisplayNode *) list->data;

      gtk_tree_store_append (dest, &iter, NULL);

      gtk_tree_store_set (dest, &iter,
                          0, node->cd_name,
                          1, node,
                          -1);
    }
}

static void
src_selection_changed (GtkTreeSelection   *sel,
                       ColorDisplayDialog *cdd)
{
  GtkTreeModel *model;
  GtkTreeIter   iter;
  const gchar  *name = NULL;

  if (gtk_tree_selection_get_selected (sel, &model, &iter))
    {
      GValue val = { 0, };

      gtk_tree_model_get_value (model, &iter, 0, &val);

      name = g_value_get_string (&val);

      g_value_unset (&val);
    }

  gtk_widget_set_sensitive (cdd->add_button, (name != NULL));
}

static void
dest_selection_changed (GtkTreeSelection   *sel,
                        ColorDisplayDialog *cdd)
{
  GtkTreeModel     *model;
  GtkTreeIter       iter;
  ColorDisplayNode *node = NULL;

  if (gtk_tree_selection_get_selected (sel, &model, &iter))
    {
      GValue val = { 0, };

      gtk_tree_model_get_value (model, &iter, 1, &val);

      node = g_value_get_pointer (&val);

      g_value_unset (&val);
    }

  gtk_widget_set_sensitive (cdd->remove_button,    (node != NULL));
  gtk_widget_set_sensitive (cdd->configure_button, (node != NULL));

  color_display_update_up_and_down (cdd);
}
