/* Shift --- image filter plug-in for The Gimp image manipulation program
 * Copyright (C) 1997 Brian Degenhardt and Federico Mena Quintero
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Please direct all comments, questions, bug reports  etc to Brian Degenhardt
 * bdegenha@ucsd.edu
 *
 * You can contact Federico Mena Quintero at quartic@polloux.fciencias.unam.mx
 * You can contact the original The Gimp authors at gimp@xcf.berkeley.edu
 */

#include "config.h"

#include <stdio.h>
#include <stdlib.h>

#include <gtk/gtk.h>

#include <libgimp/gimp.h>
#include <libgimp/gimpui.h>

#include "libgimp/stdplugins-intl.h"


/* Some useful macros */

#define SPIN_BUTTON_WIDTH 8
#define TILE_CACHE_SIZE  16
#define HORIZONTAL        0
#define VERTICAL          1

typedef struct
{
  gint shift_amount;
  gint orientation;
} ShiftValues;

typedef struct
{
  gint run;
} ShiftInterface;


/* Declare local functions.
 */
static void    query  (void);
static void    run    (gchar      *name,
		       gint        nparams,
		       GimpParam  *param,
		       gint       *nreturn_vals,
		       GimpParam **return_vals);

static void    shift  (GimpDrawable *drawable);

static gint    shift_dialog      (gint32 image_ID);
static void    shift_ok_callback (GtkWidget *widget,
				  gpointer   data);
static void    shift_amount_update_callback(GtkWidget * widget, gpointer data);

/***** Local vars *****/

GimpPlugInInfo PLUG_IN_INFO =
{
  NULL,  /* init_proc  */
  NULL,  /* quit_proc  */
  query, /* query_proc */
  run,   /* run_proc   */
};

static ShiftValues shvals =
{
  5,          /* shift amount */
  HORIZONTAL  /* orientation  */
};

static ShiftInterface shint =
{
  FALSE   /*  run  */
};

/***** Functions *****/

MAIN ()

static void
query (void)
{
  static GimpParamDef args[] =
  {
    { GIMP_PDB_INT32, "run_mode", "Interactive, non-interactive" },
    { GIMP_PDB_IMAGE, "image", "Input image (unused)" },
    { GIMP_PDB_DRAWABLE, "drawable", "Input drawable" },
    { GIMP_PDB_INT32, "shift_amount", "shift amount (0 <= shift_amount_x <= 200)" },
    { GIMP_PDB_INT32, "orientation", "vertical, horizontal orientation" }
  };

  gimp_install_procedure ("plug_in_shift",
			  "Shift the contents of the specified drawable",
			  "Shifts the pixels of the specified drawable. Each row will be displaced a random value of pixels.",
			  "Spencer Kimball and Peter Mattis, ported by Brian Degenhardt and Federico Mena Quintero",
			  "Brian Degenhardt",
			  "1997",
			  N_("<Image>/Filters/Distorts/Shift..."),
			  "RGB*, GRAY*",
			  GIMP_PLUGIN,
			  G_N_ELEMENTS (args), 0,
			  args, NULL);
}

static void
run (gchar      *name,
     gint        nparams,
     GimpParam  *param,
     gint       *nreturn_vals,
     GimpParam **return_vals)
{
  static GimpParam   values[1];
  GimpDrawable      *drawable;
  gint32             image_ID;
  GimpRunMode        run_mode;
  GimpPDBStatusType  status = GIMP_PDB_SUCCESS;

  run_mode = param[0].data.d_int32;
  image_ID = param[1].data.d_int32;

  INIT_I18N ();

  /*  Get the specified drawable  */
  drawable = gimp_drawable_get (param[2].data.d_drawable);

  *nreturn_vals = 1;
  *return_vals  = values;

  values[0].type          = GIMP_PDB_STATUS;
  values[0].data.d_status = status;

  switch (run_mode)
    {
    case GIMP_RUN_INTERACTIVE:
      /*  Possibly retrieve data  */
      gimp_get_data ("plug_in_shift", &shvals);

      /*  First acquire information with a dialog  */
      if (! shift_dialog (image_ID))
	return;
      break;

    case GIMP_RUN_NONINTERACTIVE:
      /*  Make sure all the arguments are there!  */
      if (nparams != 5)
	{
	  status = GIMP_PDB_CALLING_ERROR;
	}
      else
	{
	  shvals.shift_amount = param[3].data.d_int32;
          shvals.orientation = (param[4].data.d_int32) ? HORIZONTAL : VERTICAL;

	  if (shvals.shift_amount < 0 || shvals.shift_amount > 200)
	    status = GIMP_PDB_CALLING_ERROR;
	}
      break;

    case GIMP_RUN_WITH_LAST_VALS:
      /*  Possibly retrieve data  */
      gimp_get_data ("plug_in_shift", &shvals);
      break;

    default:
      break;
    }

  if (status == GIMP_PDB_SUCCESS)
    {
      /*  Make sure that the drawable is gray or RGB color  */
      if (gimp_drawable_is_rgb (drawable->drawable_id) ||
	  gimp_drawable_is_gray (drawable->drawable_id))
	{
	  gimp_progress_init (_("Shifting..."));

	  /*  set the tile cache size  */
	  gimp_tile_cache_ntiles (TILE_CACHE_SIZE);

	  /*  run the shift effect  */
	  shift (drawable);

	  if (run_mode != GIMP_RUN_NONINTERACTIVE)
	    gimp_displays_flush ();

	  /*  Store data  */
	  if (run_mode == GIMP_RUN_INTERACTIVE)
	    gimp_set_data ("plug_in_shift", &shvals, sizeof (ShiftValues));
	}
      else
	{
	  /* gimp_message ("shift: cannot operate on indexed color images"); */
	  status = GIMP_PDB_EXECUTION_ERROR;
	}
    }

  values[0].data.d_status = status;

  gimp_drawable_detach (drawable);
}

static void
shift (GimpDrawable *drawable)
{
  GimpPixelRgn dest_rgn;
  gpointer  pr;
  GimpPixelFetcher *pft;
  gint    width, height;
  gint    bytes;
  guchar *destline;
  guchar *dest;
  gint    x1, y1, x2, y2;
  gint    x, y;
  gint    progress, max_progress;
  gint 	  amount;
  gint 	  xdist, ydist;

  GRand *gr; /* The random number generator we're using */

  gr = g_rand_new ();

  pft = gimp_pixel_fetcher_new (drawable);

  gimp_drawable_mask_bounds (drawable->drawable_id, &x1, &y1, &x2, &y2);

  width  = drawable->width;
  height = drawable->height;
  bytes  = drawable->bpp;

  progress     = 0;
  max_progress = (x2 - x1) * (y2 - y1);

  amount = shvals.shift_amount;

  /* Shift the image.  It's a pretty simple algorithm.  If horizontal
     is selected, then every row is shifted a random number of pixels
     in the range of -shift_amount/2 to shift_amount/2.  The effect is
     just reproduced with columns if vertical is selected.  Vertical
     has been added since 0.54 so that the user doesn't have to rotate
     the image to do a vertical shift.
  */

  gimp_pixel_rgn_init (&dest_rgn, drawable,
		       x1, y1, (x2 - x1), (y2 - y1), TRUE, TRUE);
  for (pr = gimp_pixel_rgns_register (1, &dest_rgn);
       pr != NULL;
       pr = gimp_pixel_rgns_process (pr))
    {
      destline = dest_rgn.data;

      if (shvals.orientation == VERTICAL)
        {
	  for (x = dest_rgn.x; x < dest_rgn.x + dest_rgn.w; x++)
            {
	      dest = destline;
	      ydist = g_rand_int_range (gr, -(amount + 1) / 2.0, 
					(amount + 1) / 2.0 );
	      for (y = dest_rgn.y; y < dest_rgn.y + dest_rgn.h; y++)
                {
		  gimp_pixel_fetcher_get_pixel2 (pft, x, y + ydist, 
						 PIXEL_WRAP, dest);
		  dest += dest_rgn.rowstride;
                }
	      destline += bytes;
            }
        }
      else
        {
	  for (y = dest_rgn.y; y < dest_rgn.y + dest_rgn.h; y++)
            {
	      dest = destline;
	      xdist = g_rand_int_range (gr, -(amount + 1) / 2.0, 
					(amount + 1) / 2.0);
	      for (x = dest_rgn.x; x < dest_rgn.x + dest_rgn.w; x++)
                {
		  gimp_pixel_fetcher_get_pixel2 (pft, x + xdist, y, 
						 PIXEL_WRAP, dest);
		  dest += bytes;
                }
	      destline += dest_rgn.rowstride;
            }
        }
      progress += dest_rgn.w * dest_rgn.h;
      gimp_progress_update ((double) progress / (double) max_progress);
    }

  gimp_pixel_fetcher_destroy (pft);

  /*  update the region  */
  gimp_drawable_flush (drawable);
  gimp_drawable_merge_shadow (drawable->drawable_id, TRUE);
  gimp_drawable_update (drawable->drawable_id, x1, y1, (x2 - x1), (y2 - y1));
}


static gint
shift_dialog (gint32 image_ID)
{
  GtkWidget *dlg;
  GtkWidget *frame;
  GtkWidget *radio_vbox;
  GtkWidget *sep;
  GtkWidget *size_entry;
  GimpUnit   unit;
  gdouble    xres;
  gdouble    yres;

  gimp_ui_init ("shift", FALSE);

  dlg = gimp_dialog_new (_("Shift"), "shift",
			 gimp_standard_help_func, "filters/shift.html",
			 GTK_WIN_POS_MOUSE,
			 FALSE, TRUE, FALSE,

			 GTK_STOCK_CANCEL, gtk_widget_destroy,
			 NULL, 1, NULL, FALSE, TRUE,
			 GTK_STOCK_OK, shift_ok_callback,
			 NULL, NULL, NULL, TRUE, FALSE,

			 NULL);

  g_signal_connect (dlg, "destroy",
                    G_CALLBACK (gtk_main_quit),
                    NULL);

  /*  parameter settings  */
  frame = gimp_radio_group_new2 (TRUE, _("Parameter Settings"),
                                 G_CALLBACK (gimp_radio_button_update),
                                 &shvals.orientation,
                                 GINT_TO_POINTER (shvals.orientation),

                                 _("Shift _Horizontally"),
                                 GINT_TO_POINTER (HORIZONTAL), NULL,

                                 _("Shift _Vertically"),
                                 GINT_TO_POINTER (VERTICAL), NULL,

                                 NULL);

  gtk_container_set_border_width (GTK_CONTAINER (frame), 6);
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dlg)->vbox), frame, TRUE, TRUE, 0);

  radio_vbox = GTK_BIN (frame)->child;
  gtk_container_set_border_width (GTK_CONTAINER (radio_vbox), 4);

  sep = gtk_hseparator_new ();
  gtk_box_pack_start (GTK_BOX (radio_vbox), sep, FALSE, FALSE, 3);
  gtk_widget_show (sep);

  /*  Get the image resolution and unit  */
  gimp_image_get_resolution (image_ID, &xres, &yres);
  unit = gimp_image_get_unit (image_ID);

  size_entry = gimp_size_entry_new (1, unit, "%a", TRUE, FALSE, FALSE, 
                                    SPIN_BUTTON_WIDTH, 
                                    GIMP_SIZE_ENTRY_UPDATE_SIZE);

  gimp_size_entry_set_unit (GIMP_SIZE_ENTRY (size_entry), GIMP_UNIT_PIXEL);
  gimp_size_entry_set_resolution (GIMP_SIZE_ENTRY (size_entry), 0, xres, TRUE);
  gimp_size_entry_set_refval_boundaries (GIMP_SIZE_ENTRY (size_entry), 0, 
					 1.0, 200.0);
  gtk_table_set_col_spacing (GTK_TABLE (size_entry), 0, 4);
  gtk_table_set_col_spacing (GTK_TABLE (size_entry), 2, 12);
  gimp_size_entry_set_refval (GIMP_SIZE_ENTRY (size_entry), 0, 
			      (gdouble) shvals.shift_amount);
  gimp_size_entry_attach_label (GIMP_SIZE_ENTRY (size_entry), 
				_("Shift _Amount:"), 1, 0, 0.0);

  g_signal_connect (size_entry, "value_changed",
                    G_CALLBACK (shift_amount_update_callback),
                    &shvals.shift_amount);
  gtk_box_pack_start (GTK_BOX (radio_vbox), size_entry, FALSE, FALSE, 0);
  gtk_widget_show (size_entry);

  gtk_widget_show (frame);

  gtk_widget_show (dlg);

  gtk_main ();
  gdk_flush ();

  return shint.run;
}

static void 
shift_amount_update_callback(GtkWidget * widget, gpointer data)
{
  shvals.shift_amount = gimp_size_entry_get_refval (GIMP_SIZE_ENTRY (widget), 
						    0);
}

static void
shift_ok_callback (GtkWidget *widget,
		   gpointer   data)
{
  shint.run = TRUE;

  gtk_widget_destroy (GTK_WIDGET (data));
}
