/* The GIMP -- an image manipulation program
 * Copyright (C) 1995 Spencer Kimball and Peter Mattis
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <string.h>

#include "appenv.h"
#include "colormaps.h"
#include "gdisplay_ops.h"
#include "gimprc.h"
#include "image_render.h"
#include "interface.h"
#include "layers_dialog.h"
#include "layer_select.h"
#include "session.h"

/*#include "config.h"*/
#include "libgimp/gimpchainbutton.h"
#include "libgimp/gimpfileselection.h"
#include "libgimp/gimpintl.h"
#include "libgimp/gimppatheditor.h"
#include "libgimp/gimpsizeentry.h"

/*  preferences local functions  */
static void file_prefs_ok_callback (GtkWidget *, GtkWidget *);
static void file_prefs_save_callback (GtkWidget *, GtkWidget *);
static void file_prefs_cancel_callback (GtkWidget *, GtkWidget *);

static void file_prefs_toggle_callback (GtkWidget *, gpointer);
static void file_prefs_preview_size_callback (GtkWidget *, gpointer);
static void file_prefs_mem_size_unit_callback (GtkWidget *, gpointer);
/* static void file_prefs_text_callback (GtkWidget *, gpointer); */
static void file_prefs_spinbutton_callback (GtkWidget *, gpointer);
/* static void file_prefs_float_spinbutton_callback (GtkWidget *, gpointer); */
static void file_prefs_string_callback (GtkWidget *, gpointer);
static void file_prefs_filename_callback (GtkWidget *, gpointer);
static void file_prefs_path_callback (GtkWidget *, gpointer);
static void file_prefs_clear_session_info_callback (GtkWidget *, gpointer);
static void file_prefs_default_size_callback (GtkWidget *, gpointer);
static void file_prefs_default_resolution_callback (GtkWidget *, gpointer);
static void file_prefs_res_source_callback (GtkWidget *, gpointer);
static void file_prefs_monitor_resolution_callback (GtkWidget *, gpointer);

/*  static variables  */

/* static   int          last_type = RGB; ??? */

static   GtkWidget   *prefs_dlg = NULL;
static   int          old_perfectmouse;
static   int          old_transparency_type;
static   int          old_transparency_size;
static   int          old_levels_of_undo;
static   int          old_marching_speed;
static   int          old_allow_resize_windows;
static   int          old_auto_save;
static   int          old_preview_size;
static   int          old_no_cursor_updating;
static   int          old_show_tool_tips;
static   int          old_show_rulers;
static   int          old_show_statusbar;
static   int          old_cubic_interpolation;
static   int          old_confirm_on_close;
static   int          old_save_session_info;
static   int          old_save_device_status;
static   int          old_always_restore_session;
static   int          old_default_width;
static   int          old_default_height;
static   GUnit        old_default_units;
static   float        old_default_xresolution;
static   float        old_default_yresolution;
static   GUnit        old_default_resolution_units;
static   int          old_default_type;
static   int          old_stingy_memory_use;
static   int          old_tile_cache_size;
static   int          old_install_cmap;
static   int          old_cycled_marching_ants;
static   int          old_last_opened_size;
static   char *       old_temp_path;
static   char *       old_swap_path;
static   char *       old_brush_path;
static   char *       old_pattern_path;
static   char *       old_palette_path;
static   char *       old_plug_in_path;
static   char *       old_module_path;
static   char *       old_gradient_path;
static   float        old_monitor_xres;
static   float        old_monitor_yres;
static   int          old_using_xserver_resolution;
static   int          old_num_processors;
static   char *       old_image_title_format;

static   char *       edit_temp_path = NULL;
static   char *       edit_swap_path = NULL;
static   char *       edit_brush_path = NULL;
static   char *       edit_pattern_path = NULL;
static   char *       edit_palette_path = NULL;
static   char *       edit_plug_in_path = NULL;
static   char *       edit_module_path = NULL;
static   char *       edit_gradient_path = NULL;
static   int          edit_stingy_memory_use;
static   int          edit_tile_cache_size;
static   int          edit_install_cmap;
static   int          edit_cycled_marching_ants;
static   int          edit_last_opened_size;
static   int          edit_num_processors;

static   GtkWidget   *tile_cache_size_spinbutton = NULL;
static   int          divided_tile_cache_size;
static   int          mem_size_unit;
static   GtkWidget   *default_size_sizeentry = NULL;
static   GtkWidget   *default_resolution_sizeentry = NULL;
static   GtkWidget   *default_resolution_force_equal = NULL;
static   GtkWidget   *resolution_xserver_label = NULL;
static   GtkWidget   *monitor_resolution_sizeentry = NULL;
static   GtkWidget   *monitor_resolution_force_equal = NULL;
static   GtkWidget   *num_processors_spinbutton = NULL;

/* Some information regarding preferences, compiled by Raph Levien 11/3/97.
   updated by Michael Natterer 27/3/99

   The following preference items cannot be set on the fly (at least
   according to the existing pref code - it may be that changing them
   so they're set on the fly is not hard).

   temp-path
   swap-path
   brush-path
   pattern-path
   plug-in-path
   module-path
   palette-path
   gradient-path
   stingy-memory-use
   tile-cache-size
   install-cmap
   cycled-marching-ants
   last-opened-size

   All of these now have variables of the form edit_temp_path, which
   are copied from the actual variables (e.g. temp_path) the first time
   the dialog box is started.

   Variables of the form old_temp_path represent the values at the
   time the dialog is opened - a cancel copies them back from old to
   the real variables or the edit variables, depending on whether they
   can be set on the fly.

   Here are the remaining issues as I see them:

   Still no settings for default-brush, default-gradient,
   default-palette, default-pattern, gamma-correction, color-cube.
   No widget for confirm-on-close although a lot of stuff is there.

   The semantics of "save" are a little funny - it only saves the
   settings that are different from the way they were when the dialog
   was opened. So you can set something, close the window, open it
   up again, click "save" and have nothing happen. To change this
   to more intuitive semantics, we should have a whole set of init_
   variables that are set the first time the dialog is opened (along
   with the edit_ variables that are currently set). Then, the save
   callback checks against the init_ variable rather than the old_.
*/

/* Copy the string from source to destination, freeing the string stored
   in the destination if there is one there already. */
static void
file_prefs_strset (char **dst,
		   char  *src)
{
  if (*dst != NULL)
    g_free (*dst);
  *dst = g_strdup (src);
}

/* Duplicate the string, but treat NULL as the empty string. */
static char *
file_prefs_strdup (char *src)
{
  return g_strdup (src == NULL ? "" : src);
}

/* Compare two strings, but treat NULL as the empty string. */
static int
file_prefs_strcmp (char *src1,
		   char *src2)
{
  return strcmp (src1 == NULL ? "" : src1,
		 src2 == NULL ? "" : src2);
}

static void
file_prefs_ok_callback (GtkWidget *widget,
			GtkWidget *dlg)
{
  edit_tile_cache_size = mem_size_unit * divided_tile_cache_size;

  if (levels_of_undo < 0) 
    {
      g_message (_("Error: Levels of undo must be zero or greater."));
      levels_of_undo = old_levels_of_undo;
      return;
    }
  if (num_processors < 1 || num_processors > 30) 
    {
      g_message (_("Error: Number of processors must be between 1 and 30."));
      num_processors = old_num_processors;
      return;
    }
  if (marching_speed < 50)
    {
      g_message (_("Error: Marching speed must be 50 or greater."));
      marching_speed = old_marching_speed;
      return;
    }
  if (default_width < 1)
    {
      g_message (_("Error: Default width must be one or greater."));
      default_width = old_default_width;
      return;
    }
  if (default_height < 1)
    {
      g_message (_("Error: Default height must be one or greater."));
      default_height = old_default_height;
      return;
    }
  if (default_units < UNIT_INCH ||
      default_units >= gimp_unit_get_number_of_units ())
    {
      g_message (_("Error: Default units must be within unit range."));
      default_units = old_default_units;
      return;
    }
  if (default_xresolution < 1e-5 || default_yresolution < 1e-5)
    {
      g_message (_("Error: default resolution must not be zero."));
      default_xresolution = old_default_xresolution;
      default_yresolution = old_default_yresolution;
      return;
    }
  if (default_resolution_units < UNIT_INCH ||
      default_resolution_units >= gimp_unit_get_number_of_units ())
    {
      g_message (_("Error: Default units must be within unit range."));
      default_resolution_units = old_default_resolution_units;
      return;
    }
  if (monitor_xres < 1e-5 || monitor_yres < 1e-5)
    {
      g_message (_("Error: monitor resolution must not be zero."));
      monitor_xres = old_monitor_xres;
      monitor_yres = old_monitor_yres;
      return;
    }
  if (image_title_format == NULL)
    {
      g_message (_("Error: image_title_format should never be NULL."));
      image_title_format = old_image_title_format;
      return;
    }

  gtk_widget_destroy (dlg);
  prefs_dlg = NULL;
  default_size_sizeentry = NULL;
  default_resolution_sizeentry = NULL;
  default_resolution_force_equal = NULL;
  resolution_xserver_label = NULL;
  monitor_resolution_sizeentry = NULL;
  monitor_resolution_force_equal = NULL;

  if (show_tool_tips)
    gtk_tooltips_enable (tool_tips);
  else
    gtk_tooltips_disable (tool_tips);
}

static void
file_prefs_save_callback (GtkWidget *widget,
			  GtkWidget *dlg)
{
  GList *update = NULL; /* options that should be updated in .gimprc */
  GList *remove = NULL; /* options that should be commented out */
  int    save_stingy_memory_use;
  int    save_tile_cache_size;
  int    save_num_processors;
  int    save_install_cmap;
  int    save_cycled_marching_ants;
  int    save_last_opened_size;
  gchar *save_temp_path;
  gchar *save_swap_path;
  gchar *save_brush_path;
  gchar *save_pattern_path;
  gchar *save_palette_path;
  gchar *save_plug_in_path;
  gchar *save_module_path;
  gchar *save_gradient_path;
  int    restart_notification = FALSE;

  file_prefs_ok_callback (widget, dlg);

  /* Save variables so that we can restore them later */
  save_stingy_memory_use = stingy_memory_use;
  save_tile_cache_size = tile_cache_size;
  save_install_cmap = install_cmap;
  save_cycled_marching_ants = cycled_marching_ants;
  save_last_opened_size = last_opened_size;
  save_temp_path = temp_path;
  save_swap_path = swap_path;
  save_brush_path = brush_path;
  save_pattern_path = pattern_path;
  save_palette_path = palette_path;
  save_plug_in_path = plug_in_path;
  save_module_path = module_path;
  save_gradient_path = gradient_path;
  save_num_processors = num_processors;

  if (levels_of_undo != old_levels_of_undo)
    update = g_list_append (update, "undo-levels");
  if (marching_speed != old_marching_speed)
    update = g_list_append (update, "marching-ants-speed");
  if (last_opened_size != old_last_opened_size)
    update = g_list_append (update, "last-opened-size");
  if (allow_resize_windows != old_allow_resize_windows)
    {
      update = g_list_append (update, "allow-resize-windows");
      remove = g_list_append (remove, "dont-allow-resize-windows");
    }
  if (auto_save != old_auto_save)
    {
      update = g_list_append (update, "auto-save");
      remove = g_list_append (remove, "dont-auto-save");
    }
  if (no_cursor_updating != old_no_cursor_updating)
    {
      update = g_list_append (update, "cursor-updating");
      remove = g_list_append (remove, "no-cursor-updating");
    }
  if (show_tool_tips != old_show_tool_tips)
    {
      update = g_list_append (update, "show-tool-tips");
      remove = g_list_append (remove, "dont-show-tool-tips");
    }
  if (show_rulers != old_show_rulers)
    {
      update = g_list_append (update, "show-rulers");
      remove = g_list_append (remove, "dont-show-rulers");
    }
  if (show_statusbar != old_show_statusbar)
    {
      update = g_list_append (update, "show-statusbar");
      remove = g_list_append (remove, "dont-show-statusbar");
    }
  if (cubic_interpolation != old_cubic_interpolation)
    update = g_list_append (update, "cubic-interpolation");
  if (confirm_on_close != old_confirm_on_close)
    {
      update = g_list_append (update, "confirm-on-close");
      remove = g_list_append (remove, "dont-confirm-on-close");
    }
  if (save_session_info != old_save_session_info)
    {
      update = g_list_append (update, "save-session-info");
      remove = g_list_append (remove, "dont-save-session-info");
    }
  if (save_device_status!= old_save_device_status)
    {
      update = g_list_append (update, "save-device-status");
      remove = g_list_append (remove, "dont-save-device-status");
    }
  if (always_restore_session != old_always_restore_session)
    update = g_list_append (update, "always-restore-session");
  if (default_width != old_default_width ||
      default_height != old_default_height)
    update = g_list_append (update, "default-image-size");
  if (default_units != old_default_units)
    update = g_list_append (update, "default-units");
  if (ABS(default_xresolution - old_default_xresolution) > 1e-5)
    update = g_list_append (update, "default-xresolution");
  if (ABS(default_yresolution - old_default_yresolution) > 1e-5)
    update = g_list_append (update, "default-yresolution");
  if (default_resolution_units != old_default_resolution_units)
    update = g_list_append (update, "default-resolution-units");
  if (default_type != old_default_type)
    update = g_list_append (update, "default-image-type");
  if (preview_size != old_preview_size)
    update = g_list_append (update, "preview-size");
  if (perfectmouse != old_perfectmouse)
    update = g_list_append (update, "perfect-mouse");
  if (transparency_type != old_transparency_type)
    update = g_list_append (update, "transparency-type");
  if (transparency_size != old_transparency_size)
    update = g_list_append (update, "transparency-size");
  if (using_xserver_resolution != old_using_xserver_resolution ||
      ABS(monitor_xres - old_monitor_xres) > 1e-5)
    update = g_list_append (update, "monitor-xresolution");
  if (using_xserver_resolution != old_using_xserver_resolution ||
      ABS(monitor_yres - old_monitor_yres) > 1e-5)
    update = g_list_append (update, "monitor-yresolution");
  if (edit_num_processors != num_processors)
    update = g_list_append (update, "num-processors");
  if (edit_stingy_memory_use != stingy_memory_use)
    {
      update = g_list_append (update, "stingy-memory-use");
      stingy_memory_use = edit_stingy_memory_use;
      restart_notification = TRUE;
    }
  if (edit_tile_cache_size != tile_cache_size)
    {
      update = g_list_append (update, "tile-cache-size");
      tile_cache_size = edit_tile_cache_size;
      restart_notification = TRUE;
    }
  if (edit_install_cmap != old_install_cmap)
    {
      update = g_list_append (update, "install-colormap");
      install_cmap = edit_install_cmap;
      restart_notification = TRUE;
    }
  if (edit_cycled_marching_ants != cycled_marching_ants)
    {
      update = g_list_append (update, "colormap-cycling");
      cycled_marching_ants = edit_cycled_marching_ants;
      restart_notification = TRUE;
    }
  if (edit_last_opened_size != last_opened_size)
    {
      update = g_list_append (update, "last-opened-size");
      last_opened_size = edit_last_opened_size;
      restart_notification = TRUE;
    }
  if (file_prefs_strcmp (temp_path, edit_temp_path))
    {
      update = g_list_append (update, "temp-path");
      temp_path = edit_temp_path;
      restart_notification = TRUE;
    }
  if (file_prefs_strcmp (swap_path, edit_swap_path))
    {
      update = g_list_append (update, "swap-path");
      swap_path = edit_swap_path;
      restart_notification = TRUE;
    }
  if (file_prefs_strcmp (brush_path, edit_brush_path))
    {
      update = g_list_append (update, "brush-path");
      brush_path = edit_brush_path;
      restart_notification = TRUE;
    }
  if (file_prefs_strcmp (pattern_path, edit_pattern_path))
    {
      update = g_list_append (update, "pattern-path");
      pattern_path = edit_pattern_path;
      restart_notification = TRUE;
    }
  if (file_prefs_strcmp (palette_path, edit_palette_path))
    {
      update = g_list_append (update, "palette-path");
      palette_path = edit_palette_path;
      restart_notification = TRUE;
    }
  if (file_prefs_strcmp (plug_in_path, edit_plug_in_path))
    {
      update = g_list_append (update, "plug-in-path");
      plug_in_path = edit_plug_in_path;
      restart_notification = TRUE;
    }
  if (file_prefs_strcmp (module_path, edit_module_path))
    {
      update = g_list_append (update, "module-path");
      module_path = edit_module_path;
      restart_notification = TRUE;
    }
  if (file_prefs_strcmp (gradient_path, edit_gradient_path))
    {
      update = g_list_append (update, "gradient-path");
      gradient_path = edit_gradient_path;
      restart_notification = TRUE;
    }
  if (using_xserver_resolution)
    {
      /* special value of 0 for either x or y res in the gimprc file
       * means use the xserver's current resolution */
      monitor_xres = 0.0;
      monitor_yres = 0.0;
    }
  if (file_prefs_strcmp (image_title_format, old_image_title_format))
    update = g_list_append (update, "image-title-format");

  save_gimprc (&update, &remove);

  if (using_xserver_resolution)
    gdisplay_xserver_resolution (&monitor_xres, &monitor_yres);

  /* Restore variables which must not change */
  stingy_memory_use = save_stingy_memory_use;
  tile_cache_size = save_tile_cache_size;
  install_cmap = save_install_cmap;
  cycled_marching_ants = save_cycled_marching_ants;
  last_opened_size = save_last_opened_size;
  temp_path = save_temp_path;
  swap_path = save_swap_path;
  brush_path = save_brush_path;
  pattern_path = save_pattern_path;
  palette_path = save_palette_path;
  plug_in_path = save_plug_in_path;
  module_path = save_module_path;
  gradient_path = save_gradient_path;

  if (restart_notification)
    g_message (_("You will need to restart GIMP for these changes to take "
		 "effect."));

  g_list_free (update);
  g_list_free (remove);
}

static void
file_prefs_cancel_callback (GtkWidget *widget,
			    GtkWidget *dlg)
{
  gtk_widget_destroy (dlg);
  prefs_dlg = NULL;
  default_size_sizeentry = NULL;
  default_resolution_sizeentry = NULL;
  default_resolution_force_equal = NULL;
  resolution_xserver_label = NULL;
  monitor_resolution_sizeentry = NULL;
  monitor_resolution_force_equal = NULL;

  levels_of_undo = old_levels_of_undo;
  marching_speed = old_marching_speed;
  allow_resize_windows = old_allow_resize_windows;
  auto_save = old_auto_save;
  no_cursor_updating = old_no_cursor_updating;
  perfectmouse = old_perfectmouse;
  show_tool_tips = old_show_tool_tips;
  show_rulers = old_show_rulers;
  show_statusbar = old_show_statusbar;
  cubic_interpolation = old_cubic_interpolation;
  confirm_on_close = old_confirm_on_close;
  save_session_info = old_save_session_info;
  save_device_status = old_save_device_status;
  default_width = old_default_width;
  default_height = old_default_height;
  default_units = old_default_units;
  default_xresolution = old_default_xresolution;
  default_yresolution = old_default_yresolution;
  default_resolution_units = old_default_resolution_units;
  default_type = old_default_type;
  monitor_xres = old_monitor_xres;
  monitor_yres = old_monitor_yres;
  using_xserver_resolution = old_using_xserver_resolution;
  num_processors = old_num_processors;

  if (preview_size != old_preview_size)
    {
      lc_dialog_rebuild (old_preview_size);
      layer_select_update_preview_size ();
    }

  if ((transparency_type != old_transparency_type) ||
      (transparency_size != old_transparency_size))
    {
      transparency_type = old_transparency_type;
      transparency_size = old_transparency_size;

      render_setup (transparency_type, transparency_size);
      gimage_foreach ((GFunc)layer_invalidate_previews, NULL);
      gimage_invalidate_previews ();
      gdisplays_expose_full ();
      gdisplays_flush ();
    }

  edit_stingy_memory_use = old_stingy_memory_use;
  edit_tile_cache_size = old_tile_cache_size;
  edit_install_cmap = old_install_cmap;
  edit_cycled_marching_ants = old_cycled_marching_ants;
  edit_last_opened_size = old_last_opened_size;

  file_prefs_strset (&edit_temp_path, old_temp_path);
  file_prefs_strset (&edit_swap_path, old_swap_path);
  file_prefs_strset (&edit_brush_path, old_brush_path);
  file_prefs_strset (&edit_pattern_path, old_pattern_path);
  file_prefs_strset (&edit_palette_path, old_palette_path);
  file_prefs_strset (&edit_plug_in_path, old_plug_in_path);
  file_prefs_strset (&edit_module_path, old_module_path);
  file_prefs_strset (&edit_gradient_path, old_gradient_path);

  file_prefs_strset (&image_title_format, old_image_title_format);
}

static void
file_prefs_toggle_callback (GtkWidget *widget,
			    gpointer   data)
{
  int *val;

  if (data == &allow_resize_windows)
    allow_resize_windows = GTK_TOGGLE_BUTTON (widget)->active;
  else if (data == &auto_save)
    auto_save = GTK_TOGGLE_BUTTON (widget)->active;
  else if (data == &no_cursor_updating)
    no_cursor_updating = GTK_TOGGLE_BUTTON (widget)->active;
  else if (data == &perfectmouse)
    perfectmouse = GTK_TOGGLE_BUTTON (widget)->active;
  else if (data == &show_tool_tips)
    show_tool_tips = GTK_TOGGLE_BUTTON (widget)->active;
  else if (data == &show_rulers)
    show_rulers = GTK_TOGGLE_BUTTON (widget)->active;
  else if (data == &show_statusbar)
    show_statusbar = GTK_TOGGLE_BUTTON (widget)->active;
  else if (data == &cubic_interpolation)
    cubic_interpolation = GTK_TOGGLE_BUTTON (widget)->active;
  else if (data == &confirm_on_close)
    confirm_on_close = GTK_TOGGLE_BUTTON (widget)->active;
  else if (data == &save_session_info)
    save_session_info = GTK_TOGGLE_BUTTON (widget)->active;
  else if (data == &save_device_status)
    save_device_status = GTK_TOGGLE_BUTTON (widget)->active;
  else if (data == &always_restore_session)
    always_restore_session = GTK_TOGGLE_BUTTON (widget)->active;
  else if (data == &edit_stingy_memory_use)
    edit_stingy_memory_use = GTK_TOGGLE_BUTTON (widget)->active;
  else if (data == &edit_install_cmap)
    edit_install_cmap = GTK_TOGGLE_BUTTON (widget)->active;
  else if (data == &edit_cycled_marching_ants)
    edit_cycled_marching_ants = GTK_TOGGLE_BUTTON (widget)->active;
  else if (data == &default_type)
    {
      default_type = (long) gtk_object_get_user_data (GTK_OBJECT (widget));
    } 
  else if ((data == &transparency_type) ||
	   (data == &transparency_size))
    {
      val = data;
      *val = (long) gtk_object_get_user_data (GTK_OBJECT (widget));
      render_setup (transparency_type, transparency_size);
      gimage_foreach ((GFunc)layer_invalidate_previews, NULL);
      gimage_invalidate_previews ();
      gdisplays_expose_full ();
      gdisplays_flush ();
    }
}

static void
file_prefs_preview_size_callback (GtkWidget *widget,
                                  gpointer   data)
{
  lc_dialog_rebuild ((long)data);
  layer_select_update_preview_size ();
}

static void
file_prefs_mem_size_unit_callback (GtkWidget *widget,
				   gpointer   data)
{
  int new_unit;

  new_unit = (int)data;

  if (new_unit != mem_size_unit)
    {
      divided_tile_cache_size =
	divided_tile_cache_size * mem_size_unit / new_unit;
      mem_size_unit = new_unit;

      gtk_spin_button_set_value (GTK_SPIN_BUTTON (tile_cache_size_spinbutton),
				 (float)divided_tile_cache_size);
    }
}

/*  commented out because it's not used 
static void
file_prefs_text_callback (GtkWidget *widget,
			  gpointer   data)
{
  int *val;
  
  val = data;
  *val = atoi (gtk_entry_get_text (GTK_ENTRY (widget)));
}
*/

static void
file_prefs_spinbutton_callback (GtkWidget *widget,
                                gpointer   data)
{
  int *val;

  val = data;
  *val = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (widget));
}

/*  commented out because it's not used 
static void
file_prefs_float_spinbutton_callback (GtkWidget *widget,
				      gpointer   data)
{
  float *val;

  val = data;
  *val = gtk_spin_button_get_value_as_float (GTK_SPIN_BUTTON (widget));
}
*/

static void
file_prefs_string_callback (GtkWidget *widget,
			    gpointer   data)
{
  gchar **val;

  val = data;
  file_prefs_strset (val, gtk_entry_get_text (GTK_ENTRY (widget)));
}

static void
file_prefs_filename_callback (GtkWidget *widget,
			      gpointer   data)
{
  gchar **val;

  val = data;
  file_prefs_strset (val, gimp_file_selection_get_filename (GIMP_FILE_SELECTION (widget)));
}

static void
file_prefs_path_callback (GtkWidget *widget,
			  gpointer   data)
{
  gchar **val;

  val = data;
  file_prefs_strset (val, gimp_path_editor_get_path (GIMP_PATH_EDITOR (widget)));
}

static void
file_prefs_clear_session_info_callback (GtkWidget *widget,
					gpointer   data)
{
  g_list_free (session_info_updates);
  session_info_updates = NULL;
}

static void
file_prefs_default_size_callback (GtkWidget *widget,
				  gpointer   data)
{
  default_width = gimp_size_entry_get_refval (GIMP_SIZE_ENTRY (widget), 0);
  default_height = gimp_size_entry_get_refval (GIMP_SIZE_ENTRY (widget), 1);
  default_units = gimp_size_entry_get_unit (GIMP_SIZE_ENTRY (widget));
}

static void
file_prefs_default_resolution_callback (GtkWidget *widget,
					gpointer   data)
{
  static float xres = 0.0;
  static float yres = 0.0;
  float new_xres;
  float new_yres;

  new_xres = gimp_size_entry_get_refval (GIMP_SIZE_ENTRY (widget), 0);
  new_yres = gimp_size_entry_get_refval (GIMP_SIZE_ENTRY (widget), 1);

  if (gimp_chain_button_get_active (GIMP_CHAIN_BUTTON (data)))
    {
      if (new_xres != xres)
	{
	  yres = new_yres = xres = new_xres;
	  gimp_size_entry_set_refval (GIMP_SIZE_ENTRY (widget), 1, yres);
	}

      if (new_yres != yres)
	{
	  xres = new_xres = yres = new_yres;
	  gimp_size_entry_set_refval (GIMP_SIZE_ENTRY (widget), 0, xres);
	}
    }
  else
    {
      if (new_xres != xres)
	xres = new_xres;
      if (new_yres != yres)
	yres = new_yres;
    }

  gimp_size_entry_set_resolution (GIMP_SIZE_ENTRY (default_size_sizeentry),
				  0, xres, FALSE);
  gimp_size_entry_set_resolution (GIMP_SIZE_ENTRY (default_size_sizeentry),
				  1, yres, FALSE);

  default_width =
    gimp_size_entry_get_refval (GIMP_SIZE_ENTRY (default_size_sizeentry), 0);
  default_height =
    gimp_size_entry_get_refval (GIMP_SIZE_ENTRY (default_size_sizeentry), 1);
  default_xresolution = xres;
  default_yresolution = yres;
  default_resolution_units = gimp_size_entry_get_unit (GIMP_SIZE_ENTRY (widget));
}

static void
file_prefs_res_source_callback (GtkWidget *widget,
				gpointer   data)
{
  if (resolution_xserver_label)
    gtk_widget_set_sensitive (resolution_xserver_label,
			      GTK_TOGGLE_BUTTON (widget)->active);

  if (monitor_resolution_sizeentry)
    gtk_widget_set_sensitive (monitor_resolution_sizeentry,
			      ! GTK_TOGGLE_BUTTON (widget)->active);

  if (monitor_resolution_force_equal)
    gtk_widget_set_sensitive (monitor_resolution_force_equal,
			      ! GTK_TOGGLE_BUTTON (widget)->active);

  if (GTK_TOGGLE_BUTTON (widget)->active)
    {
      gdisplay_xserver_resolution (&monitor_xres, &monitor_yres);
      using_xserver_resolution = TRUE;
    }
  else
    {
      if (monitor_resolution_sizeentry)
	{
	  monitor_xres =
	    gimp_size_entry_get_refval (GIMP_SIZE_ENTRY (monitor_resolution_sizeentry), 0);
	  monitor_yres =
	    gimp_size_entry_get_refval (GIMP_SIZE_ENTRY (monitor_resolution_sizeentry), 1);
	}
      using_xserver_resolution = FALSE;
    }
}

static void
file_prefs_monitor_resolution_callback (GtkWidget *widget,
					gpointer   data)
{
  static float xres = 0.0;
  static float yres = 0.0;
  float new_xres;
  float new_yres;

  new_xres = gimp_size_entry_get_refval (GIMP_SIZE_ENTRY (widget), 0);
  new_yres = gimp_size_entry_get_refval (GIMP_SIZE_ENTRY (widget), 1);

  if (gimp_chain_button_get_active (GIMP_CHAIN_BUTTON (data)))
    {
      if (new_xres != xres)
	{
	  yres = new_yres = xres = new_xres;
	  gimp_size_entry_set_refval (GIMP_SIZE_ENTRY (widget), 1, yres);
	}

      if (new_yres != yres)
	{
	  xres = new_xres = yres = new_yres;
	  gimp_size_entry_set_refval (GIMP_SIZE_ENTRY (widget), 0, xres);
	}
    }
  else
    {
      if (new_xres != xres)
	xres = new_xres;
      if (new_yres != yres)
	yres = new_yres;
    }

  monitor_xres = xres;
  monitor_yres = yres;
}

/* ********************************************************************
 *  convenience constructors test site ;)
 */

/*  local callback of gimp_dialog_new ()  */
static int
gimp_dialog_delete_callback (GtkWidget *widget,
			     GdkEvent  *event,
			     gpointer   data) 
{
  GtkSignalFunc  cancel_callback;
  GtkWidget     *cancel_widget;

  cancel_callback =
    (GtkSignalFunc) gtk_object_get_data (GTK_OBJECT (widget),
					 "gimp_dialog_cancel_callback");
  cancel_widget =
    (GtkWidget*) gtk_object_get_data (GTK_OBJECT (widget),
				      "gimp_dialog_cancel_widget");

  /* the cancel callback has to destroy the dialog */
  if (cancel_callback)
    (* cancel_callback) (cancel_widget, data);

  return TRUE;
}

/*  this is an experimental one
 *  I tried to fold the entire dialog creation and the ActionArea stuff
 *  into one function. Might be not general enough.
 *  todo:
 *   - session management?? (probably not)
 *   - window placement
 *   - policy setting
 */
GtkWidget*
gimp_dialog_new (const gchar   *title,
		 const gchar   *wmclass_name,

		 /* this is an action_area button */
		 gchar         *label1,
		 GtkSignalFunc  callback1,
		 gpointer       data1,
		 gboolean       default_action1,
		 gboolean       connect_delete1,

		 /* more action_area buttons */
		 ...)
{
  GtkWidget     *dialog;
  GtkWidget     *hbbox;
  GtkWidget     *button;

  va_list        args;
  gchar         *label;
  GtkSignalFunc  callback;
  gpointer       data;
  gboolean       default_action;
  gboolean       connect_delete;

  gboolean       delete_connected = FALSE;

  g_return_val_if_fail (title != NULL, NULL);
  g_return_val_if_fail (wmclass_name != NULL, NULL);
  g_return_val_if_fail (label1 != NULL, NULL);

  dialog = gtk_dialog_new ();
  gtk_window_set_wmclass (GTK_WINDOW (dialog), wmclass_name, "Gimp");
  gtk_window_set_title (GTK_WINDOW (dialog), title);

  /* prepare the action_area */
  gtk_container_border_width (GTK_CONTAINER (GTK_DIALOG (dialog)->action_area),
			      2);
  gtk_box_set_homogeneous (GTK_BOX (GTK_DIALOG (dialog)->action_area), FALSE);

  hbbox = gtk_hbutton_box_new ();
  gtk_button_box_set_spacing (GTK_BUTTON_BOX (hbbox), 4);
  gtk_box_pack_end (GTK_BOX (GTK_DIALOG (dialog)->action_area), hbbox,
		    FALSE, FALSE, 0);
  gtk_widget_show (hbbox);

  label = label1;
  callback = callback1;
  data = data1;
  default_action = default_action1;
  connect_delete = connect_delete1;
  va_start (args, connect_delete1);
  while (label)
    {
      button = gtk_button_new_with_label (label);
      GTK_WIDGET_SET_FLAGS (button, GTK_CAN_DEFAULT);
      gtk_box_pack_start (GTK_BOX (hbbox), button, FALSE, FALSE, 0);

      /* pass data as user_data if data != NULL, or the dialog otherwise */
      if (callback)
	gtk_signal_connect (GTK_OBJECT (button), "clicked",
			    GTK_SIGNAL_FUNC (callback),
			    data ? data : dialog);

      if (connect_delete && callback && !delete_connected)
	{
	  gtk_object_set_data (GTK_OBJECT (dialog),
			       "gimp_dialog_cancel_callback",
			       callback);
	  gtk_object_set_data (GTK_OBJECT (dialog),
			       "gimp_dialog_cancel_widget",
			       button);

	  /* catch WM delete event */
	  gtk_signal_connect (GTK_OBJECT (dialog), "delete_event",
			      (GdkEventFunc) gimp_dialog_delete_callback,
			      data ? data : dialog);

	  delete_connected = TRUE;
	}

      if (default_action)
	gtk_widget_grab_default (button);
      gtk_widget_show (button);

      label = va_arg (args, gchar*);
      if (label)
	{
	  callback = va_arg (args, GtkSignalFunc);
	  data = va_arg (args, gpointer);
	  default_action = va_arg (args, gboolean);
	  connect_delete = va_arg (args, gboolean);
	}
    }
  va_end (args);

  /* catch WM delete event if not already done*/
  if (! delete_connected)
    gtk_signal_connect (GTK_OBJECT (dialog), "delete_event",
			(GdkEventFunc) gimp_dialog_delete_callback,
			NULL);

  return dialog;
}

GtkWidget*
gimp_option_menu_new (GtkSignalFunc  menu_item_callback,
		      gpointer       initial,

		      /* this is a menu item */
		      gchar         *label1,
		      gpointer       data1,
		      gpointer       set_data1,

		      /* more menu items */
		      ...)
{
  GtkWidget *menu;
  GtkWidget *menuitem;
  GtkWidget *optionmenu;

  va_list    args;
  gchar     *label;
  gpointer   data;
  gpointer   set_data;
  gint       i;
  gint       initial_index;

  g_return_val_if_fail (label1 != NULL, NULL);

  menu = gtk_menu_new ();

  initial_index = 0;
  label = label1;
  data = data1;
  set_data = set_data1;
  va_start (args, set_data1);
  for (i = 0; label; i++)
    {
      menuitem = gtk_menu_item_new_with_label (label);
      gtk_menu_append (GTK_MENU (menu), menuitem);
      gtk_signal_connect (GTK_OBJECT (menuitem), "activate",
			  menu_item_callback, data);
      gtk_object_set_user_data (GTK_OBJECT (menuitem), set_data);
      gtk_widget_show (menuitem);

      /* remember the initial menu item */
      if (set_data == initial)
	initial_index = i;

      label = va_arg (args, gchar*);
      if (label)
	{
	  data = va_arg (args, gpointer);
	  set_data = va_arg (args, gpointer);
	}
    }
  va_end (args);

  optionmenu = gtk_option_menu_new ();
  gtk_option_menu_set_menu (GTK_OPTION_MENU (optionmenu), menu);

  /* select the initial menu item */
  gtk_option_menu_set_history (GTK_OPTION_MENU (optionmenu), initial_index);

  return optionmenu;
}

GtkWidget*
gimp_radio_group_new (GtkSignalFunc  radio_button_callback,
		      gpointer       initial,

		      /* this is a radio button */
		      gchar         *label1,
		      gpointer       data1,
		      gpointer       set_data1,

		      /* more radio buttons */
		      ...)
{
  GtkWidget *vbox;
  GtkWidget *button;
  GSList    *group;

  va_list    args;
  gchar     *label;
  gpointer   data;
  gpointer   set_data;

  g_return_val_if_fail (label1 != NULL, NULL);

  vbox = gtk_vbox_new (FALSE, 1);
  gtk_container_set_border_width (GTK_CONTAINER (vbox), 2);
  group = NULL;

  label = label1;
  data = data1;
  set_data = set_data1;
  va_start (args, set_data1);
  while (label)
    {
      button = gtk_radio_button_new_with_label (group, label);
      group = gtk_radio_button_group (GTK_RADIO_BUTTON (button));
      gtk_box_pack_start (GTK_BOX (vbox), button, FALSE, FALSE, 0);
      gtk_signal_connect (GTK_OBJECT (button), "toggled",
			  (GtkSignalFunc) radio_button_callback,
			  data);
      gtk_object_set_user_data (GTK_OBJECT (button), set_data);

      /* press the initially active radio button */
      if (set_data == initial)
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (button), TRUE);

      gtk_widget_show (button);

      label = va_arg (args, gchar*);
      if (label)
	{
	  data = va_arg (args, gpointer);
	  set_data = va_arg (args, gpointer);
	}
    }
  va_end (args);

  return vbox;
}

/*  this might be the standard gimp spinbutton  */
GtkWidget*
gimp_spin_button_new (gfloat   value,
		      gfloat   lower,
		      gfloat   upper,
		      gfloat   step_increment,
		      gfloat   page_increment,
		      gfloat   page_size,
		      gfloat   climb_rate,
		      guint    digits)
{
  GtkWidget *spinbutton;

  spinbutton =
    gtk_spin_button_new (GTK_ADJUSTMENT (gtk_adjustment_new (value,
							     lower,
							     upper,
							     step_increment,
							     page_increment,
							     page_size)),
			 climb_rate,
			 digits);
  gtk_spin_button_set_shadow_type (GTK_SPIN_BUTTON (spinbutton),
				   GTK_SHADOW_NONE);
  gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (spinbutton), TRUE);
  gtk_widget_set_usize (spinbutton, 75, 0);

  return spinbutton;
}


/**********************************************************************
 *  preferences-specific GUI helpers
 */

/*  create a new notebook page  */
static GtkWidget*
file_prefs_notebook_append_page (GtkNotebook   *notebook,
				 gchar         *notebook_label,
				 GtkCTree      *ctree,
				 gchar         *tree_label,
				 GtkCTreeNode  *parent,
				 GtkCTreeNode **new_node,
				 gint           page_index)
{
  GtkWidget *out_vbox;
  GtkWidget *vbox;
  GtkWidget *button;
  gchar     *titles[1];

  out_vbox = gtk_vbox_new (FALSE, 0);
  gtk_widget_show (out_vbox);

  button = gtk_button_new_with_label (notebook_label);
  gtk_misc_set_alignment (GTK_MISC (GTK_BIN (button)->child), 0.0, 0.5);
  gtk_box_pack_start (GTK_BOX (out_vbox), button, FALSE, TRUE, 0);
  gtk_widget_show (button);

  vbox = gtk_vbox_new (FALSE, 2);
  gtk_container_set_border_width (GTK_CONTAINER (vbox), 4);
  gtk_container_add (GTK_CONTAINER (out_vbox), vbox);
  gtk_widget_show (vbox);

  titles[0] = tree_label;
  *new_node = gtk_ctree_insert_node (ctree, parent, NULL,
				     titles, 0,
				     NULL, NULL, NULL, NULL,
				     FALSE, TRUE);
  gtk_ctree_node_set_row_data (ctree,
			       *new_node, (gpointer) page_index);
  gtk_notebook_append_page (notebook, out_vbox, NULL);

  return vbox;
}

/*  select a notebook page  */
static void
file_pref_tree_select_callback (GtkWidget    *widget,
				GtkCTreeNode *node)
{
  GtkNotebook *notebook;
  gint         page;

  if (! GTK_CLIST (widget)->selection)
    return;

  notebook = (GtkNotebook*) gtk_object_get_user_data (GTK_OBJECT (widget));
  page = (gint) gtk_ctree_node_get_row_data (GTK_CTREE (widget), node);

  gtk_notebook_set_page (notebook, page);
}

/*  add correctly aligned label & widget to a prefs table  */
static void
file_pref_table_attach (GtkTable   *table,
			GtkLabel   *label,
			GtkWidget  *widget,
			gint        row)
{
  GtkWidget *hbox;

  gtk_misc_set_alignment (GTK_MISC (label), 1.0, 0.5);
  gtk_table_attach_defaults (table, GTK_WIDGET (label), 0, 1, row, row + 1);

  hbox = gtk_hbox_new (FALSE, 0);
  gtk_table_attach_defaults (table, hbox, 1, 2, row, row + 1);
  gtk_widget_show (hbox);
  gtk_box_pack_start (GTK_BOX (hbox), widget, FALSE, FALSE, 0);
}

/************************************************************************
 *  create the preferences dialog
 */
void
file_pref_cmd_callback (GtkWidget *widget,
			gpointer   client_data)
{
  GtkWidget     *ctree;
  gchar         *titles[1];
  GtkCTreeNode  *top_insert;
  GtkCTreeNode  *child_insert;
  gint           page_index;

  GtkWidget     *frame;
  GtkWidget     *notebook;
  GtkWidget     *vbox;
  GtkWidget     *vbox2;
  GtkWidget     *hbox;
  GtkWidget     *abox;
  GtkWidget     *label;
  GtkWidget     *button;
  GtkWidget     *fileselection;
  GtkWidget     *patheditor;
  GtkWidget     *spinbutton;
  GtkWidget     *combo;
  GtkWidget     *comboitem;
  GtkWidget     *optionmenu;
  GtkWidget     *table;
  GSList        *group;

  int            i;

  if (prefs_dlg)
    {
      gdk_window_raise (GTK_WIDGET (prefs_dlg)->window);
      return;
    }

  if (edit_temp_path == NULL)
    {
      /* first time dialog is opened - copy config vals to edit
	 variables. */
      edit_temp_path = file_prefs_strdup (temp_path);	
      edit_swap_path = file_prefs_strdup (swap_path);
      edit_brush_path = file_prefs_strdup (brush_path);
      edit_pattern_path = file_prefs_strdup (pattern_path);
      edit_palette_path = file_prefs_strdup (palette_path);
      edit_plug_in_path = file_prefs_strdup (plug_in_path);
      edit_module_path = file_prefs_strdup (module_path);
      edit_gradient_path = file_prefs_strdup (gradient_path);
      edit_stingy_memory_use = stingy_memory_use;
      edit_tile_cache_size = tile_cache_size;
      edit_install_cmap = install_cmap;
      edit_cycled_marching_ants = cycled_marching_ants;
      edit_last_opened_size = last_opened_size;
    }
  old_perfectmouse = perfectmouse;
  old_transparency_type = transparency_type;
  old_transparency_size = transparency_size;
  old_levels_of_undo = levels_of_undo;
  old_marching_speed = marching_speed;
  old_allow_resize_windows = allow_resize_windows;
  old_auto_save = auto_save;
  old_preview_size = preview_size;
  old_no_cursor_updating = no_cursor_updating;
  old_show_tool_tips = show_tool_tips;
  old_show_rulers = show_rulers;
  old_show_statusbar = show_statusbar;
  old_cubic_interpolation = cubic_interpolation;
  old_confirm_on_close = confirm_on_close;
  old_save_session_info = save_session_info;
  old_save_device_status = save_device_status;
  old_always_restore_session = always_restore_session;
  old_default_width = default_width;
  old_default_height = default_height;
  old_default_units = default_units;
  old_default_xresolution = default_xresolution;
  old_default_yresolution = default_yresolution;
  old_default_resolution_units = default_resolution_units;
  old_default_type = default_type;
  old_stingy_memory_use = edit_stingy_memory_use;
  old_tile_cache_size = edit_tile_cache_size;
  old_install_cmap = edit_install_cmap;
  old_cycled_marching_ants = edit_cycled_marching_ants;
  old_last_opened_size = edit_last_opened_size;
  old_monitor_xres = monitor_xres;
  old_monitor_yres = monitor_yres;
  old_using_xserver_resolution = using_xserver_resolution;
  old_num_processors = num_processors;
  old_image_title_format = file_prefs_strdup (image_title_format);	

  file_prefs_strset (&old_temp_path, edit_temp_path);
  file_prefs_strset (&old_swap_path, edit_swap_path);
  file_prefs_strset (&old_brush_path, edit_brush_path);
  file_prefs_strset (&old_pattern_path, edit_pattern_path);
  file_prefs_strset (&old_palette_path, edit_palette_path);
  file_prefs_strset (&old_plug_in_path, edit_plug_in_path);
  file_prefs_strset (&old_module_path, edit_module_path);
  file_prefs_strset (&old_gradient_path, edit_gradient_path);

  mem_size_unit = 1;
  for (i = 0; i < 3; i++)
    { 
      if (edit_tile_cache_size % (mem_size_unit * 1024) != 0)
	break;
      mem_size_unit *= 1024;
    }
  divided_tile_cache_size = edit_tile_cache_size / mem_size_unit;

  /* Create the dialog */
  prefs_dlg =
    gimp_dialog_new (_("Preferences"),
		     "gimp_preferences",
		     _("OK"), file_prefs_ok_callback, NULL,
		     FALSE, FALSE,
		     _("Save"), file_prefs_save_callback, NULL,
		     FALSE, FALSE,
		     _("Cancel"), file_prefs_cancel_callback, NULL,
		     TRUE, TRUE,
		     NULL);

  /* The main hbox */
  hbox = gtk_hbox_new (FALSE, 6);
  gtk_container_set_border_width (GTK_CONTAINER (hbox), 6);
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (prefs_dlg)->vbox),
		      hbox, TRUE, TRUE, 0);
  gtk_widget_show (hbox);

  /* The categories tree */
  titles[0] = _("Categories");
  ctree = gtk_ctree_new_with_titles (1, 0, titles);
  gtk_ctree_set_indent (GTK_CTREE (ctree), 15);
  gtk_widget_set_usize (ctree, 140, 0);
  gtk_box_pack_start (GTK_BOX (hbox), ctree, FALSE, FALSE, 0);
  gtk_widget_show (ctree);

  /* The main preferences notebook */
  frame = gtk_frame_new (NULL);
  gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_IN);
  gtk_box_pack_start (GTK_BOX (hbox), frame, TRUE, TRUE, 0);
  gtk_widget_show (frame);

  notebook = gtk_notebook_new ();
  gtk_notebook_set_show_tabs (GTK_NOTEBOOK (notebook), FALSE);
  gtk_notebook_set_show_border (GTK_NOTEBOOK (notebook), FALSE);
  gtk_container_add (GTK_CONTAINER (frame), notebook);

  gtk_object_set_user_data (GTK_OBJECT (ctree), notebook);
  gtk_signal_connect (GTK_OBJECT (ctree), "tree_select_row",
		      (GtkSignalFunc) file_pref_tree_select_callback, NULL);

  page_index = 0;

  /* New File page */
  vbox = file_prefs_notebook_append_page (GTK_NOTEBOOK (notebook),
					  _("New File Settings"),
					  GTK_CTREE (ctree),
					  _("New File"),
					  NULL,
					  &top_insert,
					  page_index);
  gtk_widget_show (vbox);
  page_index++;

  /* select this page in the tree */
  gtk_ctree_select (GTK_CTREE (ctree), top_insert);

  frame = gtk_frame_new (_("Default Image Size and Unit")); 
  gtk_box_pack_start (GTK_BOX (vbox), frame, FALSE, FALSE, 0);
  gtk_widget_show (frame);

  hbox = gtk_hbox_new (FALSE, 0);
  gtk_container_set_border_width (GTK_CONTAINER (hbox), 2);
  gtk_container_add (GTK_CONTAINER (frame), hbox);
  gtk_widget_show (hbox);

  default_size_sizeentry = gimp_size_entry_new (2, default_units, "%p",
						FALSE, TRUE, 75,
						GIMP_SIZE_ENTRY_UPDATE_SIZE);
  gimp_size_entry_set_resolution (GIMP_SIZE_ENTRY (default_size_sizeentry),
				  0, default_xresolution, FALSE);
  gimp_size_entry_set_resolution (GIMP_SIZE_ENTRY (default_size_sizeentry),
				  1, default_yresolution, FALSE);
  gimp_size_entry_set_refval_boundaries (GIMP_SIZE_ENTRY (default_size_sizeentry), 0, 1, 32767);
  gimp_size_entry_set_refval_boundaries (GIMP_SIZE_ENTRY (default_size_sizeentry), 1, 1, 32767);
  gimp_size_entry_set_refval (GIMP_SIZE_ENTRY (default_size_sizeentry), 0,
			      default_width);
  gimp_size_entry_set_refval (GIMP_SIZE_ENTRY (default_size_sizeentry), 1,
			      default_height);
  gimp_size_entry_attach_label (GIMP_SIZE_ENTRY (default_size_sizeentry),
				_("Width"), 0, 1, 0.0);
  gimp_size_entry_attach_label (GIMP_SIZE_ENTRY (default_size_sizeentry),
				_("Height"), 0, 2, 0.0);
  gimp_size_entry_attach_label (GIMP_SIZE_ENTRY (default_size_sizeentry),
				_("Pixels"), 1, 4, 0.0);
  gtk_signal_connect (GTK_OBJECT (default_size_sizeentry), "unit_changed",
		      (GtkSignalFunc)file_prefs_default_size_callback, NULL);
  gtk_signal_connect (GTK_OBJECT (default_size_sizeentry), "value_changed",
		      (GtkSignalFunc)file_prefs_default_size_callback, NULL);
  gtk_signal_connect (GTK_OBJECT (default_size_sizeentry), "refval_changed",
		      (GtkSignalFunc)file_prefs_default_size_callback, NULL);
  gtk_box_pack_start (GTK_BOX (hbox), default_size_sizeentry, FALSE, FALSE, 0);
  gtk_widget_show (default_size_sizeentry);

  frame = gtk_frame_new (_("Default Image Resolution and Resolution Unit"));
  gtk_box_pack_start (GTK_BOX (vbox), frame, FALSE, FALSE, 0);
  gtk_widget_show (frame);

  hbox = gtk_hbox_new (FALSE, 0);
  gtk_container_set_border_width (GTK_CONTAINER (hbox), 2);
  gtk_container_add (GTK_CONTAINER (frame), hbox);
  gtk_widget_show (hbox);

  default_resolution_force_equal = gimp_chain_button_new (GIMP_CHAIN_BOTTOM);

  default_resolution_sizeentry =
    gimp_size_entry_new (2, default_resolution_units, "%s", FALSE, TRUE, 75,
			 GIMP_SIZE_ENTRY_UPDATE_RESOLUTION);
  gimp_size_entry_set_refval_boundaries (GIMP_SIZE_ENTRY (default_resolution_sizeentry), 0, 1, 32767);
  gimp_size_entry_set_refval_boundaries (GIMP_SIZE_ENTRY (default_resolution_sizeentry), 1, 1, 32767);
  gimp_size_entry_set_refval (GIMP_SIZE_ENTRY (default_resolution_sizeentry),
			      0, default_xresolution);
  gimp_size_entry_set_refval (GIMP_SIZE_ENTRY (default_resolution_sizeentry),
			      1, default_yresolution);
  gimp_size_entry_attach_label (GIMP_SIZE_ENTRY (default_resolution_sizeentry),
				_("Horizontal"), 0, 1, 0.0);
  gimp_size_entry_attach_label (GIMP_SIZE_ENTRY (default_resolution_sizeentry),
				_("Vertical"), 0, 2, 0.0);
  gimp_size_entry_attach_label (GIMP_SIZE_ENTRY (default_resolution_sizeentry),
				_("dpi"), 1, 3, 0.0);
  gimp_size_entry_attach_label (GIMP_SIZE_ENTRY (default_resolution_sizeentry),
				_("Pixels per "), 2, 3, 0.0);
  gtk_signal_connect (GTK_OBJECT (default_resolution_sizeentry),
		      "unit_changed",
		      (GtkSignalFunc)file_prefs_default_resolution_callback,
		      default_resolution_force_equal);
  gtk_signal_connect (GTK_OBJECT (default_resolution_sizeentry),
		      "value_changed",
		      (GtkSignalFunc)file_prefs_default_resolution_callback,
		      default_resolution_force_equal);
  gtk_signal_connect (GTK_OBJECT (default_resolution_sizeentry),
		      "refval_changed",
		      (GtkSignalFunc)file_prefs_default_resolution_callback,
		      default_resolution_force_equal);
  gtk_box_pack_start (GTK_BOX (hbox), default_resolution_sizeentry,
		      FALSE, FALSE, 0);
  gtk_widget_show (default_resolution_sizeentry);

  gtk_table_attach_defaults (GTK_TABLE (default_resolution_sizeentry), 
			     default_resolution_force_equal, 1, 3, 3, 4);
  if (ABS (default_xresolution - default_yresolution) < 1e-5)
    gimp_chain_button_set_active (GIMP_CHAIN_BUTTON (default_resolution_force_equal), TRUE);
  gtk_widget_show (default_resolution_force_equal);

  hbox = gtk_hbox_new (FALSE, 2);
  gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);
  gtk_widget_show (hbox);
      
  table = gtk_table_new (1, 2, FALSE);
  gtk_container_set_border_width (GTK_CONTAINER (table), 2);
  gtk_table_set_row_spacings (GTK_TABLE (table), 2);
  gtk_table_set_col_spacings (GTK_TABLE (table), 4);
  gtk_box_pack_start (GTK_BOX (hbox), table, FALSE, FALSE, 0);
  gtk_widget_show (table);

  label = gtk_label_new (_("Default Image Type:"));
  optionmenu =
    gimp_option_menu_new (file_prefs_toggle_callback,
			  (gpointer) default_type,
			  _("RGB"),       &default_type, (gpointer) RGB,
			  _("Grayscale"), &default_type, (gpointer) GRAY,
			  NULL);
  file_pref_table_attach (GTK_TABLE (table), GTK_LABEL (label), optionmenu, 0);
  gtk_widget_show (label);
  gtk_widget_show (optionmenu);

  /* Display page */
  vbox = file_prefs_notebook_append_page (GTK_NOTEBOOK (notebook),
					  _("Display Settings"),
					  GTK_CTREE (ctree),
					  _("Display"),
					  NULL,
					  &top_insert,
					  page_index);
  gtk_widget_show (vbox);
  page_index++;

  frame = gtk_frame_new (_("Transparency")); 
  gtk_box_pack_start (GTK_BOX (vbox), frame, FALSE, FALSE, 0);
  gtk_widget_show (frame);

  hbox = gtk_hbox_new (FALSE, 2);
  gtk_container_add (GTK_CONTAINER (frame), hbox);
  gtk_widget_show (hbox);
      
  table = gtk_table_new (2, 2, FALSE);
  gtk_container_set_border_width (GTK_CONTAINER (table), 2);
  gtk_table_set_row_spacings (GTK_TABLE (table), 2);
  gtk_table_set_col_spacings (GTK_TABLE (table), 4);
  gtk_box_pack_start (GTK_BOX (hbox), table, FALSE, FALSE, 0);
  gtk_widget_show (table);

  label = gtk_label_new (_("Transparency Type:"));
  optionmenu =
    gimp_option_menu_new (file_prefs_toggle_callback,
			  (gpointer) transparency_type,
			  _("Light Checks"),
			  &transparency_type, (gpointer) LIGHT_CHECKS,
			  _("Mid-Tone Checks"),
			  &transparency_type, (gpointer) GRAY_CHECKS,
			  _("Dark Checks"),
			  &transparency_type, (gpointer) DARK_CHECKS,
			  _("White Only"),
			  &transparency_type, (gpointer) WHITE_ONLY,
			  _("Gray Only"),
			  &transparency_type, (gpointer) GRAY_ONLY,
			  _("Black Only"),
			  &transparency_type, (gpointer) BLACK_ONLY,
			  NULL);
  file_pref_table_attach (GTK_TABLE (table), GTK_LABEL (label), optionmenu, 0);
  gtk_widget_show (label);
  gtk_widget_show (optionmenu);

  label = gtk_label_new (_("Check Size:"));
  optionmenu =
    gimp_option_menu_new (file_prefs_toggle_callback,
			  (gpointer) transparency_size,
			  _("Small Checks"),
			  &transparency_size, (gpointer) SMALL_CHECKS,
			  _("Medium Checks"),
			  &transparency_size, (gpointer) MEDIUM_CHECKS,
			  _("Large Checks"), 
			  &transparency_size, (gpointer) LARGE_CHECKS,
			  NULL);
  file_pref_table_attach (GTK_TABLE (table), GTK_LABEL (label), optionmenu, 1);
  gtk_widget_show (label);
  gtk_widget_show (optionmenu);

  frame = gtk_frame_new (_("Scaling")); 
  gtk_box_pack_start (GTK_BOX (vbox), frame, FALSE, FALSE, 0);
  gtk_widget_show (frame);

  vbox2 = gtk_vbox_new (FALSE, 2);
  gtk_container_set_border_width (GTK_CONTAINER (vbox2), 2);
  gtk_container_add (GTK_CONTAINER (frame), vbox2);
  gtk_widget_show (vbox2);

  button = gtk_check_button_new_with_label(_("Cubic Interpolation"));
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (button),
				cubic_interpolation);
  gtk_box_pack_start (GTK_BOX (vbox2), button, FALSE, FALSE, 0);
  gtk_signal_connect (GTK_OBJECT (button), "toggled",
		      (GtkSignalFunc) file_prefs_toggle_callback,
		      &cubic_interpolation);
  gtk_widget_show (button);

  /* Interface */
  vbox = file_prefs_notebook_append_page (GTK_NOTEBOOK (notebook),
					  _("Interface Settings"),
					  GTK_CTREE (ctree),
					  _("Interface"),
					  NULL,
					  &top_insert,
					  page_index);
  gtk_widget_show (vbox);
  page_index++;

  frame = gtk_frame_new (_("General")); 
  gtk_box_pack_start (GTK_BOX (vbox), frame, FALSE, FALSE, 0);
  gtk_widget_show (frame);

  hbox = gtk_hbox_new (FALSE, 2);
  gtk_container_add (GTK_CONTAINER (frame), hbox);
  gtk_widget_show (hbox);
      
  table = gtk_table_new (3, 2, FALSE);
  gtk_container_set_border_width (GTK_CONTAINER (table), 2);
  gtk_table_set_row_spacings (GTK_TABLE (table), 2);
  gtk_table_set_col_spacings (GTK_TABLE (table), 4);
  gtk_box_pack_start (GTK_BOX (hbox), table, FALSE, FALSE, 0);
  gtk_widget_show (table);

  /* Don't show the Auto-save button until we really 
     have auto-saving in the gimp.

     button = gtk_check_button_new_with_label(_("Auto save"));
     gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (button),
                                   auto_save);
     gtk_box_pack_start (GTK_BOX (vbox2), button, FALSE, FALSE, 0);
     gtk_signal_connect (GTK_OBJECT (button), "toggled",
	                 (GtkSignalFunc) file_prefs_toggle_callback,
			 &auto_save);
     gtk_widget_show (button);
  */

  label = gtk_label_new (_("Preview Size:"));
  optionmenu =
    gimp_option_menu_new (file_prefs_preview_size_callback,
			  (gpointer) preview_size,
			  _("None"),   (gpointer)   0, (gpointer)   0,
			  _("Small"),  (gpointer)  32, (gpointer)  32,
			  _("Medium"), (gpointer)  64, (gpointer)  64,
			  _("Large"),  (gpointer) 128, (gpointer) 128,
			  NULL);
  file_pref_table_attach (GTK_TABLE (table), GTK_LABEL (label), optionmenu, 0);
  gtk_widget_show (label);
  gtk_widget_show (optionmenu);

  label = gtk_label_new (_("Levels of Undo:"));
  spinbutton =
    gimp_spin_button_new (levels_of_undo, 0.0, 255.0, 1.0, 5.0, 0.0, 1.0, 0.0);
  gtk_signal_connect (GTK_OBJECT (spinbutton), "changed",
		      (GtkSignalFunc) file_prefs_spinbutton_callback,
		      &levels_of_undo);
  file_pref_table_attach (GTK_TABLE (table), GTK_LABEL (label), spinbutton, 1);
  gtk_widget_show (label);
  gtk_widget_show (spinbutton);

  label = gtk_label_new (_("Recent Documents List Size:"));
  spinbutton =
    gimp_spin_button_new (last_opened_size, 0.0, 256.0, 1.0, 5.0, 0.0, 1.0, 0.0);
  gtk_signal_connect (GTK_OBJECT (spinbutton), "changed",
		      (GtkSignalFunc) file_prefs_spinbutton_callback,
		      &edit_last_opened_size);
  file_pref_table_attach (GTK_TABLE (table), GTK_LABEL (label), spinbutton, 2);
  gtk_widget_show (label);
  gtk_widget_show (spinbutton);

  frame = gtk_frame_new (_("Help System")); 
  gtk_box_pack_start (GTK_BOX (vbox), frame, FALSE, FALSE, 0);
  gtk_widget_show (frame);

  vbox2 = gtk_vbox_new (FALSE, 2);
  gtk_container_set_border_width (GTK_CONTAINER (vbox2), 2);
  gtk_container_add (GTK_CONTAINER (frame), vbox2);
  gtk_widget_show (vbox2);

  button = gtk_check_button_new_with_label(_("Show Tool Tips"));
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (button),
				show_tool_tips);
  gtk_box_pack_start (GTK_BOX (vbox2), button, FALSE, FALSE, 0);
  gtk_signal_connect (GTK_OBJECT (button), "toggled",
		      (GtkSignalFunc) file_prefs_toggle_callback,
		      &show_tool_tips);
  gtk_widget_show (button);

  /* Interface / Image Windows */
  vbox = file_prefs_notebook_append_page (GTK_NOTEBOOK (notebook),
					  _("Image Windows Settings"),
					  GTK_CTREE (ctree),
					  _("Image Windows"),
					  top_insert,
					  &child_insert,
					  page_index);
  gtk_widget_show (vbox);
  page_index++;

  frame = gtk_frame_new (_("Appearance")); 
  gtk_box_pack_start (GTK_BOX (vbox), frame, FALSE, FALSE, 0);
  gtk_widget_show (frame);

  vbox2 = gtk_vbox_new (FALSE, 2);
  gtk_container_set_border_width (GTK_CONTAINER (vbox2), 2);
  gtk_container_add (GTK_CONTAINER (frame), vbox2);
  gtk_widget_show (vbox2);

  button = gtk_check_button_new_with_label(_("Resize Window on Zoom"));
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (button),
				allow_resize_windows);
  gtk_box_pack_start (GTK_BOX (vbox2), button, FALSE, FALSE, 0);
  gtk_signal_connect (GTK_OBJECT (button), "toggled",
		      (GtkSignalFunc) file_prefs_toggle_callback,
		      &allow_resize_windows);
  gtk_widget_show (button);
      
  button = gtk_check_button_new_with_label(_("Show Rulers"));
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (button),
				show_rulers);
  gtk_box_pack_start (GTK_BOX (vbox2), button, FALSE, FALSE, 0);
  gtk_signal_connect (GTK_OBJECT (button), "toggled",
		      (GtkSignalFunc) file_prefs_toggle_callback,
		      &show_rulers);
  gtk_widget_show (button);

  button = gtk_check_button_new_with_label(_("Show Statusbar"));
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (button),
				show_statusbar);
  gtk_box_pack_start (GTK_BOX (vbox2), button, FALSE, FALSE, 0);
  gtk_signal_connect (GTK_OBJECT (button), "toggled",
		      (GtkSignalFunc) file_prefs_toggle_callback,
		      &show_statusbar);
  gtk_widget_show (button);

  hbox = gtk_hbox_new (FALSE, 0);
  gtk_box_pack_start (GTK_BOX (vbox2), hbox, FALSE, FALSE, 0);
  gtk_widget_show (hbox);

  table = gtk_table_new (2, 2, FALSE);
  gtk_container_set_border_width (GTK_CONTAINER (table), 2);
  gtk_table_set_row_spacings (GTK_TABLE (table), 2);
  gtk_table_set_col_spacings (GTK_TABLE (table), 4);
  gtk_box_pack_start (GTK_BOX (hbox), table, FALSE, FALSE, 0);
  gtk_widget_show (table);

  label = gtk_label_new (_("Marching Ants Speed:"));
  spinbutton =
    gimp_spin_button_new (marching_speed, 0.0, 32000.0, 50.0, 100.0, 0.0,
			  1.0, 0.0);
  gtk_signal_connect (GTK_OBJECT (spinbutton), "changed",
		      (GtkSignalFunc) file_prefs_spinbutton_callback,
		      &marching_speed);
  file_pref_table_attach (GTK_TABLE (table), GTK_LABEL (label), spinbutton, 0);
  gtk_widget_show (label);
  gtk_widget_show (spinbutton);

  /* The title format string */
  label = gtk_label_new (_("Image Title Format:"));

  combo = gtk_combo_new ();
  gtk_combo_set_use_arrows (GTK_COMBO (combo), FALSE);
  gtk_combo_set_value_in_list (GTK_COMBO (combo), FALSE, FALSE);
  /* Set the currently used string as "Custom" */
  comboitem = gtk_list_item_new_with_label (_("Custom"));
  gtk_combo_set_item_string (GTK_COMBO (combo), GTK_ITEM (comboitem),
			     image_title_format);
  gtk_container_add (GTK_CONTAINER (GTK_COMBO (combo)->list), comboitem);
  gtk_widget_show (comboitem);
  /* set some commonly used format strings */
  comboitem = gtk_list_item_new_with_label (_("Standard"));
  gtk_combo_set_item_string (GTK_COMBO (combo), GTK_ITEM (comboitem),
			     "%f-%p.%i (%t)");
  gtk_container_add(GTK_CONTAINER (GTK_COMBO (combo)->list), comboitem);
  gtk_widget_show (comboitem);
  comboitem = gtk_list_item_new_with_label (_("Show zoom percentage"));
  gtk_combo_set_item_string (GTK_COMBO (combo), GTK_ITEM (comboitem),
			     "%f-%p.%i (%t) %z%%");
  gtk_container_add (GTK_CONTAINER (GTK_COMBO (combo)->list), comboitem);
  gtk_widget_show (comboitem);
  comboitem = gtk_list_item_new_with_label (_("Show zoom ratio"));
  gtk_combo_set_item_string (GTK_COMBO (combo), GTK_ITEM (comboitem),
			     "%f-%p.%i (%t) %d:%s");
  gtk_container_add (GTK_CONTAINER (GTK_COMBO (combo)->list), comboitem);
  gtk_widget_show (comboitem);
  comboitem = gtk_list_item_new_with_label (_("Show reversed zoom ratio"));
  gtk_combo_set_item_string (GTK_COMBO (combo), GTK_ITEM (comboitem),
			     "%f-%p.%i (%t) %s:%d");
  gtk_container_add (GTK_CONTAINER (GTK_COMBO (combo)->list), comboitem);
  gtk_widget_show (comboitem);

  gtk_signal_connect (GTK_OBJECT (GTK_COMBO (combo)->entry), "changed",
		      GTK_SIGNAL_FUNC (file_prefs_string_callback), 
		      &image_title_format);

  file_pref_table_attach (GTK_TABLE (table), GTK_LABEL (label), combo, 1);
  gtk_widget_show (label);
  gtk_widget_show (combo);
  /* End of the title format string */

  frame = gtk_frame_new (_("Pointer Movement Feedback")); 
  gtk_box_pack_start (GTK_BOX (vbox), frame, FALSE, FALSE, 0);
  gtk_widget_show (frame);

  vbox2 = gtk_vbox_new (FALSE, 2);
  gtk_container_set_border_width (GTK_CONTAINER (vbox2), 2);
  gtk_container_add (GTK_CONTAINER (frame), vbox2);
  gtk_widget_show (vbox2);

  button = gtk_check_button_new_with_label(_("Perfect-but-slow Pointer Tracking"));
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (button),
				perfectmouse);
  gtk_box_pack_start (GTK_BOX (vbox2), button, FALSE, FALSE, 0);
  gtk_signal_connect (GTK_OBJECT (button), "toggled",
		      (GtkSignalFunc) file_prefs_toggle_callback,
		      &perfectmouse);
  gtk_widget_show (button);
      
  button = gtk_check_button_new_with_label(_("Disable Cursor Updating"));
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (button),
				no_cursor_updating);
  gtk_box_pack_start (GTK_BOX (vbox2), button, FALSE, FALSE, 0);
  gtk_signal_connect (GTK_OBJECT (button), "toggled",
		      (GtkSignalFunc) file_prefs_toggle_callback,
		      &no_cursor_updating);
  gtk_widget_show (button);

  /* Environment */
  vbox = file_prefs_notebook_append_page (GTK_NOTEBOOK (notebook),
					  _("Environment Settings"),
					  GTK_CTREE (ctree),
					  _("Environment"),
					  NULL,
					  &top_insert,
					  page_index);
  gtk_widget_show (vbox);
  page_index++;

  frame = gtk_frame_new (_("Resource Consumption")); 
  gtk_box_pack_start (GTK_BOX (vbox), frame, FALSE, FALSE, 0);
  gtk_widget_show (frame);

  vbox2 = gtk_vbox_new (FALSE, 2);
  gtk_container_set_border_width (GTK_CONTAINER (vbox2), 2);
  gtk_container_add (GTK_CONTAINER (frame), vbox2);
  gtk_widget_show (vbox2);

  button = gtk_check_button_new_with_label(_("Conservative Memory Usage"));
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (button), stingy_memory_use);
  gtk_signal_connect (GTK_OBJECT (button), "toggled",
		      (GtkSignalFunc) file_prefs_toggle_callback,
		      &edit_stingy_memory_use);
  gtk_box_pack_start (GTK_BOX (vbox2), button, FALSE, FALSE, 0);
  gtk_widget_show (button);

  hbox = gtk_hbox_new (FALSE, 0);
  gtk_box_pack_start (GTK_BOX (vbox2), hbox, FALSE, FALSE, 0);
  gtk_widget_show (hbox);

#ifdef ENABLE_MP
  table = gtk_table_new (2, 2, FALSE);
#else
  table = gtk_table_new (1, 2, FALSE);
#endif
  gtk_container_set_border_width (GTK_CONTAINER (table), 2);
  gtk_table_set_row_spacings (GTK_TABLE (table), 2);
  gtk_table_set_col_spacings (GTK_TABLE (table), 4);
  gtk_box_pack_start (GTK_BOX (hbox), table, FALSE, FALSE, 0);
  gtk_widget_show (table);

  label = gtk_label_new (_("Tile Cache Size:"));
  hbox = gtk_hbox_new (FALSE, 2);
  tile_cache_size_spinbutton =
    gimp_spin_button_new (divided_tile_cache_size,
			  0.0, (4069.0 * 1024 * 1024), 1.0, 16.0, 0.0,
			  1.0, 0.0);
  gtk_box_pack_start (GTK_BOX (hbox), tile_cache_size_spinbutton,
		      FALSE, FALSE, 0);
  gtk_signal_connect (GTK_OBJECT (tile_cache_size_spinbutton), "changed",
		      (GtkSignalFunc) file_prefs_spinbutton_callback,
		      &divided_tile_cache_size);
  gtk_widget_show (tile_cache_size_spinbutton);
      
  optionmenu =
    gimp_option_menu_new (file_prefs_mem_size_unit_callback,
			  (gpointer) mem_size_unit,
			  _("Bytes"),     (gpointer) 1, (gpointer) 1,
			  _("KiloBytes"), (gpointer) 1024, (gpointer) 1024,
			  _("MegaBytes"),
			  (gpointer) (1024*1024), (gpointer) (1024*1024),
			  NULL);
  gtk_box_pack_start (GTK_BOX (hbox), optionmenu, FALSE, FALSE, 0);
  gtk_widget_show (optionmenu);
  file_pref_table_attach (GTK_TABLE (table), GTK_LABEL (label), hbox, 0);
  gtk_widget_show (hbox);
  gtk_widget_show (label);

#ifdef ENABLE_MP
  label = gtk_label_new (_("Number of Processors to Use:"));
  num_processors_spinbutton =
    gimp_spin_button_new (num_processors, 1, 30, 1.0, 2.0, 0.0, 1.0, 0.0);
  gtk_signal_connect (GTK_OBJECT (num_processors_spinbutton), "changed",
		      (GtkSignalFunc) file_prefs_spinbutton_callback,
		      &num_processors);
  file_pref_table_attach (GTK_TABLE (table),
			  GTK_LABEL (label), num_processors_spinbutton, 1);
  gtk_widget_show (label);
  gtk_widget_show (num_processors_spinbutton);
#else
  num_processors_spinbutton = NULL;
#endif /* ENABLE_MP */

  frame = gtk_frame_new (_("8-Bit Displays")); 
  gtk_box_pack_start (GTK_BOX (vbox), frame, FALSE, FALSE, 0);
  if (g_visual->depth != 8)
    gtk_widget_set_sensitive (GTK_WIDGET (frame), FALSE);
  gtk_widget_show (frame);

  vbox2 = gtk_vbox_new (FALSE, 2);
  gtk_container_set_border_width (GTK_CONTAINER (vbox2), 2);
  gtk_container_add (GTK_CONTAINER (frame), vbox2);
  gtk_widget_show (vbox2);

  button = gtk_check_button_new_with_label(_("Install Colormap"));
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (button),
				install_cmap);
  gtk_signal_connect (GTK_OBJECT (button), "toggled",
		      (GtkSignalFunc) file_prefs_toggle_callback,
		      &edit_install_cmap);
  gtk_box_pack_start (GTK_BOX (vbox2), button, FALSE, FALSE, 0);
  gtk_widget_show (button);

  button = gtk_check_button_new_with_label(_("Colormap Cycling"));
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (button),
				cycled_marching_ants);
  gtk_signal_connect (GTK_OBJECT (button), "toggled",
		      (GtkSignalFunc) file_prefs_toggle_callback,
		      &edit_cycled_marching_ants);
  gtk_box_pack_start (GTK_BOX (vbox2), button, FALSE, FALSE, 0);
  gtk_widget_show (button);

  /* Session Management */
  vbox = file_prefs_notebook_append_page (GTK_NOTEBOOK (notebook),
					  _("Session Management"),
					  GTK_CTREE (ctree),
					  _("Session"),
					  NULL,
					  &top_insert,
					  page_index);
  gtk_widget_show (vbox);
  page_index++;

  frame = gtk_frame_new (_("Window Positions")); 
  gtk_box_pack_start (GTK_BOX (vbox), frame, FALSE, FALSE, 0);
  gtk_widget_show (frame);

  vbox2 = gtk_vbox_new (FALSE, 2);
  gtk_container_set_border_width (GTK_CONTAINER (vbox2), 2);
  gtk_container_add (GTK_CONTAINER (frame), vbox2);
  gtk_widget_show (vbox2);

  button = gtk_check_button_new_with_label (_("Save Window Positions on Exit"));
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (button),
				save_session_info);
  gtk_box_pack_start (GTK_BOX (vbox2), button, FALSE, FALSE, 0);
  gtk_signal_connect (GTK_OBJECT (button), "toggled",
		      (GtkSignalFunc) file_prefs_toggle_callback,
		      &save_session_info);
  gtk_widget_show (button);

  hbox = gtk_hbox_new (FALSE, 2);
  gtk_container_set_border_width (GTK_CONTAINER (hbox), 4);
  gtk_box_pack_start (GTK_BOX (vbox2), hbox, FALSE, FALSE, 0);
  gtk_widget_show (hbox);
      
  button = gtk_button_new_with_label (_("Clear Saved Window Positions"));
  gtk_box_pack_start (GTK_BOX (hbox), button, FALSE, FALSE, 0);
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
		      (GtkSignalFunc) file_prefs_clear_session_info_callback,
		      NULL);
  gtk_widget_show (button);

  button = gtk_check_button_new_with_label (_("Always Try to Restore Session"));
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (button),
				always_restore_session);
  gtk_box_pack_start (GTK_BOX (vbox2), button, FALSE, FALSE, 0);
  gtk_signal_connect (GTK_OBJECT (button), "toggled",
		      (GtkSignalFunc) file_prefs_toggle_callback,
		      &always_restore_session);
  gtk_widget_show (button);

  frame = gtk_frame_new (_("Devices")); 
  gtk_box_pack_start (GTK_BOX (vbox), frame, FALSE, FALSE, 0);
  gtk_widget_show (frame);

  vbox2 = gtk_vbox_new (FALSE, 2);
  gtk_container_set_border_width (GTK_CONTAINER (vbox2), 2);
  gtk_container_add (GTK_CONTAINER (frame), vbox2);
  gtk_widget_show (vbox2);

  button = gtk_check_button_new_with_label (_("Save Device Status on Exit"));
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (button),
				save_device_status);
  gtk_box_pack_start (GTK_BOX (vbox2), button, FALSE, FALSE, 0);
  gtk_signal_connect (GTK_OBJECT (button), "toggled",
		      (GtkSignalFunc) file_prefs_toggle_callback,
		      &save_device_status);
  gtk_widget_show (button);

  /* Monitor */
  vbox = file_prefs_notebook_append_page (GTK_NOTEBOOK (notebook),
					  _("Monitor Information"),
					  GTK_CTREE (ctree),
					  _("Monitor"),
					  NULL,
					  &top_insert,
					  page_index);
  gtk_widget_show (vbox);
  page_index++;

  frame = gtk_frame_new (_("Get Monitor Resolution")); 
  gtk_box_pack_start (GTK_BOX (vbox), frame, FALSE, FALSE, 0);
  gtk_widget_show (frame);

  vbox2 = gtk_vbox_new (FALSE, 2);
  gtk_container_set_border_width (GTK_CONTAINER (vbox2), 2);
  gtk_container_add (GTK_CONTAINER (frame), vbox2);
  gtk_widget_show (vbox2);

  group = NULL;
  button = gtk_radio_button_new_with_label (group, _("From X Server"));
  group = gtk_radio_button_group (GTK_RADIO_BUTTON (button));
  gtk_signal_connect (GTK_OBJECT (button), "toggled",
		      GTK_SIGNAL_FUNC (file_prefs_res_source_callback),
		      NULL);
  gtk_box_pack_start (GTK_BOX (vbox2), button, FALSE, FALSE, 0);
  gtk_widget_show (button);

  {
    float xres, yres;
    char buf[80];

    gdisplay_xserver_resolution (&xres, &yres);

    g_snprintf (buf, sizeof (buf), _("(Currently %d x %d dpi)"),
		(int) (xres + 0.5), (int) (yres + 0.5));
    resolution_xserver_label = gtk_label_new (buf);
    gtk_box_pack_start (GTK_BOX (vbox2), resolution_xserver_label,
			FALSE, FALSE, 0);
    gtk_widget_show (resolution_xserver_label);
  }

  button = gtk_radio_button_new_with_label (group, _("Manually:"));
  group = gtk_radio_button_group (GTK_RADIO_BUTTON (button));
  gtk_box_pack_start (GTK_BOX (vbox2), button, FALSE, FALSE, 0);
  gtk_widget_show (button);
  if (!using_xserver_resolution)
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (button), TRUE);

  abox = gtk_alignment_new (0.5, 0.5, 0.0, 1.0);
  gtk_box_pack_start (GTK_BOX (vbox2), abox, FALSE, FALSE, 0);
  gtk_widget_show (abox);

  monitor_resolution_force_equal = gimp_chain_button_new (GIMP_CHAIN_BOTTOM);
  monitor_resolution_sizeentry = gimp_size_entry_new (2, UNIT_INCH, "%s",
						      FALSE, TRUE, 75,
						      GIMP_SIZE_ENTRY_UPDATE_RESOLUTION);
  gimp_size_entry_set_refval_boundaries (GIMP_SIZE_ENTRY (monitor_resolution_sizeentry), 0, 1, 32767);
  gimp_size_entry_set_refval_boundaries (GIMP_SIZE_ENTRY (monitor_resolution_sizeentry), 1, 1, 32767);
  gimp_size_entry_set_refval (GIMP_SIZE_ENTRY (monitor_resolution_sizeentry),
			      0, monitor_xres);
  gimp_size_entry_set_refval (GIMP_SIZE_ENTRY (monitor_resolution_sizeentry),
			      1, monitor_yres);
  gimp_size_entry_attach_label (GIMP_SIZE_ENTRY (monitor_resolution_sizeentry),
				_("Horizontal"), 0, 1, 0.0);
  gimp_size_entry_attach_label (GIMP_SIZE_ENTRY (monitor_resolution_sizeentry),
				_("Vertical"), 0, 2, 0.0);
  gimp_size_entry_attach_label (GIMP_SIZE_ENTRY (monitor_resolution_sizeentry),
				_("dpi"), 1, 3, 0.0);
  gimp_size_entry_attach_label (GIMP_SIZE_ENTRY (monitor_resolution_sizeentry),
				_("Pixels per "), 2, 3, 0.0);
  gtk_signal_connect (GTK_OBJECT (monitor_resolution_sizeentry),
		      "value_changed",
		      (GtkSignalFunc)file_prefs_monitor_resolution_callback,
		      monitor_resolution_force_equal);
  gtk_signal_connect (GTK_OBJECT (monitor_resolution_sizeentry),
		      "refval_changed",
		      (GtkSignalFunc)file_prefs_monitor_resolution_callback,
		      monitor_resolution_force_equal);
  gtk_container_add (GTK_CONTAINER (abox), monitor_resolution_sizeentry);
  gtk_widget_show (monitor_resolution_sizeentry);

  if (ABS (monitor_xres - monitor_yres) < 1e-5)
    gimp_chain_button_set_active (GIMP_CHAIN_BUTTON (monitor_resolution_force_equal),
				  TRUE);
  gtk_table_attach_defaults (GTK_TABLE (monitor_resolution_sizeentry), 
			     monitor_resolution_force_equal, 1, 3, 3, 4);
  gtk_widget_show (monitor_resolution_force_equal);

  gtk_widget_set_sensitive (monitor_resolution_sizeentry,
			    !using_xserver_resolution);
  gtk_widget_set_sensitive (monitor_resolution_force_equal,
			    !using_xserver_resolution);

  /* Directories */
  vbox = file_prefs_notebook_append_page (GTK_NOTEBOOK (notebook),
					  _("Directories Settings"),
					  GTK_CTREE (ctree),
					  _("Directories"),
					  NULL,
					  &top_insert,
					  page_index);
  gtk_widget_show (vbox);
  page_index++;

  {
    static const struct {
      char  *label;
      char  *fs_label;
      char **mdir;
    } dirs[] = {
      { N_("Temp Dir:"), N_("Select Temp Dir"), &edit_temp_path },
      { N_("Swap Dir:"), N_("Select Swap Dir"), &edit_swap_path },
    };
    static int ndirs = sizeof (dirs) / sizeof (dirs[0]);

    table = gtk_table_new (ndirs + 1, 2, FALSE);
    gtk_table_set_row_spacings (GTK_TABLE (table), 2);
    gtk_table_set_col_spacings (GTK_TABLE (table), 4);
    gtk_box_pack_start (GTK_BOX (vbox), table, FALSE, TRUE, 0);
    gtk_widget_show (table);

    for (i = 0; i < ndirs; i++)
      {
	label = gtk_label_new (gettext(dirs[i].label));
	gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
	gtk_table_attach (GTK_TABLE (table), label, 0, 1, i, i+1,
			  GTK_FILL, GTK_FILL, 0, 0);
	gtk_widget_show (label);
	    
	fileselection = gimp_file_selection_new (gettext(dirs[i].fs_label),
						 *(dirs[i].mdir),
						 TRUE, TRUE);
	gtk_signal_connect (GTK_OBJECT (fileselection), "filename_changed",
			    (GtkSignalFunc) file_prefs_filename_callback,
			    dirs[i].mdir);
	gtk_table_attach (GTK_TABLE (table), fileselection, 1, 2, i, i+1,
			  GTK_EXPAND | GTK_FILL, 0, 0, 0);
	gtk_widget_show (fileselection); 
      }
  }

  {
    static const struct {
      char  *tree_label;
      char  *label;
      char  *fs_label;
      char **mpath;
    } paths[] = {
      { N_("Brushes"),   N_("Brushes Directories"),   N_("Select Brushes Dir"),
	&edit_brush_path },
      { N_("Gradients"), N_("Gradients Directories"), N_("Select Gradients Dir"),
	&edit_gradient_path },
      { N_("Patterns"),  N_("Patterns Directories"),  N_("Select Patterns Dir"),
	&edit_pattern_path },
      { N_("Palettes"),  N_("Palettes Directories"),  N_("Select Palettes Dir"),
	&edit_palette_path },
      { N_("Plug-Ins"),  N_("Plug-Ins Directories"),  N_("Select Plug-Ins Dir"),
	&edit_plug_in_path },
      { N_("Modules"),   N_("Modules Directories"),   N_("Select Modules Dir"),
	&edit_module_path }
    };
    static int npaths = sizeof (paths) / sizeof (paths[0]);
	
    for (i = 0; i < npaths; i++)
      {
	vbox = file_prefs_notebook_append_page (GTK_NOTEBOOK (notebook),
						gettext (paths[i].label),
						GTK_CTREE (ctree),
						gettext (paths[i].tree_label),
						top_insert,
						&child_insert,
						page_index);
	gtk_widget_show (vbox);
	page_index++;

	patheditor = gimp_path_editor_new (gettext(paths[i].fs_label),
					   *(paths[i].mpath));
	gtk_signal_connect (GTK_OBJECT (patheditor), "path_changed",
			    (GtkSignalFunc) file_prefs_path_callback,
			    paths[i].mpath);
	gtk_container_add (GTK_CONTAINER (vbox), patheditor);
	gtk_widget_show (patheditor);
      }
  }

  gtk_widget_show (notebook);
  gtk_widget_show (prefs_dlg);
}
