/*
 * ttfaux.c
 *
 * This file is part of the ttf2pk package.
 *
 * Copyright 1997-1998 by
 *    Frederic Loyer <loyer@ensta.fr>
 *    Werner Lemberg <a7971428@unet.univie.ac.at>
 */

#include "freetype.h"
#include "extend/ftxkern.h"

#include "ttf2tfm.h"
#include "newobj.h"
#include "ligkern.h"
#include "ttfenc.h"
#include "tfmaux.h"
#include "errormsg.h"


#define Macintosh_platform 1
#define Macintosh_encoding 0

#define Microsoft_platform 3
#define Microsoft_Unicode_encoding 1


TT_Engine   engine;
TT_Face     face;
TT_Instance instance;
TT_Glyph    glyph;
TT_Outline  outline;
TT_CharMap  char_map;

TT_Glyph_Metrics    metrics;
TT_Face_Properties  properties;
TT_BBox             bbox;

TT_Kerning directory;


int
readttf_kern(struct Font *fnt)
{
  register struct kern *nk;
  register struct ttfinfo *ti;
  TT_Kern_0_Pair* pairs0;
  TT_Error error;
  int i, j;

  
  error = TT_Get_Kerning_Directory(face, &directory);
  
  if (directory.nTables == 0)
    return 0;

  for (i = 0; i < directory.nTables; i++)
  {
    error = TT_Load_Kerning_Table(face, i);

    switch (directory.tables->format)
    {
    case 0:
      pairs0 = directory.tables->t.kern0.pairs;
      for (j = 0; j < directory.tables->t.kern0.nPairs; j++, pairs0++)
      {
        ti = findglyph(pairs0->left, fnt->charlist);
        if (ti == NULL)
          warning("kern char not found");
        else
        {
          nk = newkern();
          nk->succ = findglyph(pairs0->right, fnt->charlist)->adobename;
          nk->delta = transform(pairs0->value * 1000 / fnt->units_per_em, 0,
                                fnt->efactor, fnt->slant);
          nk->next = ti->kerns;
          ti->kerns = nk;
        }
      }
      break;

    default:
      break;
    }
  }
  return 1;
}


void
readttf(struct Font *fnt, char quiet)
{
  TT_Error error;
  struct ttfinfo *ti;
  int Num;
  int i, j;
  int num_cmap;
  short cmap_plat, cmap_enc;
  int index_array[257];


  /*
   *   We allocate a placeholder boundary and the `.notdef' character.
   */

  ti = newchar(fnt);
  ti->charcode = -1;
  ti->adobename = ".notdef";

  ti = newchar(fnt);
  ti->charcode = -1;
  ti->adobename = "||"; /* boundary character name */

  /* Initialize FreeType engine */

  if ((error = TT_Init_FreeType(&engine)))
    oops("While initializing engine, error code = %d", error);

  if ((error = TT_Init_Kerning_Extension(engine)))
    oops("While initializing kerning, error code = %d", error);

  /* Load face */

  error = TT_Open_Face(engine, fnt->real_ttfname, &face);
  if (error)
    oops("Cannot open %s", fnt->real_ttfname);

  /* create instance */

  error = TT_New_Instance(face, &instance);
  if (error)
    oops("Cannot create instance for %s", fnt->real_ttfname);

  /* we use a dummy glyph size of 10pt */

  error = TT_Set_Instance_CharSize(instance, 10 * 64);
  if (error)
    oops("Cannot set character set");

  /* get face properties and allocate preload arrays */
  
  TT_Get_Face_Properties(face, &properties);

  /* create glyph container */

  error = TT_New_Glyph(face, &glyph);
  if (error)
    oops("Cannot create glyph container");

  fnt->units_per_em = properties.header->Units_Per_EM;
  fnt->fixedpitch = properties.postscript->isFixedPitch;
  fnt->italicangle = properties.postscript->italicAngle / 65536.0;

  num_cmap = TT_Get_CharMap_Count(face);
  for (i = 0; i < num_cmap; i++)
  {
    error = TT_Get_CharMap_ID(face, i, &cmap_plat, &cmap_enc);
    if (error)
      oops("Cannot query cmap, error code = %d", error);
    if (cmap_plat == fnt->pid && cmap_enc == fnt->eid)
      break;
  }
  if (i == num_cmap)
    oops("Invalid platform and/or encoding ID");

  error = TT_Get_CharMap(face, i, &char_map);
  if (error)
    oops("Cannot load cmap, error code = %d", error);

  if (cmap_plat == Microsoft_platform &&
      cmap_enc == Microsoft_Unicode_encoding)
    set_encoding_scheme(encUnicode);
  else if (cmap_plat == Macintosh_platform &&
           cmap_enc == Macintosh_encoding)
    set_encoding_scheme(encMac);
  else
    set_encoding_scheme(encFontSpecific);

  if (!quiet)
  {
    printf("\n");
    printf("Glyph  Code  Glyph Name                ");
    printf("Width  llx    lly      urx    ury\n");
    printf("---------------------------------------");
    printf("---------------------------------\n");
  }

  /*   We load only glyphs with a valid cmap entry.  Nevertheless, for
   *   the default mapping, we use the first 256 glyphs addressed by
   *   ascending code points, followed by glyphs not in the cmap.
   */

  for (i = 0; i < 257; i++)
    index_array[i] = 0;

  j = 0;
  for (i = 0; i <= 0xFFFF; i++)
  {
    char *an;


    Num = TT_Char_Index(char_map, i);
          
    if (Num < 0)
      oops("Failure on cmap mapping from %s", fnt->ttfname);
    if (Num == 0)
      continue;
    if (Num <= 256)
      index_array[Num] = 1;
    error = TT_Load_Glyph(instance, glyph, Num, 0);
    if (!error)
      error = TT_Get_Glyph_Metrics(glyph, &metrics);
    if (!error)
      error = TT_Get_Glyph_Outline(glyph, &outline);
    if (!error)
      error = TT_Get_Outline_BBox(&outline, &bbox); /* we need the non-
                                                       grid-fitted bbox */
    if (!error)
    {
      an = code_to_adobename(i);
      if (strcmp(an, ".notdef") == 0)
        continue;

      ti = newchar(fnt);
      ti->charcode = i;
      ti->glyphindex = Num;
      ti->adobename = an;
      ti->llx = bbox.xMin * 1000 / fnt->units_per_em;
      ti->lly = bbox.yMin * 1000 / fnt->units_per_em;
      ti->llx = transform(ti->llx, ti->lly, fnt->efactor, fnt->slant);
      ti->urx = bbox.xMax * 1000 / fnt->units_per_em;
      ti->ury = bbox.yMax * 1000 / fnt->units_per_em;
      ti->urx = transform(ti->urx, ti->ury, fnt->efactor, fnt->slant);
      /* We need to avoid negative heights or depths.  They break accents in
         math mode, among other things.  */
      if (ti->lly > 0)
        ti->lly = 0;
      if (ti->ury < 0)
        ti->ury = 0;
      ti->width = transform(metrics.advance * 1000 / fnt->units_per_em, 0,
                            fnt->efactor, fnt->slant);

      if (!quiet)
        printf("%5d  %04x  %-25s %5d  % 5d,% 5d -- % 5d,% 5d\n",
               Num, i, ti->adobename,
               ti->width,
               ti->llx, ti->lly, ti->urx, ti->ury);

      if (j < 256)
      {
        fnt->inencptrs[j] = ti;
        ti->incode = j;
      }
      j++;
    }
  }

  for (i = 1; i <= properties.num_Glyphs; i++)
  {
    char *an;


    if (index_array[i] == 0)
    {
      error = TT_Load_Glyph(instance, glyph, i, 0);
      if (!error)
        error = TT_Get_Glyph_Metrics(glyph, &metrics);
      if (!error)
        error = TT_Get_Glyph_Outline(glyph, &outline);
      if (!error)
        error = TT_Get_Outline_BBox(&outline, &bbox);
      if (!error)
      {
        an = code_to_adobename(i | 0x10000);

        ti = newchar(fnt);
        ti->charcode = i | 0x10000;
        ti->glyphindex = i;
        ti->adobename = an;
        ti->llx = bbox.xMin * 1000 / fnt->units_per_em;
        ti->lly = bbox.yMin * 1000 / fnt->units_per_em;
        ti->llx = transform(ti->llx, ti->lly, fnt->efactor, fnt->slant);
        ti->urx = bbox.xMax * 1000 / fnt->units_per_em;
        ti->ury = bbox.yMax * 1000 / fnt->units_per_em;
        ti->urx = transform(ti->urx, ti->ury, fnt->efactor, fnt->slant);

        if (ti->lly > 0)
          ti->lly = 0;
        if (ti->ury < 0)
          ti->ury = 0;
        ti->width = transform(metrics.advance * 1000 / fnt->units_per_em, 0,
                              fnt->efactor, fnt->slant);

        if (!quiet)
          printf("%5d        %-25s %5d  % 5d,% 5d -- % 5d,% 5d\n",
                 i, ti->adobename,
                 ti->width,
                 ti->llx, ti->lly, ti->urx, ti->ury);

        if (j < 256)
        {
          fnt->inencptrs[j] = ti;
          ti->incode = j;
        }
        else
          break;
        j++;
      }
    }
  }
  readttf_kern(fnt);

  /* CLOSE Font */
}


/* end */
