/*
 *  This file is part of MagiCapture.
 *  Copyright (C) 1999 Arthur Jerijian
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * capture.c: Screen capture management routines
 */

#if HAVE_UNISTD_H
#include <unistd.h>
#endif /* HAVE_UNISTD_H */

#include <stdio.h>
#include <magick/api.h>
#include <magick/xwindows.h>
#include <glib.h>
#include <gdk/gdk.h>
#include <gtk/gtk.h>
#include <gdk_magick.h>

#include "common.h"
#include "util.h"
#include "stringtable.h"
#include "capture.h"
#include "default.xpm"

/* Global variables */

/* Screen capture preview window data structures */
static GtkWidget *capture_window = NULL;
static GtkWidget *capture_pixmap_widget = NULL;
static guint capture_ready = 0;

/* Screen capture image data structures for previewing */
static GdkPixmap *capture_pixmap = NULL;
static GdkVisual *capture_visual = NULL;
static GdkColormap *capture_colormap = NULL;

/*********************************************************************
 *
 * Capture window callback functions
 */

gint capture_window_delete_event
(
    GtkWidget *widget,
    GdkEvent *event,
    gpointer data
)
{
    gtk_widget_hide (widget);
    return (TRUE);
}

void capture_window_destroy_event (GtkWidget *widget, gpointer data)
{
    capture_destroy_pixmap ();
}

/*********************************************************************
 *
 * capture_destroy_pixmap: Destroy the screen capture preview image
 * structure and free the structure's allocated memory.
 *
 * Parameters:
 *   None.
 *
 * Return value:
 *   None.
 */

void capture_destroy_pixmap ()
{
    if (capture_pixmap != NULL)
    {
        gdk_pixmap_unref (capture_pixmap);
        capture_pixmap = NULL;
    }
}

/*********************************************************************
 *
 * capture_destroy_window: Close the screen capture window and free
 * the window's allocated memory.
 *
 * Parameters:
 *   None.
 *
 * Return value:
 *   None.
 */

void capture_destroy_window ()
{
    if (capture_visual != NULL)
    {
        gdk_visual_unref (capture_visual);
        capture_visual = NULL;
        gdk_flush ();
    }
    if (capture_colormap != NULL)
    {
        gdk_colormap_unref (capture_colormap);
        capture_colormap = NULL;
        gdk_flush ();
    }
    if (capture_window != NULL)
    {
        gtk_widget_destroy (capture_window);
        capture_window = NULL;
    }
}

/*********************************************************************
 *
 * capture_init: Initialize the screen capture variables and data
 * structures. This function must be called after initializing all
 * the toolkits and before creating the main panel window.
 *
 * Parameters:
 *   None.
 *
 * Return value:
 *   None.
 */

void capture_init ()
{
    GdkPixmap *default_pixmap;
    GdkBitmap *default_mask;
    GtkStyle *style;
    
    /* Initialize all data structure pointers to NULL. */
    
    capture_window = NULL;
    capture_pixmap_widget = NULL;
    capture_ready = 0;
    
    capture_pixmap = NULL;
    capture_visual = NULL;
    capture_colormap = NULL;

    /* Use GdkMagick's visual and colormap. */

    capture_visual = gdk_magick_get_visual ();
    if (capture_visual == NULL)
    {
        error_dialog (ERROR_VISUAL);
        capture_cleanup ();
        exit (-1);
    }
    capture_colormap = gdk_magick_get_colormap ();
    if (capture_colormap == NULL)
    {
        error_dialog (ERROR_COLORMAP);
        capture_cleanup ();
        exit (-1);
    }
    
    gtk_widget_push_visual (capture_visual);
    gtk_widget_push_colormap (capture_colormap);

#ifdef MAGICAPTURE_DEBUG
    
    /* Print out the visual type to stdout. */
    switch (capture_visual -> type)
    {
        case GDK_VISUAL_STATIC_GRAY:
            g_print ("System visual type is GDK_VISUAL_STATIC_GRAY\n");
            break;
        case GDK_VISUAL_GRAYSCALE:
            g_print ("System visual type is GDK_VISUAL_GRAYSCALE\n");    
            break;
        case GDK_VISUAL_STATIC_COLOR:
            g_print ("System visual type is GDK_VISUAL_STATIC_COLOR\n");
            g_print
            (
                "System colormap has %d colors or planes\n",
                capture_colormap -> size
            );
            break;
        case GDK_VISUAL_PSEUDO_COLOR:
            g_print ("System visual type is GDK_VISUAL_PSEUDO_COLOR\n");
            g_print
            (
                "System colormap has %d colors or planes\n",
                capture_colormap -> size
            );
            break;
        case GDK_VISUAL_TRUE_COLOR:
            g_print ("System visual type is GDK_VISUAL_TRUE_COLOR\n");
            break;
        case GDK_VISUAL_DIRECT_COLOR:
            g_print ("System visual type is GDK_VISUAL_DIRECT_COLOR\n");
            break;
        default:
            g_print ("Unknown system visual type!!!\n");
            break;
    }

#endif /* MAGICAPTURE_DEBUG */
        
    /* Create a window to display the captured image. */

    capture_window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
    if (capture_window == NULL)
    {
        error_dialog (ERROR_CAPTURE_WINDOW);
        capture_cleanup ();
        exit (-1);
    }

    gtk_signal_connect
    (
        GTK_OBJECT (capture_window),
        "delete_event",
        GTK_SIGNAL_FUNC (capture_window_delete_event),
        NULL
    );
    gtk_signal_connect
    (
        GTK_OBJECT (capture_window),
        "destroy",
        GTK_SIGNAL_FUNC (capture_window_destroy_event),
        NULL
    );
    gtk_container_set_border_width (GTK_CONTAINER (capture_window), 0);
    gtk_window_set_title (GTK_WINDOW (capture_window), TITLE_CAPTURE_WINDOW);

    /* Create a default GTK pixmap and attach it to the capture window. */
    
    style = gtk_widget_get_style (capture_window);
    default_pixmap = gdk_pixmap_colormap_create_from_xpm_d
    (
        NULL,
        capture_colormap,
        &default_mask,
        &style -> bg [GTK_STATE_NORMAL],
        (gchar **) default_xpm
    );
    capture_pixmap_widget = gtk_pixmap_new (default_pixmap, default_mask);
    if (capture_pixmap_widget == NULL)
    {
        error_dialog (ERROR_CAPTURE_IMAGE_WIDGET);
        capture_cleanup ();
        exit (-1);
    }
 
    gtk_container_add (GTK_CONTAINER (capture_window), capture_pixmap_widget);
    gtk_widget_show (capture_pixmap_widget);
}

/*********************************************************************
 *
 * capture_cleanup: Free all memory allocated to screen capture data
 * structures. This function must be called after the GTK main event
 * loop exits.
 *
 * Parameters:
 *   None.
 *
 * Return value:
 *   None.
 */

void capture_cleanup ()
{
    capture_destroy_pixmap ();
    capture_destroy_window ();
}

/*********************************************************************
 *
 * capture_do: Proceed with the screen capture!!!!!!!!!!!!!!!!!!!!
 * This function waits a specified number of seconds and calls the
 * ImageMagick XImportImage API function to perform the actual capture.
 *
 * Parameters:
 *   p_capture_image - address which will receive the pointer to the
 *     captured image.
 *   p_capture_image_info - address which will receive the pointer to
 *     the captured image info.
 *   cap_type - type of display entity to capture
 *   cap_delay - number of seconds to wait before capturing
 *
 * Return value:
 *   TRUE if the operation completed successfully, FALSE otherwise
 */

gint capture_do
(
    Image **p_capture_image,
    ImageInfo **p_capture_image_info,
    CaptureType cap_type,
    guint cap_delay
)
{
    Image *capture_image;
    ImageInfo *capture_image_info;
    XImportInfo capture_import_info;

    capture_image = NULL;
    capture_image_info = NULL;
    *p_capture_image = NULL;
    *p_capture_image_info = NULL;
    
    /* Hide the capture preview window. */

    if (capture_window != NULL)
    {
        gtk_widget_hide (capture_window);
        gdk_flush ();
    }
    
    /* Initialize the ImageMagick image structures. */

    capture_image_info = CloneImageInfo (NULL);
    GetImageInfo (capture_image_info);
    XGetImportInfo (&capture_import_info);
    capture_import_info.descend = 1;

    /* Set parameters according to the type of display entity to capture. */
    
    switch (cap_type)
    {
        case SINGLE_WINDOW:
            capture_import_info.screen = 0;
            capture_import_info.frame = 0;
            capture_import_info.descend = 1;
            strcpy (capture_image_info -> filename, "");
            break;
        case SINGLE_WINDOW_DECOR:
            capture_import_info.screen = 0;
            capture_import_info.frame = 1;
            capture_import_info.descend = 1;
            strcpy (capture_image_info -> filename, "");
            break;
        case FULL_SCREEN:
            capture_import_info.screen = 0;
            capture_import_info.frame = 1;
            capture_import_info.descend = 1;
            strcpy (capture_image_info -> filename, "root");
            break;
        default:
            error_dialog (ERROR_UNKNOWN_CAPTURE_TYPE);
            if (capture_image_info != NULL)
            {
                DestroyImageInfo (capture_image_info);
            }
            return FALSE;
            break;
    }

    /* Wait a specified number of seconds. */

    if (cap_delay > 0)
    {
        sleep (cap_delay);
    }
    
    /* Capture the image!!!!!!!!!!!!!!!!! */

    capture_image = XImportImage (capture_image_info, &capture_import_info);
    if (capture_image == NULL)
    {
        error_dialog (ERROR_MAGICK_IMAGE);
        if (capture_image_info != NULL)
        {
            DestroyImageInfo (capture_image_info);
        }
        return FALSE;
    }

    /* Feed the image pointers back to the addresses. */
    
    *p_capture_image = capture_image;
    *p_capture_image_info = capture_image_info;
    
    return TRUE;
}

/*********************************************************************
 *
 * capture_preview: Preview the captured image into a GDK window.
 *
 * Parameters:
 *   capture_image - image to preview
 *
 * Return value:
 *   TRUE if the operation completed successfully, FALSE otherwise
 */

gint capture_preview (Image *capture_image)
{
    /* Destroy the existing screen capture output pixmap. */
    
    capture_destroy_pixmap ();

    /* Use GdkMagick to render the captured image into a GDK pixmap. */

    capture_pixmap = gdk_magick_to_gdkpixmap (capture_image);
    if (capture_pixmap == NULL)
    {
        error_dialog (ERROR_PREVIEW);
        capture_ready = 0;
        return FALSE;
    }
    gdk_flush ();
    
    /*
     * Copy the rendered GDK pixmap into the window's GTK pixmap widget.
     */
    
    gtk_pixmap_set
    (
        GTK_PIXMAP (capture_pixmap_widget),
        capture_pixmap,
        NULL
    );
        
    capture_ready = 1;
    return TRUE;
}

/*********************************************************************
 *
 * capture_show: Display the screen capture preview window.
 *
 * Parameters:
 *   None.
 *
 * Return value:
 *   None.
 */

void capture_show ()
{
    gint width, height;
    
    if ((capture_ready == 0) || (capture_window == NULL))
    {
        return;
    }
    
    /* Display the capture preview window. */

    gtk_widget_show (capture_window);
    gdk_window_get_size ((GdkWindow *) capture_pixmap, &width, &height);
    gdk_window_resize (capture_window -> window, width, height);
    gdk_flush ();
}
