
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <schroedinger/schro.h>
#include <schroedinger/schrowavelet.h>
#include <schroedinger/schroorc.h>

#include <orc-test/orcprofile.h>

#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#define N 10000

uint8_t packed_data[N*16];

const int16_t _schro_table_unpack_sint[1024][10];
const int8_t test2_table_unpack_sint[256][17];
const int8_t test3_table_unpack_sint[1024][21];

int16_t a[N];
int16_t b[N];
int16_t c[N];

int16_t table[65536];

int orc_profile_get_min (OrcProfile *prof)
{
  int i;
  int min;
  min = prof->hist_time[0];
  for(i=0;i<10;i++){
    if (prof->hist_count[i] > 0) {
      if (prof->hist_time[i] < min) {
        min = prof->hist_time[i];
      }
    }
  }
  return min;
}


static void
_schro_unpack_shift_in (SchroUnpack * unpack)
{
  if (unpack->n_bits_left >= 32) {
    /* the fast path */
    if (unpack->n_bits_in_shift_register == 0) {
      unpack->shift_register =
          (unpack->data[0] << 24) | (unpack->
          data[1] << 16) | (unpack->data[2] << 8) | (unpack->data[3]);
      unpack->data += 4;
      unpack->n_bits_left -= 32;
      unpack->n_bits_in_shift_register = 32;
    } else {
      while (unpack->n_bits_in_shift_register <= 24) {
        unpack->shift_register |=
            unpack->data[0] << (24 - unpack->n_bits_in_shift_register);
        unpack->data++;
        unpack->n_bits_left -= 8;
        unpack->n_bits_in_shift_register += 8;
      }
    }
    return;
  }

  if (unpack->n_bits_left == 0) {
    unsigned int value = (unpack->guard_bit) ? 0xffffffff : 0;

    unpack->overrun += 32 - unpack->n_bits_in_shift_register;
    unpack->shift_register |= (value >> unpack->n_bits_in_shift_register);
    unpack->n_bits_in_shift_register = 32;
    return;
  }

  while (unpack->n_bits_left >= 8 && unpack->n_bits_in_shift_register <= 24) {
    unpack->shift_register |=
        unpack->data[0] << (24 - unpack->n_bits_in_shift_register);
    unpack->data++;
    unpack->n_bits_left -= 8;
    unpack->n_bits_in_shift_register += 8;
  }
  if (unpack->n_bits_left > 0 &&
      unpack->n_bits_in_shift_register + unpack->n_bits_left <= 32) {
    unsigned int value;

    value = unpack->data[0] >> (8 - unpack->n_bits_left);
    unpack->shift_register |=
        value << (32 - unpack->n_bits_in_shift_register - unpack->n_bits_left);
    unpack->data++;
    unpack->n_bits_in_shift_register += unpack->n_bits_left;
    unpack->n_bits_left = 0;
  }
}

static unsigned int
_schro_unpack_shift_out (SchroUnpack * unpack, int n)
{
  unsigned int value;

  if (n == 0)
    return 0;

  value = unpack->shift_register >> (32 - n);
  unpack->shift_register <<= n;
  unpack->n_bits_in_shift_register -= n;
  unpack->n_bits_read += n;

  return value;
}

/* test 1 */

void
test1_unpack_decode_sint_s16 (int16_t * dest, SchroUnpack * unpack, int n)
{
  int i;
  int j;
  const int16_t *table_entry;
  int x;
  int z;

  while (n > 0) {
    while (unpack->n_bits_in_shift_register < 8 + SCHRO_UNPACK_TABLE_SHIFT) {
      _schro_unpack_shift_in (unpack);
    }
    i = unpack->shift_register >> (32 - SCHRO_UNPACK_TABLE_SHIFT);
    table_entry = schro_table_unpack_sint[i];
    x = table_entry[0];
    if ((x & 0xf) == 0) {
      int y = x >> 4;

      i = (unpack->shift_register & 0xffffff) >> (24 -
          SCHRO_UNPACK_TABLE_SHIFT);
      table_entry = schro_table_unpack_sint[i];
      x = table_entry[0];
      if ((x & 0xf) == 0) {
        dest[0] = schro_unpack_decode_sint_slow (unpack);
        dest++;
        n--;
      } else {
        int bits = ((x & 0xf) >> 1) - 1;

        z = x >> 4;
        if (z > 0) {
          dest[0] = z + (y << bits);
        } else {
          dest[0] = z - (y << bits);
        }
        _schro_unpack_shift_out (unpack, (x & 0xf) + 8);
        dest++;
        n--;
      }
    } else {
      j = 0;
      do {
        dest[j] = x >> 4;
        j++;
        n--;
        x = table_entry[j];
      } while (n > 0 && x & 0xf);
      x = table_entry[j - 1];
      _schro_unpack_shift_out (unpack, x & 0xf);
      dest += j;
    }
  }
}

/* test 2 */

#define SHIFT 8
void
test2_unpack_decode_sint_s16 (int16_t * dest, SchroUnpack * unpack, int n)
{
  int i;
  int j;
  int m;

  while (n > 0) {
    while (unpack->n_bits_in_shift_register < 8 + SHIFT) {
      _schro_unpack_shift_in (unpack);
    }
    i = unpack->shift_register >> (32 - SHIFT);
    if (test2_table_unpack_sint[i][0] == 0) {
      dest[0] = schro_unpack_decode_sint_slow (unpack);
      dest++;
      n--;
    } else {
      m = MIN(n, test2_table_unpack_sint[i][0]);
      for(j=0;j<m;j++){
        dest[j] = test2_table_unpack_sint[i][1+2*j];
      }
      dest += m;
      n -= m;
      _schro_unpack_shift_out (unpack, test2_table_unpack_sint[i][2*m]);
    }
  }
}

#undef SHIFT
#define SHIFT 10
typedef const int8_t SchroUnpackTableEntry[21];

void
test3_unpack_decode_sint_s16 (int16_t * dest, SchroUnpack * unpack, int n)
{
  int i;
  int j;
  int m;
  SchroUnpackTableEntry *table = test3_table_unpack_sint;
  const int8_t *table_entry;

  while (n > 0) {
    while (unpack->n_bits_in_shift_register < 8 + SHIFT) {
      _schro_unpack_shift_in (unpack);
    }
    i = unpack->shift_register >> (32 - SHIFT);
    table_entry = table[i];
    if (table_entry[0] == 0) {
      dest[0] = schro_unpack_decode_sint_slow (unpack);
      dest++;
      n--;
    } else {
      m = MIN(n, table_entry[0]);
      for(j=0;j<m;j++){
        dest[j] = table_entry[1+2*j];
      }
      _schro_unpack_shift_out (unpack, table_entry[2*m]);
      dest += m;
      n -= m;
    }
  }
}


void
dequantise_speed (int n)
{
  OrcProfile prof1;
  OrcProfile prof2;
  OrcProfile prof3;
  double ave1;
  double ave2;
  double ave3;
  int i;
  int j;

  orc_profile_init (&prof1);
  orc_profile_init (&prof2);
  orc_profile_init (&prof3);

  for(j=0;j<N*16;j++){
    packed_data[j] = rand();
  }

  for(i=0;i<1;i++) {
    SchroUnpack unpack;
    SchroUnpack unpack2;
    SchroUnpack unpack3;

    schro_unpack_init_with_data (&unpack, packed_data, N*16, 1);

    schro_unpack_copy (&unpack2, &unpack);
    schro_unpack_copy (&unpack3, &unpack);

    orc_profile_start (&prof1);
    schro_unpack_decode_sint_s16 (a, &unpack, N);
    orc_profile_stop (&prof1);

    orc_profile_start (&prof2);
    test3_unpack_decode_sint_s16 (b, &unpack2, N);
    orc_profile_stop (&prof2);

    orc_profile_start (&prof3);
    test3_unpack_decode_sint_s16 (c, &unpack3, N);
    orc_profile_stop (&prof3);

    if (memcmp (a, b, N*sizeof(int16_t))) {
      fprintf(stderr, "unpack error\n");
      exit (1);
    }
  }

  ave1 = orc_profile_get_min (&prof1);
  ave2 = orc_profile_get_min (&prof2);
  ave3 = orc_profile_get_min (&prof3);
  printf("%d %g %g %g\n", n, ave1, ave2, ave3);
}


int
main (int argc, char *argv[])
{
  int i;

  schro_init();
  orc_init();

  for(i=0;i<200;i++){
    dequantise_speed (i);
  }

  return 0;
}


const int8_t test2_table_unpack_sint[256][17] = {
  /*   0 */ { 0, },
  /*   1 */ { 0, },
  /*   2 */ { 1, 7, 8, },
  /*   3 */ { 1, -7, 8, },
  /*   4 */ { 0, },
  /*   5 */ { 0, },
  /*   6 */ { 1, 8, 8, },
  /*   7 */ { 1, -8, 8, },
  /*   8 */ { 1, 3, 6, },
  /*   9 */ { 1, 3, 6, },
  /*  10 */ { 2, 3, 6, 0, 7, },
  /*  11 */ { 3, 3, 6, 0, 7, 0, 8, },
  /*  12 */ { 1, -3, 6, },
  /*  13 */ { 1, -3, 6, },
  /*  14 */ { 2, -3, 6, 0, 7, },
  /*  15 */ { 3, -3, 6, 0, 7, 0, 8, },
  /*  16 */ { 0, },
  /*  17 */ { 0, },
  /*  18 */ { 1, 9, 8, },
  /*  19 */ { 1, -9, 8, },
  /*  20 */ { 0, },
  /*  21 */ { 0, },
  /*  22 */ { 1, 10, 8, },
  /*  23 */ { 1, -10, 8, },
  /*  24 */ { 1, 4, 6, },
  /*  25 */ { 1, 4, 6, },
  /*  26 */ { 2, 4, 6, 0, 7, },
  /*  27 */ { 3, 4, 6, 0, 7, 0, 8, },
  /*  28 */ { 1, -4, 6, },
  /*  29 */ { 1, -4, 6, },
  /*  30 */ { 2, -4, 6, 0, 7, },
  /*  31 */ { 3, -4, 6, 0, 7, 0, 8, },
  /*  32 */ { 1, 1, 4, },
  /*  33 */ { 1, 1, 4, },
  /*  34 */ { 2, 1, 4, 1, 8, },
  /*  35 */ { 2, 1, 4, -1, 8, },
  /*  36 */ { 1, 1, 4, },
  /*  37 */ { 1, 1, 4, },
  /*  38 */ { 2, 1, 4, 2, 8, },
  /*  39 */ { 2, 1, 4, -2, 8, },
  /*  40 */ { 2, 1, 4, 0, 5, },
  /*  41 */ { 2, 1, 4, 0, 5, },
  /*  42 */ { 2, 1, 4, 0, 5, },
  /*  43 */ { 2, 1, 4, 0, 5, },
  /*  44 */ { 3, 1, 4, 0, 5, 0, 6, },
  /*  45 */ { 3, 1, 4, 0, 5, 0, 6, },
  /*  46 */ { 4, 1, 4, 0, 5, 0, 6, 0, 7, },
  /*  47 */ { 5, 1, 4, 0, 5, 0, 6, 0, 7, 0, 8, },
  /*  48 */ { 1, -1, 4, },
  /*  49 */ { 1, -1, 4, },
  /*  50 */ { 2, -1, 4, 1, 8, },
  /*  51 */ { 2, -1, 4, -1, 8, },
  /*  52 */ { 1, -1, 4, },
  /*  53 */ { 1, -1, 4, },
  /*  54 */ { 2, -1, 4, 2, 8, },
  /*  55 */ { 2, -1, 4, -2, 8, },
  /*  56 */ { 2, -1, 4, 0, 5, },
  /*  57 */ { 2, -1, 4, 0, 5, },
  /*  58 */ { 2, -1, 4, 0, 5, },
  /*  59 */ { 2, -1, 4, 0, 5, },
  /*  60 */ { 3, -1, 4, 0, 5, 0, 6, },
  /*  61 */ { 3, -1, 4, 0, 5, 0, 6, },
  /*  62 */ { 4, -1, 4, 0, 5, 0, 6, 0, 7, },
  /*  63 */ { 5, -1, 4, 0, 5, 0, 6, 0, 7, 0, 8, },
  /*  64 */ { 0, },
  /*  65 */ { 0, },
  /*  66 */ { 1, 11, 8, },
  /*  67 */ { 1, -11, 8, },
  /*  68 */ { 0, },
  /*  69 */ { 0, },
  /*  70 */ { 1, 12, 8, },
  /*  71 */ { 1, -12, 8, },
  /*  72 */ { 1, 5, 6, },
  /*  73 */ { 1, 5, 6, },
  /*  74 */ { 2, 5, 6, 0, 7, },
  /*  75 */ { 3, 5, 6, 0, 7, 0, 8, },
  /*  76 */ { 1, -5, 6, },
  /*  77 */ { 1, -5, 6, },
  /*  78 */ { 2, -5, 6, 0, 7, },
  /*  79 */ { 3, -5, 6, 0, 7, 0, 8, },
  /*  80 */ { 0, },
  /*  81 */ { 0, },
  /*  82 */ { 1, 13, 8, },
  /*  83 */ { 1, -13, 8, },
  /*  84 */ { 0, },
  /*  85 */ { 0, },
  /*  86 */ { 1, 14, 8, },
  /*  87 */ { 1, -14, 8, },
  /*  88 */ { 1, 6, 6, },
  /*  89 */ { 1, 6, 6, },
  /*  90 */ { 2, 6, 6, 0, 7, },
  /*  91 */ { 3, 6, 6, 0, 7, 0, 8, },
  /*  92 */ { 1, -6, 6, },
  /*  93 */ { 1, -6, 6, },
  /*  94 */ { 2, -6, 6, 0, 7, },
  /*  95 */ { 3, -6, 6, 0, 7, 0, 8, },
  /*  96 */ { 1, 2, 4, },
  /*  97 */ { 1, 2, 4, },
  /*  98 */ { 2, 2, 4, 1, 8, },
  /*  99 */ { 2, 2, 4, -1, 8, },
  /* 100 */ { 1, 2, 4, },
  /* 101 */ { 1, 2, 4, },
  /* 102 */ { 2, 2, 4, 2, 8, },
  /* 103 */ { 2, 2, 4, -2, 8, },
  /* 104 */ { 2, 2, 4, 0, 5, },
  /* 105 */ { 2, 2, 4, 0, 5, },
  /* 106 */ { 2, 2, 4, 0, 5, },
  /* 107 */ { 2, 2, 4, 0, 5, },
  /* 108 */ { 3, 2, 4, 0, 5, 0, 6, },
  /* 109 */ { 3, 2, 4, 0, 5, 0, 6, },
  /* 110 */ { 4, 2, 4, 0, 5, 0, 6, 0, 7, },
  /* 111 */ { 5, 2, 4, 0, 5, 0, 6, 0, 7, 0, 8, },
  /* 112 */ { 1, -2, 4, },
  /* 113 */ { 1, -2, 4, },
  /* 114 */ { 2, -2, 4, 1, 8, },
  /* 115 */ { 2, -2, 4, -1, 8, },
  /* 116 */ { 1, -2, 4, },
  /* 117 */ { 1, -2, 4, },
  /* 118 */ { 2, -2, 4, 2, 8, },
  /* 119 */ { 2, -2, 4, -2, 8, },
  /* 120 */ { 2, -2, 4, 0, 5, },
  /* 121 */ { 2, -2, 4, 0, 5, },
  /* 122 */ { 2, -2, 4, 0, 5, },
  /* 123 */ { 2, -2, 4, 0, 5, },
  /* 124 */ { 3, -2, 4, 0, 5, 0, 6, },
  /* 125 */ { 3, -2, 4, 0, 5, 0, 6, },
  /* 126 */ { 4, -2, 4, 0, 5, 0, 6, 0, 7, },
  /* 127 */ { 5, -2, 4, 0, 5, 0, 6, 0, 7, 0, 8, },
  /* 128 */ { 1, 0, 1, },
  /* 129 */ { 1, 0, 1, },
  /* 130 */ { 1, 0, 1, },
  /* 131 */ { 1, 0, 1, },
  /* 132 */ { 2, 0, 1, 3, 7, },
  /* 133 */ { 3, 0, 1, 3, 7, 0, 8, },
  /* 134 */ { 2, 0, 1, -3, 7, },
  /* 135 */ { 3, 0, 1, -3, 7, 0, 8, },
  /* 136 */ { 1, 0, 1, },
  /* 137 */ { 1, 0, 1, },
  /* 138 */ { 1, 0, 1, },
  /* 139 */ { 1, 0, 1, },
  /* 140 */ { 2, 0, 1, 4, 7, },
  /* 141 */ { 3, 0, 1, 4, 7, 0, 8, },
  /* 142 */ { 2, 0, 1, -4, 7, },
  /* 143 */ { 3, 0, 1, -4, 7, 0, 8, },
  /* 144 */ { 2, 0, 1, 1, 5, },
  /* 145 */ { 2, 0, 1, 1, 5, },
  /* 146 */ { 2, 0, 1, 1, 5, },
  /* 147 */ { 2, 0, 1, 1, 5, },
  /* 148 */ { 3, 0, 1, 1, 5, 0, 6, },
  /* 149 */ { 3, 0, 1, 1, 5, 0, 6, },
  /* 150 */ { 4, 0, 1, 1, 5, 0, 6, 0, 7, },
  /* 151 */ { 5, 0, 1, 1, 5, 0, 6, 0, 7, 0, 8, },
  /* 152 */ { 2, 0, 1, -1, 5, },
  /* 153 */ { 2, 0, 1, -1, 5, },
  /* 154 */ { 2, 0, 1, -1, 5, },
  /* 155 */ { 2, 0, 1, -1, 5, },
  /* 156 */ { 3, 0, 1, -1, 5, 0, 6, },
  /* 157 */ { 3, 0, 1, -1, 5, 0, 6, },
  /* 158 */ { 4, 0, 1, -1, 5, 0, 6, 0, 7, },
  /* 159 */ { 5, 0, 1, -1, 5, 0, 6, 0, 7, 0, 8, },
  /* 160 */ { 1, 0, 1, },
  /* 161 */ { 1, 0, 1, },
  /* 162 */ { 1, 0, 1, },
  /* 163 */ { 1, 0, 1, },
  /* 164 */ { 2, 0, 1, 5, 7, },
  /* 165 */ { 3, 0, 1, 5, 7, 0, 8, },
  /* 166 */ { 2, 0, 1, -5, 7, },
  /* 167 */ { 3, 0, 1, -5, 7, 0, 8, },
  /* 168 */ { 1, 0, 1, },
  /* 169 */ { 1, 0, 1, },
  /* 170 */ { 1, 0, 1, },
  /* 171 */ { 1, 0, 1, },
  /* 172 */ { 2, 0, 1, 6, 7, },
  /* 173 */ { 3, 0, 1, 6, 7, 0, 8, },
  /* 174 */ { 2, 0, 1, -6, 7, },
  /* 175 */ { 3, 0, 1, -6, 7, 0, 8, },
  /* 176 */ { 2, 0, 1, 2, 5, },
  /* 177 */ { 2, 0, 1, 2, 5, },
  /* 178 */ { 2, 0, 1, 2, 5, },
  /* 179 */ { 2, 0, 1, 2, 5, },
  /* 180 */ { 3, 0, 1, 2, 5, 0, 6, },
  /* 181 */ { 3, 0, 1, 2, 5, 0, 6, },
  /* 182 */ { 4, 0, 1, 2, 5, 0, 6, 0, 7, },
  /* 183 */ { 5, 0, 1, 2, 5, 0, 6, 0, 7, 0, 8, },
  /* 184 */ { 2, 0, 1, -2, 5, },
  /* 185 */ { 2, 0, 1, -2, 5, },
  /* 186 */ { 2, 0, 1, -2, 5, },
  /* 187 */ { 2, 0, 1, -2, 5, },
  /* 188 */ { 3, 0, 1, -2, 5, 0, 6, },
  /* 189 */ { 3, 0, 1, -2, 5, 0, 6, },
  /* 190 */ { 4, 0, 1, -2, 5, 0, 6, 0, 7, },
  /* 191 */ { 5, 0, 1, -2, 5, 0, 6, 0, 7, 0, 8, },
  /* 192 */ { 2, 0, 1, 0, 2, },
  /* 193 */ { 2, 0, 1, 0, 2, },
  /* 194 */ { 3, 0, 1, 0, 2, 3, 8, },
  /* 195 */ { 3, 0, 1, 0, 2, -3, 8, },
  /* 196 */ { 2, 0, 1, 0, 2, },
  /* 197 */ { 2, 0, 1, 0, 2, },
  /* 198 */ { 3, 0, 1, 0, 2, 4, 8, },
  /* 199 */ { 3, 0, 1, 0, 2, -4, 8, },
  /* 200 */ { 3, 0, 1, 0, 2, 1, 6, },
  /* 201 */ { 3, 0, 1, 0, 2, 1, 6, },
  /* 202 */ { 4, 0, 1, 0, 2, 1, 6, 0, 7, },
  /* 203 */ { 5, 0, 1, 0, 2, 1, 6, 0, 7, 0, 8, },
  /* 204 */ { 3, 0, 1, 0, 2, -1, 6, },
  /* 205 */ { 3, 0, 1, 0, 2, -1, 6, },
  /* 206 */ { 4, 0, 1, 0, 2, -1, 6, 0, 7, },
  /* 207 */ { 5, 0, 1, 0, 2, -1, 6, 0, 7, 0, 8, },
  /* 208 */ { 2, 0, 1, 0, 2, },
  /* 209 */ { 2, 0, 1, 0, 2, },
  /* 210 */ { 3, 0, 1, 0, 2, 5, 8, },
  /* 211 */ { 3, 0, 1, 0, 2, -5, 8, },
  /* 212 */ { 2, 0, 1, 0, 2, },
  /* 213 */ { 2, 0, 1, 0, 2, },
  /* 214 */ { 3, 0, 1, 0, 2, 6, 8, },
  /* 215 */ { 3, 0, 1, 0, 2, -6, 8, },
  /* 216 */ { 3, 0, 1, 0, 2, 2, 6, },
  /* 217 */ { 3, 0, 1, 0, 2, 2, 6, },
  /* 218 */ { 4, 0, 1, 0, 2, 2, 6, 0, 7, },
  /* 219 */ { 5, 0, 1, 0, 2, 2, 6, 0, 7, 0, 8, },
  /* 220 */ { 3, 0, 1, 0, 2, -2, 6, },
  /* 221 */ { 3, 0, 1, 0, 2, -2, 6, },
  /* 222 */ { 4, 0, 1, 0, 2, -2, 6, 0, 7, },
  /* 223 */ { 5, 0, 1, 0, 2, -2, 6, 0, 7, 0, 8, },
  /* 224 */ { 3, 0, 1, 0, 2, 0, 3, },
  /* 225 */ { 3, 0, 1, 0, 2, 0, 3, },
  /* 226 */ { 3, 0, 1, 0, 2, 0, 3, },
  /* 227 */ { 3, 0, 1, 0, 2, 0, 3, },
  /* 228 */ { 4, 0, 1, 0, 2, 0, 3, 1, 7, },
  /* 229 */ { 5, 0, 1, 0, 2, 0, 3, 1, 7, 0, 8, },
  /* 230 */ { 4, 0, 1, 0, 2, 0, 3, -1, 7, },
  /* 231 */ { 5, 0, 1, 0, 2, 0, 3, -1, 7, 0, 8, },
  /* 232 */ { 3, 0, 1, 0, 2, 0, 3, },
  /* 233 */ { 3, 0, 1, 0, 2, 0, 3, },
  /* 234 */ { 3, 0, 1, 0, 2, 0, 3, },
  /* 235 */ { 3, 0, 1, 0, 2, 0, 3, },
  /* 236 */ { 4, 0, 1, 0, 2, 0, 3, 2, 7, },
  /* 237 */ { 5, 0, 1, 0, 2, 0, 3, 2, 7, 0, 8, },
  /* 238 */ { 4, 0, 1, 0, 2, 0, 3, -2, 7, },
  /* 239 */ { 5, 0, 1, 0, 2, 0, 3, -2, 7, 0, 8, },
  /* 240 */ { 4, 0, 1, 0, 2, 0, 3, 0, 4, },
  /* 241 */ { 4, 0, 1, 0, 2, 0, 3, 0, 4, },
  /* 242 */ { 5, 0, 1, 0, 2, 0, 3, 0, 4, 1, 8, },
  /* 243 */ { 5, 0, 1, 0, 2, 0, 3, 0, 4, -1, 8, },
  /* 244 */ { 4, 0, 1, 0, 2, 0, 3, 0, 4, },
  /* 245 */ { 4, 0, 1, 0, 2, 0, 3, 0, 4, },
  /* 246 */ { 5, 0, 1, 0, 2, 0, 3, 0, 4, 2, 8, },
  /* 247 */ { 5, 0, 1, 0, 2, 0, 3, 0, 4, -2, 8, },
  /* 248 */ { 5, 0, 1, 0, 2, 0, 3, 0, 4, 0, 5, },
  /* 249 */ { 5, 0, 1, 0, 2, 0, 3, 0, 4, 0, 5, },
  /* 250 */ { 5, 0, 1, 0, 2, 0, 3, 0, 4, 0, 5, },
  /* 251 */ { 5, 0, 1, 0, 2, 0, 3, 0, 4, 0, 5, },
  /* 252 */ { 6, 0, 1, 0, 2, 0, 3, 0, 4, 0, 5, 0, 6, },
  /* 253 */ { 6, 0, 1, 0, 2, 0, 3, 0, 4, 0, 5, 0, 6, },
  /* 254 */ { 7, 0, 1, 0, 2, 0, 3, 0, 4, 0, 5, 0, 6, 0, 7, },
  /* 255 */ { 8, 0, 1, 0, 2, 0, 3, 0, 4, 0, 5, 0, 6, 0, 7, 0, 8, },
};



const int8_t test3_table_unpack_sint[1024][21] = {
  /*   0 */ { 0, },
  /*   1 */ { 0, },
  /*   2 */ { 1, 15, 10, },
  /*   3 */ { 1, -15, 10, },
  /*   4 */ { 0, },
  /*   5 */ { 0, },
  /*   6 */ { 1, 16, 10, },
  /*   7 */ { 1, -16, 10, },
  /*   8 */ { 1, 7, 8, },
  /*   9 */ { 1, 7, 8, },
  /*  10 */ { 2, 7, 8, 0, 9, },
  /*  11 */ { 3, 7, 8, 0, 9, 0, 10, },
  /*  12 */ { 1, -7, 8, },
  /*  13 */ { 1, -7, 8, },
  /*  14 */ { 2, -7, 8, 0, 9, },
  /*  15 */ { 3, -7, 8, 0, 9, 0, 10, },
  /*  16 */ { 0, },
  /*  17 */ { 0, },
  /*  18 */ { 1, 17, 10, },
  /*  19 */ { 1, -17, 10, },
  /*  20 */ { 0, },
  /*  21 */ { 0, },
  /*  22 */ { 1, 18, 10, },
  /*  23 */ { 1, -18, 10, },
  /*  24 */ { 1, 8, 8, },
  /*  25 */ { 1, 8, 8, },
  /*  26 */ { 2, 8, 8, 0, 9, },
  /*  27 */ { 3, 8, 8, 0, 9, 0, 10, },
  /*  28 */ { 1, -8, 8, },
  /*  29 */ { 1, -8, 8, },
  /*  30 */ { 2, -8, 8, 0, 9, },
  /*  31 */ { 3, -8, 8, 0, 9, 0, 10, },
  /*  32 */ { 1, 3, 6, },
  /*  33 */ { 1, 3, 6, },
  /*  34 */ { 2, 3, 6, 1, 10, },
  /*  35 */ { 2, 3, 6, -1, 10, },
  /*  36 */ { 1, 3, 6, },
  /*  37 */ { 1, 3, 6, },
  /*  38 */ { 2, 3, 6, 2, 10, },
  /*  39 */ { 2, 3, 6, -2, 10, },
  /*  40 */ { 2, 3, 6, 0, 7, },
  /*  41 */ { 2, 3, 6, 0, 7, },
  /*  42 */ { 2, 3, 6, 0, 7, },
  /*  43 */ { 2, 3, 6, 0, 7, },
  /*  44 */ { 3, 3, 6, 0, 7, 0, 8, },
  /*  45 */ { 3, 3, 6, 0, 7, 0, 8, },
  /*  46 */ { 4, 3, 6, 0, 7, 0, 8, 0, 9, },
  /*  47 */ { 5, 3, 6, 0, 7, 0, 8, 0, 9, 0, 10, },
  /*  48 */ { 1, -3, 6, },
  /*  49 */ { 1, -3, 6, },
  /*  50 */ { 2, -3, 6, 1, 10, },
  /*  51 */ { 2, -3, 6, -1, 10, },
  /*  52 */ { 1, -3, 6, },
  /*  53 */ { 1, -3, 6, },
  /*  54 */ { 2, -3, 6, 2, 10, },
  /*  55 */ { 2, -3, 6, -2, 10, },
  /*  56 */ { 2, -3, 6, 0, 7, },
  /*  57 */ { 2, -3, 6, 0, 7, },
  /*  58 */ { 2, -3, 6, 0, 7, },
  /*  59 */ { 2, -3, 6, 0, 7, },
  /*  60 */ { 3, -3, 6, 0, 7, 0, 8, },
  /*  61 */ { 3, -3, 6, 0, 7, 0, 8, },
  /*  62 */ { 4, -3, 6, 0, 7, 0, 8, 0, 9, },
  /*  63 */ { 5, -3, 6, 0, 7, 0, 8, 0, 9, 0, 10, },
  /*  64 */ { 0, },
  /*  65 */ { 0, },
  /*  66 */ { 1, 19, 10, },
  /*  67 */ { 1, -19, 10, },
  /*  68 */ { 0, },
  /*  69 */ { 0, },
  /*  70 */ { 1, 20, 10, },
  /*  71 */ { 1, -20, 10, },
  /*  72 */ { 1, 9, 8, },
  /*  73 */ { 1, 9, 8, },
  /*  74 */ { 2, 9, 8, 0, 9, },
  /*  75 */ { 3, 9, 8, 0, 9, 0, 10, },
  /*  76 */ { 1, -9, 8, },
  /*  77 */ { 1, -9, 8, },
  /*  78 */ { 2, -9, 8, 0, 9, },
  /*  79 */ { 3, -9, 8, 0, 9, 0, 10, },
  /*  80 */ { 0, },
  /*  81 */ { 0, },
  /*  82 */ { 1, 21, 10, },
  /*  83 */ { 1, -21, 10, },
  /*  84 */ { 0, },
  /*  85 */ { 0, },
  /*  86 */ { 1, 22, 10, },
  /*  87 */ { 1, -22, 10, },
  /*  88 */ { 1, 10, 8, },
  /*  89 */ { 1, 10, 8, },
  /*  90 */ { 2, 10, 8, 0, 9, },
  /*  91 */ { 3, 10, 8, 0, 9, 0, 10, },
  /*  92 */ { 1, -10, 8, },
  /*  93 */ { 1, -10, 8, },
  /*  94 */ { 2, -10, 8, 0, 9, },
  /*  95 */ { 3, -10, 8, 0, 9, 0, 10, },
  /*  96 */ { 1, 4, 6, },
  /*  97 */ { 1, 4, 6, },
  /*  98 */ { 2, 4, 6, 1, 10, },
  /*  99 */ { 2, 4, 6, -1, 10, },
  /* 100 */ { 1, 4, 6, },
  /* 101 */ { 1, 4, 6, },
  /* 102 */ { 2, 4, 6, 2, 10, },
  /* 103 */ { 2, 4, 6, -2, 10, },
  /* 104 */ { 2, 4, 6, 0, 7, },
  /* 105 */ { 2, 4, 6, 0, 7, },
  /* 106 */ { 2, 4, 6, 0, 7, },
  /* 107 */ { 2, 4, 6, 0, 7, },
  /* 108 */ { 3, 4, 6, 0, 7, 0, 8, },
  /* 109 */ { 3, 4, 6, 0, 7, 0, 8, },
  /* 110 */ { 4, 4, 6, 0, 7, 0, 8, 0, 9, },
  /* 111 */ { 5, 4, 6, 0, 7, 0, 8, 0, 9, 0, 10, },
  /* 112 */ { 1, -4, 6, },
  /* 113 */ { 1, -4, 6, },
  /* 114 */ { 2, -4, 6, 1, 10, },
  /* 115 */ { 2, -4, 6, -1, 10, },
  /* 116 */ { 1, -4, 6, },
  /* 117 */ { 1, -4, 6, },
  /* 118 */ { 2, -4, 6, 2, 10, },
  /* 119 */ { 2, -4, 6, -2, 10, },
  /* 120 */ { 2, -4, 6, 0, 7, },
  /* 121 */ { 2, -4, 6, 0, 7, },
  /* 122 */ { 2, -4, 6, 0, 7, },
  /* 123 */ { 2, -4, 6, 0, 7, },
  /* 124 */ { 3, -4, 6, 0, 7, 0, 8, },
  /* 125 */ { 3, -4, 6, 0, 7, 0, 8, },
  /* 126 */ { 4, -4, 6, 0, 7, 0, 8, 0, 9, },
  /* 127 */ { 5, -4, 6, 0, 7, 0, 8, 0, 9, 0, 10, },
  /* 128 */ { 1, 1, 4, },
  /* 129 */ { 1, 1, 4, },
  /* 130 */ { 2, 1, 4, 3, 10, },
  /* 131 */ { 2, 1, 4, -3, 10, },
  /* 132 */ { 1, 1, 4, },
  /* 133 */ { 1, 1, 4, },
  /* 134 */ { 2, 1, 4, 4, 10, },
  /* 135 */ { 2, 1, 4, -4, 10, },
  /* 136 */ { 2, 1, 4, 1, 8, },
  /* 137 */ { 2, 1, 4, 1, 8, },
  /* 138 */ { 3, 1, 4, 1, 8, 0, 9, },
  /* 139 */ { 4, 1, 4, 1, 8, 0, 9, 0, 10, },
  /* 140 */ { 2, 1, 4, -1, 8, },
  /* 141 */ { 2, 1, 4, -1, 8, },
  /* 142 */ { 3, 1, 4, -1, 8, 0, 9, },
  /* 143 */ { 4, 1, 4, -1, 8, 0, 9, 0, 10, },
  /* 144 */ { 1, 1, 4, },
  /* 145 */ { 1, 1, 4, },
  /* 146 */ { 2, 1, 4, 5, 10, },
  /* 147 */ { 2, 1, 4, -5, 10, },
  /* 148 */ { 1, 1, 4, },
  /* 149 */ { 1, 1, 4, },
  /* 150 */ { 2, 1, 4, 6, 10, },
  /* 151 */ { 2, 1, 4, -6, 10, },
  /* 152 */ { 2, 1, 4, 2, 8, },
  /* 153 */ { 2, 1, 4, 2, 8, },
  /* 154 */ { 3, 1, 4, 2, 8, 0, 9, },
  /* 155 */ { 4, 1, 4, 2, 8, 0, 9, 0, 10, },
  /* 156 */ { 2, 1, 4, -2, 8, },
  /* 157 */ { 2, 1, 4, -2, 8, },
  /* 158 */ { 3, 1, 4, -2, 8, 0, 9, },
  /* 159 */ { 4, 1, 4, -2, 8, 0, 9, 0, 10, },
  /* 160 */ { 2, 1, 4, 0, 5, },
  /* 161 */ { 2, 1, 4, 0, 5, },
  /* 162 */ { 2, 1, 4, 0, 5, },
  /* 163 */ { 2, 1, 4, 0, 5, },
  /* 164 */ { 3, 1, 4, 0, 5, 1, 9, },
  /* 165 */ { 4, 1, 4, 0, 5, 1, 9, 0, 10, },
  /* 166 */ { 3, 1, 4, 0, 5, -1, 9, },
  /* 167 */ { 4, 1, 4, 0, 5, -1, 9, 0, 10, },
  /* 168 */ { 2, 1, 4, 0, 5, },
  /* 169 */ { 2, 1, 4, 0, 5, },
  /* 170 */ { 2, 1, 4, 0, 5, },
  /* 171 */ { 2, 1, 4, 0, 5, },
  /* 172 */ { 3, 1, 4, 0, 5, 2, 9, },
  /* 173 */ { 4, 1, 4, 0, 5, 2, 9, 0, 10, },
  /* 174 */ { 3, 1, 4, 0, 5, -2, 9, },
  /* 175 */ { 4, 1, 4, 0, 5, -2, 9, 0, 10, },
  /* 176 */ { 3, 1, 4, 0, 5, 0, 6, },
  /* 177 */ { 3, 1, 4, 0, 5, 0, 6, },
  /* 178 */ { 4, 1, 4, 0, 5, 0, 6, 1, 10, },
  /* 179 */ { 4, 1, 4, 0, 5, 0, 6, -1, 10, },
  /* 180 */ { 3, 1, 4, 0, 5, 0, 6, },
  /* 181 */ { 3, 1, 4, 0, 5, 0, 6, },
  /* 182 */ { 4, 1, 4, 0, 5, 0, 6, 2, 10, },
  /* 183 */ { 4, 1, 4, 0, 5, 0, 6, -2, 10, },
  /* 184 */ { 4, 1, 4, 0, 5, 0, 6, 0, 7, },
  /* 185 */ { 4, 1, 4, 0, 5, 0, 6, 0, 7, },
  /* 186 */ { 4, 1, 4, 0, 5, 0, 6, 0, 7, },
  /* 187 */ { 4, 1, 4, 0, 5, 0, 6, 0, 7, },
  /* 188 */ { 5, 1, 4, 0, 5, 0, 6, 0, 7, 0, 8, },
  /* 189 */ { 5, 1, 4, 0, 5, 0, 6, 0, 7, 0, 8, },
  /* 190 */ { 6, 1, 4, 0, 5, 0, 6, 0, 7, 0, 8, 0, 9, },
  /* 191 */ { 7, 1, 4, 0, 5, 0, 6, 0, 7, 0, 8, 0, 9, 0, 10, },
  /* 192 */ { 1, -1, 4, },
  /* 193 */ { 1, -1, 4, },
  /* 194 */ { 2, -1, 4, 3, 10, },
  /* 195 */ { 2, -1, 4, -3, 10, },
  /* 196 */ { 1, -1, 4, },
  /* 197 */ { 1, -1, 4, },
  /* 198 */ { 2, -1, 4, 4, 10, },
  /* 199 */ { 2, -1, 4, -4, 10, },
  /* 200 */ { 2, -1, 4, 1, 8, },
  /* 201 */ { 2, -1, 4, 1, 8, },
  /* 202 */ { 3, -1, 4, 1, 8, 0, 9, },
  /* 203 */ { 4, -1, 4, 1, 8, 0, 9, 0, 10, },
  /* 204 */ { 2, -1, 4, -1, 8, },
  /* 205 */ { 2, -1, 4, -1, 8, },
  /* 206 */ { 3, -1, 4, -1, 8, 0, 9, },
  /* 207 */ { 4, -1, 4, -1, 8, 0, 9, 0, 10, },
  /* 208 */ { 1, -1, 4, },
  /* 209 */ { 1, -1, 4, },
  /* 210 */ { 2, -1, 4, 5, 10, },
  /* 211 */ { 2, -1, 4, -5, 10, },
  /* 212 */ { 1, -1, 4, },
  /* 213 */ { 1, -1, 4, },
  /* 214 */ { 2, -1, 4, 6, 10, },
  /* 215 */ { 2, -1, 4, -6, 10, },
  /* 216 */ { 2, -1, 4, 2, 8, },
  /* 217 */ { 2, -1, 4, 2, 8, },
  /* 218 */ { 3, -1, 4, 2, 8, 0, 9, },
  /* 219 */ { 4, -1, 4, 2, 8, 0, 9, 0, 10, },
  /* 220 */ { 2, -1, 4, -2, 8, },
  /* 221 */ { 2, -1, 4, -2, 8, },
  /* 222 */ { 3, -1, 4, -2, 8, 0, 9, },
  /* 223 */ { 4, -1, 4, -2, 8, 0, 9, 0, 10, },
  /* 224 */ { 2, -1, 4, 0, 5, },
  /* 225 */ { 2, -1, 4, 0, 5, },
  /* 226 */ { 2, -1, 4, 0, 5, },
  /* 227 */ { 2, -1, 4, 0, 5, },
  /* 228 */ { 3, -1, 4, 0, 5, 1, 9, },
  /* 229 */ { 4, -1, 4, 0, 5, 1, 9, 0, 10, },
  /* 230 */ { 3, -1, 4, 0, 5, -1, 9, },
  /* 231 */ { 4, -1, 4, 0, 5, -1, 9, 0, 10, },
  /* 232 */ { 2, -1, 4, 0, 5, },
  /* 233 */ { 2, -1, 4, 0, 5, },
  /* 234 */ { 2, -1, 4, 0, 5, },
  /* 235 */ { 2, -1, 4, 0, 5, },
  /* 236 */ { 3, -1, 4, 0, 5, 2, 9, },
  /* 237 */ { 4, -1, 4, 0, 5, 2, 9, 0, 10, },
  /* 238 */ { 3, -1, 4, 0, 5, -2, 9, },
  /* 239 */ { 4, -1, 4, 0, 5, -2, 9, 0, 10, },
  /* 240 */ { 3, -1, 4, 0, 5, 0, 6, },
  /* 241 */ { 3, -1, 4, 0, 5, 0, 6, },
  /* 242 */ { 4, -1, 4, 0, 5, 0, 6, 1, 10, },
  /* 243 */ { 4, -1, 4, 0, 5, 0, 6, -1, 10, },
  /* 244 */ { 3, -1, 4, 0, 5, 0, 6, },
  /* 245 */ { 3, -1, 4, 0, 5, 0, 6, },
  /* 246 */ { 4, -1, 4, 0, 5, 0, 6, 2, 10, },
  /* 247 */ { 4, -1, 4, 0, 5, 0, 6, -2, 10, },
  /* 248 */ { 4, -1, 4, 0, 5, 0, 6, 0, 7, },
  /* 249 */ { 4, -1, 4, 0, 5, 0, 6, 0, 7, },
  /* 250 */ { 4, -1, 4, 0, 5, 0, 6, 0, 7, },
  /* 251 */ { 4, -1, 4, 0, 5, 0, 6, 0, 7, },
  /* 252 */ { 5, -1, 4, 0, 5, 0, 6, 0, 7, 0, 8, },
  /* 253 */ { 5, -1, 4, 0, 5, 0, 6, 0, 7, 0, 8, },
  /* 254 */ { 6, -1, 4, 0, 5, 0, 6, 0, 7, 0, 8, 0, 9, },
  /* 255 */ { 7, -1, 4, 0, 5, 0, 6, 0, 7, 0, 8, 0, 9, 0, 10, },
  /* 256 */ { 0, },
  /* 257 */ { 0, },
  /* 258 */ { 1, 23, 10, },
  /* 259 */ { 1, -23, 10, },
  /* 260 */ { 0, },
  /* 261 */ { 0, },
  /* 262 */ { 1, 24, 10, },
  /* 263 */ { 1, -24, 10, },
  /* 264 */ { 1, 11, 8, },
  /* 265 */ { 1, 11, 8, },
  /* 266 */ { 2, 11, 8, 0, 9, },
  /* 267 */ { 3, 11, 8, 0, 9, 0, 10, },
  /* 268 */ { 1, -11, 8, },
  /* 269 */ { 1, -11, 8, },
  /* 270 */ { 2, -11, 8, 0, 9, },
  /* 271 */ { 3, -11, 8, 0, 9, 0, 10, },
  /* 272 */ { 0, },
  /* 273 */ { 0, },
  /* 274 */ { 1, 25, 10, },
  /* 275 */ { 1, -25, 10, },
  /* 276 */ { 0, },
  /* 277 */ { 0, },
  /* 278 */ { 1, 26, 10, },
  /* 279 */ { 1, -26, 10, },
  /* 280 */ { 1, 12, 8, },
  /* 281 */ { 1, 12, 8, },
  /* 282 */ { 2, 12, 8, 0, 9, },
  /* 283 */ { 3, 12, 8, 0, 9, 0, 10, },
  /* 284 */ { 1, -12, 8, },
  /* 285 */ { 1, -12, 8, },
  /* 286 */ { 2, -12, 8, 0, 9, },
  /* 287 */ { 3, -12, 8, 0, 9, 0, 10, },
  /* 288 */ { 1, 5, 6, },
  /* 289 */ { 1, 5, 6, },
  /* 290 */ { 2, 5, 6, 1, 10, },
  /* 291 */ { 2, 5, 6, -1, 10, },
  /* 292 */ { 1, 5, 6, },
  /* 293 */ { 1, 5, 6, },
  /* 294 */ { 2, 5, 6, 2, 10, },
  /* 295 */ { 2, 5, 6, -2, 10, },
  /* 296 */ { 2, 5, 6, 0, 7, },
  /* 297 */ { 2, 5, 6, 0, 7, },
  /* 298 */ { 2, 5, 6, 0, 7, },
  /* 299 */ { 2, 5, 6, 0, 7, },
  /* 300 */ { 3, 5, 6, 0, 7, 0, 8, },
  /* 301 */ { 3, 5, 6, 0, 7, 0, 8, },
  /* 302 */ { 4, 5, 6, 0, 7, 0, 8, 0, 9, },
  /* 303 */ { 5, 5, 6, 0, 7, 0, 8, 0, 9, 0, 10, },
  /* 304 */ { 1, -5, 6, },
  /* 305 */ { 1, -5, 6, },
  /* 306 */ { 2, -5, 6, 1, 10, },
  /* 307 */ { 2, -5, 6, -1, 10, },
  /* 308 */ { 1, -5, 6, },
  /* 309 */ { 1, -5, 6, },
  /* 310 */ { 2, -5, 6, 2, 10, },
  /* 311 */ { 2, -5, 6, -2, 10, },
  /* 312 */ { 2, -5, 6, 0, 7, },
  /* 313 */ { 2, -5, 6, 0, 7, },
  /* 314 */ { 2, -5, 6, 0, 7, },
  /* 315 */ { 2, -5, 6, 0, 7, },
  /* 316 */ { 3, -5, 6, 0, 7, 0, 8, },
  /* 317 */ { 3, -5, 6, 0, 7, 0, 8, },
  /* 318 */ { 4, -5, 6, 0, 7, 0, 8, 0, 9, },
  /* 319 */ { 5, -5, 6, 0, 7, 0, 8, 0, 9, 0, 10, },
  /* 320 */ { 0, },
  /* 321 */ { 0, },
  /* 322 */ { 1, 27, 10, },
  /* 323 */ { 1, -27, 10, },
  /* 324 */ { 0, },
  /* 325 */ { 0, },
  /* 326 */ { 1, 28, 10, },
  /* 327 */ { 1, -28, 10, },
  /* 328 */ { 1, 13, 8, },
  /* 329 */ { 1, 13, 8, },
  /* 330 */ { 2, 13, 8, 0, 9, },
  /* 331 */ { 3, 13, 8, 0, 9, 0, 10, },
  /* 332 */ { 1, -13, 8, },
  /* 333 */ { 1, -13, 8, },
  /* 334 */ { 2, -13, 8, 0, 9, },
  /* 335 */ { 3, -13, 8, 0, 9, 0, 10, },
  /* 336 */ { 0, },
  /* 337 */ { 0, },
  /* 338 */ { 1, 29, 10, },
  /* 339 */ { 1, -29, 10, },
  /* 340 */ { 0, },
  /* 341 */ { 0, },
  /* 342 */ { 1, 30, 10, },
  /* 343 */ { 1, -30, 10, },
  /* 344 */ { 1, 14, 8, },
  /* 345 */ { 1, 14, 8, },
  /* 346 */ { 2, 14, 8, 0, 9, },
  /* 347 */ { 3, 14, 8, 0, 9, 0, 10, },
  /* 348 */ { 1, -14, 8, },
  /* 349 */ { 1, -14, 8, },
  /* 350 */ { 2, -14, 8, 0, 9, },
  /* 351 */ { 3, -14, 8, 0, 9, 0, 10, },
  /* 352 */ { 1, 6, 6, },
  /* 353 */ { 1, 6, 6, },
  /* 354 */ { 2, 6, 6, 1, 10, },
  /* 355 */ { 2, 6, 6, -1, 10, },
  /* 356 */ { 1, 6, 6, },
  /* 357 */ { 1, 6, 6, },
  /* 358 */ { 2, 6, 6, 2, 10, },
  /* 359 */ { 2, 6, 6, -2, 10, },
  /* 360 */ { 2, 6, 6, 0, 7, },
  /* 361 */ { 2, 6, 6, 0, 7, },
  /* 362 */ { 2, 6, 6, 0, 7, },
  /* 363 */ { 2, 6, 6, 0, 7, },
  /* 364 */ { 3, 6, 6, 0, 7, 0, 8, },
  /* 365 */ { 3, 6, 6, 0, 7, 0, 8, },
  /* 366 */ { 4, 6, 6, 0, 7, 0, 8, 0, 9, },
  /* 367 */ { 5, 6, 6, 0, 7, 0, 8, 0, 9, 0, 10, },
  /* 368 */ { 1, -6, 6, },
  /* 369 */ { 1, -6, 6, },
  /* 370 */ { 2, -6, 6, 1, 10, },
  /* 371 */ { 2, -6, 6, -1, 10, },
  /* 372 */ { 1, -6, 6, },
  /* 373 */ { 1, -6, 6, },
  /* 374 */ { 2, -6, 6, 2, 10, },
  /* 375 */ { 2, -6, 6, -2, 10, },
  /* 376 */ { 2, -6, 6, 0, 7, },
  /* 377 */ { 2, -6, 6, 0, 7, },
  /* 378 */ { 2, -6, 6, 0, 7, },
  /* 379 */ { 2, -6, 6, 0, 7, },
  /* 380 */ { 3, -6, 6, 0, 7, 0, 8, },
  /* 381 */ { 3, -6, 6, 0, 7, 0, 8, },
  /* 382 */ { 4, -6, 6, 0, 7, 0, 8, 0, 9, },
  /* 383 */ { 5, -6, 6, 0, 7, 0, 8, 0, 9, 0, 10, },
  /* 384 */ { 1, 2, 4, },
  /* 385 */ { 1, 2, 4, },
  /* 386 */ { 2, 2, 4, 3, 10, },
  /* 387 */ { 2, 2, 4, -3, 10, },
  /* 388 */ { 1, 2, 4, },
  /* 389 */ { 1, 2, 4, },
  /* 390 */ { 2, 2, 4, 4, 10, },
  /* 391 */ { 2, 2, 4, -4, 10, },
  /* 392 */ { 2, 2, 4, 1, 8, },
  /* 393 */ { 2, 2, 4, 1, 8, },
  /* 394 */ { 3, 2, 4, 1, 8, 0, 9, },
  /* 395 */ { 4, 2, 4, 1, 8, 0, 9, 0, 10, },
  /* 396 */ { 2, 2, 4, -1, 8, },
  /* 397 */ { 2, 2, 4, -1, 8, },
  /* 398 */ { 3, 2, 4, -1, 8, 0, 9, },
  /* 399 */ { 4, 2, 4, -1, 8, 0, 9, 0, 10, },
  /* 400 */ { 1, 2, 4, },
  /* 401 */ { 1, 2, 4, },
  /* 402 */ { 2, 2, 4, 5, 10, },
  /* 403 */ { 2, 2, 4, -5, 10, },
  /* 404 */ { 1, 2, 4, },
  /* 405 */ { 1, 2, 4, },
  /* 406 */ { 2, 2, 4, 6, 10, },
  /* 407 */ { 2, 2, 4, -6, 10, },
  /* 408 */ { 2, 2, 4, 2, 8, },
  /* 409 */ { 2, 2, 4, 2, 8, },
  /* 410 */ { 3, 2, 4, 2, 8, 0, 9, },
  /* 411 */ { 4, 2, 4, 2, 8, 0, 9, 0, 10, },
  /* 412 */ { 2, 2, 4, -2, 8, },
  /* 413 */ { 2, 2, 4, -2, 8, },
  /* 414 */ { 3, 2, 4, -2, 8, 0, 9, },
  /* 415 */ { 4, 2, 4, -2, 8, 0, 9, 0, 10, },
  /* 416 */ { 2, 2, 4, 0, 5, },
  /* 417 */ { 2, 2, 4, 0, 5, },
  /* 418 */ { 2, 2, 4, 0, 5, },
  /* 419 */ { 2, 2, 4, 0, 5, },
  /* 420 */ { 3, 2, 4, 0, 5, 1, 9, },
  /* 421 */ { 4, 2, 4, 0, 5, 1, 9, 0, 10, },
  /* 422 */ { 3, 2, 4, 0, 5, -1, 9, },
  /* 423 */ { 4, 2, 4, 0, 5, -1, 9, 0, 10, },
  /* 424 */ { 2, 2, 4, 0, 5, },
  /* 425 */ { 2, 2, 4, 0, 5, },
  /* 426 */ { 2, 2, 4, 0, 5, },
  /* 427 */ { 2, 2, 4, 0, 5, },
  /* 428 */ { 3, 2, 4, 0, 5, 2, 9, },
  /* 429 */ { 4, 2, 4, 0, 5, 2, 9, 0, 10, },
  /* 430 */ { 3, 2, 4, 0, 5, -2, 9, },
  /* 431 */ { 4, 2, 4, 0, 5, -2, 9, 0, 10, },
  /* 432 */ { 3, 2, 4, 0, 5, 0, 6, },
  /* 433 */ { 3, 2, 4, 0, 5, 0, 6, },
  /* 434 */ { 4, 2, 4, 0, 5, 0, 6, 1, 10, },
  /* 435 */ { 4, 2, 4, 0, 5, 0, 6, -1, 10, },
  /* 436 */ { 3, 2, 4, 0, 5, 0, 6, },
  /* 437 */ { 3, 2, 4, 0, 5, 0, 6, },
  /* 438 */ { 4, 2, 4, 0, 5, 0, 6, 2, 10, },
  /* 439 */ { 4, 2, 4, 0, 5, 0, 6, -2, 10, },
  /* 440 */ { 4, 2, 4, 0, 5, 0, 6, 0, 7, },
  /* 441 */ { 4, 2, 4, 0, 5, 0, 6, 0, 7, },
  /* 442 */ { 4, 2, 4, 0, 5, 0, 6, 0, 7, },
  /* 443 */ { 4, 2, 4, 0, 5, 0, 6, 0, 7, },
  /* 444 */ { 5, 2, 4, 0, 5, 0, 6, 0, 7, 0, 8, },
  /* 445 */ { 5, 2, 4, 0, 5, 0, 6, 0, 7, 0, 8, },
  /* 446 */ { 6, 2, 4, 0, 5, 0, 6, 0, 7, 0, 8, 0, 9, },
  /* 447 */ { 7, 2, 4, 0, 5, 0, 6, 0, 7, 0, 8, 0, 9, 0, 10, },
  /* 448 */ { 1, -2, 4, },
  /* 449 */ { 1, -2, 4, },
  /* 450 */ { 2, -2, 4, 3, 10, },
  /* 451 */ { 2, -2, 4, -3, 10, },
  /* 452 */ { 1, -2, 4, },
  /* 453 */ { 1, -2, 4, },
  /* 454 */ { 2, -2, 4, 4, 10, },
  /* 455 */ { 2, -2, 4, -4, 10, },
  /* 456 */ { 2, -2, 4, 1, 8, },
  /* 457 */ { 2, -2, 4, 1, 8, },
  /* 458 */ { 3, -2, 4, 1, 8, 0, 9, },
  /* 459 */ { 4, -2, 4, 1, 8, 0, 9, 0, 10, },
  /* 460 */ { 2, -2, 4, -1, 8, },
  /* 461 */ { 2, -2, 4, -1, 8, },
  /* 462 */ { 3, -2, 4, -1, 8, 0, 9, },
  /* 463 */ { 4, -2, 4, -1, 8, 0, 9, 0, 10, },
  /* 464 */ { 1, -2, 4, },
  /* 465 */ { 1, -2, 4, },
  /* 466 */ { 2, -2, 4, 5, 10, },
  /* 467 */ { 2, -2, 4, -5, 10, },
  /* 468 */ { 1, -2, 4, },
  /* 469 */ { 1, -2, 4, },
  /* 470 */ { 2, -2, 4, 6, 10, },
  /* 471 */ { 2, -2, 4, -6, 10, },
  /* 472 */ { 2, -2, 4, 2, 8, },
  /* 473 */ { 2, -2, 4, 2, 8, },
  /* 474 */ { 3, -2, 4, 2, 8, 0, 9, },
  /* 475 */ { 4, -2, 4, 2, 8, 0, 9, 0, 10, },
  /* 476 */ { 2, -2, 4, -2, 8, },
  /* 477 */ { 2, -2, 4, -2, 8, },
  /* 478 */ { 3, -2, 4, -2, 8, 0, 9, },
  /* 479 */ { 4, -2, 4, -2, 8, 0, 9, 0, 10, },
  /* 480 */ { 2, -2, 4, 0, 5, },
  /* 481 */ { 2, -2, 4, 0, 5, },
  /* 482 */ { 2, -2, 4, 0, 5, },
  /* 483 */ { 2, -2, 4, 0, 5, },
  /* 484 */ { 3, -2, 4, 0, 5, 1, 9, },
  /* 485 */ { 4, -2, 4, 0, 5, 1, 9, 0, 10, },
  /* 486 */ { 3, -2, 4, 0, 5, -1, 9, },
  /* 487 */ { 4, -2, 4, 0, 5, -1, 9, 0, 10, },
  /* 488 */ { 2, -2, 4, 0, 5, },
  /* 489 */ { 2, -2, 4, 0, 5, },
  /* 490 */ { 2, -2, 4, 0, 5, },
  /* 491 */ { 2, -2, 4, 0, 5, },
  /* 492 */ { 3, -2, 4, 0, 5, 2, 9, },
  /* 493 */ { 4, -2, 4, 0, 5, 2, 9, 0, 10, },
  /* 494 */ { 3, -2, 4, 0, 5, -2, 9, },
  /* 495 */ { 4, -2, 4, 0, 5, -2, 9, 0, 10, },
  /* 496 */ { 3, -2, 4, 0, 5, 0, 6, },
  /* 497 */ { 3, -2, 4, 0, 5, 0, 6, },
  /* 498 */ { 4, -2, 4, 0, 5, 0, 6, 1, 10, },
  /* 499 */ { 4, -2, 4, 0, 5, 0, 6, -1, 10, },
  /* 500 */ { 3, -2, 4, 0, 5, 0, 6, },
  /* 501 */ { 3, -2, 4, 0, 5, 0, 6, },
  /* 502 */ { 4, -2, 4, 0, 5, 0, 6, 2, 10, },
  /* 503 */ { 4, -2, 4, 0, 5, 0, 6, -2, 10, },
  /* 504 */ { 4, -2, 4, 0, 5, 0, 6, 0, 7, },
  /* 505 */ { 4, -2, 4, 0, 5, 0, 6, 0, 7, },
  /* 506 */ { 4, -2, 4, 0, 5, 0, 6, 0, 7, },
  /* 507 */ { 4, -2, 4, 0, 5, 0, 6, 0, 7, },
  /* 508 */ { 5, -2, 4, 0, 5, 0, 6, 0, 7, 0, 8, },
  /* 509 */ { 5, -2, 4, 0, 5, 0, 6, 0, 7, 0, 8, },
  /* 510 */ { 6, -2, 4, 0, 5, 0, 6, 0, 7, 0, 8, 0, 9, },
  /* 511 */ { 7, -2, 4, 0, 5, 0, 6, 0, 7, 0, 8, 0, 9, 0, 10, },
  /* 512 */ { 1, 0, 1, },
  /* 513 */ { 1, 0, 1, },
  /* 514 */ { 1, 0, 1, },
  /* 515 */ { 1, 0, 1, },
  /* 516 */ { 2, 0, 1, 7, 9, },
  /* 517 */ { 3, 0, 1, 7, 9, 0, 10, },
  /* 518 */ { 2, 0, 1, -7, 9, },
  /* 519 */ { 3, 0, 1, -7, 9, 0, 10, },
  /* 520 */ { 1, 0, 1, },
  /* 521 */ { 1, 0, 1, },
  /* 522 */ { 1, 0, 1, },
  /* 523 */ { 1, 0, 1, },
  /* 524 */ { 2, 0, 1, 8, 9, },
  /* 525 */ { 3, 0, 1, 8, 9, 0, 10, },
  /* 526 */ { 2, 0, 1, -8, 9, },
  /* 527 */ { 3, 0, 1, -8, 9, 0, 10, },
  /* 528 */ { 2, 0, 1, 3, 7, },
  /* 529 */ { 2, 0, 1, 3, 7, },
  /* 530 */ { 2, 0, 1, 3, 7, },
  /* 531 */ { 2, 0, 1, 3, 7, },
  /* 532 */ { 3, 0, 1, 3, 7, 0, 8, },
  /* 533 */ { 3, 0, 1, 3, 7, 0, 8, },
  /* 534 */ { 4, 0, 1, 3, 7, 0, 8, 0, 9, },
  /* 535 */ { 5, 0, 1, 3, 7, 0, 8, 0, 9, 0, 10, },
  /* 536 */ { 2, 0, 1, -3, 7, },
  /* 537 */ { 2, 0, 1, -3, 7, },
  /* 538 */ { 2, 0, 1, -3, 7, },
  /* 539 */ { 2, 0, 1, -3, 7, },
  /* 540 */ { 3, 0, 1, -3, 7, 0, 8, },
  /* 541 */ { 3, 0, 1, -3, 7, 0, 8, },
  /* 542 */ { 4, 0, 1, -3, 7, 0, 8, 0, 9, },
  /* 543 */ { 5, 0, 1, -3, 7, 0, 8, 0, 9, 0, 10, },
  /* 544 */ { 1, 0, 1, },
  /* 545 */ { 1, 0, 1, },
  /* 546 */ { 1, 0, 1, },
  /* 547 */ { 1, 0, 1, },
  /* 548 */ { 2, 0, 1, 9, 9, },
  /* 549 */ { 3, 0, 1, 9, 9, 0, 10, },
  /* 550 */ { 2, 0, 1, -9, 9, },
  /* 551 */ { 3, 0, 1, -9, 9, 0, 10, },
  /* 552 */ { 1, 0, 1, },
  /* 553 */ { 1, 0, 1, },
  /* 554 */ { 1, 0, 1, },
  /* 555 */ { 1, 0, 1, },
  /* 556 */ { 2, 0, 1, 10, 9, },
  /* 557 */ { 3, 0, 1, 10, 9, 0, 10, },
  /* 558 */ { 2, 0, 1, -10, 9, },
  /* 559 */ { 3, 0, 1, -10, 9, 0, 10, },
  /* 560 */ { 2, 0, 1, 4, 7, },
  /* 561 */ { 2, 0, 1, 4, 7, },
  /* 562 */ { 2, 0, 1, 4, 7, },
  /* 563 */ { 2, 0, 1, 4, 7, },
  /* 564 */ { 3, 0, 1, 4, 7, 0, 8, },
  /* 565 */ { 3, 0, 1, 4, 7, 0, 8, },
  /* 566 */ { 4, 0, 1, 4, 7, 0, 8, 0, 9, },
  /* 567 */ { 5, 0, 1, 4, 7, 0, 8, 0, 9, 0, 10, },
  /* 568 */ { 2, 0, 1, -4, 7, },
  /* 569 */ { 2, 0, 1, -4, 7, },
  /* 570 */ { 2, 0, 1, -4, 7, },
  /* 571 */ { 2, 0, 1, -4, 7, },
  /* 572 */ { 3, 0, 1, -4, 7, 0, 8, },
  /* 573 */ { 3, 0, 1, -4, 7, 0, 8, },
  /* 574 */ { 4, 0, 1, -4, 7, 0, 8, 0, 9, },
  /* 575 */ { 5, 0, 1, -4, 7, 0, 8, 0, 9, 0, 10, },
  /* 576 */ { 2, 0, 1, 1, 5, },
  /* 577 */ { 2, 0, 1, 1, 5, },
  /* 578 */ { 2, 0, 1, 1, 5, },
  /* 579 */ { 2, 0, 1, 1, 5, },
  /* 580 */ { 3, 0, 1, 1, 5, 1, 9, },
  /* 581 */ { 4, 0, 1, 1, 5, 1, 9, 0, 10, },
  /* 582 */ { 3, 0, 1, 1, 5, -1, 9, },
  /* 583 */ { 4, 0, 1, 1, 5, -1, 9, 0, 10, },
  /* 584 */ { 2, 0, 1, 1, 5, },
  /* 585 */ { 2, 0, 1, 1, 5, },
  /* 586 */ { 2, 0, 1, 1, 5, },
  /* 587 */ { 2, 0, 1, 1, 5, },
  /* 588 */ { 3, 0, 1, 1, 5, 2, 9, },
  /* 589 */ { 4, 0, 1, 1, 5, 2, 9, 0, 10, },
  /* 590 */ { 3, 0, 1, 1, 5, -2, 9, },
  /* 591 */ { 4, 0, 1, 1, 5, -2, 9, 0, 10, },
  /* 592 */ { 3, 0, 1, 1, 5, 0, 6, },
  /* 593 */ { 3, 0, 1, 1, 5, 0, 6, },
  /* 594 */ { 4, 0, 1, 1, 5, 0, 6, 1, 10, },
  /* 595 */ { 4, 0, 1, 1, 5, 0, 6, -1, 10, },
  /* 596 */ { 3, 0, 1, 1, 5, 0, 6, },
  /* 597 */ { 3, 0, 1, 1, 5, 0, 6, },
  /* 598 */ { 4, 0, 1, 1, 5, 0, 6, 2, 10, },
  /* 599 */ { 4, 0, 1, 1, 5, 0, 6, -2, 10, },
  /* 600 */ { 4, 0, 1, 1, 5, 0, 6, 0, 7, },
  /* 601 */ { 4, 0, 1, 1, 5, 0, 6, 0, 7, },
  /* 602 */ { 4, 0, 1, 1, 5, 0, 6, 0, 7, },
  /* 603 */ { 4, 0, 1, 1, 5, 0, 6, 0, 7, },
  /* 604 */ { 5, 0, 1, 1, 5, 0, 6, 0, 7, 0, 8, },
  /* 605 */ { 5, 0, 1, 1, 5, 0, 6, 0, 7, 0, 8, },
  /* 606 */ { 6, 0, 1, 1, 5, 0, 6, 0, 7, 0, 8, 0, 9, },
  /* 607 */ { 7, 0, 1, 1, 5, 0, 6, 0, 7, 0, 8, 0, 9, 0, 10, },
  /* 608 */ { 2, 0, 1, -1, 5, },
  /* 609 */ { 2, 0, 1, -1, 5, },
  /* 610 */ { 2, 0, 1, -1, 5, },
  /* 611 */ { 2, 0, 1, -1, 5, },
  /* 612 */ { 3, 0, 1, -1, 5, 1, 9, },
  /* 613 */ { 4, 0, 1, -1, 5, 1, 9, 0, 10, },
  /* 614 */ { 3, 0, 1, -1, 5, -1, 9, },
  /* 615 */ { 4, 0, 1, -1, 5, -1, 9, 0, 10, },
  /* 616 */ { 2, 0, 1, -1, 5, },
  /* 617 */ { 2, 0, 1, -1, 5, },
  /* 618 */ { 2, 0, 1, -1, 5, },
  /* 619 */ { 2, 0, 1, -1, 5, },
  /* 620 */ { 3, 0, 1, -1, 5, 2, 9, },
  /* 621 */ { 4, 0, 1, -1, 5, 2, 9, 0, 10, },
  /* 622 */ { 3, 0, 1, -1, 5, -2, 9, },
  /* 623 */ { 4, 0, 1, -1, 5, -2, 9, 0, 10, },
  /* 624 */ { 3, 0, 1, -1, 5, 0, 6, },
  /* 625 */ { 3, 0, 1, -1, 5, 0, 6, },
  /* 626 */ { 4, 0, 1, -1, 5, 0, 6, 1, 10, },
  /* 627 */ { 4, 0, 1, -1, 5, 0, 6, -1, 10, },
  /* 628 */ { 3, 0, 1, -1, 5, 0, 6, },
  /* 629 */ { 3, 0, 1, -1, 5, 0, 6, },
  /* 630 */ { 4, 0, 1, -1, 5, 0, 6, 2, 10, },
  /* 631 */ { 4, 0, 1, -1, 5, 0, 6, -2, 10, },
  /* 632 */ { 4, 0, 1, -1, 5, 0, 6, 0, 7, },
  /* 633 */ { 4, 0, 1, -1, 5, 0, 6, 0, 7, },
  /* 634 */ { 4, 0, 1, -1, 5, 0, 6, 0, 7, },
  /* 635 */ { 4, 0, 1, -1, 5, 0, 6, 0, 7, },
  /* 636 */ { 5, 0, 1, -1, 5, 0, 6, 0, 7, 0, 8, },
  /* 637 */ { 5, 0, 1, -1, 5, 0, 6, 0, 7, 0, 8, },
  /* 638 */ { 6, 0, 1, -1, 5, 0, 6, 0, 7, 0, 8, 0, 9, },
  /* 639 */ { 7, 0, 1, -1, 5, 0, 6, 0, 7, 0, 8, 0, 9, 0, 10, },
  /* 640 */ { 1, 0, 1, },
  /* 641 */ { 1, 0, 1, },
  /* 642 */ { 1, 0, 1, },
  /* 643 */ { 1, 0, 1, },
  /* 644 */ { 2, 0, 1, 11, 9, },
  /* 645 */ { 3, 0, 1, 11, 9, 0, 10, },
  /* 646 */ { 2, 0, 1, -11, 9, },
  /* 647 */ { 3, 0, 1, -11, 9, 0, 10, },
  /* 648 */ { 1, 0, 1, },
  /* 649 */ { 1, 0, 1, },
  /* 650 */ { 1, 0, 1, },
  /* 651 */ { 1, 0, 1, },
  /* 652 */ { 2, 0, 1, 12, 9, },
  /* 653 */ { 3, 0, 1, 12, 9, 0, 10, },
  /* 654 */ { 2, 0, 1, -12, 9, },
  /* 655 */ { 3, 0, 1, -12, 9, 0, 10, },
  /* 656 */ { 2, 0, 1, 5, 7, },
  /* 657 */ { 2, 0, 1, 5, 7, },
  /* 658 */ { 2, 0, 1, 5, 7, },
  /* 659 */ { 2, 0, 1, 5, 7, },
  /* 660 */ { 3, 0, 1, 5, 7, 0, 8, },
  /* 661 */ { 3, 0, 1, 5, 7, 0, 8, },
  /* 662 */ { 4, 0, 1, 5, 7, 0, 8, 0, 9, },
  /* 663 */ { 5, 0, 1, 5, 7, 0, 8, 0, 9, 0, 10, },
  /* 664 */ { 2, 0, 1, -5, 7, },
  /* 665 */ { 2, 0, 1, -5, 7, },
  /* 666 */ { 2, 0, 1, -5, 7, },
  /* 667 */ { 2, 0, 1, -5, 7, },
  /* 668 */ { 3, 0, 1, -5, 7, 0, 8, },
  /* 669 */ { 3, 0, 1, -5, 7, 0, 8, },
  /* 670 */ { 4, 0, 1, -5, 7, 0, 8, 0, 9, },
  /* 671 */ { 5, 0, 1, -5, 7, 0, 8, 0, 9, 0, 10, },
  /* 672 */ { 1, 0, 1, },
  /* 673 */ { 1, 0, 1, },
  /* 674 */ { 1, 0, 1, },
  /* 675 */ { 1, 0, 1, },
  /* 676 */ { 2, 0, 1, 13, 9, },
  /* 677 */ { 3, 0, 1, 13, 9, 0, 10, },
  /* 678 */ { 2, 0, 1, -13, 9, },
  /* 679 */ { 3, 0, 1, -13, 9, 0, 10, },
  /* 680 */ { 1, 0, 1, },
  /* 681 */ { 1, 0, 1, },
  /* 682 */ { 1, 0, 1, },
  /* 683 */ { 1, 0, 1, },
  /* 684 */ { 2, 0, 1, 14, 9, },
  /* 685 */ { 3, 0, 1, 14, 9, 0, 10, },
  /* 686 */ { 2, 0, 1, -14, 9, },
  /* 687 */ { 3, 0, 1, -14, 9, 0, 10, },
  /* 688 */ { 2, 0, 1, 6, 7, },
  /* 689 */ { 2, 0, 1, 6, 7, },
  /* 690 */ { 2, 0, 1, 6, 7, },
  /* 691 */ { 2, 0, 1, 6, 7, },
  /* 692 */ { 3, 0, 1, 6, 7, 0, 8, },
  /* 693 */ { 3, 0, 1, 6, 7, 0, 8, },
  /* 694 */ { 4, 0, 1, 6, 7, 0, 8, 0, 9, },
  /* 695 */ { 5, 0, 1, 6, 7, 0, 8, 0, 9, 0, 10, },
  /* 696 */ { 2, 0, 1, -6, 7, },
  /* 697 */ { 2, 0, 1, -6, 7, },
  /* 698 */ { 2, 0, 1, -6, 7, },
  /* 699 */ { 2, 0, 1, -6, 7, },
  /* 700 */ { 3, 0, 1, -6, 7, 0, 8, },
  /* 701 */ { 3, 0, 1, -6, 7, 0, 8, },
  /* 702 */ { 4, 0, 1, -6, 7, 0, 8, 0, 9, },
  /* 703 */ { 5, 0, 1, -6, 7, 0, 8, 0, 9, 0, 10, },
  /* 704 */ { 2, 0, 1, 2, 5, },
  /* 705 */ { 2, 0, 1, 2, 5, },
  /* 706 */ { 2, 0, 1, 2, 5, },
  /* 707 */ { 2, 0, 1, 2, 5, },
  /* 708 */ { 3, 0, 1, 2, 5, 1, 9, },
  /* 709 */ { 4, 0, 1, 2, 5, 1, 9, 0, 10, },
  /* 710 */ { 3, 0, 1, 2, 5, -1, 9, },
  /* 711 */ { 4, 0, 1, 2, 5, -1, 9, 0, 10, },
  /* 712 */ { 2, 0, 1, 2, 5, },
  /* 713 */ { 2, 0, 1, 2, 5, },
  /* 714 */ { 2, 0, 1, 2, 5, },
  /* 715 */ { 2, 0, 1, 2, 5, },
  /* 716 */ { 3, 0, 1, 2, 5, 2, 9, },
  /* 717 */ { 4, 0, 1, 2, 5, 2, 9, 0, 10, },
  /* 718 */ { 3, 0, 1, 2, 5, -2, 9, },
  /* 719 */ { 4, 0, 1, 2, 5, -2, 9, 0, 10, },
  /* 720 */ { 3, 0, 1, 2, 5, 0, 6, },
  /* 721 */ { 3, 0, 1, 2, 5, 0, 6, },
  /* 722 */ { 4, 0, 1, 2, 5, 0, 6, 1, 10, },
  /* 723 */ { 4, 0, 1, 2, 5, 0, 6, -1, 10, },
  /* 724 */ { 3, 0, 1, 2, 5, 0, 6, },
  /* 725 */ { 3, 0, 1, 2, 5, 0, 6, },
  /* 726 */ { 4, 0, 1, 2, 5, 0, 6, 2, 10, },
  /* 727 */ { 4, 0, 1, 2, 5, 0, 6, -2, 10, },
  /* 728 */ { 4, 0, 1, 2, 5, 0, 6, 0, 7, },
  /* 729 */ { 4, 0, 1, 2, 5, 0, 6, 0, 7, },
  /* 730 */ { 4, 0, 1, 2, 5, 0, 6, 0, 7, },
  /* 731 */ { 4, 0, 1, 2, 5, 0, 6, 0, 7, },
  /* 732 */ { 5, 0, 1, 2, 5, 0, 6, 0, 7, 0, 8, },
  /* 733 */ { 5, 0, 1, 2, 5, 0, 6, 0, 7, 0, 8, },
  /* 734 */ { 6, 0, 1, 2, 5, 0, 6, 0, 7, 0, 8, 0, 9, },
  /* 735 */ { 7, 0, 1, 2, 5, 0, 6, 0, 7, 0, 8, 0, 9, 0, 10, },
  /* 736 */ { 2, 0, 1, -2, 5, },
  /* 737 */ { 2, 0, 1, -2, 5, },
  /* 738 */ { 2, 0, 1, -2, 5, },
  /* 739 */ { 2, 0, 1, -2, 5, },
  /* 740 */ { 3, 0, 1, -2, 5, 1, 9, },
  /* 741 */ { 4, 0, 1, -2, 5, 1, 9, 0, 10, },
  /* 742 */ { 3, 0, 1, -2, 5, -1, 9, },
  /* 743 */ { 4, 0, 1, -2, 5, -1, 9, 0, 10, },
  /* 744 */ { 2, 0, 1, -2, 5, },
  /* 745 */ { 2, 0, 1, -2, 5, },
  /* 746 */ { 2, 0, 1, -2, 5, },
  /* 747 */ { 2, 0, 1, -2, 5, },
  /* 748 */ { 3, 0, 1, -2, 5, 2, 9, },
  /* 749 */ { 4, 0, 1, -2, 5, 2, 9, 0, 10, },
  /* 750 */ { 3, 0, 1, -2, 5, -2, 9, },
  /* 751 */ { 4, 0, 1, -2, 5, -2, 9, 0, 10, },
  /* 752 */ { 3, 0, 1, -2, 5, 0, 6, },
  /* 753 */ { 3, 0, 1, -2, 5, 0, 6, },
  /* 754 */ { 4, 0, 1, -2, 5, 0, 6, 1, 10, },
  /* 755 */ { 4, 0, 1, -2, 5, 0, 6, -1, 10, },
  /* 756 */ { 3, 0, 1, -2, 5, 0, 6, },
  /* 757 */ { 3, 0, 1, -2, 5, 0, 6, },
  /* 758 */ { 4, 0, 1, -2, 5, 0, 6, 2, 10, },
  /* 759 */ { 4, 0, 1, -2, 5, 0, 6, -2, 10, },
  /* 760 */ { 4, 0, 1, -2, 5, 0, 6, 0, 7, },
  /* 761 */ { 4, 0, 1, -2, 5, 0, 6, 0, 7, },
  /* 762 */ { 4, 0, 1, -2, 5, 0, 6, 0, 7, },
  /* 763 */ { 4, 0, 1, -2, 5, 0, 6, 0, 7, },
  /* 764 */ { 5, 0, 1, -2, 5, 0, 6, 0, 7, 0, 8, },
  /* 765 */ { 5, 0, 1, -2, 5, 0, 6, 0, 7, 0, 8, },
  /* 766 */ { 6, 0, 1, -2, 5, 0, 6, 0, 7, 0, 8, 0, 9, },
  /* 767 */ { 7, 0, 1, -2, 5, 0, 6, 0, 7, 0, 8, 0, 9, 0, 10, },
  /* 768 */ { 2, 0, 1, 0, 2, },
  /* 769 */ { 2, 0, 1, 0, 2, },
  /* 770 */ { 3, 0, 1, 0, 2, 7, 10, },
  /* 771 */ { 3, 0, 1, 0, 2, -7, 10, },
  /* 772 */ { 2, 0, 1, 0, 2, },
  /* 773 */ { 2, 0, 1, 0, 2, },
  /* 774 */ { 3, 0, 1, 0, 2, 8, 10, },
  /* 775 */ { 3, 0, 1, 0, 2, -8, 10, },
  /* 776 */ { 3, 0, 1, 0, 2, 3, 8, },
  /* 777 */ { 3, 0, 1, 0, 2, 3, 8, },
  /* 778 */ { 4, 0, 1, 0, 2, 3, 8, 0, 9, },
  /* 779 */ { 5, 0, 1, 0, 2, 3, 8, 0, 9, 0, 10, },
  /* 780 */ { 3, 0, 1, 0, 2, -3, 8, },
  /* 781 */ { 3, 0, 1, 0, 2, -3, 8, },
  /* 782 */ { 4, 0, 1, 0, 2, -3, 8, 0, 9, },
  /* 783 */ { 5, 0, 1, 0, 2, -3, 8, 0, 9, 0, 10, },
  /* 784 */ { 2, 0, 1, 0, 2, },
  /* 785 */ { 2, 0, 1, 0, 2, },
  /* 786 */ { 3, 0, 1, 0, 2, 9, 10, },
  /* 787 */ { 3, 0, 1, 0, 2, -9, 10, },
  /* 788 */ { 2, 0, 1, 0, 2, },
  /* 789 */ { 2, 0, 1, 0, 2, },
  /* 790 */ { 3, 0, 1, 0, 2, 10, 10, },
  /* 791 */ { 3, 0, 1, 0, 2, -10, 10, },
  /* 792 */ { 3, 0, 1, 0, 2, 4, 8, },
  /* 793 */ { 3, 0, 1, 0, 2, 4, 8, },
  /* 794 */ { 4, 0, 1, 0, 2, 4, 8, 0, 9, },
  /* 795 */ { 5, 0, 1, 0, 2, 4, 8, 0, 9, 0, 10, },
  /* 796 */ { 3, 0, 1, 0, 2, -4, 8, },
  /* 797 */ { 3, 0, 1, 0, 2, -4, 8, },
  /* 798 */ { 4, 0, 1, 0, 2, -4, 8, 0, 9, },
  /* 799 */ { 5, 0, 1, 0, 2, -4, 8, 0, 9, 0, 10, },
  /* 800 */ { 3, 0, 1, 0, 2, 1, 6, },
  /* 801 */ { 3, 0, 1, 0, 2, 1, 6, },
  /* 802 */ { 4, 0, 1, 0, 2, 1, 6, 1, 10, },
  /* 803 */ { 4, 0, 1, 0, 2, 1, 6, -1, 10, },
  /* 804 */ { 3, 0, 1, 0, 2, 1, 6, },
  /* 805 */ { 3, 0, 1, 0, 2, 1, 6, },
  /* 806 */ { 4, 0, 1, 0, 2, 1, 6, 2, 10, },
  /* 807 */ { 4, 0, 1, 0, 2, 1, 6, -2, 10, },
  /* 808 */ { 4, 0, 1, 0, 2, 1, 6, 0, 7, },
  /* 809 */ { 4, 0, 1, 0, 2, 1, 6, 0, 7, },
  /* 810 */ { 4, 0, 1, 0, 2, 1, 6, 0, 7, },
  /* 811 */ { 4, 0, 1, 0, 2, 1, 6, 0, 7, },
  /* 812 */ { 5, 0, 1, 0, 2, 1, 6, 0, 7, 0, 8, },
  /* 813 */ { 5, 0, 1, 0, 2, 1, 6, 0, 7, 0, 8, },
  /* 814 */ { 6, 0, 1, 0, 2, 1, 6, 0, 7, 0, 8, 0, 9, },
  /* 815 */ { 7, 0, 1, 0, 2, 1, 6, 0, 7, 0, 8, 0, 9, 0, 10, },
  /* 816 */ { 3, 0, 1, 0, 2, -1, 6, },
  /* 817 */ { 3, 0, 1, 0, 2, -1, 6, },
  /* 818 */ { 4, 0, 1, 0, 2, -1, 6, 1, 10, },
  /* 819 */ { 4, 0, 1, 0, 2, -1, 6, -1, 10, },
  /* 820 */ { 3, 0, 1, 0, 2, -1, 6, },
  /* 821 */ { 3, 0, 1, 0, 2, -1, 6, },
  /* 822 */ { 4, 0, 1, 0, 2, -1, 6, 2, 10, },
  /* 823 */ { 4, 0, 1, 0, 2, -1, 6, -2, 10, },
  /* 824 */ { 4, 0, 1, 0, 2, -1, 6, 0, 7, },
  /* 825 */ { 4, 0, 1, 0, 2, -1, 6, 0, 7, },
  /* 826 */ { 4, 0, 1, 0, 2, -1, 6, 0, 7, },
  /* 827 */ { 4, 0, 1, 0, 2, -1, 6, 0, 7, },
  /* 828 */ { 5, 0, 1, 0, 2, -1, 6, 0, 7, 0, 8, },
  /* 829 */ { 5, 0, 1, 0, 2, -1, 6, 0, 7, 0, 8, },
  /* 830 */ { 6, 0, 1, 0, 2, -1, 6, 0, 7, 0, 8, 0, 9, },
  /* 831 */ { 7, 0, 1, 0, 2, -1, 6, 0, 7, 0, 8, 0, 9, 0, 10, },
  /* 832 */ { 2, 0, 1, 0, 2, },
  /* 833 */ { 2, 0, 1, 0, 2, },
  /* 834 */ { 3, 0, 1, 0, 2, 11, 10, },
  /* 835 */ { 3, 0, 1, 0, 2, -11, 10, },
  /* 836 */ { 2, 0, 1, 0, 2, },
  /* 837 */ { 2, 0, 1, 0, 2, },
  /* 838 */ { 3, 0, 1, 0, 2, 12, 10, },
  /* 839 */ { 3, 0, 1, 0, 2, -12, 10, },
  /* 840 */ { 3, 0, 1, 0, 2, 5, 8, },
  /* 841 */ { 3, 0, 1, 0, 2, 5, 8, },
  /* 842 */ { 4, 0, 1, 0, 2, 5, 8, 0, 9, },
  /* 843 */ { 5, 0, 1, 0, 2, 5, 8, 0, 9, 0, 10, },
  /* 844 */ { 3, 0, 1, 0, 2, -5, 8, },
  /* 845 */ { 3, 0, 1, 0, 2, -5, 8, },
  /* 846 */ { 4, 0, 1, 0, 2, -5, 8, 0, 9, },
  /* 847 */ { 5, 0, 1, 0, 2, -5, 8, 0, 9, 0, 10, },
  /* 848 */ { 2, 0, 1, 0, 2, },
  /* 849 */ { 2, 0, 1, 0, 2, },
  /* 850 */ { 3, 0, 1, 0, 2, 13, 10, },
  /* 851 */ { 3, 0, 1, 0, 2, -13, 10, },
  /* 852 */ { 2, 0, 1, 0, 2, },
  /* 853 */ { 2, 0, 1, 0, 2, },
  /* 854 */ { 3, 0, 1, 0, 2, 14, 10, },
  /* 855 */ { 3, 0, 1, 0, 2, -14, 10, },
  /* 856 */ { 3, 0, 1, 0, 2, 6, 8, },
  /* 857 */ { 3, 0, 1, 0, 2, 6, 8, },
  /* 858 */ { 4, 0, 1, 0, 2, 6, 8, 0, 9, },
  /* 859 */ { 5, 0, 1, 0, 2, 6, 8, 0, 9, 0, 10, },
  /* 860 */ { 3, 0, 1, 0, 2, -6, 8, },
  /* 861 */ { 3, 0, 1, 0, 2, -6, 8, },
  /* 862 */ { 4, 0, 1, 0, 2, -6, 8, 0, 9, },
  /* 863 */ { 5, 0, 1, 0, 2, -6, 8, 0, 9, 0, 10, },
  /* 864 */ { 3, 0, 1, 0, 2, 2, 6, },
  /* 865 */ { 3, 0, 1, 0, 2, 2, 6, },
  /* 866 */ { 4, 0, 1, 0, 2, 2, 6, 1, 10, },
  /* 867 */ { 4, 0, 1, 0, 2, 2, 6, -1, 10, },
  /* 868 */ { 3, 0, 1, 0, 2, 2, 6, },
  /* 869 */ { 3, 0, 1, 0, 2, 2, 6, },
  /* 870 */ { 4, 0, 1, 0, 2, 2, 6, 2, 10, },
  /* 871 */ { 4, 0, 1, 0, 2, 2, 6, -2, 10, },
  /* 872 */ { 4, 0, 1, 0, 2, 2, 6, 0, 7, },
  /* 873 */ { 4, 0, 1, 0, 2, 2, 6, 0, 7, },
  /* 874 */ { 4, 0, 1, 0, 2, 2, 6, 0, 7, },
  /* 875 */ { 4, 0, 1, 0, 2, 2, 6, 0, 7, },
  /* 876 */ { 5, 0, 1, 0, 2, 2, 6, 0, 7, 0, 8, },
  /* 877 */ { 5, 0, 1, 0, 2, 2, 6, 0, 7, 0, 8, },
  /* 878 */ { 6, 0, 1, 0, 2, 2, 6, 0, 7, 0, 8, 0, 9, },
  /* 879 */ { 7, 0, 1, 0, 2, 2, 6, 0, 7, 0, 8, 0, 9, 0, 10, },
  /* 880 */ { 3, 0, 1, 0, 2, -2, 6, },
  /* 881 */ { 3, 0, 1, 0, 2, -2, 6, },
  /* 882 */ { 4, 0, 1, 0, 2, -2, 6, 1, 10, },
  /* 883 */ { 4, 0, 1, 0, 2, -2, 6, -1, 10, },
  /* 884 */ { 3, 0, 1, 0, 2, -2, 6, },
  /* 885 */ { 3, 0, 1, 0, 2, -2, 6, },
  /* 886 */ { 4, 0, 1, 0, 2, -2, 6, 2, 10, },
  /* 887 */ { 4, 0, 1, 0, 2, -2, 6, -2, 10, },
  /* 888 */ { 4, 0, 1, 0, 2, -2, 6, 0, 7, },
  /* 889 */ { 4, 0, 1, 0, 2, -2, 6, 0, 7, },
  /* 890 */ { 4, 0, 1, 0, 2, -2, 6, 0, 7, },
  /* 891 */ { 4, 0, 1, 0, 2, -2, 6, 0, 7, },
  /* 892 */ { 5, 0, 1, 0, 2, -2, 6, 0, 7, 0, 8, },
  /* 893 */ { 5, 0, 1, 0, 2, -2, 6, 0, 7, 0, 8, },
  /* 894 */ { 6, 0, 1, 0, 2, -2, 6, 0, 7, 0, 8, 0, 9, },
  /* 895 */ { 7, 0, 1, 0, 2, -2, 6, 0, 7, 0, 8, 0, 9, 0, 10, },
  /* 896 */ { 3, 0, 1, 0, 2, 0, 3, },
  /* 897 */ { 3, 0, 1, 0, 2, 0, 3, },
  /* 898 */ { 3, 0, 1, 0, 2, 0, 3, },
  /* 899 */ { 3, 0, 1, 0, 2, 0, 3, },
  /* 900 */ { 4, 0, 1, 0, 2, 0, 3, 3, 9, },
  /* 901 */ { 5, 0, 1, 0, 2, 0, 3, 3, 9, 0, 10, },
  /* 902 */ { 4, 0, 1, 0, 2, 0, 3, -3, 9, },
  /* 903 */ { 5, 0, 1, 0, 2, 0, 3, -3, 9, 0, 10, },
  /* 904 */ { 3, 0, 1, 0, 2, 0, 3, },
  /* 905 */ { 3, 0, 1, 0, 2, 0, 3, },
  /* 906 */ { 3, 0, 1, 0, 2, 0, 3, },
  /* 907 */ { 3, 0, 1, 0, 2, 0, 3, },
  /* 908 */ { 4, 0, 1, 0, 2, 0, 3, 4, 9, },
  /* 909 */ { 5, 0, 1, 0, 2, 0, 3, 4, 9, 0, 10, },
  /* 910 */ { 4, 0, 1, 0, 2, 0, 3, -4, 9, },
  /* 911 */ { 5, 0, 1, 0, 2, 0, 3, -4, 9, 0, 10, },
  /* 912 */ { 4, 0, 1, 0, 2, 0, 3, 1, 7, },
  /* 913 */ { 4, 0, 1, 0, 2, 0, 3, 1, 7, },
  /* 914 */ { 4, 0, 1, 0, 2, 0, 3, 1, 7, },
  /* 915 */ { 4, 0, 1, 0, 2, 0, 3, 1, 7, },
  /* 916 */ { 5, 0, 1, 0, 2, 0, 3, 1, 7, 0, 8, },
  /* 917 */ { 5, 0, 1, 0, 2, 0, 3, 1, 7, 0, 8, },
  /* 918 */ { 6, 0, 1, 0, 2, 0, 3, 1, 7, 0, 8, 0, 9, },
  /* 919 */ { 7, 0, 1, 0, 2, 0, 3, 1, 7, 0, 8, 0, 9, 0, 10, },
  /* 920 */ { 4, 0, 1, 0, 2, 0, 3, -1, 7, },
  /* 921 */ { 4, 0, 1, 0, 2, 0, 3, -1, 7, },
  /* 922 */ { 4, 0, 1, 0, 2, 0, 3, -1, 7, },
  /* 923 */ { 4, 0, 1, 0, 2, 0, 3, -1, 7, },
  /* 924 */ { 5, 0, 1, 0, 2, 0, 3, -1, 7, 0, 8, },
  /* 925 */ { 5, 0, 1, 0, 2, 0, 3, -1, 7, 0, 8, },
  /* 926 */ { 6, 0, 1, 0, 2, 0, 3, -1, 7, 0, 8, 0, 9, },
  /* 927 */ { 7, 0, 1, 0, 2, 0, 3, -1, 7, 0, 8, 0, 9, 0, 10, },
  /* 928 */ { 3, 0, 1, 0, 2, 0, 3, },
  /* 929 */ { 3, 0, 1, 0, 2, 0, 3, },
  /* 930 */ { 3, 0, 1, 0, 2, 0, 3, },
  /* 931 */ { 3, 0, 1, 0, 2, 0, 3, },
  /* 932 */ { 4, 0, 1, 0, 2, 0, 3, 5, 9, },
  /* 933 */ { 5, 0, 1, 0, 2, 0, 3, 5, 9, 0, 10, },
  /* 934 */ { 4, 0, 1, 0, 2, 0, 3, -5, 9, },
  /* 935 */ { 5, 0, 1, 0, 2, 0, 3, -5, 9, 0, 10, },
  /* 936 */ { 3, 0, 1, 0, 2, 0, 3, },
  /* 937 */ { 3, 0, 1, 0, 2, 0, 3, },
  /* 938 */ { 3, 0, 1, 0, 2, 0, 3, },
  /* 939 */ { 3, 0, 1, 0, 2, 0, 3, },
  /* 940 */ { 4, 0, 1, 0, 2, 0, 3, 6, 9, },
  /* 941 */ { 5, 0, 1, 0, 2, 0, 3, 6, 9, 0, 10, },
  /* 942 */ { 4, 0, 1, 0, 2, 0, 3, -6, 9, },
  /* 943 */ { 5, 0, 1, 0, 2, 0, 3, -6, 9, 0, 10, },
  /* 944 */ { 4, 0, 1, 0, 2, 0, 3, 2, 7, },
  /* 945 */ { 4, 0, 1, 0, 2, 0, 3, 2, 7, },
  /* 946 */ { 4, 0, 1, 0, 2, 0, 3, 2, 7, },
  /* 947 */ { 4, 0, 1, 0, 2, 0, 3, 2, 7, },
  /* 948 */ { 5, 0, 1, 0, 2, 0, 3, 2, 7, 0, 8, },
  /* 949 */ { 5, 0, 1, 0, 2, 0, 3, 2, 7, 0, 8, },
  /* 950 */ { 6, 0, 1, 0, 2, 0, 3, 2, 7, 0, 8, 0, 9, },
  /* 951 */ { 7, 0, 1, 0, 2, 0, 3, 2, 7, 0, 8, 0, 9, 0, 10, },
  /* 952 */ { 4, 0, 1, 0, 2, 0, 3, -2, 7, },
  /* 953 */ { 4, 0, 1, 0, 2, 0, 3, -2, 7, },
  /* 954 */ { 4, 0, 1, 0, 2, 0, 3, -2, 7, },
  /* 955 */ { 4, 0, 1, 0, 2, 0, 3, -2, 7, },
  /* 956 */ { 5, 0, 1, 0, 2, 0, 3, -2, 7, 0, 8, },
  /* 957 */ { 5, 0, 1, 0, 2, 0, 3, -2, 7, 0, 8, },
  /* 958 */ { 6, 0, 1, 0, 2, 0, 3, -2, 7, 0, 8, 0, 9, },
  /* 959 */ { 7, 0, 1, 0, 2, 0, 3, -2, 7, 0, 8, 0, 9, 0, 10, },
  /* 960 */ { 4, 0, 1, 0, 2, 0, 3, 0, 4, },
  /* 961 */ { 4, 0, 1, 0, 2, 0, 3, 0, 4, },
  /* 962 */ { 5, 0, 1, 0, 2, 0, 3, 0, 4, 3, 10, },
  /* 963 */ { 5, 0, 1, 0, 2, 0, 3, 0, 4, -3, 10, },
  /* 964 */ { 4, 0, 1, 0, 2, 0, 3, 0, 4, },
  /* 965 */ { 4, 0, 1, 0, 2, 0, 3, 0, 4, },
  /* 966 */ { 5, 0, 1, 0, 2, 0, 3, 0, 4, 4, 10, },
  /* 967 */ { 5, 0, 1, 0, 2, 0, 3, 0, 4, -4, 10, },
  /* 968 */ { 5, 0, 1, 0, 2, 0, 3, 0, 4, 1, 8, },
  /* 969 */ { 5, 0, 1, 0, 2, 0, 3, 0, 4, 1, 8, },
  /* 970 */ { 6, 0, 1, 0, 2, 0, 3, 0, 4, 1, 8, 0, 9, },
  /* 971 */ { 7, 0, 1, 0, 2, 0, 3, 0, 4, 1, 8, 0, 9, 0, 10, },
  /* 972 */ { 5, 0, 1, 0, 2, 0, 3, 0, 4, -1, 8, },
  /* 973 */ { 5, 0, 1, 0, 2, 0, 3, 0, 4, -1, 8, },
  /* 974 */ { 6, 0, 1, 0, 2, 0, 3, 0, 4, -1, 8, 0, 9, },
  /* 975 */ { 7, 0, 1, 0, 2, 0, 3, 0, 4, -1, 8, 0, 9, 0, 10, },
  /* 976 */ { 4, 0, 1, 0, 2, 0, 3, 0, 4, },
  /* 977 */ { 4, 0, 1, 0, 2, 0, 3, 0, 4, },
  /* 978 */ { 5, 0, 1, 0, 2, 0, 3, 0, 4, 5, 10, },
  /* 979 */ { 5, 0, 1, 0, 2, 0, 3, 0, 4, -5, 10, },
  /* 980 */ { 4, 0, 1, 0, 2, 0, 3, 0, 4, },
  /* 981 */ { 4, 0, 1, 0, 2, 0, 3, 0, 4, },
  /* 982 */ { 5, 0, 1, 0, 2, 0, 3, 0, 4, 6, 10, },
  /* 983 */ { 5, 0, 1, 0, 2, 0, 3, 0, 4, -6, 10, },
  /* 984 */ { 5, 0, 1, 0, 2, 0, 3, 0, 4, 2, 8, },
  /* 985 */ { 5, 0, 1, 0, 2, 0, 3, 0, 4, 2, 8, },
  /* 986 */ { 6, 0, 1, 0, 2, 0, 3, 0, 4, 2, 8, 0, 9, },
  /* 987 */ { 7, 0, 1, 0, 2, 0, 3, 0, 4, 2, 8, 0, 9, 0, 10, },
  /* 988 */ { 5, 0, 1, 0, 2, 0, 3, 0, 4, -2, 8, },
  /* 989 */ { 5, 0, 1, 0, 2, 0, 3, 0, 4, -2, 8, },
  /* 990 */ { 6, 0, 1, 0, 2, 0, 3, 0, 4, -2, 8, 0, 9, },
  /* 991 */ { 7, 0, 1, 0, 2, 0, 3, 0, 4, -2, 8, 0, 9, 0, 10, },
  /* 992 */ { 5, 0, 1, 0, 2, 0, 3, 0, 4, 0, 5, },
  /* 993 */ { 5, 0, 1, 0, 2, 0, 3, 0, 4, 0, 5, },
  /* 994 */ { 5, 0, 1, 0, 2, 0, 3, 0, 4, 0, 5, },
  /* 995 */ { 5, 0, 1, 0, 2, 0, 3, 0, 4, 0, 5, },
  /* 996 */ { 6, 0, 1, 0, 2, 0, 3, 0, 4, 0, 5, 1, 9, },
  /* 997 */ { 7, 0, 1, 0, 2, 0, 3, 0, 4, 0, 5, 1, 9, 0, 10, },
  /* 998 */ { 6, 0, 1, 0, 2, 0, 3, 0, 4, 0, 5, -1, 9, },
  /* 999 */ { 7, 0, 1, 0, 2, 0, 3, 0, 4, 0, 5, -1, 9, 0, 10, },
  /* 1000 */ { 5, 0, 1, 0, 2, 0, 3, 0, 4, 0, 5, },
  /* 1001 */ { 5, 0, 1, 0, 2, 0, 3, 0, 4, 0, 5, },
  /* 1002 */ { 5, 0, 1, 0, 2, 0, 3, 0, 4, 0, 5, },
  /* 1003 */ { 5, 0, 1, 0, 2, 0, 3, 0, 4, 0, 5, },
  /* 1004 */ { 6, 0, 1, 0, 2, 0, 3, 0, 4, 0, 5, 2, 9, },
  /* 1005 */ { 7, 0, 1, 0, 2, 0, 3, 0, 4, 0, 5, 2, 9, 0, 10, },
  /* 1006 */ { 6, 0, 1, 0, 2, 0, 3, 0, 4, 0, 5, -2, 9, },
  /* 1007 */ { 7, 0, 1, 0, 2, 0, 3, 0, 4, 0, 5, -2, 9, 0, 10, },
  /* 1008 */ { 6, 0, 1, 0, 2, 0, 3, 0, 4, 0, 5, 0, 6, },
  /* 1009 */ { 6, 0, 1, 0, 2, 0, 3, 0, 4, 0, 5, 0, 6, },
  /* 1010 */ { 7, 0, 1, 0, 2, 0, 3, 0, 4, 0, 5, 0, 6, 1, 10, },
  /* 1011 */ { 7, 0, 1, 0, 2, 0, 3, 0, 4, 0, 5, 0, 6, -1, 10, },
  /* 1012 */ { 6, 0, 1, 0, 2, 0, 3, 0, 4, 0, 5, 0, 6, },
  /* 1013 */ { 6, 0, 1, 0, 2, 0, 3, 0, 4, 0, 5, 0, 6, },
  /* 1014 */ { 7, 0, 1, 0, 2, 0, 3, 0, 4, 0, 5, 0, 6, 2, 10, },
  /* 1015 */ { 7, 0, 1, 0, 2, 0, 3, 0, 4, 0, 5, 0, 6, -2, 10, },
  /* 1016 */ { 7, 0, 1, 0, 2, 0, 3, 0, 4, 0, 5, 0, 6, 0, 7, },
  /* 1017 */ { 7, 0, 1, 0, 2, 0, 3, 0, 4, 0, 5, 0, 6, 0, 7, },
  /* 1018 */ { 7, 0, 1, 0, 2, 0, 3, 0, 4, 0, 5, 0, 6, 0, 7, },
  /* 1019 */ { 7, 0, 1, 0, 2, 0, 3, 0, 4, 0, 5, 0, 6, 0, 7, },
  /* 1020 */ { 8, 0, 1, 0, 2, 0, 3, 0, 4, 0, 5, 0, 6, 0, 7, 0, 8, },
  /* 1021 */ { 8, 0, 1, 0, 2, 0, 3, 0, 4, 0, 5, 0, 6, 0, 7, 0, 8, },
  /* 1022 */ { 9, 0, 1, 0, 2, 0, 3, 0, 4, 0, 5, 0, 6, 0, 7, 0, 8, 0, 9, },
  /* 1023 */ { 10, 0, 1, 0, 2, 0, 3, 0, 4, 0, 5, 0, 6, 0, 7, 0, 8, 0, 9, 0, 10, },
};
